/* valamemberaccess.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valamemberaccess.h>
#include <gee/arraylist.h>
#include <gee/readonlylist.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valasymbol.h>
#include <vala/valaformalparameter.h>
#include <vala/valascope.h>
#include <vala/valareport.h>
#include <vala/valacreationmethod.h>
#include <vala/valamethod.h>
#include <vala/valaproperty.h>
#include <vala/valaconstructor.h>
#include <vala/valadestructor.h>
#include <vala/valamember.h>
#include <vala/valasourcefile.h>
#include <vala/valausingdirective.h>
#include <vala/valapointertype.h>
#include <vala/valavaluetype.h>
#include <vala/valapointerindirection.h>
#include <vala/valabaseaccess.h>
#include <vala/valanamespace.h>
#include <vala/valatypesymbol.h>
#include <vala/valaobjecttype.h>
#include <vala/valamethodcall.h>
#include <vala/valaexpressionstatement.h>
#include <vala/valavoidtype.h>
#include <vala/valadynamicmethod.h>
#include <vala/valaerrortype.h>
#include <vala/valaerrordomain.h>
#include <vala/valaerrorcode.h>
#include <vala/valaassignment.h>
#include <vala/valadynamicsignal.h>
#include <vala/valadynamicproperty.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valablock.h>
#include <vala/valafield.h>
#include <vala/valasignal.h>
#include <vala/valafieldprototype.h>
#include <vala/valainvalidtype.h>
#include <vala/valaconstant.h>
#include <vala/valalocalvariable.h>




struct _ValaMemberAccessPrivate {
	char* _member_name;
	gboolean _pointer_member_access;
	gboolean _prototype_access;
	gboolean _creation_member;
	gboolean _qualified;
	ValaExpression* _inner;
	GeeList* type_argument_list;
};

#define VALA_MEMBER_ACCESS_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_MEMBER_ACCESS, ValaMemberAccessPrivate))
enum  {
	VALA_MEMBER_ACCESS_DUMMY_PROPERTY
};
static void vala_member_access_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_member_access_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_member_access_real_to_string (ValaCodeNode* base);
static void vala_member_access_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gboolean vala_member_access_real_is_pure (ValaExpression* base);
static void vala_member_access_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gboolean vala_member_access_real_is_constant (ValaExpression* base);
static gboolean vala_member_access_real_is_non_null (ValaExpression* base);
static gboolean vala_member_access_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static void vala_member_access_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection);
static void vala_member_access_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection);
static gpointer vala_member_access_parent_class = NULL;
static void vala_member_access_finalize (ValaCodeNode* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new member access expression.
 *
 * @param inner            parent of the member
 * @param member_name      member name
 * @param source_reference reference to source code
 * @return                 newly created member access expression
 */
ValaMemberAccess* vala_member_access_construct (GType object_type, ValaExpression* inner, const char* member_name, ValaSourceReference* source_reference) {
	ValaMemberAccess* self;
	g_return_val_if_fail (member_name != NULL, NULL);
	self = (ValaMemberAccess*) g_type_create_instance (object_type);
	vala_member_access_set_inner (self, inner);
	vala_member_access_set_member_name (self, member_name);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	return self;
}


ValaMemberAccess* vala_member_access_new (ValaExpression* inner, const char* member_name, ValaSourceReference* source_reference) {
	return vala_member_access_construct (VALA_TYPE_MEMBER_ACCESS, inner, member_name, source_reference);
}


ValaMemberAccess* vala_member_access_construct_simple (GType object_type, const char* member_name, ValaSourceReference* source_reference) {
	ValaMemberAccess* self;
	g_return_val_if_fail (member_name != NULL, NULL);
	self = (ValaMemberAccess*) g_type_create_instance (object_type);
	vala_member_access_set_member_name (self, member_name);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	return self;
}


ValaMemberAccess* vala_member_access_new_simple (const char* member_name, ValaSourceReference* source_reference) {
	return vala_member_access_construct_simple (VALA_TYPE_MEMBER_ACCESS, member_name, source_reference);
}


ValaMemberAccess* vala_member_access_construct_pointer (GType object_type, ValaExpression* inner, const char* member_name, ValaSourceReference* source_reference) {
	ValaMemberAccess* self;
	g_return_val_if_fail (inner != NULL, NULL);
	g_return_val_if_fail (member_name != NULL, NULL);
	self = (ValaMemberAccess*) g_type_create_instance (object_type);
	vala_member_access_set_inner (self, inner);
	vala_member_access_set_member_name (self, member_name);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	vala_member_access_set_pointer_member_access (self, TRUE);
	return self;
}


ValaMemberAccess* vala_member_access_new_pointer (ValaExpression* inner, const char* member_name, ValaSourceReference* source_reference) {
	return vala_member_access_construct_pointer (VALA_TYPE_MEMBER_ACCESS, inner, member_name, source_reference);
}


/**
 * Appends the specified type as generic type argument.
 *
 * @param arg a type reference
 */
void vala_member_access_add_type_argument (ValaMemberAccess* self, ValaDataType* arg) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (arg != NULL);
	gee_collection_add ((GeeCollection*) self->priv->type_argument_list, arg);
	vala_code_node_set_parent_node ((ValaCodeNode*) arg, (ValaCodeNode*) self);
}


/**
 * Returns a copy of the list of generic type arguments.
 *
 * @return type argument list
 */
GeeList* vala_member_access_get_type_arguments (ValaMemberAccess* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_DATA_TYPE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->type_argument_list);
}


static void vala_member_access_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaMemberAccess * self;
	self = (ValaMemberAccess*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_member_access (visitor, self);
	vala_code_visitor_visit_expression (visitor, (ValaExpression*) self);
}


static void vala_member_access_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaMemberAccess * self;
	self = (ValaMemberAccess*) base;
	g_return_if_fail (visitor != NULL);
	if (vala_member_access_get_inner (self) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_member_access_get_inner (self), visitor);
	}
	{
		GeeIterator* _type_arg_it;
		_type_arg_it = gee_iterable_iterator ((GeeIterable*) self->priv->type_argument_list);
		while (gee_iterator_next (_type_arg_it)) {
			ValaDataType* type_arg;
			type_arg = (ValaDataType*) gee_iterator_get (_type_arg_it);
			vala_code_node_accept ((ValaCodeNode*) type_arg, visitor);
			(type_arg == NULL) ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL));
		}
		(_type_arg_it == NULL) ? NULL : (_type_arg_it = (gee_collection_object_unref (_type_arg_it), NULL));
	}
}


static char* vala_member_access_real_to_string (ValaCodeNode* base) {
	ValaMemberAccess * self;
	self = (ValaMemberAccess*) base;
	if (vala_member_access_get_inner (self) == NULL) {
		const char* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = self->priv->_member_name, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
	} else {
		char* _tmp2;
		char* _tmp3;
		_tmp2 = NULL;
		_tmp3 = NULL;
		return (_tmp3 = g_strdup_printf ("%s.%s", _tmp2 = vala_code_node_to_string ((ValaCodeNode*) vala_member_access_get_inner (self)), self->priv->_member_name), _tmp2 = (g_free (_tmp2), NULL), _tmp3);
	}
}


static void vala_member_access_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaMemberAccess * self;
	self = (ValaMemberAccess*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_member_access_get_inner (self) == old_node) {
		vala_member_access_set_inner (self, new_node);
	}
}


static gboolean vala_member_access_real_is_pure (ValaExpression* base) {
	ValaMemberAccess * self;
	gboolean _tmp0;
	gboolean _tmp1;
	self = (ValaMemberAccess*) base;
	_tmp0 = FALSE;
	_tmp1 = FALSE;
	if (vala_member_access_get_inner (self) == NULL) {
		_tmp1 = TRUE;
	} else {
		_tmp1 = vala_expression_is_pure (vala_member_access_get_inner (self));
	}
	if (_tmp1) {
		_tmp0 = !VALA_IS_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) self));
	} else {
		_tmp0 = FALSE;
	}
	/* accessing property could have side-effects*/
	return _tmp0;
}


static void vala_member_access_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaMemberAccess * self;
	self = (ValaMemberAccess*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	{
		gint i;
		i = 0;
		for (; i < gee_collection_get_size ((GeeCollection*) self->priv->type_argument_list); i++) {
			ValaDataType* _tmp0;
			gboolean _tmp1;
			_tmp0 = NULL;
			if ((_tmp1 = (_tmp0 = (ValaDataType*) gee_list_get (self->priv->type_argument_list, i)) == old_type, (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)), _tmp1)) {
				gee_list_set (self->priv->type_argument_list, i, new_type);
				return;
			}
		}
	}
}


static gboolean vala_member_access_real_is_constant (ValaExpression* base) {
	ValaMemberAccess * self;
	self = (ValaMemberAccess*) base;
	if (VALA_IS_CONSTANT (vala_expression_get_symbol_reference ((ValaExpression*) self))) {
		return TRUE;
	} else {
		return FALSE;
	}
}


static gboolean vala_member_access_real_is_non_null (ValaExpression* base) {
	ValaMemberAccess * self;
	ValaConstant* _tmp1;
	ValaSymbol* _tmp0;
	ValaConstant* c;
	self = (ValaMemberAccess*) base;
	_tmp1 = NULL;
	_tmp0 = NULL;
	c = (_tmp1 = (_tmp0 = vala_expression_get_symbol_reference ((ValaExpression*) self), VALA_IS_CONSTANT (_tmp0) ? ((ValaConstant*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	if (c != NULL) {
		gboolean _tmp2;
		return (_tmp2 = !vala_data_type_get_nullable (vala_constant_get_type_reference (c)), (c == NULL) ? NULL : (c = (vala_code_node_unref (c), NULL)), _tmp2);
	} else {
		gboolean _tmp3;
		return (_tmp3 = FALSE, (c == NULL) ? NULL : (c = (vala_code_node_unref (c), NULL)), _tmp3);
	}
	(c == NULL) ? NULL : (c = (vala_code_node_unref (c), NULL));
}


static gboolean vala_member_access_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaMemberAccess * self;
	ValaSymbol* base_symbol;
	ValaFormalParameter* this_parameter;
	gboolean may_access_instance_members;
	ValaSymbol* _tmp76;
	ValaSymbol* member;
	ValaSymbolAccessibility access;
	gboolean instance;
	gboolean klass;
	gboolean _tmp113;
	gboolean _tmp114;
	gboolean _tmp146;
	self = (ValaMemberAccess*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	if (vala_member_access_get_inner (self) != NULL) {
		vala_code_node_check ((ValaCodeNode*) vala_member_access_get_inner (self), analyzer);
	}
	{
		GeeIterator* _type_arg_it;
		_type_arg_it = gee_iterable_iterator ((GeeIterable*) self->priv->type_argument_list);
		while (gee_iterator_next (_type_arg_it)) {
			ValaDataType* type_arg;
			type_arg = (ValaDataType*) gee_iterator_get (_type_arg_it);
			vala_code_node_check ((ValaCodeNode*) type_arg, analyzer);
			(type_arg == NULL) ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL));
		}
		(_type_arg_it == NULL) ? NULL : (_type_arg_it = (gee_collection_object_unref (_type_arg_it), NULL));
	}
	base_symbol = NULL;
	this_parameter = NULL;
	may_access_instance_members = FALSE;
	vala_expression_set_symbol_reference ((ValaExpression*) self, NULL);
	if (self->priv->_qualified) {
		ValaSymbol* _tmp2;
		ValaSymbol* _tmp1;
		ValaSymbol* _tmp3;
		_tmp2 = NULL;
		_tmp1 = NULL;
		base_symbol = (_tmp2 = (_tmp1 = analyzer->root_symbol, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (base_symbol == NULL) ? NULL : (base_symbol = (vala_code_node_unref (base_symbol), NULL)), _tmp2);
		_tmp3 = NULL;
		vala_expression_set_symbol_reference ((ValaExpression*) self, _tmp3 = vala_scope_lookup (vala_symbol_get_scope (analyzer->root_symbol), self->priv->_member_name));
		(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL));
	} else {
		if (vala_member_access_get_inner (self) == NULL) {
			ValaSymbol* _tmp6;
			ValaSymbol* _tmp5;
			ValaSymbol* _tmp7;
			ValaSymbol* sym;
			if (_vala_strcmp0 (self->priv->_member_name, "this") == 0) {
				if (!vala_semantic_analyzer_is_in_instance_method (analyzer)) {
					gboolean _tmp4;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "This access invalid outside of instance methods");
					return (_tmp4 = FALSE, (base_symbol == NULL) ? NULL : (base_symbol = (vala_code_node_unref (base_symbol), NULL)), (this_parameter == NULL) ? NULL : (this_parameter = (vala_code_node_unref (this_parameter), NULL)), _tmp4);
				}
			}
			_tmp6 = NULL;
			_tmp5 = NULL;
			base_symbol = (_tmp6 = (_tmp5 = vala_semantic_analyzer_get_current_symbol (analyzer), (_tmp5 == NULL) ? NULL : vala_code_node_ref (_tmp5)), (base_symbol == NULL) ? NULL : (base_symbol = (vala_code_node_unref (base_symbol), NULL)), _tmp6);
			_tmp7 = NULL;
			sym = (_tmp7 = vala_semantic_analyzer_get_current_symbol (analyzer), (_tmp7 == NULL) ? NULL : vala_code_node_ref (_tmp7));
			while (TRUE) {
				gboolean _tmp8;
				ValaSymbol* _tmp24;
				ValaSymbol* _tmp26;
				ValaSymbol* _tmp25;
				_tmp8 = FALSE;
				if (sym != NULL) {
					_tmp8 = vala_expression_get_symbol_reference ((ValaExpression*) self) == NULL;
				} else {
					_tmp8 = FALSE;
				}
				if (!_tmp8) {
					break;
				}
				if (this_parameter == NULL) {
					if (VALA_IS_CREATION_METHOD (sym)) {
						ValaCreationMethod* _tmp9;
						ValaCreationMethod* cm;
						ValaFormalParameter* _tmp11;
						ValaFormalParameter* _tmp10;
						_tmp9 = NULL;
						cm = (_tmp9 = VALA_CREATION_METHOD (sym), (_tmp9 == NULL) ? NULL : vala_code_node_ref (_tmp9));
						_tmp11 = NULL;
						_tmp10 = NULL;
						this_parameter = (_tmp11 = (_tmp10 = vala_method_get_this_parameter ((ValaMethod*) cm), (_tmp10 == NULL) ? NULL : vala_code_node_ref (_tmp10)), (this_parameter == NULL) ? NULL : (this_parameter = (vala_code_node_unref (this_parameter), NULL)), _tmp11);
						may_access_instance_members = TRUE;
						(cm == NULL) ? NULL : (cm = (vala_code_node_unref (cm), NULL));
					} else {
						if (VALA_IS_PROPERTY (sym)) {
							ValaProperty* _tmp12;
							ValaProperty* prop;
							ValaFormalParameter* _tmp14;
							ValaFormalParameter* _tmp13;
							_tmp12 = NULL;
							prop = (_tmp12 = VALA_PROPERTY (sym), (_tmp12 == NULL) ? NULL : vala_code_node_ref (_tmp12));
							_tmp14 = NULL;
							_tmp13 = NULL;
							this_parameter = (_tmp14 = (_tmp13 = vala_property_get_this_parameter (prop), (_tmp13 == NULL) ? NULL : vala_code_node_ref (_tmp13)), (this_parameter == NULL) ? NULL : (this_parameter = (vala_code_node_unref (this_parameter), NULL)), _tmp14);
							may_access_instance_members = TRUE;
							(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
						} else {
							if (VALA_IS_CONSTRUCTOR (sym)) {
								ValaConstructor* _tmp15;
								ValaConstructor* c;
								ValaFormalParameter* _tmp17;
								ValaFormalParameter* _tmp16;
								_tmp15 = NULL;
								c = (_tmp15 = VALA_CONSTRUCTOR (sym), (_tmp15 == NULL) ? NULL : vala_code_node_ref (_tmp15));
								_tmp17 = NULL;
								_tmp16 = NULL;
								this_parameter = (_tmp17 = (_tmp16 = vala_constructor_get_this_parameter (c), (_tmp16 == NULL) ? NULL : vala_code_node_ref (_tmp16)), (this_parameter == NULL) ? NULL : (this_parameter = (vala_code_node_unref (this_parameter), NULL)), _tmp17);
								may_access_instance_members = TRUE;
								(c == NULL) ? NULL : (c = (vala_code_node_unref (c), NULL));
							} else {
								if (VALA_IS_DESTRUCTOR (sym)) {
									ValaDestructor* _tmp18;
									ValaDestructor* d;
									ValaFormalParameter* _tmp20;
									ValaFormalParameter* _tmp19;
									_tmp18 = NULL;
									d = (_tmp18 = VALA_DESTRUCTOR (sym), (_tmp18 == NULL) ? NULL : vala_code_node_ref (_tmp18));
									_tmp20 = NULL;
									_tmp19 = NULL;
									this_parameter = (_tmp20 = (_tmp19 = vala_destructor_get_this_parameter (d), (_tmp19 == NULL) ? NULL : vala_code_node_ref (_tmp19)), (this_parameter == NULL) ? NULL : (this_parameter = (vala_code_node_unref (this_parameter), NULL)), _tmp20);
									may_access_instance_members = TRUE;
									(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
								} else {
									if (VALA_IS_METHOD (sym)) {
										ValaMethod* _tmp21;
										ValaMethod* m;
										ValaFormalParameter* _tmp23;
										ValaFormalParameter* _tmp22;
										_tmp21 = NULL;
										m = (_tmp21 = VALA_METHOD (sym), (_tmp21 == NULL) ? NULL : vala_code_node_ref (_tmp21));
										_tmp23 = NULL;
										_tmp22 = NULL;
										this_parameter = (_tmp23 = (_tmp22 = vala_method_get_this_parameter (m), (_tmp22 == NULL) ? NULL : vala_code_node_ref (_tmp22)), (this_parameter == NULL) ? NULL : (this_parameter = (vala_code_node_unref (this_parameter), NULL)), _tmp23);
										may_access_instance_members = vala_method_get_binding (m) == MEMBER_BINDING_INSTANCE;
										(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
									}
								}
							}
						}
					}
				}
				_tmp24 = NULL;
				vala_expression_set_symbol_reference ((ValaExpression*) self, _tmp24 = vala_semantic_analyzer_symbol_lookup_inherited (sym, self->priv->_member_name));
				(_tmp24 == NULL) ? NULL : (_tmp24 = (vala_code_node_unref (_tmp24), NULL));
				_tmp26 = NULL;
				_tmp25 = NULL;
				sym = (_tmp26 = (_tmp25 = vala_symbol_get_parent_symbol (sym), (_tmp25 == NULL) ? NULL : vala_code_node_ref (_tmp25)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp26);
			}
			if (vala_expression_get_symbol_reference ((ValaExpression*) self) == NULL) {
				{
					GeeList* _tmp27;
					GeeIterator* _tmp28;
					GeeIterator* _ns_it;
					_tmp27 = NULL;
					_tmp28 = NULL;
					_ns_it = (_tmp28 = gee_iterable_iterator ((GeeIterable*) (_tmp27 = vala_source_file_get_using_directives (vala_semantic_analyzer_get_current_source_file (analyzer)))), (_tmp27 == NULL) ? NULL : (_tmp27 = (gee_collection_object_unref (_tmp27), NULL)), _tmp28);
					while (gee_iterator_next (_ns_it)) {
						ValaUsingDirective* ns;
						ValaSymbol* local_sym;
						ns = (ValaUsingDirective*) gee_iterator_get (_ns_it);
						local_sym = vala_scope_lookup (vala_symbol_get_scope (vala_using_directive_get_namespace_symbol (ns)), self->priv->_member_name);
						if (local_sym != NULL) {
							if (vala_expression_get_symbol_reference ((ValaExpression*) self) != NULL) {
								char* _tmp31;
								char* _tmp30;
								char* _tmp29;
								gboolean _tmp32;
								vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
								_tmp31 = NULL;
								_tmp30 = NULL;
								_tmp29 = NULL;
								vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp31 = g_strdup_printf ("`%s' is an ambiguous reference between `%s' and `%s'", self->priv->_member_name, _tmp29 = vala_symbol_get_full_name (vala_expression_get_symbol_reference ((ValaExpression*) self)), _tmp30 = vala_symbol_get_full_name (local_sym)));
								_tmp31 = (g_free (_tmp31), NULL);
								_tmp30 = (g_free (_tmp30), NULL);
								_tmp29 = (g_free (_tmp29), NULL);
								return (_tmp32 = FALSE, (ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL)), (local_sym == NULL) ? NULL : (local_sym = (vala_code_node_unref (local_sym), NULL)), (_ns_it == NULL) ? NULL : (_ns_it = (gee_collection_object_unref (_ns_it), NULL)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), (base_symbol == NULL) ? NULL : (base_symbol = (vala_code_node_unref (base_symbol), NULL)), (this_parameter == NULL) ? NULL : (this_parameter = (vala_code_node_unref (this_parameter), NULL)), _tmp32);
							}
							vala_expression_set_symbol_reference ((ValaExpression*) self, local_sym);
						}
						(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
						(local_sym == NULL) ? NULL : (local_sym = (vala_code_node_unref (local_sym), NULL));
					}
					(_ns_it == NULL) ? NULL : (_ns_it = (gee_collection_object_unref (_ns_it), NULL));
				}
			}
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
		} else {
			gboolean _tmp42;
			gboolean _tmp48;
			gboolean _tmp53;
			gboolean _tmp54;
			if (vala_code_node_get_error ((ValaCodeNode*) vala_member_access_get_inner (self))) {
				gboolean _tmp33;
				/* if there was an error in the inner expression, skip this check */
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				return (_tmp33 = FALSE, (base_symbol == NULL) ? NULL : (base_symbol = (vala_code_node_unref (base_symbol), NULL)), (this_parameter == NULL) ? NULL : (this_parameter = (vala_code_node_unref (this_parameter), NULL)), _tmp33);
			}
			if (self->priv->_pointer_member_access) {
				ValaPointerType* _tmp35;
				ValaDataType* _tmp34;
				ValaPointerType* pointer_type;
				gboolean _tmp36;
				_tmp35 = NULL;
				_tmp34 = NULL;
				pointer_type = (_tmp35 = (_tmp34 = vala_expression_get_value_type (vala_member_access_get_inner (self)), VALA_IS_POINTER_TYPE (_tmp34) ? ((ValaPointerType*) _tmp34) : NULL), (_tmp35 == NULL) ? NULL : vala_code_node_ref (_tmp35));
				_tmp36 = FALSE;
				if (pointer_type != NULL) {
					_tmp36 = VALA_IS_VALUE_TYPE (vala_pointer_type_get_base_type (pointer_type));
				} else {
					_tmp36 = FALSE;
				}
				if (_tmp36) {
					ValaPointerIndirection* _tmp37;
					/* transform foo->bar to (*foo).bar*/
					_tmp37 = NULL;
					vala_member_access_set_inner (self, (ValaExpression*) (_tmp37 = vala_pointer_indirection_new (vala_member_access_get_inner (self), vala_code_node_get_source_reference ((ValaCodeNode*) self))));
					(_tmp37 == NULL) ? NULL : (_tmp37 = (vala_code_node_unref (_tmp37), NULL));
					vala_code_node_check ((ValaCodeNode*) vala_member_access_get_inner (self), analyzer);
					vala_member_access_set_pointer_member_access (self, FALSE);
				}
				(pointer_type == NULL) ? NULL : (pointer_type = (vala_code_node_unref (pointer_type), NULL));
			}
			if (VALA_IS_MEMBER_ACCESS (vala_member_access_get_inner (self))) {
				ValaMemberAccess* _tmp38;
				ValaMemberAccess* ma;
				_tmp38 = NULL;
				ma = (_tmp38 = VALA_MEMBER_ACCESS (vala_member_access_get_inner (self)), (_tmp38 == NULL) ? NULL : vala_code_node_ref (_tmp38));
				if (ma->priv->_prototype_access) {
					char* _tmp40;
					char* _tmp39;
					gboolean _tmp41;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp40 = NULL;
					_tmp39 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp40 = g_strdup_printf ("Access to instance member `%s' denied", _tmp39 = vala_symbol_get_full_name (vala_expression_get_symbol_reference (vala_member_access_get_inner (self)))));
					_tmp40 = (g_free (_tmp40), NULL);
					_tmp39 = (g_free (_tmp39), NULL);
					return (_tmp41 = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), (base_symbol == NULL) ? NULL : (base_symbol = (vala_code_node_unref (base_symbol), NULL)), (this_parameter == NULL) ? NULL : (this_parameter = (vala_code_node_unref (this_parameter), NULL)), _tmp41);
				}
				(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
			}
			_tmp42 = FALSE;
			if (VALA_IS_MEMBER_ACCESS (vala_member_access_get_inner (self))) {
				_tmp42 = TRUE;
			} else {
				_tmp42 = VALA_IS_BASE_ACCESS (vala_member_access_get_inner (self));
			}
			if (_tmp42) {
				ValaSymbol* _tmp44;
				ValaSymbol* _tmp43;
				gboolean _tmp45;
				_tmp44 = NULL;
				_tmp43 = NULL;
				base_symbol = (_tmp44 = (_tmp43 = vala_expression_get_symbol_reference (vala_member_access_get_inner (self)), (_tmp43 == NULL) ? NULL : vala_code_node_ref (_tmp43)), (base_symbol == NULL) ? NULL : (base_symbol = (vala_code_node_unref (base_symbol), NULL)), _tmp44);
				_tmp45 = FALSE;
				if (vala_expression_get_symbol_reference ((ValaExpression*) self) == NULL) {
					gboolean _tmp46;
					_tmp46 = FALSE;
					if (VALA_IS_NAMESPACE (base_symbol)) {
						_tmp46 = TRUE;
					} else {
						_tmp46 = VALA_IS_TYPESYMBOL (base_symbol);
					}
					_tmp45 = _tmp46;
				} else {
					_tmp45 = FALSE;
				}
				if (_tmp45) {
					ValaSymbol* _tmp47;
					_tmp47 = NULL;
					vala_expression_set_symbol_reference ((ValaExpression*) self, _tmp47 = vala_scope_lookup (vala_symbol_get_scope (base_symbol), self->priv->_member_name));
					(_tmp47 == NULL) ? NULL : (_tmp47 = (vala_code_node_unref (_tmp47), NULL));
					if (VALA_IS_BASE_ACCESS (vala_member_access_get_inner (self))) {
						/* inner expression is base access
						 access to instance members of the base type possible*/
						may_access_instance_members = TRUE;
					}
				}
			}
			_tmp48 = FALSE;
			if (vala_expression_get_symbol_reference ((ValaExpression*) self) == NULL) {
				_tmp48 = vala_expression_get_value_type (vala_member_access_get_inner (self)) != NULL;
			} else {
				_tmp48 = FALSE;
			}
			if (_tmp48) {
				if (self->priv->_pointer_member_access) {
					ValaSymbol* _tmp49;
					_tmp49 = NULL;
					vala_expression_set_symbol_reference ((ValaExpression*) self, _tmp49 = vala_data_type_get_pointer_member (vala_expression_get_value_type (vala_member_access_get_inner (self)), self->priv->_member_name));
					(_tmp49 == NULL) ? NULL : (_tmp49 = (vala_code_node_unref (_tmp49), NULL));
				} else {
					ValaSymbol* _tmp52;
					if (vala_data_type_get_data_type (vala_expression_get_value_type (vala_member_access_get_inner (self))) != NULL) {
						ValaSymbol* _tmp51;
						ValaSymbol* _tmp50;
						_tmp51 = NULL;
						_tmp50 = NULL;
						base_symbol = (_tmp51 = (_tmp50 = (ValaSymbol*) vala_data_type_get_data_type (vala_expression_get_value_type (vala_member_access_get_inner (self))), (_tmp50 == NULL) ? NULL : vala_code_node_ref (_tmp50)), (base_symbol == NULL) ? NULL : (base_symbol = (vala_code_node_unref (base_symbol), NULL)), _tmp51);
					}
					_tmp52 = NULL;
					vala_expression_set_symbol_reference ((ValaExpression*) self, _tmp52 = vala_data_type_get_member (vala_expression_get_value_type (vala_member_access_get_inner (self)), self->priv->_member_name));
					(_tmp52 == NULL) ? NULL : (_tmp52 = (vala_code_node_unref (_tmp52), NULL));
				}
				if (vala_expression_get_symbol_reference ((ValaExpression*) self) != NULL) {
					/* inner expression is variable, field, or parameter
					 access to instance members of the corresponding type possible*/
					may_access_instance_members = TRUE;
				}
			}
			_tmp53 = FALSE;
			_tmp54 = FALSE;
			if (vala_expression_get_symbol_reference ((ValaExpression*) self) == NULL) {
				_tmp54 = vala_expression_get_value_type (vala_member_access_get_inner (self)) != NULL;
			} else {
				_tmp54 = FALSE;
			}
			if (_tmp54) {
				_tmp53 = vala_data_type_get_is_dynamic (vala_expression_get_value_type (vala_member_access_get_inner (self)));
			} else {
				_tmp53 = FALSE;
			}
			if (_tmp53) {
				ValaObjectType* _tmp55;
				ValaObjectType* dynamic_object_type;
				/* allow late bound members for dynamic types*/
				_tmp55 = NULL;
				dynamic_object_type = (_tmp55 = VALA_OBJECT_TYPE (vala_expression_get_value_type (vala_member_access_get_inner (self))), (_tmp55 == NULL) ? NULL : vala_code_node_ref (_tmp55));
				if (VALA_IS_METHOD_CALL (vala_code_node_get_parent_node ((ValaCodeNode*) self))) {
					ValaMethodCall* _tmp56;
					ValaMethodCall* invoc;
					_tmp56 = NULL;
					invoc = (_tmp56 = VALA_METHOD_CALL (vala_code_node_get_parent_node ((ValaCodeNode*) self)), (_tmp56 == NULL) ? NULL : vala_code_node_ref (_tmp56));
					if (vala_method_call_get_call (invoc) == VALA_EXPRESSION (self)) {
						ValaDataType* ret_type;
						ValaDynamicMethod* m;
						ValaErrorType* _tmp60;
						ValaFormalParameter* _tmp61;
						/* dynamic method*/
						ret_type = NULL;
						if (vala_expression_get_target_type ((ValaExpression*) invoc) != NULL) {
							ValaDataType* _tmp57;
							_tmp57 = NULL;
							ret_type = (_tmp57 = vala_data_type_copy (vala_expression_get_target_type ((ValaExpression*) invoc)), (ret_type == NULL) ? NULL : (ret_type = (vala_code_node_unref (ret_type), NULL)), _tmp57);
							vala_data_type_set_value_owned (ret_type, TRUE);
						} else {
							if (VALA_IS_EXPRESSION_STATEMENT (vala_code_node_get_parent_node ((ValaCodeNode*) invoc))) {
								ValaDataType* _tmp58;
								_tmp58 = NULL;
								ret_type = (_tmp58 = (ValaDataType*) vala_void_type_new (NULL), (ret_type == NULL) ? NULL : (ret_type = (vala_code_node_unref (ret_type), NULL)), _tmp58);
							} else {
								ValaDataType* _tmp59;
								/* expect dynamic object of the same type*/
								_tmp59 = NULL;
								ret_type = (_tmp59 = vala_data_type_copy (vala_expression_get_value_type (vala_member_access_get_inner (self))), (ret_type == NULL) ? NULL : (ret_type = (vala_code_node_unref (ret_type), NULL)), _tmp59);
							}
						}
						m = vala_dynamic_method_new (vala_expression_get_value_type (vala_member_access_get_inner (self)), self->priv->_member_name, ret_type, vala_code_node_get_source_reference ((ValaCodeNode*) self));
						vala_dynamic_method_set_invocation (m, invoc);
						_tmp60 = NULL;
						vala_code_node_add_error_type ((ValaCodeNode*) m, (ValaDataType*) (_tmp60 = vala_error_type_new (NULL, NULL, NULL)));
						(_tmp60 == NULL) ? NULL : (_tmp60 = (vala_code_node_unref (_tmp60), NULL));
						vala_symbol_set_access ((ValaSymbol*) m, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
						_tmp61 = NULL;
						vala_method_add_parameter ((ValaMethod*) m, _tmp61 = vala_formal_parameter_new_with_ellipsis (NULL));
						(_tmp61 == NULL) ? NULL : (_tmp61 = (vala_code_node_unref (_tmp61), NULL));
						vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) vala_object_type_get_type_symbol (dynamic_object_type)), NULL, (ValaSymbol*) m);
						vala_expression_set_symbol_reference ((ValaExpression*) self, (ValaSymbol*) m);
						(ret_type == NULL) ? NULL : (ret_type = (vala_code_node_unref (ret_type), NULL));
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
					}
					(invoc == NULL) ? NULL : (invoc = (vala_code_node_unref (invoc), NULL));
				} else {
					if (VALA_IS_ASSIGNMENT (vala_code_node_get_parent_node ((ValaCodeNode*) self))) {
						ValaAssignment* _tmp62;
						ValaAssignment* a;
						gboolean _tmp63;
						_tmp62 = NULL;
						a = (_tmp62 = VALA_ASSIGNMENT (vala_code_node_get_parent_node ((ValaCodeNode*) self)), (_tmp62 == NULL) ? NULL : vala_code_node_ref (_tmp62));
						_tmp63 = FALSE;
						if (vala_assignment_get_left (a) == VALA_EXPRESSION (self)) {
							gboolean _tmp64;
							_tmp64 = FALSE;
							if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_ADD) {
								_tmp64 = TRUE;
							} else {
								_tmp64 = vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_SUB;
							}
							_tmp63 = _tmp64;
						} else {
							_tmp63 = FALSE;
						}
						if (_tmp63) {
							ValaVoidType* _tmp65;
							ValaDynamicSignal* _tmp66;
							ValaDynamicSignal* s;
							/* dynamic signal*/
							_tmp65 = NULL;
							_tmp66 = NULL;
							s = (_tmp66 = vala_dynamic_signal_new (vala_expression_get_value_type (vala_member_access_get_inner (self)), self->priv->_member_name, (ValaDataType*) (_tmp65 = vala_void_type_new (NULL)), vala_code_node_get_source_reference ((ValaCodeNode*) self)), (_tmp65 == NULL) ? NULL : (_tmp65 = (vala_code_node_unref (_tmp65), NULL)), _tmp66);
							vala_dynamic_signal_set_handler (s, vala_assignment_get_right (a));
							vala_symbol_set_access ((ValaSymbol*) s, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
							vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) vala_object_type_get_type_symbol (dynamic_object_type)), NULL, (ValaSymbol*) s);
							vala_expression_set_symbol_reference ((ValaExpression*) self, (ValaSymbol*) s);
							(s == NULL) ? NULL : (s = (vala_code_node_unref (s), NULL));
						} else {
							if (vala_assignment_get_left (a) == VALA_EXPRESSION (self)) {
								ValaDynamicProperty* prop;
								ValaPropertyAccessor* _tmp67;
								/* dynamic property assignment*/
								prop = vala_dynamic_property_new (vala_expression_get_value_type (vala_member_access_get_inner (self)), self->priv->_member_name, vala_code_node_get_source_reference ((ValaCodeNode*) self));
								vala_symbol_set_access ((ValaSymbol*) prop, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
								_tmp67 = NULL;
								vala_property_set_set_accessor ((ValaProperty*) prop, _tmp67 = vala_property_accessor_new (FALSE, TRUE, FALSE, NULL, NULL, vala_code_node_get_source_reference ((ValaCodeNode*) prop)));
								(_tmp67 == NULL) ? NULL : (_tmp67 = (vala_code_node_unref (_tmp67), NULL));
								vala_property_accessor_set_access (vala_property_get_set_accessor ((ValaProperty*) prop), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
								vala_symbol_set_owner ((ValaSymbol*) prop, vala_symbol_get_scope ((ValaSymbol*) vala_data_type_get_data_type (vala_expression_get_value_type (vala_member_access_get_inner (self)))));
								vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) vala_object_type_get_type_symbol (dynamic_object_type)), NULL, (ValaSymbol*) prop);
								vala_expression_set_symbol_reference ((ValaExpression*) self, (ValaSymbol*) prop);
								(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
							}
						}
						(a == NULL) ? NULL : (a = (vala_code_node_unref (a), NULL));
					}
				}
				if (vala_expression_get_symbol_reference ((ValaExpression*) self) == NULL) {
					ValaDynamicProperty* prop;
					ValaPropertyAccessor* _tmp70;
					ValaDataType* _tmp69;
					/* dynamic property read access*/
					prop = vala_dynamic_property_new (vala_expression_get_value_type (vala_member_access_get_inner (self)), self->priv->_member_name, vala_code_node_get_source_reference ((ValaCodeNode*) self));
					if (vala_expression_get_target_type ((ValaExpression*) self) != NULL) {
						vala_property_set_property_type ((ValaProperty*) prop, vala_expression_get_target_type ((ValaExpression*) self));
					} else {
						ValaDataType* _tmp68;
						/* expect dynamic object of the same type*/
						_tmp68 = NULL;
						vala_property_set_property_type ((ValaProperty*) prop, _tmp68 = vala_data_type_copy (vala_expression_get_value_type (vala_member_access_get_inner (self))));
						(_tmp68 == NULL) ? NULL : (_tmp68 = (vala_code_node_unref (_tmp68), NULL));
					}
					vala_symbol_set_access ((ValaSymbol*) prop, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
					_tmp70 = NULL;
					_tmp69 = NULL;
					vala_property_set_get_accessor ((ValaProperty*) prop, _tmp70 = vala_property_accessor_new (TRUE, FALSE, FALSE, _tmp69 = vala_data_type_copy (vala_property_get_property_type ((ValaProperty*) prop)), NULL, vala_code_node_get_source_reference ((ValaCodeNode*) prop)));
					(_tmp70 == NULL) ? NULL : (_tmp70 = (vala_code_node_unref (_tmp70), NULL));
					(_tmp69 == NULL) ? NULL : (_tmp69 = (vala_code_node_unref (_tmp69), NULL));
					vala_property_accessor_set_access (vala_property_get_get_accessor ((ValaProperty*) prop), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
					vala_symbol_set_owner ((ValaSymbol*) prop, vala_symbol_get_scope ((ValaSymbol*) vala_data_type_get_data_type (vala_expression_get_value_type (vala_member_access_get_inner (self)))));
					vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) vala_object_type_get_type_symbol (dynamic_object_type)), NULL, (ValaSymbol*) prop);
					vala_expression_set_symbol_reference ((ValaExpression*) self, (ValaSymbol*) prop);
					(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				}
				if (vala_expression_get_symbol_reference ((ValaExpression*) self) != NULL) {
					may_access_instance_members = TRUE;
				}
				(dynamic_object_type == NULL) ? NULL : (dynamic_object_type = (vala_code_node_unref (dynamic_object_type), NULL));
			}
		}
	}
	if (vala_expression_get_symbol_reference ((ValaExpression*) self) == NULL) {
		char* base_type_name;
		gboolean _tmp71;
		char* _tmp74;
		gboolean _tmp75;
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		base_type_name = g_strdup ("(null)");
		_tmp71 = FALSE;
		if (vala_member_access_get_inner (self) != NULL) {
			_tmp71 = vala_expression_get_value_type (vala_member_access_get_inner (self)) != NULL;
		} else {
			_tmp71 = FALSE;
		}
		if (_tmp71) {
			char* _tmp72;
			_tmp72 = NULL;
			base_type_name = (_tmp72 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_member_access_get_inner (self))), base_type_name = (g_free (base_type_name), NULL), _tmp72);
		} else {
			if (base_symbol != NULL) {
				char* _tmp73;
				_tmp73 = NULL;
				base_type_name = (_tmp73 = vala_symbol_get_full_name (base_symbol), base_type_name = (g_free (base_type_name), NULL), _tmp73);
			}
		}
		_tmp74 = NULL;
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp74 = g_strdup_printf ("The name `%s' does not exist in the context of `%s'", self->priv->_member_name, base_type_name));
		_tmp74 = (g_free (_tmp74), NULL);
		return (_tmp75 = FALSE, base_type_name = (g_free (base_type_name), NULL), (base_symbol == NULL) ? NULL : (base_symbol = (vala_code_node_unref (base_symbol), NULL)), (this_parameter == NULL) ? NULL : (this_parameter = (vala_code_node_unref (this_parameter), NULL)), _tmp75);
	}
	_tmp76 = NULL;
	member = (_tmp76 = vala_expression_get_symbol_reference ((ValaExpression*) self), (_tmp76 == NULL) ? NULL : vala_code_node_ref (_tmp76));
	access = VALA_SYMBOL_ACCESSIBILITY_PUBLIC;
	instance = FALSE;
	klass = FALSE;
	if (!vala_code_node_check ((ValaCodeNode*) member, analyzer)) {
		gboolean _tmp77;
		return (_tmp77 = FALSE, (base_symbol == NULL) ? NULL : (base_symbol = (vala_code_node_unref (base_symbol), NULL)), (this_parameter == NULL) ? NULL : (this_parameter = (vala_code_node_unref (this_parameter), NULL)), (member == NULL) ? NULL : (member = (vala_code_node_unref (member), NULL)), _tmp77);
	}
	if (VALA_IS_FIELD (member)) {
		ValaField* _tmp78;
		ValaField* f;
		_tmp78 = NULL;
		f = (_tmp78 = VALA_FIELD (member), (_tmp78 == NULL) ? NULL : vala_code_node_ref (_tmp78));
		access = vala_symbol_get_access ((ValaSymbol*) f);
		instance = vala_field_get_binding (f) == MEMBER_BINDING_INSTANCE;
		klass = vala_field_get_binding (f) == MEMBER_BINDING_CLASS;
		(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
	} else {
		if (VALA_IS_METHOD (member)) {
			ValaMethod* _tmp79;
			ValaMethod* m;
			_tmp79 = NULL;
			m = (_tmp79 = VALA_METHOD (member), (_tmp79 == NULL) ? NULL : vala_code_node_ref (_tmp79));
			if (vala_method_get_base_method (m) != NULL) {
				ValaMethod* _tmp81;
				ValaMethod* _tmp80;
				ValaSymbol* _tmp83;
				ValaSymbol* _tmp82;
				/* refer to base method to inherit default arguments*/
				_tmp81 = NULL;
				_tmp80 = NULL;
				m = (_tmp81 = (_tmp80 = vala_method_get_base_method (m), (_tmp80 == NULL) ? NULL : vala_code_node_ref (_tmp80)), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), _tmp81);
				if (vala_method_get_signal_reference (m) != NULL) {
					/* method is class/default handler for a signal
					 let signal deal with member access*/
					vala_expression_set_symbol_reference ((ValaExpression*) self, (ValaSymbol*) vala_method_get_signal_reference (m));
				} else {
					vala_expression_set_symbol_reference ((ValaExpression*) self, (ValaSymbol*) m);
				}
				_tmp83 = NULL;
				_tmp82 = NULL;
				member = (_tmp83 = (_tmp82 = vala_expression_get_symbol_reference ((ValaExpression*) self), (_tmp82 == NULL) ? NULL : vala_code_node_ref (_tmp82)), (member == NULL) ? NULL : (member = (vala_code_node_unref (member), NULL)), _tmp83);
			} else {
				if (vala_method_get_base_interface_method (m) != NULL) {
					ValaMethod* _tmp85;
					ValaMethod* _tmp84;
					ValaSymbol* _tmp87;
					ValaSymbol* _tmp86;
					/* refer to base method to inherit default arguments*/
					_tmp85 = NULL;
					_tmp84 = NULL;
					m = (_tmp85 = (_tmp84 = vala_method_get_base_interface_method (m), (_tmp84 == NULL) ? NULL : vala_code_node_ref (_tmp84)), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), _tmp85);
					vala_expression_set_symbol_reference ((ValaExpression*) self, (ValaSymbol*) m);
					_tmp87 = NULL;
					_tmp86 = NULL;
					member = (_tmp87 = (_tmp86 = vala_expression_get_symbol_reference ((ValaExpression*) self), (_tmp86 == NULL) ? NULL : vala_code_node_ref (_tmp86)), (member == NULL) ? NULL : (member = (vala_code_node_unref (member), NULL)), _tmp87);
				}
			}
			access = vala_symbol_get_access ((ValaSymbol*) m);
			if (!VALA_IS_CREATION_METHOD (m)) {
				instance = vala_method_get_binding (m) == MEMBER_BINDING_INSTANCE;
			}
			klass = vala_method_get_binding (m) == MEMBER_BINDING_CLASS;
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		} else {
			if (VALA_IS_PROPERTY (member)) {
				ValaProperty* _tmp88;
				ValaProperty* prop;
				_tmp88 = NULL;
				prop = (_tmp88 = VALA_PROPERTY (member), (_tmp88 == NULL) ? NULL : vala_code_node_ref (_tmp88));
				if (!vala_code_node_check ((ValaCodeNode*) prop, analyzer)) {
					gboolean _tmp89;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					return (_tmp89 = FALSE, (prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL)), (base_symbol == NULL) ? NULL : (base_symbol = (vala_code_node_unref (base_symbol), NULL)), (this_parameter == NULL) ? NULL : (this_parameter = (vala_code_node_unref (this_parameter), NULL)), (member == NULL) ? NULL : (member = (vala_code_node_unref (member), NULL)), _tmp89);
				}
				if (vala_property_get_base_property (prop) != NULL) {
					ValaProperty* _tmp91;
					ValaProperty* _tmp90;
					ValaSymbol* _tmp93;
					ValaSymbol* _tmp92;
					/* refer to base property*/
					_tmp91 = NULL;
					_tmp90 = NULL;
					prop = (_tmp91 = (_tmp90 = vala_property_get_base_property (prop), (_tmp90 == NULL) ? NULL : vala_code_node_ref (_tmp90)), (prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL)), _tmp91);
					vala_expression_set_symbol_reference ((ValaExpression*) self, (ValaSymbol*) prop);
					_tmp93 = NULL;
					_tmp92 = NULL;
					member = (_tmp93 = (_tmp92 = vala_expression_get_symbol_reference ((ValaExpression*) self), (_tmp92 == NULL) ? NULL : vala_code_node_ref (_tmp92)), (member == NULL) ? NULL : (member = (vala_code_node_unref (member), NULL)), _tmp93);
				} else {
					if (vala_property_get_base_interface_property (prop) != NULL) {
						ValaProperty* _tmp95;
						ValaProperty* _tmp94;
						ValaSymbol* _tmp97;
						ValaSymbol* _tmp96;
						/* refer to base property*/
						_tmp95 = NULL;
						_tmp94 = NULL;
						prop = (_tmp95 = (_tmp94 = vala_property_get_base_interface_property (prop), (_tmp94 == NULL) ? NULL : vala_code_node_ref (_tmp94)), (prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL)), _tmp95);
						vala_expression_set_symbol_reference ((ValaExpression*) self, (ValaSymbol*) prop);
						_tmp97 = NULL;
						_tmp96 = NULL;
						member = (_tmp97 = (_tmp96 = vala_expression_get_symbol_reference ((ValaExpression*) self), (_tmp96 == NULL) ? NULL : vala_code_node_ref (_tmp96)), (member == NULL) ? NULL : (member = (vala_code_node_unref (member), NULL)), _tmp97);
					}
				}
				access = vala_symbol_get_access ((ValaSymbol*) prop);
				if (vala_expression_get_lvalue ((ValaExpression*) self)) {
					if (vala_property_get_set_accessor (prop) == NULL) {
						char* _tmp99;
						char* _tmp98;
						gboolean _tmp100;
						vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
						_tmp99 = NULL;
						_tmp98 = NULL;
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp99 = g_strdup_printf ("Property `%s' is read-only", _tmp98 = vala_symbol_get_full_name ((ValaSymbol*) prop)));
						_tmp99 = (g_free (_tmp99), NULL);
						_tmp98 = (g_free (_tmp98), NULL);
						return (_tmp100 = FALSE, (prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL)), (base_symbol == NULL) ? NULL : (base_symbol = (vala_code_node_unref (base_symbol), NULL)), (this_parameter == NULL) ? NULL : (this_parameter = (vala_code_node_unref (this_parameter), NULL)), (member == NULL) ? NULL : (member = (vala_code_node_unref (member), NULL)), _tmp100);
					}
					if (vala_symbol_get_access ((ValaSymbol*) prop) == VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
						access = vala_property_accessor_get_access (vala_property_get_set_accessor (prop));
					} else {
						gboolean _tmp101;
						_tmp101 = FALSE;
						if (vala_symbol_get_access ((ValaSymbol*) prop) == VALA_SYMBOL_ACCESSIBILITY_PROTECTED) {
							_tmp101 = vala_property_accessor_get_access (vala_property_get_set_accessor (prop)) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC;
						} else {
							_tmp101 = FALSE;
						}
						if (_tmp101) {
							access = vala_property_accessor_get_access (vala_property_get_set_accessor (prop));
						}
					}
				} else {
					if (vala_property_get_get_accessor (prop) == NULL) {
						char* _tmp103;
						char* _tmp102;
						gboolean _tmp104;
						vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
						_tmp103 = NULL;
						_tmp102 = NULL;
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp103 = g_strdup_printf ("Property `%s' is write-only", _tmp102 = vala_symbol_get_full_name ((ValaSymbol*) prop)));
						_tmp103 = (g_free (_tmp103), NULL);
						_tmp102 = (g_free (_tmp102), NULL);
						return (_tmp104 = FALSE, (prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL)), (base_symbol == NULL) ? NULL : (base_symbol = (vala_code_node_unref (base_symbol), NULL)), (this_parameter == NULL) ? NULL : (this_parameter = (vala_code_node_unref (this_parameter), NULL)), (member == NULL) ? NULL : (member = (vala_code_node_unref (member), NULL)), _tmp104);
					}
					if (vala_symbol_get_access ((ValaSymbol*) prop) == VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
						access = vala_property_accessor_get_access (vala_property_get_get_accessor (prop));
					} else {
						gboolean _tmp105;
						_tmp105 = FALSE;
						if (vala_symbol_get_access ((ValaSymbol*) prop) == VALA_SYMBOL_ACCESSIBILITY_PROTECTED) {
							_tmp105 = vala_property_accessor_get_access (vala_property_get_get_accessor (prop)) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC;
						} else {
							_tmp105 = FALSE;
						}
						if (_tmp105) {
							access = vala_property_accessor_get_access (vala_property_get_get_accessor (prop));
						}
					}
				}
				instance = vala_property_get_binding (prop) == MEMBER_BINDING_INSTANCE;
				(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
			} else {
				if (VALA_IS_SIGNAL (member)) {
					instance = TRUE;
				}
			}
		}
	}
	vala_symbol_set_used (member, TRUE);
	if (access == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		ValaSymbol* _tmp106;
		ValaSymbol* target_type;
		gboolean in_target_type;
		_tmp106 = NULL;
		target_type = (_tmp106 = vala_symbol_get_parent_symbol (member), (_tmp106 == NULL) ? NULL : vala_code_node_ref (_tmp106));
		in_target_type = FALSE;
		{
			ValaSymbol* _tmp107;
			ValaSymbol* this_symbol;
			ValaSymbol* _tmp109;
			ValaSymbol* _tmp108;
			_tmp107 = NULL;
			this_symbol = (_tmp107 = vala_semantic_analyzer_get_current_symbol (analyzer), (_tmp107 == NULL) ? NULL : vala_code_node_ref (_tmp107));
			_tmp109 = NULL;
			_tmp108 = NULL;
			for (; this_symbol != NULL; this_symbol = (_tmp109 = (_tmp108 = vala_symbol_get_parent_symbol (this_symbol), (_tmp108 == NULL) ? NULL : vala_code_node_ref (_tmp108)), (this_symbol == NULL) ? NULL : (this_symbol = (vala_code_node_unref (this_symbol), NULL)), _tmp109)) {
				if (target_type == this_symbol) {
					in_target_type = TRUE;
					break;
				}
			}
			(this_symbol == NULL) ? NULL : (this_symbol = (vala_code_node_unref (this_symbol), NULL));
		}
		if (!in_target_type) {
			char* _tmp111;
			char* _tmp110;
			gboolean _tmp112;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			_tmp111 = NULL;
			_tmp110 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp111 = g_strdup_printf ("Access to private member `%s' denied", _tmp110 = vala_symbol_get_full_name (member)));
			_tmp111 = (g_free (_tmp111), NULL);
			_tmp110 = (g_free (_tmp110), NULL);
			return (_tmp112 = FALSE, (target_type == NULL) ? NULL : (target_type = (vala_code_node_unref (target_type), NULL)), (base_symbol == NULL) ? NULL : (base_symbol = (vala_code_node_unref (base_symbol), NULL)), (this_parameter == NULL) ? NULL : (this_parameter = (vala_code_node_unref (this_parameter), NULL)), (member == NULL) ? NULL : (member = (vala_code_node_unref (member), NULL)), _tmp112);
		}
		(target_type == NULL) ? NULL : (target_type = (vala_code_node_unref (target_type), NULL));
	}
	_tmp113 = FALSE;
	_tmp114 = FALSE;
	if (instance) {
		_tmp114 = TRUE;
	} else {
		_tmp114 = klass;
	}
	if (_tmp114) {
		_tmp113 = !may_access_instance_members;
	} else {
		_tmp113 = FALSE;
	}
	if (_tmp113) {
		vala_member_access_set_prototype_access (self, TRUE);
		if (VALA_IS_METHOD (vala_expression_get_symbol_reference ((ValaExpression*) self))) {
			ValaDataType* _tmp115;
			/* also set static type for prototype access
			 required when using instance methods as delegates in constants
			 TODO replace by MethodPrototype*/
			_tmp115 = NULL;
			vala_expression_set_value_type ((ValaExpression*) self, _tmp115 = vala_semantic_analyzer_get_value_type_for_symbol (analyzer, vala_expression_get_symbol_reference ((ValaExpression*) self), vala_expression_get_lvalue ((ValaExpression*) self)));
			(_tmp115 == NULL) ? NULL : (_tmp115 = (vala_code_node_unref (_tmp115), NULL));
		} else {
			if (VALA_IS_FIELD (vala_expression_get_symbol_reference ((ValaExpression*) self))) {
				ValaFieldPrototype* _tmp116;
				_tmp116 = NULL;
				vala_expression_set_value_type ((ValaExpression*) self, (ValaDataType*) (_tmp116 = vala_field_prototype_new (VALA_FIELD (vala_expression_get_symbol_reference ((ValaExpression*) self)))));
				(_tmp116 == NULL) ? NULL : (_tmp116 = (vala_code_node_unref (_tmp116), NULL));
			} else {
				ValaInvalidType* _tmp117;
				_tmp117 = NULL;
				vala_expression_set_value_type ((ValaExpression*) self, (ValaDataType*) (_tmp117 = vala_invalid_type_new ()));
				(_tmp117 == NULL) ? NULL : (_tmp117 = (vala_code_node_unref (_tmp117), NULL));
			}
		}
	} else {
		gboolean _tmp118;
		ValaDataType* _tmp121;
		gboolean _tmp122;
		_tmp118 = FALSE;
		if (instance) {
			_tmp118 = vala_member_access_get_inner (self) == NULL;
		} else {
			_tmp118 = FALSE;
		}
		/* implicit this access*/
		if (_tmp118) {
			ValaMemberAccess* _tmp119;
			ValaDataType* _tmp120;
			_tmp119 = NULL;
			vala_member_access_set_inner (self, (ValaExpression*) (_tmp119 = vala_member_access_new (NULL, "this", vala_code_node_get_source_reference ((ValaCodeNode*) self))));
			(_tmp119 == NULL) ? NULL : (_tmp119 = (vala_code_node_unref (_tmp119), NULL));
			_tmp120 = NULL;
			vala_expression_set_value_type (vala_member_access_get_inner (self), _tmp120 = vala_data_type_copy (vala_formal_parameter_get_parameter_type (this_parameter)));
			(_tmp120 == NULL) ? NULL : (_tmp120 = (vala_code_node_unref (_tmp120), NULL));
			vala_expression_set_symbol_reference (vala_member_access_get_inner (self), (ValaSymbol*) this_parameter);
		}
		_tmp121 = NULL;
		vala_expression_set_formal_value_type ((ValaExpression*) self, _tmp121 = vala_semantic_analyzer_get_value_type_for_symbol (analyzer, vala_expression_get_symbol_reference ((ValaExpression*) self), vala_expression_get_lvalue ((ValaExpression*) self)));
		(_tmp121 == NULL) ? NULL : (_tmp121 = (vala_code_node_unref (_tmp121), NULL));
		_tmp122 = FALSE;
		if (vala_member_access_get_inner (self) != NULL) {
			_tmp122 = vala_expression_get_formal_value_type ((ValaExpression*) self) != NULL;
		} else {
			_tmp122 = FALSE;
		}
		if (_tmp122) {
			ValaDataType* _tmp123;
			_tmp123 = NULL;
			vala_expression_set_value_type ((ValaExpression*) self, _tmp123 = vala_data_type_get_actual_type (vala_expression_get_formal_value_type ((ValaExpression*) self), vala_expression_get_value_type (vala_member_access_get_inner (self)), (ValaCodeNode*) self));
			(_tmp123 == NULL) ? NULL : (_tmp123 = (vala_code_node_unref (_tmp123), NULL));
		} else {
			vala_expression_set_value_type ((ValaExpression*) self, vala_expression_get_formal_value_type ((ValaExpression*) self));
		}
		if (VALA_IS_METHOD (vala_expression_get_symbol_reference ((ValaExpression*) self))) {
			ValaMethod* _tmp124;
			ValaMethod* m;
			ValaMethod* base_method;
			gboolean _tmp131;
			_tmp124 = NULL;
			m = (_tmp124 = VALA_METHOD (vala_expression_get_symbol_reference ((ValaExpression*) self)), (_tmp124 == NULL) ? NULL : vala_code_node_ref (_tmp124));
			base_method = NULL;
			if (vala_method_get_base_method (m) != NULL) {
				ValaMethod* _tmp126;
				ValaMethod* _tmp125;
				_tmp126 = NULL;
				_tmp125 = NULL;
				base_method = (_tmp126 = (_tmp125 = vala_method_get_base_method (m), (_tmp125 == NULL) ? NULL : vala_code_node_ref (_tmp125)), (base_method == NULL) ? NULL : (base_method = (vala_code_node_unref (base_method), NULL)), _tmp126);
			} else {
				if (vala_method_get_base_interface_method (m) != NULL) {
					ValaMethod* _tmp128;
					ValaMethod* _tmp127;
					_tmp128 = NULL;
					_tmp127 = NULL;
					base_method = (_tmp128 = (_tmp127 = vala_method_get_base_interface_method (m), (_tmp127 == NULL) ? NULL : vala_code_node_ref (_tmp127)), (base_method == NULL) ? NULL : (base_method = (vala_code_node_unref (base_method), NULL)), _tmp128);
				} else {
					ValaMethod* _tmp130;
					ValaMethod* _tmp129;
					_tmp130 = NULL;
					_tmp129 = NULL;
					base_method = (_tmp130 = (_tmp129 = m, (_tmp129 == NULL) ? NULL : vala_code_node_ref (_tmp129)), (base_method == NULL) ? NULL : (base_method = (vala_code_node_unref (base_method), NULL)), _tmp130);
				}
			}
			_tmp131 = FALSE;
			if (instance) {
				_tmp131 = vala_symbol_get_parent_symbol ((ValaSymbol*) base_method) != NULL;
			} else {
				_tmp131 = FALSE;
			}
			if (_tmp131) {
				ValaDataType* _tmp132;
				_tmp132 = NULL;
				vala_expression_set_target_type (vala_member_access_get_inner (self), _tmp132 = vala_semantic_analyzer_get_data_type_for_symbol (VALA_TYPESYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) base_method))));
				(_tmp132 == NULL) ? NULL : (_tmp132 = (vala_code_node_unref (_tmp132), NULL));
			}
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
			(base_method == NULL) ? NULL : (base_method = (vala_code_node_unref (base_method), NULL));
		} else {
			if (VALA_IS_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) self))) {
				ValaProperty* _tmp133;
				ValaProperty* prop;
				ValaProperty* base_property;
				gboolean _tmp140;
				_tmp133 = NULL;
				prop = (_tmp133 = VALA_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) self)), (_tmp133 == NULL) ? NULL : vala_code_node_ref (_tmp133));
				base_property = NULL;
				if (vala_property_get_base_property (prop) != NULL) {
					ValaProperty* _tmp135;
					ValaProperty* _tmp134;
					_tmp135 = NULL;
					_tmp134 = NULL;
					base_property = (_tmp135 = (_tmp134 = vala_property_get_base_property (prop), (_tmp134 == NULL) ? NULL : vala_code_node_ref (_tmp134)), (base_property == NULL) ? NULL : (base_property = (vala_code_node_unref (base_property), NULL)), _tmp135);
				} else {
					if (vala_property_get_base_interface_property (prop) != NULL) {
						ValaProperty* _tmp137;
						ValaProperty* _tmp136;
						_tmp137 = NULL;
						_tmp136 = NULL;
						base_property = (_tmp137 = (_tmp136 = vala_property_get_base_interface_property (prop), (_tmp136 == NULL) ? NULL : vala_code_node_ref (_tmp136)), (base_property == NULL) ? NULL : (base_property = (vala_code_node_unref (base_property), NULL)), _tmp137);
					} else {
						ValaProperty* _tmp139;
						ValaProperty* _tmp138;
						_tmp139 = NULL;
						_tmp138 = NULL;
						base_property = (_tmp139 = (_tmp138 = prop, (_tmp138 == NULL) ? NULL : vala_code_node_ref (_tmp138)), (base_property == NULL) ? NULL : (base_property = (vala_code_node_unref (base_property), NULL)), _tmp139);
					}
				}
				_tmp140 = FALSE;
				if (instance) {
					_tmp140 = vala_symbol_get_parent_symbol ((ValaSymbol*) base_property) != NULL;
				} else {
					_tmp140 = FALSE;
				}
				if (_tmp140) {
					ValaDataType* _tmp141;
					_tmp141 = NULL;
					vala_expression_set_target_type (vala_member_access_get_inner (self), _tmp141 = vala_semantic_analyzer_get_data_type_for_symbol (VALA_TYPESYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) base_property))));
					(_tmp141 == NULL) ? NULL : (_tmp141 = (vala_code_node_unref (_tmp141), NULL));
				}
				(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				(base_property == NULL) ? NULL : (base_property = (vala_code_node_unref (base_property), NULL));
			} else {
				gboolean _tmp142;
				gboolean _tmp143;
				gboolean _tmp144;
				_tmp142 = FALSE;
				_tmp143 = FALSE;
				_tmp144 = FALSE;
				if (VALA_IS_FIELD (vala_expression_get_symbol_reference ((ValaExpression*) self))) {
					_tmp144 = TRUE;
				} else {
					_tmp144 = VALA_IS_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) self));
				}
				if (_tmp144) {
					_tmp143 = instance;
				} else {
					_tmp143 = FALSE;
				}
				if (_tmp143) {
					_tmp142 = vala_symbol_get_parent_symbol (vala_expression_get_symbol_reference ((ValaExpression*) self)) != NULL;
				} else {
					_tmp142 = FALSE;
				}
				if (_tmp142) {
					ValaDataType* _tmp145;
					_tmp145 = NULL;
					vala_expression_set_target_type (vala_member_access_get_inner (self), _tmp145 = vala_semantic_analyzer_get_data_type_for_symbol (VALA_TYPESYMBOL (vala_symbol_get_parent_symbol (vala_expression_get_symbol_reference ((ValaExpression*) self)))));
					(_tmp145 == NULL) ? NULL : (_tmp145 = (vala_code_node_unref (_tmp145), NULL));
				}
			}
		}
	}
	vala_source_file_add_symbol_dependency (vala_semantic_analyzer_get_current_source_file (analyzer), vala_expression_get_symbol_reference ((ValaExpression*) self), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	return (_tmp146 = !vala_code_node_get_error ((ValaCodeNode*) self), (base_symbol == NULL) ? NULL : (base_symbol = (vala_code_node_unref (base_symbol), NULL)), (this_parameter == NULL) ? NULL : (this_parameter = (vala_code_node_unref (this_parameter), NULL)), (member == NULL) ? NULL : (member = (vala_code_node_unref (member), NULL)), _tmp146);
}


static void vala_member_access_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaMemberAccess * self;
	self = (ValaMemberAccess*) base;
	g_return_if_fail (collection != NULL);
	if (vala_member_access_get_inner (self) != NULL) {
		vala_code_node_get_defined_variables ((ValaCodeNode*) vala_member_access_get_inner (self), collection);
	}
}


static void vala_member_access_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaMemberAccess * self;
	ValaLocalVariable* _tmp1;
	ValaSymbol* _tmp0;
	ValaLocalVariable* local;
	self = (ValaMemberAccess*) base;
	g_return_if_fail (collection != NULL);
	if (vala_member_access_get_inner (self) != NULL) {
		vala_code_node_get_used_variables ((ValaCodeNode*) vala_member_access_get_inner (self), collection);
	}
	_tmp1 = NULL;
	_tmp0 = NULL;
	local = (_tmp1 = (_tmp0 = vala_expression_get_symbol_reference ((ValaExpression*) self), VALA_IS_LOCAL_VARIABLE (_tmp0) ? ((ValaLocalVariable*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	if (local != NULL) {
		gee_collection_add (collection, local);
	}
	(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
}


ValaExpression* vala_member_access_get_inner (ValaMemberAccess* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_inner;
}


void vala_member_access_set_inner (ValaMemberAccess* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_inner = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_inner == NULL) ? NULL : (self->priv->_inner = (vala_code_node_unref (self->priv->_inner), NULL)), _tmp2);
	if (self->priv->_inner != NULL) {
		vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_inner, (ValaCodeNode*) self);
	}
}


const char* vala_member_access_get_member_name (ValaMemberAccess* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_member_name;
}


void vala_member_access_set_member_name (ValaMemberAccess* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_member_name = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_member_name = (g_free (self->priv->_member_name), NULL), _tmp2);
}


gboolean vala_member_access_get_pointer_member_access (ValaMemberAccess* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_pointer_member_access;
}


void vala_member_access_set_pointer_member_access (ValaMemberAccess* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_pointer_member_access = value;
}


gboolean vala_member_access_get_prototype_access (ValaMemberAccess* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_prototype_access;
}


void vala_member_access_set_prototype_access (ValaMemberAccess* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_prototype_access = value;
}


gboolean vala_member_access_get_creation_member (ValaMemberAccess* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_creation_member;
}


void vala_member_access_set_creation_member (ValaMemberAccess* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_creation_member = value;
}


gboolean vala_member_access_get_qualified (ValaMemberAccess* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_qualified;
}


void vala_member_access_set_qualified (ValaMemberAccess* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_qualified = value;
}


static void vala_member_access_class_init (ValaMemberAccessClass * klass) {
	vala_member_access_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_member_access_finalize;
	g_type_class_add_private (klass, sizeof (ValaMemberAccessPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_member_access_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_member_access_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->to_string = vala_member_access_real_to_string;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_member_access_real_replace_expression;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_member_access_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_member_access_real_replace_type;
	VALA_EXPRESSION_CLASS (klass)->is_constant = vala_member_access_real_is_constant;
	VALA_EXPRESSION_CLASS (klass)->is_non_null = vala_member_access_real_is_non_null;
	VALA_CODE_NODE_CLASS (klass)->check = vala_member_access_real_check;
	VALA_CODE_NODE_CLASS (klass)->get_defined_variables = vala_member_access_real_get_defined_variables;
	VALA_CODE_NODE_CLASS (klass)->get_used_variables = vala_member_access_real_get_used_variables;
}


static void vala_member_access_instance_init (ValaMemberAccess * self) {
	self->priv = VALA_MEMBER_ACCESS_GET_PRIVATE (self);
	self->priv->type_argument_list = (GeeList*) gee_array_list_new (VALA_TYPE_DATA_TYPE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
}


static void vala_member_access_finalize (ValaCodeNode* obj) {
	ValaMemberAccess * self;
	self = VALA_MEMBER_ACCESS (obj);
	self->priv->_member_name = (g_free (self->priv->_member_name), NULL);
	(self->priv->_inner == NULL) ? NULL : (self->priv->_inner = (vala_code_node_unref (self->priv->_inner), NULL));
	(self->priv->type_argument_list == NULL) ? NULL : (self->priv->type_argument_list = (gee_collection_object_unref (self->priv->type_argument_list), NULL));
	VALA_CODE_NODE_CLASS (vala_member_access_parent_class)->finalize (obj);
}


GType vala_member_access_get_type (void) {
	static GType vala_member_access_type_id = 0;
	if (vala_member_access_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaMemberAccessClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_member_access_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaMemberAccess), 0, (GInstanceInitFunc) vala_member_access_instance_init, NULL };
		vala_member_access_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaMemberAccess", &g_define_type_info, 0);
	}
	return vala_member_access_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




