/* valasignal.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valasignal.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valaformalparameter.h>
#include <vala/valamethod.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valadelegate.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>
#include <vala/valasemanticanalyzer.h>




struct _ValaSignalPrivate {
	gboolean _has_emitter;
	gboolean _is_virtual;
	GeeList* parameters;
	ValaMethod* generated_method;
	char* cname;
	gboolean lock_used;
	ValaDataType* _return_type;
};

#define VALA_SIGNAL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SIGNAL, ValaSignalPrivate))
enum  {
	VALA_SIGNAL_DUMMY_PROPERTY
};
static void vala_signal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_signal_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_signal_real_get_lock_used (ValaLockable* base);
static void vala_signal_real_set_lock_used (ValaLockable* base, gboolean used);
static void vala_signal_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gboolean vala_signal_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_signal_parent_class = NULL;
static ValaLockableIface* vala_signal_vala_lockable_parent_iface = NULL;
static void vala_signal_finalize (ValaCodeNode* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new signal.
 *
 * @param name        signal name
 * @param return_type signal return type
 * @param source      reference to source code
 * @return            newly created signal
 */
ValaSignal* vala_signal_construct (GType object_type, const char* name, ValaDataType* return_type, ValaSourceReference* source_reference) {
	ValaSignal* self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (return_type != NULL, NULL);
	self = (ValaSignal*) vala_member_construct (object_type, name, source_reference);
	vala_signal_set_return_type (self, return_type);
	return self;
}


ValaSignal* vala_signal_new (const char* name, ValaDataType* return_type, ValaSourceReference* source_reference) {
	return vala_signal_construct (VALA_TYPE_SIGNAL, name, return_type, source_reference);
}


/**
 * Appends parameter to signal handler.
 *
 * @param param a formal parameter
 */
void vala_signal_add_parameter (ValaSignal* self, ValaFormalParameter* param) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (param != NULL);
	/* default C parameter position*/
	vala_formal_parameter_set_cparameter_position (param, (double) (gee_collection_get_size ((GeeCollection*) self->priv->parameters) + 1));
	vala_formal_parameter_set_carray_length_parameter_position (param, vala_formal_parameter_get_cparameter_position (param) + 0.1);
	vala_formal_parameter_set_cdelegate_target_parameter_position (param, vala_formal_parameter_get_cparameter_position (param) + 0.1);
	gee_collection_add ((GeeCollection*) self->priv->parameters, param);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) param), (ValaSymbol*) param);
}


GeeList* vala_signal_get_parameters (ValaSignal* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_FORMAL_PARAMETER, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->parameters);
}


/**
 * Returns generated delegate to be used for signal handlers.
 *
 * @return delegate
 */
ValaDelegate* vala_signal_get_delegate (ValaSignal* self, ValaDataType* sender_type, ValaCodeNode* node_reference) {
	ValaDataType* actual_return_type;
	ValaDelegate* generated_delegate;
	ValaDataType* sender_param_type;
	ValaFormalParameter* sender_param;
	ValaDelegate* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (sender_type != NULL, NULL);
	g_return_val_if_fail (node_reference != NULL, NULL);
	actual_return_type = vala_data_type_get_actual_type (vala_signal_get_return_type (self), sender_type, node_reference);
	generated_delegate = vala_delegate_new (NULL, actual_return_type, NULL);
	vala_delegate_set_has_target (generated_delegate, TRUE);
	/* sender parameter is never null and doesn't own its value*/
	sender_param_type = vala_data_type_copy (sender_type);
	vala_data_type_set_value_owned (sender_param_type, FALSE);
	vala_data_type_set_nullable (sender_param_type, FALSE);
	sender_param = vala_formal_parameter_new ("_sender", sender_param_type, NULL);
	vala_delegate_add_parameter (generated_delegate, sender_param);
	{
		GeeIterator* _param_it;
		_param_it = gee_iterable_iterator ((GeeIterable*) self->priv->parameters);
		while (gee_iterator_next (_param_it)) {
			ValaFormalParameter* param;
			ValaFormalParameter* actual_param;
			ValaDataType* _tmp0;
			param = (ValaFormalParameter*) gee_iterator_get (_param_it);
			actual_param = vala_formal_parameter_copy (param);
			_tmp0 = NULL;
			vala_formal_parameter_set_parameter_type (actual_param, _tmp0 = vala_data_type_get_actual_type (vala_formal_parameter_get_parameter_type (actual_param), sender_type, node_reference));
			(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL));
			vala_delegate_add_parameter (generated_delegate, actual_param);
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
			(actual_param == NULL) ? NULL : (actual_param = (vala_code_node_unref (actual_param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), NULL, (ValaSymbol*) generated_delegate);
	_tmp1 = NULL;
	return (_tmp1 = generated_delegate, (actual_return_type == NULL) ? NULL : (actual_return_type = (vala_code_node_unref (actual_return_type), NULL)), (sender_param_type == NULL) ? NULL : (sender_param_type = (vala_code_node_unref (sender_param_type), NULL)), (sender_param == NULL) ? NULL : (sender_param = (vala_code_node_unref (sender_param), NULL)), _tmp1);
}


/**
 * Returns the name of this signal as it is used in C code.
 *
 * @return the name to be used in C code
 */
char* vala_signal_get_cname (ValaSignal* self) {
	const char* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->cname == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->cname = (_tmp0 = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name ((ValaSymbol*) self)), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->cname, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1));
}


void vala_signal_set_cname (ValaSignal* self, const char* cname) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cname != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cname = (_tmp1 = (_tmp0 = cname, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp1);
}


/**
 * Returns the string literal of this signal to be used in C code.
 *
 * @return string literal to be used in C code
 */
ValaCCodeConstant* vala_signal_get_canonical_cconstant (ValaSignal* self, const char* detail) {
	GString* str;
	char* i;
	ValaCCodeConstant* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	str = g_string_new ("\"");
	i = vala_signal_get_cname (self);
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		char* _tmp1;
		const char* _tmp0;
		c = g_utf8_get_char (i);
		if (c == '_') {
			g_string_append_c (str, '-');
		} else {
			g_string_append_unichar (str, c);
		}
		_tmp1 = NULL;
		_tmp0 = NULL;
		i = (_tmp1 = (_tmp0 = g_utf8_next_char (i), (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), i = (g_free (i), NULL), _tmp1);
	}
	if (detail != NULL) {
		g_string_append (str, "::");
		g_string_append (str, detail);
	}
	g_string_append_c (str, '"');
	_tmp2 = NULL;
	return (_tmp2 = vala_ccode_constant_new (str->str), (str == NULL) ? NULL : (str = (g_string_free (str, TRUE), NULL)), i = (g_free (i), NULL), _tmp2);
}


static void vala_signal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaSignal * self;
	self = (ValaSignal*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_member (visitor, (ValaMember*) self);
	vala_code_visitor_visit_signal (visitor, self);
}


static void vala_signal_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaSignal * self;
	self = (ValaSignal*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_signal_get_return_type (self), visitor);
	{
		GeeIterator* _param_it;
		_param_it = gee_iterable_iterator ((GeeIterable*) self->priv->parameters);
		while (gee_iterator_next (_param_it)) {
			ValaFormalParameter* param;
			param = (ValaFormalParameter*) gee_iterator_get (_param_it);
			vala_code_node_accept ((ValaCodeNode*) param, visitor);
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
}


/**
 * Process all associated attributes.
 */
void vala_signal_process_attributes (ValaSignal* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = (ValaAttribute*) a_it->data, (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "HasEmitter") == 0) {
					vala_signal_set_has_emitter (self, TRUE);
				}
				(a == NULL) ? NULL : (a = (vala_code_node_unref (a), NULL));
			}
		}
	}
}


static gboolean vala_signal_real_get_lock_used (ValaLockable* base) {
	ValaSignal * self;
	self = (ValaSignal*) base;
	return self->priv->lock_used;
}


static void vala_signal_real_set_lock_used (ValaLockable* base, gboolean used) {
	ValaSignal * self;
	self = (ValaSignal*) base;
	self->priv->lock_used = used;
}


static void vala_signal_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaSignal * self;
	self = (ValaSignal*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	if (vala_signal_get_return_type (self) == old_type) {
		vala_signal_set_return_type (self, new_type);
	}
}


ValaMethod* vala_signal_get_method_handler (ValaSignal* self) {
	ValaMethod* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	g_assert (self->priv->_is_virtual);
	if (self->priv->generated_method == NULL) {
		ValaMethod* _tmp0;
		_tmp0 = NULL;
		self->priv->generated_method = (_tmp0 = vala_method_new (vala_symbol_get_name ((ValaSymbol*) self), vala_signal_get_return_type (self), vala_code_node_get_source_reference ((ValaCodeNode*) self)), (self->priv->generated_method == NULL) ? NULL : (self->priv->generated_method = (vala_code_node_unref (self->priv->generated_method), NULL)), _tmp0);
		vala_method_set_is_virtual (self->priv->generated_method, TRUE);
		vala_method_set_vfunc_name (self->priv->generated_method, vala_symbol_get_name ((ValaSymbol*) self));
		{
			GeeIterator* _param_it;
			_param_it = gee_iterable_iterator ((GeeIterable*) self->priv->parameters);
			while (gee_iterator_next (_param_it)) {
				ValaFormalParameter* param;
				param = (ValaFormalParameter*) gee_iterator_get (_param_it);
				vala_method_add_parameter (self->priv->generated_method, param);
				(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
			}
			(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
		}
		vala_scope_add (vala_symbol_get_scope (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), NULL, (ValaSymbol*) self->priv->generated_method);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->generated_method, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
}


static gboolean vala_signal_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaSignal * self;
	self = (ValaSignal*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_signal_process_attributes (self);
	vala_code_node_check ((ValaCodeNode*) vala_signal_get_return_type (self), analyzer);
	{
		GeeIterator* _param_it;
		_param_it = gee_iterable_iterator ((GeeIterable*) self->priv->parameters);
		while (gee_iterator_next (_param_it)) {
			ValaFormalParameter* param;
			param = (ValaFormalParameter*) gee_iterator_get (_param_it);
			vala_code_node_check ((ValaCodeNode*) param, analyzer);
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


ValaDataType* vala_signal_get_return_type (ValaSignal* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_return_type;
}


void vala_signal_set_return_type (ValaSignal* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_return_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_return_type == NULL) ? NULL : (self->priv->_return_type = (vala_code_node_unref (self->priv->_return_type), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_return_type, (ValaCodeNode*) self);
}


gboolean vala_signal_get_has_emitter (ValaSignal* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_has_emitter;
}


void vala_signal_set_has_emitter (ValaSignal* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_has_emitter = value;
}


gboolean vala_signal_get_is_virtual (ValaSignal* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_virtual;
}


void vala_signal_set_is_virtual (ValaSignal* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_virtual = value;
}


static void vala_signal_class_init (ValaSignalClass * klass) {
	vala_signal_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_signal_finalize;
	g_type_class_add_private (klass, sizeof (ValaSignalPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_signal_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_signal_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_signal_real_replace_type;
	VALA_CODE_NODE_CLASS (klass)->check = vala_signal_real_check;
}


static void vala_signal_vala_lockable_interface_init (ValaLockableIface * iface) {
	vala_signal_vala_lockable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_lock_used = vala_signal_real_get_lock_used;
	iface->set_lock_used = vala_signal_real_set_lock_used;
}


static void vala_signal_instance_init (ValaSignal * self) {
	self->priv = VALA_SIGNAL_GET_PRIVATE (self);
	self->priv->parameters = (GeeList*) gee_array_list_new (VALA_TYPE_FORMAL_PARAMETER, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->lock_used = FALSE;
}


static void vala_signal_finalize (ValaCodeNode* obj) {
	ValaSignal * self;
	self = VALA_SIGNAL (obj);
	(self->priv->parameters == NULL) ? NULL : (self->priv->parameters = (gee_collection_object_unref (self->priv->parameters), NULL));
	(self->priv->generated_method == NULL) ? NULL : (self->priv->generated_method = (vala_code_node_unref (self->priv->generated_method), NULL));
	self->priv->cname = (g_free (self->priv->cname), NULL);
	(self->priv->_return_type == NULL) ? NULL : (self->priv->_return_type = (vala_code_node_unref (self->priv->_return_type), NULL));
	VALA_CODE_NODE_CLASS (vala_signal_parent_class)->finalize (obj);
}


GType vala_signal_get_type (void) {
	static GType vala_signal_type_id = 0;
	if (vala_signal_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSignalClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_signal_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSignal), 0, (GInstanceInitFunc) vala_signal_instance_init, NULL };
		static const GInterfaceInfo vala_lockable_info = { (GInterfaceInitFunc) vala_signal_vala_lockable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_signal_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaSignal", &g_define_type_info, 0);
		g_type_add_interface_static (vala_signal_type_id, VALA_TYPE_LOCKABLE, &vala_lockable_info);
	}
	return vala_signal_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




