/* valasymbol.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valasymbol.h>
#include <gee/arraylist.h>
#include <vala/valascope.h>
#include <vala/valasourcereference.h>
#include <vala/valasourcefile.h>
#include <vala/valanamespace.h>




struct _ValaSymbolPrivate {
	char* _name;
	gboolean _active;
	ValaSymbolAccessibility _access;
	gboolean _external;
	ValaScope* _owner;
	ValaScope* _scope;
};

#define VALA_SYMBOL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SYMBOL, ValaSymbolPrivate))
enum  {
	VALA_SYMBOL_DUMMY_PROPERTY
};
static char* vala_symbol_real_get_cprefix (ValaSymbol* self);
static char* vala_symbol_real_get_lower_case_cname (ValaSymbol* self, const char* infix);
static char* vala_symbol_real_get_lower_case_cprefix (ValaSymbol* self);
static GeeList* vala_symbol_real_get_cheader_filenames (ValaSymbol* self);
static gpointer vala_symbol_parent_class = NULL;
static void vala_symbol_finalize (ValaCodeNode* obj);



/**
 * Check if this symbol is just internal API (and therefore doesn't need 
 * to be listed in header files for instance) by traversing parent symbols
 * and checking their accessibility.
 */
gboolean vala_symbol_is_internal_symbol (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	if (!self->priv->_external && vala_symbol_get_external_package (self)) {
		/* non-external symbols in VAPI files are internal symbols*/
		return TRUE;
	}
	{
		ValaSymbol* _tmp1;
		ValaSymbol* sym;
		ValaSymbol* _tmp3;
		ValaSymbol* _tmp2;
		_tmp1 = NULL;
		sym = (_tmp1 = self, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1)));
		_tmp3 = NULL;
		_tmp2 = NULL;
		for (; NULL != sym; sym = (_tmp3 = (_tmp2 = vala_symbol_get_parent_symbol (sym), (_tmp2 == NULL ? NULL : vala_code_node_ref (_tmp2))), (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp3)) {
			if (VALA_SYMBOL_ACCESSIBILITY_PRIVATE == sym->priv->_access) {
				gboolean _tmp4;
				return (_tmp4 = TRUE, (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp4);
			}
		}
		(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
	}
	return FALSE;
}


ValaSymbol* vala_symbol_construct (GType object_type, const char* name, ValaSourceReference* source_reference) {
	ValaSymbol* self;
	ValaScope* _tmp0;
	self = ((ValaSymbol*) (g_type_create_instance (object_type)));
	vala_symbol_set_name (self, name);
	vala_code_node_set_source_reference (((ValaCodeNode*) (self)), source_reference);
	_tmp0 = NULL;
	self->priv->_scope = (_tmp0 = vala_scope_new (self), (self->priv->_scope == NULL ? NULL : (self->priv->_scope = (vala_scope_unref (self->priv->_scope), NULL))), _tmp0);
	return self;
}


ValaSymbol* vala_symbol_new (const char* name, ValaSourceReference* source_reference) {
	return vala_symbol_construct (VALA_TYPE_SYMBOL, name, source_reference);
}


/**
 * Returns the fully expanded name of this symbol for use in
 * human-readable messages.
 *
 * @return full name
 */
char* vala_symbol_get_full_name (ValaSymbol* self) {
	char* _tmp3;
	gboolean _tmp4;
	char* _tmp7;
	char* _tmp8;
	g_return_val_if_fail (self != NULL, NULL);
	if (vala_symbol_get_parent_symbol (self) == NULL) {
		const char* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = self->priv->_name, (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
	}
	if (self->priv->_name == NULL) {
		return vala_symbol_get_full_name (vala_symbol_get_parent_symbol (self));
	}
	_tmp3 = NULL;
	if ((_tmp4 = (_tmp3 = vala_symbol_get_full_name (vala_symbol_get_parent_symbol (self))) == NULL, (_tmp3 = (g_free (_tmp3), NULL)), _tmp4)) {
		const char* _tmp5;
		_tmp5 = NULL;
		return (_tmp5 = self->priv->_name, (_tmp5 == NULL ? NULL : g_strdup (_tmp5)));
	}
	_tmp7 = NULL;
	_tmp8 = NULL;
	return (_tmp8 = g_strdup_printf ("%s.%s", (_tmp7 = vala_symbol_get_full_name (vala_symbol_get_parent_symbol (self))), self->priv->_name), (_tmp7 = (g_free (_tmp7), NULL)), _tmp8);
}


/**
 * Returns the camel case string to be prepended to the name of members
 * of this symbol when used in C code.
 *
 * @return the camel case prefix to be used in C code
 */
static char* vala_symbol_real_get_cprefix (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_name == NULL) {
		return g_strdup ("");
	} else {
		const char* _tmp1;
		_tmp1 = NULL;
		return (_tmp1 = self->priv->_name, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
	}
}


char* vala_symbol_get_cprefix (ValaSymbol* self) {
	return VALA_SYMBOL_GET_CLASS (self)->get_cprefix (self);
}


/**
 * Returns the C name of this symbol in lower case. Words are
 * separated by underscores. The lower case C name of the parent symbol
 * is prefix of the result, if there is one.
 *
 * @param infix a string to be placed between namespace and data type
 *              name or null
 * @return      the lower case name to be used in C code
 */
static char* vala_symbol_real_get_lower_case_cname (ValaSymbol* self, const char* infix) {
	g_return_val_if_fail (self != NULL, NULL);
	return NULL;
}


char* vala_symbol_get_lower_case_cname (ValaSymbol* self, const char* infix) {
	return VALA_SYMBOL_GET_CLASS (self)->get_lower_case_cname (self, infix);
}


/**
 * Returns the string to be prefixed to members of this symbol in
 * lower case when used in C code.
 *
 * @return      the lower case prefix to be used in C code
 */
static char* vala_symbol_real_get_lower_case_cprefix (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return g_strdup ("");
}


char* vala_symbol_get_lower_case_cprefix (ValaSymbol* self) {
	return VALA_SYMBOL_GET_CLASS (self)->get_lower_case_cprefix (self);
}


/**
 * Returns a list of C header filenames users of this symbol must
 * include.
 *
 * @return list of C header filenames for this symbol
 */
static GeeList* vala_symbol_real_get_cheader_filenames (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return ((GeeList*) (gee_array_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, g_direct_equal)));
}


GeeList* vala_symbol_get_cheader_filenames (ValaSymbol* self) {
	return VALA_SYMBOL_GET_CLASS (self)->get_cheader_filenames (self);
}


/**
 * Converts a string from CamelCase to lower_case.
 *
 * @param camel_case a string in camel case
 * @return           the specified string converted to lower case
 */
char* vala_symbol_camel_case_to_lower_case (const char* camel_case) {
	GString* result;
	const char* i;
	gboolean first;
	const char* _tmp0;
	char* _tmp1;
	g_return_val_if_fail (camel_case != NULL, NULL);
	result = g_string_new ("");
	i = camel_case;
	first = TRUE;
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		c = g_utf8_get_char (i);
		if (g_unichar_isupper (c) && !first) {
			const char* t;
			gboolean prev_upper;
			gboolean next_upper;
			/* current character is upper case and
			 * we're not at the beginning */
			t = g_utf8_prev_char (i);
			prev_upper = g_unichar_isupper (g_utf8_get_char (t));
			t = g_utf8_next_char (i);
			next_upper = g_unichar_isupper (g_utf8_get_char (t));
			if (!prev_upper || (g_utf8_strlen (i, -1) >= 2 && !next_upper)) {
				glong len;
				/* previous character wasn't upper case or
				 * next character isn't upper case*/
				len = g_utf8_strlen (result->str, -1);
				if (len != 1 && g_utf8_get_char (g_utf8_offset_to_pointer (result->str, len - 2)) != '_') {
					/* we're not creating 1 character words */
					g_string_append_c (result, '_');
				}
			}
		}
		g_string_append_unichar (result, g_unichar_tolower (c));
		first = FALSE;
		i = g_utf8_next_char (i);
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = (_tmp0 = result->str, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (result == NULL ? NULL : (result = (g_string_free (result, TRUE), NULL))), _tmp1);
}


/**
 * Converts a string from lower_case to CamelCase.
 *
 * @param lower_case a string in lower case
 * @return           the specified string converted to camel case
 */
char* vala_symbol_lower_case_to_camel_case (const char* lower_case) {
	GString* result;
	const char* i;
	gboolean last_underscore;
	const char* _tmp2;
	char* _tmp3;
	g_return_val_if_fail (lower_case != NULL, NULL);
	result = g_string_new ("");
	i = lower_case;
	last_underscore = TRUE;
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		c = g_utf8_get_char (i);
		if (c == '_') {
			last_underscore = TRUE;
		} else {
			if (g_unichar_isupper (c)) {
				const char* _tmp0;
				char* _tmp1;
				/* original string is not lower_case, don't apply transformation*/
				_tmp0 = NULL;
				_tmp1 = NULL;
				return (_tmp1 = (_tmp0 = lower_case, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (result == NULL ? NULL : (result = (g_string_free (result, TRUE), NULL))), _tmp1);
			} else {
				if (last_underscore) {
					g_string_append_unichar (result, g_unichar_toupper (c));
					last_underscore = FALSE;
				} else {
					g_string_append_unichar (result, c);
				}
			}
		}
		i = g_utf8_next_char (i);
	}
	_tmp2 = NULL;
	_tmp3 = NULL;
	return (_tmp3 = (_tmp2 = result->str, (_tmp2 == NULL ? NULL : g_strdup (_tmp2))), (result == NULL ? NULL : (result = (g_string_free (result, TRUE), NULL))), _tmp3);
}


/* get the top scope from where this symbol is still accessible*/
ValaScope* vala_symbol_get_top_accessible_scope (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_access != VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
		ValaScope* _tmp0;
		ValaScope* scope;
		/* private symbols are accessible within the scope where the symbol has been declared*/
		_tmp0 = NULL;
		scope = (_tmp0 = vala_symbol_get_owner (self), (_tmp0 == NULL ? NULL : vala_scope_ref (_tmp0)));
		while (scope != NULL && VALA_IS_NAMESPACE (vala_scope_get_owner (scope)) && vala_scope_get_owner (scope)->priv->_name != NULL) {
			ValaScope* _tmp2;
			ValaScope* _tmp1;
			_tmp2 = NULL;
			_tmp1 = NULL;
			scope = (_tmp2 = (_tmp1 = vala_scope_get_parent_scope (scope), (_tmp1 == NULL ? NULL : vala_scope_ref (_tmp1))), (scope == NULL ? NULL : (scope = (vala_scope_unref (scope), NULL))), _tmp2);
		}
		return scope;
	}
	if (vala_symbol_get_parent_symbol (self) == NULL) {
		return NULL;
	}
	/* if this is a public symbol, it's equally accessible as the parent symbol*/
	return vala_symbol_get_top_accessible_scope (vala_symbol_get_parent_symbol (self));
}


ValaSymbol* vala_symbol_get_parent_symbol (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (vala_symbol_get_owner (self) == NULL) {
		return NULL;
	} else {
		return vala_scope_get_owner (vala_symbol_get_owner (self));
	}
}


ValaScope* vala_symbol_get_owner (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_owner;
}


void vala_symbol_set_owner (ValaSymbol* self, ValaScope* value) {
	g_return_if_fail (self != NULL);
	self->priv->_owner = value;
	vala_scope_set_parent_scope (self->priv->_scope, value);
}


const char* vala_symbol_get_name (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_name;
}


void vala_symbol_set_name (ValaSymbol* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_name = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_name = (g_free (self->priv->_name), NULL)), _tmp2);
}


gboolean vala_symbol_get_active (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_active;
}


void vala_symbol_set_active (ValaSymbol* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_active = value;
}


ValaSymbolAccessibility vala_symbol_get_access (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_access;
}


void vala_symbol_set_access (ValaSymbol* self, ValaSymbolAccessibility value) {
	g_return_if_fail (self != NULL);
	self->priv->_access = value;
}


ValaScope* vala_symbol_get_scope (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_scope;
}


gboolean vala_symbol_get_external (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_external;
}


void vala_symbol_set_external (ValaSymbol* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_external = value;
}


gboolean vala_symbol_get_external_package (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return (vala_code_node_get_source_reference (((ValaCodeNode*) (self))) != NULL && vala_source_file_get_external_package (vala_source_reference_get_file (vala_code_node_get_source_reference (((ValaCodeNode*) (self))))));
}


static void vala_symbol_class_init (ValaSymbolClass * klass) {
	vala_symbol_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_symbol_finalize;
	g_type_class_add_private (klass, sizeof (ValaSymbolPrivate));
	VALA_SYMBOL_CLASS (klass)->get_cprefix = vala_symbol_real_get_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cname = vala_symbol_real_get_lower_case_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_symbol_real_get_lower_case_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_cheader_filenames = vala_symbol_real_get_cheader_filenames;
}


static void vala_symbol_instance_init (ValaSymbol * self) {
	self->priv = VALA_SYMBOL_GET_PRIVATE (self);
	self->priv->_active = TRUE;
}


static void vala_symbol_finalize (ValaCodeNode* obj) {
	ValaSymbol * self;
	self = VALA_SYMBOL (obj);
	self->priv->_name = (g_free (self->priv->_name), NULL);
	(self->priv->_scope == NULL ? NULL : (self->priv->_scope = (vala_scope_unref (self->priv->_scope), NULL)));
	VALA_CODE_NODE_CLASS (vala_symbol_parent_class)->finalize (obj);
}


GType vala_symbol_get_type (void) {
	static GType vala_symbol_type_id = 0;
	if (vala_symbol_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSymbolClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_symbol_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSymbol), 0, (GInstanceInitFunc) vala_symbol_instance_init, NULL };
		vala_symbol_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaSymbol", &g_define_type_info, G_TYPE_FLAG_ABSTRACT);
	}
	return vala_symbol_type_id;
}



GType vala_symbol_accessibility_get_type (void) {
	static GType vala_symbol_accessibility_type_id = 0;
	if (G_UNLIKELY (vala_symbol_accessibility_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_SYMBOL_ACCESSIBILITY_PRIVATE, "VALA_SYMBOL_ACCESSIBILITY_PRIVATE", "private"}, {VALA_SYMBOL_ACCESSIBILITY_INTERNAL, "VALA_SYMBOL_ACCESSIBILITY_INTERNAL", "internal"}, {VALA_SYMBOL_ACCESSIBILITY_PROTECTED, "VALA_SYMBOL_ACCESSIBILITY_PROTECTED", "protected"}, {VALA_SYMBOL_ACCESSIBILITY_PUBLIC, "VALA_SYMBOL_ACCESSIBILITY_PUBLIC", "public"}, {0, NULL, NULL}};
		vala_symbol_accessibility_type_id = g_enum_register_static ("ValaSymbolAccessibility", values);
	}
	return vala_symbol_accessibility_type_id;
}




