/* valainvocationexpression.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valainvocationexpression.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>




struct _ValaInvocationExpressionPrivate {
	ValaCCodeExpression* _delegate_target;
	GeeList* argument_list;
	GeeList* array_sizes;
};

#define VALA_INVOCATION_EXPRESSION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_INVOCATION_EXPRESSION, ValaInvocationExpressionPrivate))
enum  {
	VALA_INVOCATION_EXPRESSION_DUMMY_PROPERTY
};
static void vala_invocation_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_invocation_expression_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_invocation_expression_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gboolean vala_invocation_expression_real_is_pure (ValaExpression* base);
static gpointer vala_invocation_expression_parent_class = NULL;
static void vala_invocation_expression_finalize (ValaCodeNode* obj);



/**
 * Creates a new invocation expression.
 *
 * @param call             method to call
 * @param source_reference reference to source code
 * @return                 newly created invocation expression
 */
ValaInvocationExpression* vala_invocation_expression_construct (GType object_type, ValaExpression* call, ValaSourceReference* source_reference) {
	ValaInvocationExpression* self;
	g_return_val_if_fail (call != NULL, NULL);
	self = ((ValaInvocationExpression*) (g_type_create_instance (object_type)));
	vala_code_node_set_source_reference (((ValaCodeNode*) (self)), source_reference);
	vala_invocation_expression_set_call (self, call);
	return self;
}


ValaInvocationExpression* vala_invocation_expression_new (ValaExpression* call, ValaSourceReference* source_reference) {
	return vala_invocation_expression_construct (VALA_TYPE_INVOCATION_EXPRESSION, call, source_reference);
}


/**
 * Appends the specified expression to the list of arguments.
 *
 * @param arg an argument
 */
void vala_invocation_expression_add_argument (ValaInvocationExpression* self, ValaExpression* arg) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (arg != NULL);
	gee_collection_add (((GeeCollection*) (self->priv->argument_list)), arg);
	vala_code_node_set_parent_node (((ValaCodeNode*) (arg)), ((ValaCodeNode*) (self)));
}


/**
 * Returns a copy of the argument list.
 *
 * @return argument list
 */
GeeList* vala_invocation_expression_get_argument_list (ValaInvocationExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return ((GeeList*) (gee_read_only_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->argument_list)));
}


/**
 * Add an array size C code expression.
 */
void vala_invocation_expression_append_array_size (ValaInvocationExpression* self, ValaCCodeExpression* size) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (size != NULL);
	gee_collection_add (((GeeCollection*) (self->priv->array_sizes)), size);
}


/**
 * Get the C code expression for array sizes for all dimensions
 * ascending from left to right.
 */
GeeList* vala_invocation_expression_get_array_sizes (ValaInvocationExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return ((GeeList*) (gee_read_only_list_new (VALA_TYPE_CCODE_EXPRESSION, ((GBoxedCopyFunc) (vala_ccode_node_ref)), vala_ccode_node_unref, self->priv->array_sizes)));
}


static void vala_invocation_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaInvocationExpression * self;
	self = ((ValaInvocationExpression*) (base));
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_invocation_expression (visitor, self);
	vala_code_visitor_visit_expression (visitor, ((ValaExpression*) (self)));
}


static void vala_invocation_expression_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaInvocationExpression * self;
	self = ((ValaInvocationExpression*) (base));
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept (((ValaCodeNode*) (vala_invocation_expression_get_call (self))), visitor);
	{
		GeeList* expr_collection;
		int expr_it;
		expr_collection = self->priv->argument_list;
		for (expr_it = 0; expr_it < gee_collection_get_size (GEE_COLLECTION (expr_collection)); expr_it = expr_it + 1) {
			ValaExpression* expr;
			expr = ((ValaExpression*) (gee_list_get (GEE_LIST (expr_collection), expr_it)));
			{
				vala_code_node_accept (((ValaCodeNode*) (expr)), visitor);
				(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
			}
		}
	}
}


static void vala_invocation_expression_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaInvocationExpression * self;
	gint index;
	self = ((ValaInvocationExpression*) (base));
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_invocation_expression_get_call (self) == old_node) {
		vala_invocation_expression_set_call (self, new_node);
	}
	index = gee_list_index_of (self->priv->argument_list, old_node);
	if (index >= 0 && vala_code_node_get_parent_node (((ValaCodeNode*) (new_node))) == NULL) {
		gee_list_set (((GeeList*) (self->priv->argument_list)), index, new_node);
		vala_code_node_set_parent_node (((ValaCodeNode*) (new_node)), ((ValaCodeNode*) (self)));
	}
}


static gboolean vala_invocation_expression_real_is_pure (ValaExpression* base) {
	ValaInvocationExpression * self;
	self = ((ValaInvocationExpression*) (base));
	return FALSE;
}


ValaExpression* vala_invocation_expression_get_call (ValaInvocationExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->_call;
}


void vala_invocation_expression_set_call (ValaInvocationExpression* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->_call = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->_call == NULL ? NULL : (self->_call = (vala_code_node_unref (self->_call), NULL))), _tmp2);
	vala_code_node_set_parent_node (((ValaCodeNode*) (self->_call)), ((ValaCodeNode*) (self)));
}


ValaCCodeExpression* vala_invocation_expression_get_delegate_target (ValaInvocationExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_delegate_target;
}


void vala_invocation_expression_set_delegate_target (ValaInvocationExpression* self, ValaCCodeExpression* value) {
	ValaCCodeExpression* _tmp2;
	ValaCCodeExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_delegate_target = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_ccode_node_ref (_tmp1))), (self->priv->_delegate_target == NULL ? NULL : (self->priv->_delegate_target = (vala_ccode_node_unref (self->priv->_delegate_target), NULL))), _tmp2);
}


static void vala_invocation_expression_class_init (ValaInvocationExpressionClass * klass) {
	vala_invocation_expression_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_invocation_expression_finalize;
	g_type_class_add_private (klass, sizeof (ValaInvocationExpressionPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_invocation_expression_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_invocation_expression_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_invocation_expression_real_replace_expression;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_invocation_expression_real_is_pure;
}


static void vala_invocation_expression_instance_init (ValaInvocationExpression * self) {
	self->priv = VALA_INVOCATION_EXPRESSION_GET_PRIVATE (self);
	self->priv->argument_list = ((GeeList*) (gee_array_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal)));
	self->priv->array_sizes = ((GeeList*) (gee_array_list_new (VALA_TYPE_CCODE_EXPRESSION, ((GBoxedCopyFunc) (vala_ccode_node_ref)), vala_ccode_node_unref, g_direct_equal)));
}


static void vala_invocation_expression_finalize (ValaCodeNode* obj) {
	ValaInvocationExpression * self;
	self = VALA_INVOCATION_EXPRESSION (obj);
	(self->priv->_delegate_target == NULL ? NULL : (self->priv->_delegate_target = (vala_ccode_node_unref (self->priv->_delegate_target), NULL)));
	(self->_call == NULL ? NULL : (self->_call = (vala_code_node_unref (self->_call), NULL)));
	(self->priv->argument_list == NULL ? NULL : (self->priv->argument_list = (gee_collection_object_unref (self->priv->argument_list), NULL)));
	(self->priv->array_sizes == NULL ? NULL : (self->priv->array_sizes = (gee_collection_object_unref (self->priv->array_sizes), NULL)));
	VALA_CODE_NODE_CLASS (vala_invocation_expression_parent_class)->finalize (obj);
}


GType vala_invocation_expression_get_type (void) {
	static GType vala_invocation_expression_type_id = 0;
	if (vala_invocation_expression_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaInvocationExpressionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_invocation_expression_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaInvocationExpression), 0, (GInstanceInitFunc) vala_invocation_expression_instance_init, NULL };
		vala_invocation_expression_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaInvocationExpression", &g_define_type_info, 0);
	}
	return vala_invocation_expression_type_id;
}




