/* valascanner.vala
 *
 * Copyright (C) 2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valascanner.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>




struct _ValaScannerPrivate {
	ValaSourceFile* _source_file;
	gchar* current;
	gchar* end;
	gint line;
	gint column;
	char* _comment;
};

#define VALA_SCANNER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SCANNER, ValaScannerPrivate))
enum  {
	VALA_SCANNER_DUMMY_PROPERTY,
	VALA_SCANNER_SOURCE_FILE
};
static gboolean vala_scanner_is_ident_char (ValaScanner* self, gchar c);
static ValaTokenType vala_scanner_get_identifier_or_keyword (ValaScanner* self, gchar* begin, gint len);
static gboolean vala_scanner_matches (ValaScanner* self, gchar* begin, const char* keyword);
static gboolean vala_scanner_whitespace (ValaScanner* self);
static gboolean vala_scanner_comment (ValaScanner* self);
static void vala_scanner_space (ValaScanner* self);
static void vala_scanner_push_comment (ValaScanner* self, const char* comment_item, gboolean file_comment);
static void vala_scanner_set_source_file (ValaScanner* self, ValaSourceFile* value);
static GObject * vala_scanner_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer vala_scanner_parent_class = NULL;
static void vala_scanner_finalize (GObject * obj);



ValaScanner* vala_scanner_new (ValaSourceFile* source_file) {
	GParameter * __params;
	GParameter * __params_it;
	ValaScanner * self;
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (source_file), NULL);
	__params = g_new0 (GParameter, 1);
	__params_it = __params;
	__params_it->name = "source-file";
	g_value_init (&__params_it->value, VALA_TYPE_SOURCE_FILE);
	g_value_set_object (&__params_it->value, source_file);
	__params_it++;
	self = g_object_newv (VALA_TYPE_SCANNER, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


static gboolean vala_scanner_is_ident_char (ValaScanner* self, gchar c) {
	g_return_val_if_fail (VALA_IS_SCANNER (self), FALSE);
	return (g_ascii_isalnum (c) || c == '_');
}


static ValaTokenType vala_scanner_get_identifier_or_keyword (ValaScanner* self, gchar* begin, gint len) {
	gint _tmp80;
	g_return_val_if_fail (VALA_IS_SCANNER (self), 0);
	_tmp80 = len;
	if (_tmp80 == 2)
	do {
		gchar _tmp6;
		_tmp6 = begin[0];
		if (_tmp6 == 'a')
		do {
			if (vala_scanner_matches (self, begin, "as")) {
				return VALA_TOKEN_TYPE_AS;
			}
			break;
		} while (0); else if (_tmp6 == 'd')
		do {
			if (vala_scanner_matches (self, begin, "do")) {
				return VALA_TOKEN_TYPE_DO;
			}
			break;
		} while (0); else if (_tmp6 == 'i')
		do {
			gchar _tmp5;
			_tmp5 = begin[1];
			if (_tmp5 == 'f')
			do {
				return VALA_TOKEN_TYPE_IF;
			} while (0); else if (_tmp5 == 'n')
			do {
				return VALA_TOKEN_TYPE_IN;
			} while (0); else if (_tmp5 == 's')
			do {
				return VALA_TOKEN_TYPE_IS;
			} while (0);
			break;
		} while (0);
		break;
	} while (0); else if (_tmp80 == 3)
	do {
		gchar _tmp15;
		_tmp15 = begin[0];
		if (_tmp15 == 'f')
		do {
			if (vala_scanner_matches (self, begin, "for")) {
				return VALA_TOKEN_TYPE_FOR;
			}
			break;
		} while (0); else if (_tmp15 == 'g')
		do {
			if (vala_scanner_matches (self, begin, "get")) {
				return VALA_TOKEN_TYPE_GET;
			}
			break;
		} while (0); else if (_tmp15 == 'n')
		do {
			if (vala_scanner_matches (self, begin, "new")) {
				return VALA_TOKEN_TYPE_NEW;
			}
			break;
		} while (0); else if (_tmp15 == 'o')
		do {
			if (vala_scanner_matches (self, begin, "out")) {
				return VALA_TOKEN_TYPE_OUT;
			}
			break;
		} while (0); else if (_tmp15 == 'r')
		do {
			if (vala_scanner_matches (self, begin, "ref")) {
				return VALA_TOKEN_TYPE_REF;
			}
			break;
		} while (0); else if (_tmp15 == 's')
		do {
			if (vala_scanner_matches (self, begin, "set")) {
				return VALA_TOKEN_TYPE_SET;
			}
			break;
		} while (0); else if (_tmp15 == 't')
		do {
			if (vala_scanner_matches (self, begin, "try")) {
				return VALA_TOKEN_TYPE_TRY;
			}
			break;
		} while (0); else if (_tmp15 == 'v')
		do {
			if (vala_scanner_matches (self, begin, "var")) {
				return VALA_TOKEN_TYPE_VAR;
			}
			break;
		} while (0);
		break;
	} while (0); else if (_tmp80 == 4)
	do {
		gchar _tmp28;
		_tmp28 = begin[0];
		if (_tmp28 == 'b')
		do {
			if (vala_scanner_matches (self, begin, "base")) {
				return VALA_TOKEN_TYPE_BASE;
			}
			break;
		} while (0); else if (_tmp28 == 'c')
		do {
			if (vala_scanner_matches (self, begin, "case")) {
				return VALA_TOKEN_TYPE_CASE;
			}
			break;
		} while (0); else if (_tmp28 == 'e')
		do {
			gchar _tmp20;
			_tmp20 = begin[1];
			if (_tmp20 == 'l')
			do {
				if (vala_scanner_matches (self, begin, "else")) {
					return VALA_TOKEN_TYPE_ELSE;
				}
				break;
			} while (0); else if (_tmp20 == 'n')
			do {
				if (vala_scanner_matches (self, begin, "enum")) {
					return VALA_TOKEN_TYPE_ENUM;
				}
				break;
			} while (0);
			break;
		} while (0); else if (_tmp28 == 'l')
		do {
			if (vala_scanner_matches (self, begin, "lock")) {
				return VALA_TOKEN_TYPE_LOCK;
			}
			break;
		} while (0); else if (_tmp28 == 'n')
		do {
			if (vala_scanner_matches (self, begin, "null")) {
				return VALA_TOKEN_TYPE_NULL;
			}
			break;
		} while (0); else if (_tmp28 == 't')
		do {
			gchar _tmp25;
			_tmp25 = begin[1];
			if (_tmp25 == 'h')
			do {
				if (vala_scanner_matches (self, begin, "this")) {
					return VALA_TOKEN_TYPE_THIS;
				}
				break;
			} while (0); else if (_tmp25 == 'r')
			do {
				if (vala_scanner_matches (self, begin, "true")) {
					return VALA_TOKEN_TYPE_TRUE;
				}
				break;
			} while (0);
			break;
		} while (0); else if (_tmp28 == 'v')
		do {
			if (vala_scanner_matches (self, begin, "void")) {
				return VALA_TOKEN_TYPE_VOID;
			}
			break;
		} while (0); else if (_tmp28 == 'w')
		do {
			if (vala_scanner_matches (self, begin, "weak")) {
				return VALA_TOKEN_TYPE_WEAK;
			}
			break;
		} while (0);
		break;
	} while (0); else if (_tmp80 == 5)
	do {
		gchar _tmp38;
		_tmp38 = begin[0];
		if (_tmp38 == 'b')
		do {
			if (vala_scanner_matches (self, begin, "break")) {
				return VALA_TOKEN_TYPE_BREAK;
			}
			break;
		} while (0); else if (_tmp38 == 'c')
		do {
			gchar _tmp33;
			_tmp33 = begin[1];
			if (_tmp33 == 'a')
			do {
				if (vala_scanner_matches (self, begin, "catch")) {
					return VALA_TOKEN_TYPE_CATCH;
				}
				break;
			} while (0); else if (_tmp33 == 'l')
			do {
				if (vala_scanner_matches (self, begin, "class")) {
					return VALA_TOKEN_TYPE_CLASS;
				}
				break;
			} while (0); else if (_tmp33 == 'o')
			do {
				if (vala_scanner_matches (self, begin, "const")) {
					return VALA_TOKEN_TYPE_CONST;
				}
				break;
			} while (0);
			break;
		} while (0); else if (_tmp38 == 'f')
		do {
			if (vala_scanner_matches (self, begin, "false")) {
				return VALA_TOKEN_TYPE_FALSE;
			}
			break;
		} while (0); else if (_tmp38 == 't')
		do {
			if (vala_scanner_matches (self, begin, "throw")) {
				return VALA_TOKEN_TYPE_THROW;
			}
			break;
		} while (0); else if (_tmp38 == 'u')
		do {
			if (vala_scanner_matches (self, begin, "using")) {
				return VALA_TOKEN_TYPE_USING;
			}
			break;
		} while (0); else if (_tmp38 == 'w')
		do {
			if (vala_scanner_matches (self, begin, "while")) {
				return VALA_TOKEN_TYPE_WHILE;
			}
			break;
		} while (0);
		break;
	} while (0); else if (_tmp80 == 6)
	do {
		gchar _tmp55;
		_tmp55 = begin[0];
		if (_tmp55 == 'd')
		do {
			if (vala_scanner_matches (self, begin, "delete")) {
				return VALA_TOKEN_TYPE_DELETE;
			}
			break;
		} while (0); else if (_tmp55 == 'e')
		do {
			if (vala_scanner_matches (self, begin, "extern")) {
				return VALA_TOKEN_TYPE_EXTERN;
			}
			break;
		} while (0); else if (_tmp55 == 'i')
		do {
			if (vala_scanner_matches (self, begin, "inline")) {
				return VALA_TOKEN_TYPE_INLINE;
			}
			break;
		} while (0); else if (_tmp55 == 'p')
		do {
			if (vala_scanner_matches (self, begin, "public")) {
				return VALA_TOKEN_TYPE_PUBLIC;
			}
			break;
		} while (0); else if (_tmp55 == 'r')
		do {
			if (vala_scanner_matches (self, begin, "return")) {
				return VALA_TOKEN_TYPE_RETURN;
			}
			break;
		} while (0); else if (_tmp55 == 's')
		do {
			gchar _tmp51;
			_tmp51 = begin[1];
			if (_tmp51 == 'i')
			do {
				gchar _tmp46;
				_tmp46 = begin[2];
				if (_tmp46 == 'g')
				do {
					if (vala_scanner_matches (self, begin, "signal")) {
						return VALA_TOKEN_TYPE_SIGNAL;
					}
					break;
				} while (0); else if (_tmp46 == 'z')
				do {
					if (vala_scanner_matches (self, begin, "sizeof")) {
						return VALA_TOKEN_TYPE_SIZEOF;
					}
					break;
				} while (0);
				break;
			} while (0); else if (_tmp51 == 't')
			do {
				gchar _tmp49;
				_tmp49 = begin[2];
				if (_tmp49 == 'a')
				do {
					if (vala_scanner_matches (self, begin, "static")) {
						return VALA_TOKEN_TYPE_STATIC;
					}
					break;
				} while (0); else if (_tmp49 == 'r')
				do {
					if (vala_scanner_matches (self, begin, "struct")) {
						return VALA_TOKEN_TYPE_STRUCT;
					}
					break;
				} while (0);
				break;
			} while (0); else if (_tmp51 == 'w')
			do {
				if (vala_scanner_matches (self, begin, "switch")) {
					return VALA_TOKEN_TYPE_SWITCH;
				}
				break;
			} while (0);
			break;
		} while (0); else if (_tmp55 == 't')
		do {
			gchar _tmp54;
			_tmp54 = begin[1];
			if (_tmp54 == 'h')
			do {
				if (vala_scanner_matches (self, begin, "throws")) {
					return VALA_TOKEN_TYPE_THROWS;
				}
				break;
			} while (0); else if (_tmp54 == 'y')
			do {
				if (vala_scanner_matches (self, begin, "typeof")) {
					return VALA_TOKEN_TYPE_TYPEOF;
				}
				break;
			} while (0);
			break;
		} while (0);
		break;
	} while (0); else if (_tmp80 == 7)
	do {
		gchar _tmp65;
		_tmp65 = begin[0];
		if (_tmp65 == 'd')
		do {
			gchar _tmp58;
			_tmp58 = begin[1];
			if (_tmp58 == 'e')
			do {
				if (vala_scanner_matches (self, begin, "default")) {
					return VALA_TOKEN_TYPE_DEFAULT;
				}
				break;
			} while (0); else if (_tmp58 == 'y')
			do {
				if (vala_scanner_matches (self, begin, "dynamic")) {
					return VALA_TOKEN_TYPE_DYNAMIC;
				}
				break;
			} while (0);
			break;
		} while (0); else if (_tmp65 == 'e')
		do {
			if (vala_scanner_matches (self, begin, "ensures")) {
				return VALA_TOKEN_TYPE_ENSURES;
			}
			break;
		} while (0); else if (_tmp65 == 'f')
		do {
			gchar _tmp62;
			_tmp62 = begin[1];
			if (_tmp62 == 'i')
			do {
				if (vala_scanner_matches (self, begin, "finally")) {
					return VALA_TOKEN_TYPE_FINALLY;
				}
				break;
			} while (0); else if (_tmp62 == 'o')
			do {
				if (vala_scanner_matches (self, begin, "foreach")) {
					return VALA_TOKEN_TYPE_FOREACH;
				}
				break;
			} while (0);
			break;
		} while (0); else if (_tmp65 == 'p')
		do {
			if (vala_scanner_matches (self, begin, "private")) {
				return VALA_TOKEN_TYPE_PRIVATE;
			}
			break;
		} while (0); else if (_tmp65 == 'v')
		do {
			if (vala_scanner_matches (self, begin, "virtual")) {
				return VALA_TOKEN_TYPE_VIRTUAL;
			}
			break;
		} while (0);
		break;
	} while (0); else if (_tmp80 == 8)
	do {
		gchar _tmp73;
		_tmp73 = begin[0];
		if (_tmp73 == 'a')
		do {
			if (vala_scanner_matches (self, begin, "abstract")) {
				return VALA_TOKEN_TYPE_ABSTRACT;
			}
			break;
		} while (0); else if (_tmp73 == 'c')
		do {
			if (vala_scanner_matches (self, begin, "continue")) {
				return VALA_TOKEN_TYPE_CONTINUE;
			}
			break;
		} while (0); else if (_tmp73 == 'd')
		do {
			if (vala_scanner_matches (self, begin, "delegate")) {
				return VALA_TOKEN_TYPE_DELEGATE;
			}
			break;
		} while (0); else if (_tmp73 == 'i')
		do {
			if (vala_scanner_matches (self, begin, "internal")) {
				return VALA_TOKEN_TYPE_INTERNAL;
			}
			break;
		} while (0); else if (_tmp73 == 'o')
		do {
			if (vala_scanner_matches (self, begin, "override")) {
				return VALA_TOKEN_TYPE_OVERRIDE;
			}
			break;
		} while (0); else if (_tmp73 == 'r')
		do {
			if (vala_scanner_matches (self, begin, "requires")) {
				return VALA_TOKEN_TYPE_REQUIRES;
			}
			break;
		} while (0); else if (_tmp73 == 'v')
		do {
			if (vala_scanner_matches (self, begin, "volatile")) {
				return VALA_TOKEN_TYPE_VOLATILE;
			}
			break;
		} while (0);
		break;
	} while (0); else if (_tmp80 == 9)
	do {
		gchar _tmp78;
		_tmp78 = begin[0];
		if (_tmp78 == 'c')
		do {
			if (vala_scanner_matches (self, begin, "construct")) {
				return VALA_TOKEN_TYPE_CONSTRUCT;
			}
			break;
		} while (0); else if (_tmp78 == 'i')
		do {
			if (vala_scanner_matches (self, begin, "interface")) {
				return VALA_TOKEN_TYPE_INTERFACE;
			}
			break;
		} while (0); else if (_tmp78 == 'n')
		do {
			if (vala_scanner_matches (self, begin, "namespace")) {
				return VALA_TOKEN_TYPE_NAMESPACE;
			}
			break;
		} while (0); else if (_tmp78 == 'p')
		do {
			if (vala_scanner_matches (self, begin, "protected")) {
				return VALA_TOKEN_TYPE_PROTECTED;
			}
			break;
		} while (0);
		break;
	} while (0); else if (_tmp80 == 11)
	do {
		if (vala_scanner_matches (self, begin, "errordomain")) {
			return VALA_TOKEN_TYPE_ERRORDOMAIN;
		}
		break;
	} while (0);
	return VALA_TOKEN_TYPE_IDENTIFIER;
}


ValaTokenType vala_scanner_read_token (ValaScanner* self, ValaSourceLocation* token_begin, ValaSourceLocation* token_end) {
	ValaTokenType type;
	gchar* begin;
	gint token_length_in_chars;
	g_return_val_if_fail (VALA_IS_SCANNER (self), 0);
	vala_scanner_space (self);
	type = 0;
	begin = self->priv->current;
	(*token_begin).pos = begin;
	(*token_begin).line = self->priv->line;
	(*token_begin).column = self->priv->column;
	token_length_in_chars = -1;
	if (self->priv->current >= self->priv->end) {
		type = VALA_TOKEN_TYPE_EOF;
	} else {
		if (g_ascii_isalpha (self->priv->current[0]) || self->priv->current[0] == '_') {
			gint len;
			len = 0;
			while (self->priv->current < self->priv->end && vala_scanner_is_ident_char (self, self->priv->current[0])) {
				self->priv->current++;
				len++;
			}
			type = vala_scanner_get_identifier_or_keyword (self, begin, len);
		} else {
			if (self->priv->current[0] == '@') {
				gint len;
				(*token_begin).pos++;
				/* @ is not part of the identifier*/
				self->priv->current++;
				len = 0;
				while (self->priv->current < self->priv->end && vala_scanner_is_ident_char (self, self->priv->current[0])) {
					self->priv->current++;
					len++;
				}
				type = VALA_TOKEN_TYPE_IDENTIFIER;
			} else {
				if (g_ascii_isdigit (self->priv->current[0])) {
					while (self->priv->current < self->priv->end && g_ascii_isdigit (self->priv->current[0])) {
						self->priv->current++;
					}
					type = VALA_TOKEN_TYPE_INTEGER_LITERAL;
					if (self->priv->current < self->priv->end && g_ascii_tolower (self->priv->current[0]) == 'l') {
						self->priv->current++;
						if (self->priv->current < self->priv->end && g_ascii_tolower (self->priv->current[0]) == 'l') {
							self->priv->current++;
						}
					} else {
						if (self->priv->current < self->priv->end && g_ascii_tolower (self->priv->current[0]) == 'u') {
							self->priv->current++;
							if (self->priv->current < self->priv->end && g_ascii_tolower (self->priv->current[0]) == 'l') {
								self->priv->current++;
								if (self->priv->current < self->priv->end && g_ascii_tolower (self->priv->current[0]) == 'l') {
									self->priv->current++;
								}
							}
						} else {
							if (self->priv->current < self->priv->end && self->priv->current[0] == '.') {
								self->priv->current++;
								while (self->priv->current < self->priv->end && g_ascii_isdigit (self->priv->current[0])) {
									self->priv->current++;
								}
								if (self->priv->current < self->priv->end && g_ascii_tolower (self->priv->current[0]) == 'e') {
									self->priv->current++;
									if (self->priv->current < self->priv->end && (self->priv->current[0] == '+' || self->priv->current[0] == '-')) {
										self->priv->current++;
									}
									while (self->priv->current < self->priv->end && g_ascii_isdigit (self->priv->current[0])) {
										self->priv->current++;
									}
								}
								if (self->priv->current < self->priv->end && g_ascii_tolower (self->priv->current[0]) == 'f') {
									self->priv->current++;
								}
								type = VALA_TOKEN_TYPE_REAL_LITERAL;
							} else {
								if (self->priv->current < self->priv->end && self->priv->current == begin + 1 && begin[0] == '0' && begin[1] == 'x' && g_ascii_isxdigit (begin[2])) {
									/* hexadecimal integer literal*/
									self->priv->current++;
									while (self->priv->current < self->priv->end && g_ascii_isxdigit (self->priv->current[0])) {
										self->priv->current++;
									}
								} else {
									if (self->priv->current < self->priv->end && vala_scanner_is_ident_char (self, self->priv->current[0])) {
										while (self->priv->current < self->priv->end && vala_scanner_is_ident_char (self, self->priv->current[0])) {
											self->priv->current++;
										}
										type = VALA_TOKEN_TYPE_IDENTIFIER;
									}
								}
							}
						}
					}
				} else {
					gchar _tmp14;
					_tmp14 = self->priv->current[0];
					if (_tmp14 == '{')
					do {
						type = VALA_TOKEN_TYPE_OPEN_BRACE;
						self->priv->current++;
						break;
					} while (0); else if (_tmp14 == '}')
					do {
						type = VALA_TOKEN_TYPE_CLOSE_BRACE;
						self->priv->current++;
						break;
					} while (0); else if (_tmp14 == '(')
					do {
						type = VALA_TOKEN_TYPE_OPEN_PARENS;
						self->priv->current++;
						break;
					} while (0); else if (_tmp14 == ')')
					do {
						type = VALA_TOKEN_TYPE_CLOSE_PARENS;
						self->priv->current++;
						break;
					} while (0); else if (_tmp14 == '[')
					do {
						type = VALA_TOKEN_TYPE_OPEN_BRACKET;
						self->priv->current++;
						break;
					} while (0); else if (_tmp14 == ']')
					do {
						type = VALA_TOKEN_TYPE_CLOSE_BRACKET;
						self->priv->current++;
						break;
					} while (0); else if (_tmp14 == '.')
					do {
						type = VALA_TOKEN_TYPE_DOT;
						self->priv->current++;
						if (self->priv->current < self->priv->end - 1) {
							if (self->priv->current[0] == '.' && self->priv->current[1] == '.') {
								type = VALA_TOKEN_TYPE_ELLIPSIS;
								self->priv->current = self->priv->current + (2);
							}
						}
						break;
					} while (0); else if (_tmp14 == ':')
					do {
						type = VALA_TOKEN_TYPE_COLON;
						self->priv->current++;
						if (self->priv->current < self->priv->end && self->priv->current[0] == ':') {
							type = VALA_TOKEN_TYPE_DOUBLE_COLON;
							self->priv->current++;
						}
						break;
					} while (0); else if (_tmp14 == ',')
					do {
						type = VALA_TOKEN_TYPE_COMMA;
						self->priv->current++;
						break;
					} while (0); else if (_tmp14 == ';')
					do {
						type = VALA_TOKEN_TYPE_SEMICOLON;
						self->priv->current++;
						break;
					} while (0); else if (_tmp14 == '#')
					do {
						type = VALA_TOKEN_TYPE_HASH;
						self->priv->current++;
						break;
					} while (0); else if (_tmp14 == '?')
					do {
						type = VALA_TOKEN_TYPE_INTERR;
						self->priv->current++;
						break;
					} while (0); else if (_tmp14 == '|')
					do {
						type = VALA_TOKEN_TYPE_BITWISE_OR;
						self->priv->current++;
						if (self->priv->current < self->priv->end) {
							gchar _tmp0;
							_tmp0 = self->priv->current[0];
							if (_tmp0 == '=')
							do {
								type = VALA_TOKEN_TYPE_ASSIGN_BITWISE_OR;
								self->priv->current++;
								break;
							} while (0); else if (_tmp0 == '|')
							do {
								type = VALA_TOKEN_TYPE_OP_OR;
								self->priv->current++;
								break;
							} while (0);
						}
						break;
					} while (0); else if (_tmp14 == '&')
					do {
						type = VALA_TOKEN_TYPE_BITWISE_AND;
						self->priv->current++;
						if (self->priv->current < self->priv->end) {
							gchar _tmp1;
							_tmp1 = self->priv->current[0];
							if (_tmp1 == '=')
							do {
								type = VALA_TOKEN_TYPE_ASSIGN_BITWISE_AND;
								self->priv->current++;
								break;
							} while (0); else if (_tmp1 == '&')
							do {
								type = VALA_TOKEN_TYPE_OP_AND;
								self->priv->current++;
								break;
							} while (0);
						}
						break;
					} while (0); else if (_tmp14 == '^')
					do {
						type = VALA_TOKEN_TYPE_CARRET;
						self->priv->current++;
						if (self->priv->current < self->priv->end && self->priv->current[0] == '=') {
							type = VALA_TOKEN_TYPE_ASSIGN_BITWISE_XOR;
							self->priv->current++;
						}
						break;
					} while (0); else if (_tmp14 == '~')
					do {
						type = VALA_TOKEN_TYPE_TILDE;
						self->priv->current++;
						break;
					} while (0); else if (_tmp14 == '=')
					do {
						type = VALA_TOKEN_TYPE_ASSIGN;
						self->priv->current++;
						if (self->priv->current < self->priv->end) {
							gchar _tmp2;
							_tmp2 = self->priv->current[0];
							if (_tmp2 == '=')
							do {
								type = VALA_TOKEN_TYPE_OP_EQ;
								self->priv->current++;
								break;
							} while (0); else if (_tmp2 == '>')
							do {
								type = VALA_TOKEN_TYPE_LAMBDA;
								self->priv->current++;
								break;
							} while (0);
						}
						break;
					} while (0); else if (_tmp14 == '<')
					do {
						type = VALA_TOKEN_TYPE_OP_LT;
						self->priv->current++;
						if (self->priv->current < self->priv->end) {
							gchar _tmp3;
							_tmp3 = self->priv->current[0];
							if (_tmp3 == '=')
							do {
								type = VALA_TOKEN_TYPE_OP_LE;
								self->priv->current++;
								break;
							} while (0); else if (_tmp3 == '<')
							do {
								type = VALA_TOKEN_TYPE_OP_SHIFT_LEFT;
								self->priv->current++;
								if (self->priv->current < self->priv->end && self->priv->current[0] == '=') {
									type = VALA_TOKEN_TYPE_ASSIGN_SHIFT_LEFT;
									self->priv->current++;
								}
								break;
							} while (0);
						}
						break;
					} while (0); else if (_tmp14 == '>')
					do {
						type = VALA_TOKEN_TYPE_OP_GT;
						self->priv->current++;
						if (self->priv->current < self->priv->end && self->priv->current[0] == '=') {
							type = VALA_TOKEN_TYPE_OP_GE;
							self->priv->current++;
						}
						break;
					} while (0); else if (_tmp14 == '!')
					do {
						type = VALA_TOKEN_TYPE_OP_NEG;
						self->priv->current++;
						if (self->priv->current < self->priv->end && self->priv->current[0] == '=') {
							type = VALA_TOKEN_TYPE_OP_NE;
							self->priv->current++;
						}
						break;
					} while (0); else if (_tmp14 == '+')
					do {
						type = VALA_TOKEN_TYPE_PLUS;
						self->priv->current++;
						if (self->priv->current < self->priv->end) {
							gchar _tmp4;
							_tmp4 = self->priv->current[0];
							if (_tmp4 == '=')
							do {
								type = VALA_TOKEN_TYPE_ASSIGN_ADD;
								self->priv->current++;
								break;
							} while (0); else if (_tmp4 == '+')
							do {
								type = VALA_TOKEN_TYPE_OP_INC;
								self->priv->current++;
								break;
							} while (0);
						}
						break;
					} while (0); else if (_tmp14 == '-')
					do {
						type = VALA_TOKEN_TYPE_MINUS;
						self->priv->current++;
						if (self->priv->current < self->priv->end) {
							gchar _tmp5;
							_tmp5 = self->priv->current[0];
							if (_tmp5 == '=')
							do {
								type = VALA_TOKEN_TYPE_ASSIGN_SUB;
								self->priv->current++;
								break;
							} while (0); else if (_tmp5 == '-')
							do {
								type = VALA_TOKEN_TYPE_OP_DEC;
								self->priv->current++;
								break;
							} while (0); else if (_tmp5 == '>')
							do {
								type = VALA_TOKEN_TYPE_OP_PTR;
								self->priv->current++;
								break;
							} while (0);
						}
						break;
					} while (0); else if (_tmp14 == '*')
					do {
						type = VALA_TOKEN_TYPE_STAR;
						self->priv->current++;
						if (self->priv->current < self->priv->end && self->priv->current[0] == '=') {
							type = VALA_TOKEN_TYPE_ASSIGN_MUL;
							self->priv->current++;
						}
						break;
					} while (0); else if (_tmp14 == '/')
					do {
						type = VALA_TOKEN_TYPE_DIV;
						self->priv->current++;
						if (self->priv->current < self->priv->end && self->priv->current[0] == '=') {
							type = VALA_TOKEN_TYPE_ASSIGN_DIV;
							self->priv->current++;
						}
						break;
					} while (0); else if (_tmp14 == '%')
					do {
						type = VALA_TOKEN_TYPE_PERCENT;
						self->priv->current++;
						if (self->priv->current < self->priv->end && self->priv->current[0] == '=') {
							type = VALA_TOKEN_TYPE_ASSIGN_PERCENT;
							self->priv->current++;
						}
						break;
					} while (0); else if (_tmp14 == '\'' || _tmp14 == '"')
					do {
						if (begin[0] == '\'') {
							type = VALA_TOKEN_TYPE_CHARACTER_LITERAL;
						} else {
							if (self->priv->current < self->priv->end - 6 && begin[1] == '"' && begin[2] == '"') {
								type = VALA_TOKEN_TYPE_VERBATIM_STRING_LITERAL;
								token_length_in_chars = 6;
								self->priv->current = self->priv->current + (3);
								while (self->priv->current < self->priv->end - 4) {
									if (self->priv->current[0] == '"' && self->priv->current[1] == '"' && self->priv->current[2] == '"') {
										break;
									} else {
										if (self->priv->current[0] == '\n') {
											self->priv->current++;
											self->priv->line++;
											self->priv->column = 1;
											token_length_in_chars = 3;
										} else {
											gunichar u;
											u = g_utf8_get_char_validated ((((const char*) (self->priv->current))), ((glong) ((self->priv->end - self->priv->current))));
											if (u != ((gunichar) ((-1)))) {
												self->priv->current = self->priv->current + (g_unichar_to_utf8 (u, NULL));
												token_length_in_chars++;
											} else {
												ValaSourceReference* _tmp6;
												_tmp6 = NULL;
												vala_report_error ((_tmp6 = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column + token_length_in_chars, self->priv->line, self->priv->column + token_length_in_chars)), "invalid UTF-8 character");
												(_tmp6 == NULL ? NULL : (_tmp6 = (g_object_unref (_tmp6), NULL)));
											}
										}
									}
								}
								if (self->priv->current[0] == '"' && self->priv->current[1] == '"' && self->priv->current[2] == '"') {
									self->priv->current = self->priv->current + (3);
								} else {
									ValaSourceReference* _tmp7;
									_tmp7 = NULL;
									vala_report_error ((_tmp7 = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column + token_length_in_chars, self->priv->line, self->priv->column + token_length_in_chars)), "syntax error, expected \"\"\"");
									(_tmp7 == NULL ? NULL : (_tmp7 = (g_object_unref (_tmp7), NULL)));
								}
								break;
							} else {
								type = VALA_TOKEN_TYPE_STRING_LITERAL;
							}
						}
						token_length_in_chars = 2;
						self->priv->current++;
						while (self->priv->current < self->priv->end && self->priv->current[0] != begin[0]) {
							if (self->priv->current[0] == '\\') {
								self->priv->current++;
								token_length_in_chars++;
								if (self->priv->current < self->priv->end && self->priv->current[0] == 'x') {
									/* hexadecimal escape character*/
									self->priv->current++;
									token_length_in_chars++;
									while (self->priv->current < self->priv->end && g_ascii_isxdigit (self->priv->current[0])) {
										self->priv->current++;
										token_length_in_chars++;
									}
								} else {
									self->priv->current++;
									token_length_in_chars++;
								}
							} else {
								if (self->priv->current[0] == '\n') {
									break;
								} else {
									gunichar u;
									u = g_utf8_get_char_validated ((((const char*) (self->priv->current))), ((glong) ((self->priv->end - self->priv->current))));
									if (u != ((gunichar) ((-1)))) {
										self->priv->current = self->priv->current + (g_unichar_to_utf8 (u, NULL));
										token_length_in_chars++;
									} else {
										ValaSourceReference* _tmp8;
										_tmp8 = NULL;
										vala_report_error ((_tmp8 = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column + token_length_in_chars, self->priv->line, self->priv->column + token_length_in_chars)), "invalid UTF-8 character");
										(_tmp8 == NULL ? NULL : (_tmp8 = (g_object_unref (_tmp8), NULL)));
									}
								}
							}
						}
						if (self->priv->current < self->priv->end && self->priv->current[0] != '\n') {
							self->priv->current++;
						} else {
							char* _tmp10;
							ValaSourceReference* _tmp9;
							_tmp10 = NULL;
							_tmp9 = NULL;
							vala_report_error ((_tmp9 = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column + token_length_in_chars, self->priv->line, self->priv->column + token_length_in_chars)), (_tmp10 = g_strdup_printf ("syntax error, expected %c", ((gint) (begin[0])))));
							_tmp10 = (g_free (_tmp10), NULL);
							(_tmp9 == NULL ? NULL : (_tmp9 = (g_object_unref (_tmp9), NULL)));
						}
						break;
					} while (0); else
					do {
						gunichar u;
						u = g_utf8_get_char_validated ((((const char*) (self->priv->current))), ((glong) ((self->priv->end - self->priv->current))));
						if (u != ((gunichar) ((-1)))) {
							ValaSourceReference* _tmp11;
							self->priv->current = self->priv->current + (g_unichar_to_utf8 (u, NULL));
							_tmp11 = NULL;
							vala_report_error ((_tmp11 = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column)), "syntax error, unexpected character");
							(_tmp11 == NULL ? NULL : (_tmp11 = (g_object_unref (_tmp11), NULL)));
						} else {
							ValaSourceReference* _tmp12;
							self->priv->current++;
							_tmp12 = NULL;
							vala_report_error ((_tmp12 = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column)), "invalid UTF-8 character");
							(_tmp12 == NULL ? NULL : (_tmp12 = (g_object_unref (_tmp12), NULL)));
						}
						self->priv->column++;
						return vala_scanner_read_token (self, &(*token_begin), &(*token_end));
					} while (0);
				}
			}
		}
	}
	if (token_length_in_chars < 0) {
		self->priv->column = self->priv->column + (((gint) ((self->priv->current - begin))));
	} else {
		self->priv->column = self->priv->column + (token_length_in_chars);
	}
	(*token_end).pos = self->priv->current;
	(*token_end).line = self->priv->line;
	(*token_end).column = self->priv->column - 1;
	return type;
}


static gboolean vala_scanner_matches (ValaScanner* self, gchar* begin, const char* keyword) {
	gchar* keyword_array;
	glong len;
	g_return_val_if_fail (VALA_IS_SCANNER (self), FALSE);
	g_return_val_if_fail (keyword != NULL, FALSE);
	keyword_array = keyword;
	len = g_utf8_strlen (keyword, -1);
	{
		gint i;
		i = 0;
		for (; i < len; i++) {
			if (begin[i] != keyword_array[i]) {
				return FALSE;
			}
		}
	}
	return TRUE;
}


static gboolean vala_scanner_whitespace (ValaScanner* self) {
	gboolean found;
	g_return_val_if_fail (VALA_IS_SCANNER (self), FALSE);
	found = FALSE;
	while (self->priv->current < self->priv->end && g_ascii_isspace (self->priv->current[0])) {
		if (self->priv->current[0] == '\n') {
			self->priv->line++;
			self->priv->column = 0;
		}
		found = TRUE;
		self->priv->current++;
		self->priv->column++;
	}
	return found;
}


static gboolean vala_scanner_comment (ValaScanner* self) {
	g_return_val_if_fail (VALA_IS_SCANNER (self), FALSE);
	if (self->priv->current > self->priv->end - 2 || self->priv->current[0] != '/' || (self->priv->current[1] != '/' && self->priv->current[1] != '*')) {
		return FALSE;
	}
	if (self->priv->current[1] == '/') {
		gchar* begin;
		char* _tmp1;
		/* single-line comment*/
		self->priv->current = self->priv->current + (2);
		begin = self->priv->current;
		while (self->priv->current < self->priv->end && self->priv->current[0] != '\n') {
			self->priv->current++;
		}
		_tmp1 = NULL;
		vala_scanner_push_comment (self, (_tmp1 = g_strndup ((((const char*) (begin))), ((gulong) (((glong) ((self->priv->current - begin))))))), self->priv->line == 1);
		_tmp1 = (g_free (_tmp1), NULL);
	} else {
		gchar* begin;
		gint begin_line;
		char* _tmp4;
		/* delimited comment*/
		self->priv->current = self->priv->current + (2);
		begin = self->priv->current;
		begin_line = self->priv->line;
		while (self->priv->current < self->priv->end - 1 && (self->priv->current[0] != '*' || self->priv->current[1] != '/')) {
			if (self->priv->current[0] == '\n') {
				self->priv->line++;
				self->priv->column = 0;
			}
			self->priv->current++;
			self->priv->column++;
		}
		if (self->priv->current == self->priv->end - 1) {
			ValaSourceReference* _tmp2;
			_tmp2 = NULL;
			vala_report_error ((_tmp2 = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column)), "syntax error, expected */");
			(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
			return TRUE;
		}
		_tmp4 = NULL;
		vala_scanner_push_comment (self, (_tmp4 = g_strndup ((((const char*) (begin))), ((gulong) (((glong) ((self->priv->current - begin))))))), begin_line == 1);
		_tmp4 = (g_free (_tmp4), NULL);
		self->priv->current = self->priv->current + (2);
		self->priv->column = self->priv->column + (2);
	}
	return TRUE;
}


static void vala_scanner_space (ValaScanner* self) {
	g_return_if_fail (VALA_IS_SCANNER (self));
	while (vala_scanner_whitespace (self) || vala_scanner_comment (self)) {
	}
}


static void vala_scanner_push_comment (ValaScanner* self, const char* comment_item, gboolean file_comment) {
	g_return_if_fail (VALA_IS_SCANNER (self));
	g_return_if_fail (comment_item != NULL);
	if (self->priv->_comment == NULL) {
		char* _tmp1;
		const char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->_comment = (_tmp1 = (_tmp0 = comment_item, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->_comment = (g_free (self->priv->_comment), NULL)), _tmp1);
	} else {
		char* _tmp2;
		_tmp2 = NULL;
		self->priv->_comment = (_tmp2 = g_strdup_printf ("%s\n%s", self->priv->_comment, comment_item), (self->priv->_comment = (g_free (self->priv->_comment), NULL)), _tmp2);
	}
	if (file_comment) {
		char* _tmp3;
		vala_source_file_set_comment (self->priv->_source_file, self->priv->_comment);
		_tmp3 = NULL;
		self->priv->_comment = (_tmp3 = NULL, (self->priv->_comment = (g_free (self->priv->_comment), NULL)), _tmp3);
	}
}


/**
 * Clears and returns the content of the comment stack.
 *
 * @return saved comment
 */
char* vala_scanner_pop_comment (ValaScanner* self) {
	GString* result;
	char* _tmp1;
	const char* index;
	const char* _tmp2;
	char* _tmp3;
	g_return_val_if_fail (VALA_IS_SCANNER (self), NULL);
	if (self->priv->_comment == NULL) {
		return NULL;
	}
	result = g_string_new (self->priv->_comment);
	_tmp1 = NULL;
	self->priv->_comment = (_tmp1 = NULL, (self->priv->_comment = (g_free (self->priv->_comment), NULL)), _tmp1);
	index = NULL;
	while ((index = g_utf8_strchr (result->str, ((glong) (-1)), ((gunichar) ('\t')))) != NULL) {
		g_string_erase (result, g_utf8_pointer_to_offset (result->str, index), ((glong) (1)));
	}
	_tmp2 = NULL;
	_tmp3 = NULL;
	return (_tmp3 = (_tmp2 = result->str, (_tmp2 == NULL ? NULL : g_strdup (_tmp2))), (result == NULL ? NULL : (result = (g_string_free (result, TRUE), NULL))), _tmp3);
}


ValaSourceFile* vala_scanner_get_source_file (ValaScanner* self) {
	g_return_val_if_fail (VALA_IS_SCANNER (self), NULL);
	return self->priv->_source_file;
}


static void vala_scanner_set_source_file (ValaScanner* self, ValaSourceFile* value) {
	ValaSourceFile* _tmp2;
	ValaSourceFile* _tmp1;
	g_return_if_fail (VALA_IS_SCANNER (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_source_file = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_source_file == NULL ? NULL : (self->priv->_source_file = (g_object_unref (self->priv->_source_file), NULL))), _tmp2);
	g_object_notify (((GObject *) (self)), "source-file");
}


static GObject * vala_scanner_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	ValaScannerClass * klass;
	GObjectClass * parent_class;
	ValaScanner * self;
	klass = VALA_SCANNER_CLASS (g_type_class_peek (VALA_TYPE_SCANNER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = VALA_SCANNER (obj);
	{
		gchar* begin;
		begin = vala_source_file_get_mapped_contents (self->priv->_source_file);
		self->priv->end = begin + vala_source_file_get_mapped_length (self->priv->_source_file);
		self->priv->current = begin;
		self->priv->line = 1;
		self->priv->column = 1;
	}
	return obj;
}


static void vala_scanner_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaScanner * self;
	self = VALA_SCANNER (object);
	switch (property_id) {
		case VALA_SCANNER_SOURCE_FILE:
		g_value_set_object (value, vala_scanner_get_source_file (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_scanner_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaScanner * self;
	self = VALA_SCANNER (object);
	switch (property_id) {
		case VALA_SCANNER_SOURCE_FILE:
		vala_scanner_set_source_file (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_scanner_class_init (ValaScannerClass * klass) {
	vala_scanner_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaScannerPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_scanner_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_scanner_set_property;
	G_OBJECT_CLASS (klass)->constructor = vala_scanner_constructor;
	G_OBJECT_CLASS (klass)->finalize = vala_scanner_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SCANNER_SOURCE_FILE, g_param_spec_object ("source-file", "source-file", "source-file", VALA_TYPE_SOURCE_FILE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}


static void vala_scanner_instance_init (ValaScanner * self) {
	self->priv = VALA_SCANNER_GET_PRIVATE (self);
}


static void vala_scanner_finalize (GObject * obj) {
	ValaScanner * self;
	self = VALA_SCANNER (obj);
	(self->priv->_source_file == NULL ? NULL : (self->priv->_source_file = (g_object_unref (self->priv->_source_file), NULL)));
	self->priv->_comment = (g_free (self->priv->_comment), NULL);
	G_OBJECT_CLASS (vala_scanner_parent_class)->finalize (obj);
}


GType vala_scanner_get_type (void) {
	static GType vala_scanner_type_id = 0;
	if (vala_scanner_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaScannerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_scanner_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaScanner), 0, (GInstanceInitFunc) vala_scanner_instance_init };
		vala_scanner_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaScanner", &g_define_type_info, 0);
	}
	return vala_scanner_type_id;
}




