/* valagidlwriter.vala
 *
 * Copyright (C) 2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <gobject/valagidlwriter.h>
#include <stdio.h>
#include <vala/valatypesymbol.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valacodenode.h>
#include <vala/valadatatype.h>
#include <vala/valainterfaceinstancetype.h>
#include <gee/collection.h>
#include <vala/valaclassinstancetype.h>
#include <vala/valadelegatetype.h>
#include <vala/valaformalparameter.h>
#include <vala/valamember.h>
#include <gobject/valaccodegenerator.h>




struct _ValaGIdlWriterPrivate {
	ValaCodeContext* context;
	FILE* stream;
	gint indent;
	ValaTypesymbol* gobject_type;
};

#define VALA_GIDL_WRITER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_GIDL_WRITER, ValaGIdlWriterPrivate))
enum  {
	VALA_GIDL_WRITER_DUMMY_PROPERTY
};
static void vala_gidl_writer_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns);
static void vala_gidl_writer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_gidl_writer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_gidl_writer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_gidl_writer_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en);
static void vala_gidl_writer_real_visit_enum_value (ValaCodeVisitor* base, ValaEnumValue* ev);
static void vala_gidl_writer_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* edomain);
static void vala_gidl_writer_real_visit_error_code (ValaCodeVisitor* base, ValaErrorCode* ecode);
static void vala_gidl_writer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c);
static void vala_gidl_writer_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static char* vala_gidl_writer_get_gidl_type_name (ValaGIdlWriter* self, ValaDataType* type);
static void vala_gidl_writer_write_params (ValaGIdlWriter* self, GeeCollection* params, ValaDataType* instance_type);
static void vala_gidl_writer_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* cb);
static void vala_gidl_writer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_gidl_writer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_gidl_writer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_gidl_writer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig);
static void vala_gidl_writer_write_indent (ValaGIdlWriter* self);
static void vala_gidl_writer_write_return_type (ValaGIdlWriter* self, ValaDataType* type);
static gboolean vala_gidl_writer_check_accessibility (ValaGIdlWriter* self, ValaSymbol* sym);
static gpointer vala_gidl_writer_parent_class = NULL;
static void vala_gidl_writer_dispose (GObject * obj);



/**
 * Writes the public interface of the specified code context into the
 * specified file.
 *
 * @param context  a code context
 * @param filename a relative or absolute filename
 */
void vala_gidl_writer_write_file (ValaGIdlWriter* self, ValaCodeContext* context, const char* filename) {
	ValaCodeContext* _tmp1;
	ValaCodeContext* _tmp0;
	ValaNamespace* _tmp2;
	ValaNamespace* root_symbol;
	ValaSymbol* glib_ns;
	ValaTypesymbol* _tmp3;
	FILE* _tmp4;
	FILE* _tmp5;
	g_return_if_fail (VALA_IS_GIDL_WRITER (self));
	g_return_if_fail (VALA_IS_CODE_CONTEXT (context));
	g_return_if_fail (filename != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->context = (_tmp1 = (_tmp0 = context, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL))), _tmp1);
	_tmp2 = NULL;
	root_symbol = (_tmp2 = vala_code_context_get_root (context), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)));
	glib_ns = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (root_symbol)), "GLib");
	_tmp3 = NULL;
	self->priv->gobject_type = (_tmp3 = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "Object")), (self->priv->gobject_type == NULL ? NULL : (self->priv->gobject_type = (g_object_unref (self->priv->gobject_type), NULL))), _tmp3);
	_tmp4 = NULL;
	self->priv->stream = (_tmp4 = fopen (filename, "w"), (self->priv->stream == NULL ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL))), _tmp4);
	fprintf (self->priv->stream, "<?xml version=\"1.0\"?>\n");
	fprintf (self->priv->stream, "<api version=\"1.0\">\n");
	vala_code_context_accept (context, VALA_CODE_VISITOR (self));
	fprintf (self->priv->stream, "</api>\n");
	_tmp5 = NULL;
	self->priv->stream = (_tmp5 = NULL, (self->priv->stream == NULL ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL))), _tmp5);
	(root_symbol == NULL ? NULL : (root_symbol = (g_object_unref (root_symbol), NULL)));
	(glib_ns == NULL ? NULL : (glib_ns = (g_object_unref (glib_ns), NULL)));
}


static void vala_gidl_writer_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns) {
	ValaGIdlWriter * self;
	self = VALA_GIDL_WRITER (base);
	g_return_if_fail (VALA_IS_NAMESPACE (ns));
	if (vala_symbol_get_external_package (VALA_SYMBOL (ns))) {
		return;
	}
	if (vala_symbol_get_name (VALA_SYMBOL (ns)) == NULL) {
		/* global namespace*/
		vala_code_node_accept_children (VALA_CODE_NODE (ns), VALA_CODE_VISITOR (self));
		return;
	}
	vala_gidl_writer_write_indent (self);
	fprintf (self->priv->stream, "<namespace name=\"%s\">\n", vala_symbol_get_name (VALA_SYMBOL (ns)));
	self->priv->indent++;
	vala_code_node_accept_children (VALA_CODE_NODE (ns), VALA_CODE_VISITOR (self));
	self->priv->indent--;
	vala_gidl_writer_write_indent (self);
	fprintf (self->priv->stream, "</namespace>\n");
}


static void vala_gidl_writer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl) {
	ValaGIdlWriter * self;
	self = VALA_GIDL_WRITER (base);
	g_return_if_fail (VALA_IS_CLASS (cl));
	if (vala_symbol_get_external_package (VALA_SYMBOL (cl))) {
		return;
	}
	if (!vala_gidl_writer_check_accessibility (self, VALA_SYMBOL (cl))) {
		return;
	}
	if (vala_typesymbol_is_subtype_of (VALA_TYPESYMBOL (cl), self->priv->gobject_type)) {
		char* _tmp0;
		char* _tmp1;
		char* _tmp2;
		gboolean first;
		vala_gidl_writer_write_indent (self);
		fprintf (self->priv->stream, "<object name=\"%s\"", vala_symbol_get_name (VALA_SYMBOL (cl)));
		_tmp0 = NULL;
		fprintf (self->priv->stream, " parent=\"%s\"", (_tmp0 = vala_symbol_get_full_name (VALA_SYMBOL (vala_class_get_base_class (cl)))));
		_tmp0 = (g_free (_tmp0), NULL);
		_tmp1 = NULL;
		fprintf (self->priv->stream, " type-name=\"%s\"", (_tmp1 = vala_typesymbol_get_cname (VALA_TYPESYMBOL (cl), FALSE)));
		_tmp1 = (g_free (_tmp1), NULL);
		_tmp2 = NULL;
		fprintf (self->priv->stream, " get-type=\"%sget_type\"", (_tmp2 = vala_symbol_get_lower_case_cprefix (VALA_SYMBOL (cl))));
		_tmp2 = (g_free (_tmp2), NULL);
		fprintf (self->priv->stream, ">\n");
		self->priv->indent++;
		/* write implemented interfaces*/
		first = TRUE;
		{
			GeeCollection* base_type_collection;
			GeeIterator* base_type_it;
			base_type_collection = vala_class_get_base_types (cl);
			base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
			while (gee_iterator_next (base_type_it)) {
				ValaDataType* base_type;
				base_type = ((ValaDataType*) gee_iterator_get (base_type_it));
				{
					ValaInterfaceInstanceType* _tmp4;
					ValaDataType* _tmp3;
					ValaInterfaceInstanceType* iface_type;
					_tmp4 = NULL;
					_tmp3 = NULL;
					iface_type = (_tmp4 = (_tmp3 = base_type, (VALA_IS_INTERFACE_INSTANCE_TYPE (_tmp3) ? ((ValaInterfaceInstanceType*) _tmp3) : NULL)), (_tmp4 == NULL ? NULL : g_object_ref (_tmp4)));
					if (iface_type != NULL) {
						char* _tmp5;
						if (first) {
							vala_gidl_writer_write_indent (self);
							fprintf (self->priv->stream, "<implements>\n");
							self->priv->indent++;
							first = FALSE;
						}
						vala_gidl_writer_write_indent (self);
						_tmp5 = NULL;
						fprintf (self->priv->stream, "<interface name=\"%s\"/>\n", (_tmp5 = vala_symbol_get_full_name (VALA_SYMBOL (vala_interface_instance_type_get_interface_symbol (iface_type)))));
						_tmp5 = (g_free (_tmp5), NULL);
					}
					(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
					(iface_type == NULL ? NULL : (iface_type = (g_object_unref (iface_type), NULL)));
				}
			}
			(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
			(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
		}
		if (!first) {
			self->priv->indent--;
			vala_gidl_writer_write_indent (self);
			fprintf (self->priv->stream, "</implements>\n");
		}
		vala_code_node_accept_children (VALA_CODE_NODE (cl), VALA_CODE_VISITOR (self));
		self->priv->indent--;
		vala_gidl_writer_write_indent (self);
		fprintf (self->priv->stream, "</object>\n");
	} else {
		vala_gidl_writer_write_indent (self);
		fprintf (self->priv->stream, "<struct name=\"%s\"", vala_symbol_get_name (VALA_SYMBOL (cl)));
		fprintf (self->priv->stream, ">\n");
		self->priv->indent++;
		vala_code_node_accept_children (VALA_CODE_NODE (cl), VALA_CODE_VISITOR (self));
		self->priv->indent--;
		vala_gidl_writer_write_indent (self);
		fprintf (self->priv->stream, "</struct>\n");
	}
}


static void vala_gidl_writer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st) {
	ValaGIdlWriter * self;
	char* _tmp0;
	self = VALA_GIDL_WRITER (base);
	g_return_if_fail (VALA_IS_STRUCT (st));
	if (vala_symbol_get_external_package (VALA_SYMBOL (st))) {
		return;
	}
	if (!vala_gidl_writer_check_accessibility (self, VALA_SYMBOL (st))) {
		return;
	}
	vala_gidl_writer_write_indent (self);
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<struct name=\"%s\"", (_tmp0 = vala_typesymbol_get_cname (VALA_TYPESYMBOL (st), FALSE)));
	_tmp0 = (g_free (_tmp0), NULL);
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	vala_code_node_accept_children (VALA_CODE_NODE (st), VALA_CODE_VISITOR (self));
	self->priv->indent--;
	vala_gidl_writer_write_indent (self);
	fprintf (self->priv->stream, "</struct>\n");
}


static void vala_gidl_writer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface) {
	ValaGIdlWriter * self;
	char* _tmp0;
	char* _tmp1;
	GeeCollection* _tmp2;
	gboolean _tmp3;
	self = VALA_GIDL_WRITER (base);
	g_return_if_fail (VALA_IS_INTERFACE (iface));
	if (vala_symbol_get_external_package (VALA_SYMBOL (iface))) {
		return;
	}
	if (!vala_gidl_writer_check_accessibility (self, VALA_SYMBOL (iface))) {
		return;
	}
	vala_gidl_writer_write_indent (self);
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<interface name=\"%s\"", (_tmp0 = vala_typesymbol_get_cname (VALA_TYPESYMBOL (iface), FALSE)));
	_tmp0 = (g_free (_tmp0), NULL);
	_tmp1 = NULL;
	fprintf (self->priv->stream, " get-type=\"%sget_type\"", (_tmp1 = vala_symbol_get_lower_case_cprefix (VALA_SYMBOL (iface))));
	_tmp1 = (g_free (_tmp1), NULL);
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	/* write prerequisites*/
	_tmp2 = NULL;
	if ((_tmp3 = gee_collection_get_size ((_tmp2 = vala_interface_get_prerequisites (iface))) > 0, (_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL))), _tmp3)) {
		vala_gidl_writer_write_indent (self);
		fprintf (self->priv->stream, "<requires>\n");
		self->priv->indent++;
		{
			GeeCollection* base_type_collection;
			GeeIterator* base_type_it;
			base_type_collection = vala_interface_get_prerequisites (iface);
			base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
			while (gee_iterator_next (base_type_it)) {
				ValaDataType* base_type;
				base_type = ((ValaDataType*) gee_iterator_get (base_type_it));
				{
					ValaClassInstanceType* _tmp5;
					ValaDataType* _tmp4;
					ValaClassInstanceType* class_type;
					ValaInterfaceInstanceType* _tmp7;
					ValaDataType* _tmp6;
					ValaInterfaceInstanceType* iface_type;
					_tmp5 = NULL;
					_tmp4 = NULL;
					class_type = (_tmp5 = (_tmp4 = base_type, (VALA_IS_CLASS_INSTANCE_TYPE (_tmp4) ? ((ValaClassInstanceType*) _tmp4) : NULL)), (_tmp5 == NULL ? NULL : g_object_ref (_tmp5)));
					_tmp7 = NULL;
					_tmp6 = NULL;
					iface_type = (_tmp7 = (_tmp6 = base_type, (VALA_IS_INTERFACE_INSTANCE_TYPE (_tmp6) ? ((ValaInterfaceInstanceType*) _tmp6) : NULL)), (_tmp7 == NULL ? NULL : g_object_ref (_tmp7)));
					if (class_type != NULL) {
						char* _tmp8;
						vala_gidl_writer_write_indent (self);
						_tmp8 = NULL;
						fprintf (self->priv->stream, "<object name=\"%s\"/>\n", (_tmp8 = vala_symbol_get_full_name (VALA_SYMBOL (vala_class_instance_type_get_class_symbol (class_type)))));
						_tmp8 = (g_free (_tmp8), NULL);
					} else {
						if (iface_type != NULL) {
							char* _tmp9;
							vala_gidl_writer_write_indent (self);
							_tmp9 = NULL;
							fprintf (self->priv->stream, "<interface name=\"%s\"/>\n", (_tmp9 = vala_symbol_get_full_name (VALA_SYMBOL (vala_interface_instance_type_get_interface_symbol (iface_type)))));
							_tmp9 = (g_free (_tmp9), NULL);
						} else {
							g_assert_not_reached ();
						}
					}
					(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
					(class_type == NULL ? NULL : (class_type = (g_object_unref (class_type), NULL)));
					(iface_type == NULL ? NULL : (iface_type = (g_object_unref (iface_type), NULL)));
				}
			}
			(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
			(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
		}
		self->priv->indent--;
		vala_gidl_writer_write_indent (self);
		fprintf (self->priv->stream, "</requires>\n");
	}
	vala_code_node_accept_children (VALA_CODE_NODE (iface), VALA_CODE_VISITOR (self));
	self->priv->indent--;
	vala_gidl_writer_write_indent (self);
	fprintf (self->priv->stream, "</interface>\n");
}


static void vala_gidl_writer_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en) {
	ValaGIdlWriter * self;
	char* _tmp0;
	char* _tmp1;
	self = VALA_GIDL_WRITER (base);
	g_return_if_fail (VALA_IS_ENUM (en));
	if (vala_symbol_get_external_package (VALA_SYMBOL (en))) {
		return;
	}
	if (!vala_gidl_writer_check_accessibility (self, VALA_SYMBOL (en))) {
		return;
	}
	vala_gidl_writer_write_indent (self);
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<enum name=\"%s\"", (_tmp0 = vala_typesymbol_get_cname (VALA_TYPESYMBOL (en), FALSE)));
	_tmp0 = (g_free (_tmp0), NULL);
	_tmp1 = NULL;
	fprintf (self->priv->stream, " get-type=\"%sget_type\"", (_tmp1 = vala_symbol_get_lower_case_cprefix (VALA_SYMBOL (en))));
	_tmp1 = (g_free (_tmp1), NULL);
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	vala_code_node_accept_children (VALA_CODE_NODE (en), VALA_CODE_VISITOR (self));
	self->priv->indent--;
	vala_gidl_writer_write_indent (self);
	fprintf (self->priv->stream, "</enum>\n");
}


static void vala_gidl_writer_real_visit_enum_value (ValaCodeVisitor* base, ValaEnumValue* ev) {
	ValaGIdlWriter * self;
	char* _tmp0;
	self = VALA_GIDL_WRITER (base);
	g_return_if_fail (VALA_IS_ENUM_VALUE (ev));
	vala_gidl_writer_write_indent (self);
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<member name=\"%s\"/>\n", (_tmp0 = vala_enum_value_get_cname (ev)));
	_tmp0 = (g_free (_tmp0), NULL);
}


static void vala_gidl_writer_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* edomain) {
	ValaGIdlWriter * self;
	char* _tmp0;
	self = VALA_GIDL_WRITER (base);
	g_return_if_fail (VALA_IS_ERROR_DOMAIN (edomain));
	if (vala_symbol_get_external_package (VALA_SYMBOL (edomain))) {
		return;
	}
	if (!vala_gidl_writer_check_accessibility (self, VALA_SYMBOL (edomain))) {
		return;
	}
	vala_gidl_writer_write_indent (self);
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<errordomain name=\"%s\"", (_tmp0 = vala_typesymbol_get_cname (VALA_TYPESYMBOL (edomain), FALSE)));
	_tmp0 = (g_free (_tmp0), NULL);
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	vala_code_node_accept_children (VALA_CODE_NODE (edomain), VALA_CODE_VISITOR (self));
	self->priv->indent--;
	vala_gidl_writer_write_indent (self);
	fprintf (self->priv->stream, "</errordomain>\n");
}


static void vala_gidl_writer_real_visit_error_code (ValaCodeVisitor* base, ValaErrorCode* ecode) {
	ValaGIdlWriter * self;
	char* _tmp0;
	self = VALA_GIDL_WRITER (base);
	g_return_if_fail (VALA_IS_ERROR_CODE (ecode));
	vala_gidl_writer_write_indent (self);
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<member name=\"%s\"/>\n", (_tmp0 = vala_error_code_get_cname (ecode)));
	_tmp0 = (g_free (_tmp0), NULL);
}


static void vala_gidl_writer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c) {
	ValaGIdlWriter * self;
	char* _tmp0;
	self = VALA_GIDL_WRITER (base);
	g_return_if_fail (VALA_IS_CONSTANT (c));
	if (vala_symbol_get_external_package (VALA_SYMBOL (c))) {
		return;
	}
	if (!vala_gidl_writer_check_accessibility (self, VALA_SYMBOL (c))) {
		return;
	}
	vala_gidl_writer_write_indent (self);
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<constant name=\"%s\"/>\n", (_tmp0 = vala_constant_get_cname (c)));
	_tmp0 = (g_free (_tmp0), NULL);
}


static void vala_gidl_writer_real_visit_field (ValaCodeVisitor* base, ValaField* f) {
	ValaGIdlWriter * self;
	char* _tmp0;
	self = VALA_GIDL_WRITER (base);
	g_return_if_fail (VALA_IS_FIELD (f));
	if (vala_symbol_get_external_package (VALA_SYMBOL (f))) {
		return;
	}
	if (!vala_gidl_writer_check_accessibility (self, VALA_SYMBOL (f))) {
		return;
	}
	vala_gidl_writer_write_indent (self);
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<field name=\"%s\"/>\n", (_tmp0 = vala_field_get_cname (f)));
	_tmp0 = (g_free (_tmp0), NULL);
}


static char* vala_gidl_writer_get_gidl_type_name (ValaGIdlWriter* self, ValaDataType* type) {
	char* gidl_type;
	g_return_val_if_fail (VALA_IS_GIDL_WRITER (self), NULL);
	g_return_val_if_fail (VALA_IS_DATA_TYPE (type), NULL);
	/* workaround to get GIDL-specific type name*/
	gidl_type = vala_data_type_get_cname (type);
	if (vala_data_type_get_data_type (type) != NULL) {
		char* cname;
		cname = vala_typesymbol_get_cname (vala_data_type_get_data_type (type), FALSE);
		if (g_str_has_prefix (gidl_type, cname)) {
			char* _tmp3;
			char* _tmp2;
			char* _tmp1;
			char* _tmp0;
			_tmp3 = NULL;
			_tmp2 = NULL;
			_tmp1 = NULL;
			_tmp0 = NULL;
			gidl_type = (_tmp3 = g_strconcat ((_tmp0 = vala_symbol_get_full_name (VALA_SYMBOL (vala_data_type_get_data_type (type)))), (_tmp2 = g_utf8_offset_to_pointer (gidl_type, g_utf8_strlen (cname, -1)), g_strndup (_tmp2, g_utf8_offset_to_pointer (_tmp2, g_utf8_strlen (gidl_type, -1) - g_utf8_strlen (cname, -1)) - _tmp2)), NULL), (gidl_type = (g_free (gidl_type), NULL)), _tmp3);
			_tmp1 = (g_free (_tmp1), NULL);
			_tmp0 = (g_free (_tmp0), NULL);
		}
		cname = (g_free (cname), NULL);
	} else {
		if (VALA_IS_DELEGATE_TYPE (type)) {
			ValaDelegateType* _tmp4;
			ValaDelegateType* dt;
			char* cname;
			_tmp4 = NULL;
			dt = (_tmp4 = VALA_DELEGATE_TYPE (type), (_tmp4 == NULL ? NULL : g_object_ref (_tmp4)));
			cname = vala_data_type_get_cname (VALA_DATA_TYPE (dt));
			if (g_str_has_prefix (gidl_type, cname)) {
				char* _tmp8;
				char* _tmp7;
				char* _tmp6;
				char* _tmp5;
				_tmp8 = NULL;
				_tmp7 = NULL;
				_tmp6 = NULL;
				_tmp5 = NULL;
				gidl_type = (_tmp8 = g_strconcat ((_tmp5 = vala_symbol_get_full_name (VALA_SYMBOL (vala_delegate_type_get_delegate_symbol (dt)))), (_tmp7 = g_utf8_offset_to_pointer (gidl_type, g_utf8_strlen (cname, -1)), g_strndup (_tmp7, g_utf8_offset_to_pointer (_tmp7, g_utf8_strlen (gidl_type, -1) - g_utf8_strlen (cname, -1)) - _tmp7)), NULL), (gidl_type = (g_free (gidl_type), NULL)), _tmp8);
				_tmp6 = (g_free (_tmp6), NULL);
				_tmp5 = (g_free (_tmp5), NULL);
			}
			(dt == NULL ? NULL : (dt = (g_object_unref (dt), NULL)));
			cname = (g_free (cname), NULL);
		}
	}
	return gidl_type;
}


static void vala_gidl_writer_write_params (ValaGIdlWriter* self, GeeCollection* params, ValaDataType* instance_type) {
	g_return_if_fail (VALA_IS_GIDL_WRITER (self));
	g_return_if_fail (GEE_IS_COLLECTION (params));
	g_return_if_fail (instance_type == NULL || VALA_IS_DATA_TYPE (instance_type));
	vala_gidl_writer_write_indent (self);
	fprintf (self->priv->stream, "<parameters>\n");
	self->priv->indent++;
	if (instance_type != NULL) {
		char* _tmp0;
		vala_gidl_writer_write_indent (self);
		_tmp0 = NULL;
		fprintf (self->priv->stream, "<parameter name=\"self\" type=\"%s\"/>\n", (_tmp0 = vala_gidl_writer_get_gidl_type_name (self, instance_type)));
		_tmp0 = (g_free (_tmp0), NULL);
	}
	{
		GeeCollection* param_collection;
		GeeIterator* param_it;
		param_collection = params;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = ((ValaFormalParameter*) gee_iterator_get (param_it));
			{
				char* _tmp1;
				vala_gidl_writer_write_indent (self);
				_tmp1 = NULL;
				fprintf (self->priv->stream, "<parameter name=\"%s\" type=\"%s\"", vala_symbol_get_name (VALA_SYMBOL (param)), (_tmp1 = vala_gidl_writer_get_gidl_type_name (self, vala_formal_parameter_get_type_reference (param))));
				_tmp1 = (g_free (_tmp1), NULL);
				if (vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_REF) {
					fprintf (self->priv->stream, " direction=\"inout\"");
					/* in/out paramter*/
					if (vala_data_type_get_takes_ownership (vala_formal_parameter_get_type_reference (param))) {
						fprintf (self->priv->stream, " transfer=\"full\"");
					}
				} else {
					if (vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_OUT) {
						/* out paramter*/
						fprintf (self->priv->stream, " direction=\"out\"");
						if (vala_data_type_get_takes_ownership (vala_formal_parameter_get_type_reference (param))) {
							fprintf (self->priv->stream, " transfer=\"full\"");
						}
					} else {
						/* normal in paramter*/
						if (vala_data_type_get_transfers_ownership (vala_formal_parameter_get_type_reference (param))) {
							fprintf (self->priv->stream, " transfer=\"full\"");
						}
					}
				}
				fprintf (self->priv->stream, "/>\n");
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
	self->priv->indent--;
	vala_gidl_writer_write_indent (self);
	fprintf (self->priv->stream, "</parameters>\n");
}


static void vala_gidl_writer_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* cb) {
	ValaGIdlWriter * self;
	char* _tmp0;
	GeeList* _tmp1;
	self = VALA_GIDL_WRITER (base);
	g_return_if_fail (VALA_IS_DELEGATE (cb));
	if (vala_symbol_get_external_package (VALA_SYMBOL (cb))) {
		return;
	}
	if (!vala_gidl_writer_check_accessibility (self, VALA_SYMBOL (cb))) {
		return;
	}
	vala_gidl_writer_write_indent (self);
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<callback name=\"%s\"", (_tmp0 = vala_typesymbol_get_cname (VALA_TYPESYMBOL (cb), FALSE)));
	_tmp0 = (g_free (_tmp0), NULL);
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	_tmp1 = NULL;
	vala_gidl_writer_write_params (self, GEE_COLLECTION ((_tmp1 = vala_delegate_get_parameters (cb))), NULL);
	(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
	vala_gidl_writer_write_return_type (self, vala_delegate_get_return_type (cb));
	self->priv->indent--;
	vala_gidl_writer_write_indent (self);
	fprintf (self->priv->stream, "</callback>\n");
}


static void vala_gidl_writer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m) {
	ValaGIdlWriter * self;
	char* _tmp0;
	ValaDataType* instance_type;
	GeeCollection* _tmp2;
	self = VALA_GIDL_WRITER (base);
	g_return_if_fail (VALA_IS_METHOD (m));
	if (vala_symbol_get_external_package (VALA_SYMBOL (m))) {
		return;
	}
	/* don't write interface implementation unless it's an abstract or virtual method*/
	if (!vala_gidl_writer_check_accessibility (self, VALA_SYMBOL (m)) || vala_method_get_overrides (m) || (vala_method_get_base_interface_method (m) != NULL && !vala_method_get_is_abstract (m) && !vala_method_get_is_virtual (m))) {
		return;
	}
	vala_gidl_writer_write_indent (self);
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<method name=\"%s\" symbol=\"%s\"", vala_symbol_get_name (VALA_SYMBOL (m)), (_tmp0 = vala_method_get_cname (m)));
	_tmp0 = (g_free (_tmp0), NULL);
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	instance_type = NULL;
	if (vala_method_get_binding (m) == MEMBER_BINDING_INSTANCE) {
		ValaDataType* _tmp1;
		_tmp1 = NULL;
		instance_type = (_tmp1 = vala_ccode_generator_get_data_type_for_symbol (VALA_TYPESYMBOL (vala_symbol_get_parent_symbol (VALA_SYMBOL (m)))), (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), _tmp1);
	}
	_tmp2 = NULL;
	vala_gidl_writer_write_params (self, (_tmp2 = vala_method_get_parameters (m)), instance_type);
	(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
	vala_gidl_writer_write_return_type (self, vala_method_get_return_type (m));
	self->priv->indent--;
	vala_gidl_writer_write_indent (self);
	fprintf (self->priv->stream, "</method>\n");
	(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
}


static void vala_gidl_writer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m) {
	ValaGIdlWriter * self;
	char* name;
	char* _tmp2;
	GeeCollection* _tmp3;
	ValaDataType* _tmp4;
	self = VALA_GIDL_WRITER (base);
	g_return_if_fail (VALA_IS_CREATION_METHOD (m));
	if (vala_symbol_get_external_package (VALA_SYMBOL (m))) {
		return;
	}
	if (!vala_gidl_writer_check_accessibility (self, VALA_SYMBOL (m))) {
		return;
	}
	name = g_strdup ("new");
	if (g_str_has_prefix (vala_symbol_get_name (VALA_SYMBOL (m)), ".new.")) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		name = (_tmp1 = (_tmp0 = g_utf8_offset_to_pointer (vala_symbol_get_name (VALA_SYMBOL (m)), ((glong) 5)), g_strndup (_tmp0, g_utf8_offset_to_pointer (_tmp0, g_utf8_strlen (vala_symbol_get_name (VALA_SYMBOL (m)), -1) - 5) - _tmp0)), (name = (g_free (name), NULL)), _tmp1);
	}
	vala_gidl_writer_write_indent (self);
	_tmp2 = NULL;
	fprintf (self->priv->stream, "<constructor name=\"%s\" symbol=\"%s\"", name, (_tmp2 = vala_method_get_cname (VALA_METHOD (m))));
	_tmp2 = (g_free (_tmp2), NULL);
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	_tmp3 = NULL;
	vala_gidl_writer_write_params (self, (_tmp3 = vala_method_get_parameters (VALA_METHOD (m))), NULL);
	(_tmp3 == NULL ? NULL : (_tmp3 = (g_object_unref (_tmp3), NULL)));
	_tmp4 = NULL;
	vala_gidl_writer_write_return_type (self, (_tmp4 = vala_ccode_generator_get_data_type_for_symbol (VALA_TYPESYMBOL (vala_symbol_get_parent_symbol (VALA_SYMBOL (m))))));
	(_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL)));
	self->priv->indent--;
	vala_gidl_writer_write_indent (self);
	fprintf (self->priv->stream, "</constructor>\n");
	name = (g_free (name), NULL);
}


static void vala_gidl_writer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop) {
	ValaGIdlWriter * self;
	char* _tmp0;
	self = VALA_GIDL_WRITER (base);
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	if (!vala_gidl_writer_check_accessibility (self, VALA_SYMBOL (prop)) || vala_property_get_overrides (prop) || vala_property_get_base_interface_property (prop) != NULL) {
		return;
	}
	vala_gidl_writer_write_indent (self);
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<property name=\"%s\" type=\"%s\"", vala_symbol_get_name (VALA_SYMBOL (prop)), (_tmp0 = vala_gidl_writer_get_gidl_type_name (self, vala_property_get_type_reference (prop))));
	_tmp0 = (g_free (_tmp0), NULL);
	if (vala_property_get_get_accessor (prop) != NULL) {
		fprintf (self->priv->stream, " readable=\"1\"");
	}
	if (vala_property_get_set_accessor (prop) != NULL) {
		fprintf (self->priv->stream, " writable=\"1\"");
	}
	fprintf (self->priv->stream, "/>\n");
}


static void vala_gidl_writer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig) {
	ValaGIdlWriter * self;
	char* _tmp0;
	GeeList* _tmp1;
	self = VALA_GIDL_WRITER (base);
	g_return_if_fail (VALA_IS_SIGNAL (sig));
	if (!vala_gidl_writer_check_accessibility (self, VALA_SYMBOL (sig))) {
		return;
	}
	vala_gidl_writer_write_indent (self);
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<signal name=\"%s\"", (_tmp0 = vala_signal_get_cname (sig)));
	_tmp0 = (g_free (_tmp0), NULL);
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	_tmp1 = NULL;
	vala_gidl_writer_write_params (self, GEE_COLLECTION ((_tmp1 = vala_signal_get_parameters (sig))), NULL);
	(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
	vala_gidl_writer_write_return_type (self, vala_signal_get_return_type (sig));
	self->priv->indent--;
	vala_gidl_writer_write_indent (self);
	fprintf (self->priv->stream, "</signal>\n");
}


static void vala_gidl_writer_write_indent (ValaGIdlWriter* self) {
	gint i;
	g_return_if_fail (VALA_IS_GIDL_WRITER (self));
	i = 0;
	for (i = 0; i < self->priv->indent; i++) {
		fputc ('\t', self->priv->stream);
	}
}


static void vala_gidl_writer_write_return_type (ValaGIdlWriter* self, ValaDataType* type) {
	char* _tmp0;
	g_return_if_fail (VALA_IS_GIDL_WRITER (self));
	g_return_if_fail (VALA_IS_DATA_TYPE (type));
	vala_gidl_writer_write_indent (self);
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<return-type type=\"%s\"", (_tmp0 = vala_gidl_writer_get_gidl_type_name (self, type)));
	_tmp0 = (g_free (_tmp0), NULL);
	if (vala_data_type_get_transfers_ownership (type)) {
		fprintf (self->priv->stream, " transfer=\"full\"");
	}
	fprintf (self->priv->stream, "/>\n");
}


static gboolean vala_gidl_writer_check_accessibility (ValaGIdlWriter* self, ValaSymbol* sym) {
	g_return_val_if_fail (VALA_IS_GIDL_WRITER (self), FALSE);
	g_return_val_if_fail (VALA_IS_SYMBOL (sym), FALSE);
	if (vala_symbol_get_access (sym) == VALA_SYMBOL_ACCESSIBILITY_PUBLIC || vala_symbol_get_access (sym) == VALA_SYMBOL_ACCESSIBILITY_PROTECTED) {
		return TRUE;
	}
	return FALSE;
}


/**
 * Code visitor generating .gidl file for the public interface.
 */
ValaGIdlWriter* vala_gidl_writer_new (void) {
	ValaGIdlWriter * self;
	self = g_object_newv (VALA_TYPE_GIDL_WRITER, 0, NULL);
	return self;
}


static void vala_gidl_writer_class_init (ValaGIdlWriterClass * klass) {
	vala_gidl_writer_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaGIdlWriterPrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_gidl_writer_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_namespace = vala_gidl_writer_real_visit_namespace;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_gidl_writer_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_gidl_writer_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_gidl_writer_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum = vala_gidl_writer_real_visit_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum_value = vala_gidl_writer_real_visit_enum_value;
	VALA_CODE_VISITOR_CLASS (klass)->visit_error_domain = vala_gidl_writer_real_visit_error_domain;
	VALA_CODE_VISITOR_CLASS (klass)->visit_error_code = vala_gidl_writer_real_visit_error_code;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constant = vala_gidl_writer_real_visit_constant;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_gidl_writer_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_delegate = vala_gidl_writer_real_visit_delegate;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_gidl_writer_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_creation_method = vala_gidl_writer_real_visit_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_gidl_writer_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_signal = vala_gidl_writer_real_visit_signal;
}


static void vala_gidl_writer_init (ValaGIdlWriter * self) {
	self->priv = VALA_GIDL_WRITER_GET_PRIVATE (self);
}


static void vala_gidl_writer_dispose (GObject * obj) {
	ValaGIdlWriter * self;
	self = VALA_GIDL_WRITER (obj);
	(self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL)));
	(self->priv->stream == NULL ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL)));
	(self->priv->gobject_type == NULL ? NULL : (self->priv->gobject_type = (g_object_unref (self->priv->gobject_type), NULL)));
	G_OBJECT_CLASS (vala_gidl_writer_parent_class)->dispose (obj);
}


GType vala_gidl_writer_get_type (void) {
	static GType vala_gidl_writer_type_id = 0;
	if (G_UNLIKELY (vala_gidl_writer_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaGIdlWriterClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_gidl_writer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaGIdlWriter), 0, (GInstanceInitFunc) vala_gidl_writer_init };
		vala_gidl_writer_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaGIdlWriter", &g_define_type_info, 0);
	}
	return vala_gidl_writer_type_id;
}




