// TTY-Grin
// Copyright (C) 2001 Daniel Beer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include "grinlisp.h"
#include "unix.h"
#include "tgws.h"

grinlisp *config;
int config_old_directory=0;

void
config_swapdir(void) {
  int x=open(".", O_RDONLY);
  if(config_old_directory>0) {
    fchdir(config_old_directory);
    close(config_old_directory);
  }
  config_old_directory=x;
}

void
config_lisp_error(const lisp::error& e) {
  string out(_("LISP error: "));
  out+=e.get_comment();
  if(!e.get_cause().eq(lisp::nil)) {
    out+=": ";
    out+=e.get_cause().print();
  }

  tgws::error_message(out.c_str());
}

void
config_lisp_error(const lisp::return_from& r) {
  string out(_("Invalid return-from: "));
  out+=r.get_block_name().print();
  tgws::error_message(out.c_str());
}

lisp::object
config_execute(lisp::library& lib, int argc, lisp::object *argv) {
  lisp::object r;
  tgws::suspend();
  try {
    lisp::object f=lib.get_function("execute");
    r=lisp::as_function(f).funcall(lib, argc, argv);
  }
  catch(lisp::error e) {
    tgws::unsuspend();
    throw e;
  }
  tgws::unsuspend();
  return r;
}

enum tgws::which_style
config_color_name(const char *name) {
  static struct {
    const char *name;
    enum tgws::which_style style;
  } names[]={
    {"TEXT", tgws::TEXT},
    {"ENTRY", tgws::ENTRY},
    {"ENTRY-DISABLED", tgws::ENTRY_DISABLED},
    {"LIST", tgws::LIST},
    {"SELECTION", tgws::SELECTION},
    {"TITLE", tgws::TITLE},
    {"EDIT", tgws::EDIT},
    {"MENU", tgws::MENU},
    {"FRAME", tgws::FRAME},
    {"KEY", tgws::KEY},
    {"QUOTED", tgws::QUOTED},
    {"SIGNATURE", tgws::SIGNATURE},
    {0, tgws::SIGNATURE}
  };

  for(int i=0;names[i].name;i++)
    if(!strcmp(name, names[i].name)) return names[i].style;
  throw lisp::error(_("Unknown attribute"));
}

const char *
config_color_to_text(int color) {
  static struct {
    const char *name;
    int color;
  } colors[]={
    {"BLACK", COLOR_BLACK},
    {"RED", COLOR_RED},
    {"GREEN", COLOR_GREEN},
    {"YELLOW", COLOR_YELLOW},
    {"BLUE", COLOR_BLUE},
    {"MAGENTA", COLOR_MAGENTA},
    {"CYAN", COLOR_CYAN},
    {"WHITE", COLOR_WHITE},
    {0, 0}
  };

  for(int i=0;colors[i].name;i++)
    if(colors[i].color==color) return colors[i].name;

  return "UNKNOWN";
}

lisp::object
config_color(lisp::library& lib, int argc, lisp::object *argv) {
  if(argc<1) throw lisp::error(_("Insufficient arguments"));
  if(argc>1) throw lisp::error(_("Excess arguments"));

  enum tgws::which_style w=config_color_name(as_atom(argv[0]).get_text());
  lisp::structure *c=new lisp::structure("COLOR",
					 lib.getstruct("COLOR"));

  short fg, bg;
  if(pair_content(w+1, &fg, &bg)==ERR) {
    (*c)[0]=lisp::object(new lisp::atom("WHITE"));
    (*c)[1]=lisp::object(new lisp::atom("BLACK"));
  } else {
    (*c)[0]=lisp::object(new lisp::atom(config_color_to_text(fg)));
    (*c)[1]=lisp::object(new lisp::atom(config_color_to_text(bg)));
  }

  int attr=tgws::default_style.get_attr(w);
  if(attr&A_BOLD) (*c)[2]=lisp::t;
  if(attr&A_REVERSE) (*c)[3]=lisp::t;
  
  return lisp::object(c);
}

int
config_text_to_color(const char *text) {
  static struct {
    const char *name;
    int color;
  } colors[]={
    {"BLACK", COLOR_BLACK},
    {"RED", COLOR_RED},
    {"GREEN", COLOR_GREEN},
    {"YELLOW", COLOR_YELLOW},
    {"BLUE", COLOR_BLUE},
    {"MAGENTA", COLOR_MAGENTA},
    {"CYAN", COLOR_CYAN},
    {"WHITE", COLOR_WHITE},
    {0, 0}
  };

  for(int i=0;colors[i].name;i++)
    if(!strcmp(colors[i].name, text)) return colors[i].color;

  throw lisp::error(_("Unknown color"));
}

lisp::object
config_setf_color(lisp::library& lib, int argc, lisp::object *argv) {
  if(argc<2) throw lisp::error(_("Insufficient arguments"));
  if(argc>2) throw lisp::error(_("Excess arguments"));

  enum tgws::which_style w=config_color_name(as_atom(argv[1]).get_text());
  lisp::object fg=as_structure(argv[0])[0];
  lisp::object bg=assume_structure(argv[0])[1];
  lisp::object bold=assume_structure(argv[0])[2];
  lisp::object reverse=assume_structure(argv[0])[3];

  init_pair(w+1, fg.eq(lisp::nil)?COLOR_WHITE:
	    config_text_to_color(as_atom(fg).get_text()),
	    bg.eq(lisp::nil)?COLOR_BLACK:
	    config_text_to_color(as_atom(bg).get_text()));
  int x=COLOR_PAIR(w+1);
  if(!bold.eq(lisp::nil)) x|=A_BOLD;
  if(!reverse.eq(lisp::nil)) x|=A_REVERSE;
  tgws::default_style.set_attr(w, x);

  return argv[0];
}

void
config_init(void) {
  config=new grinlisp;

  { // Colours
    lisp::structdef c;
    c.push_back("FG");
    c.push_back("BG");
    c.push_back("BOLD");
    c.push_back("REVERSE");
    config->defstruct("COLOR", c);
    config->set_function("COLOR", lisp::object
			 (new lisp::system_function(config_color)));
    config->set_function("SETF COLOR", lisp::object
			 (new lisp::system_function(config_setf_color)));
  }

  { // Defaults
    config->set_variable("FROM", lisp::object(new lisp::str(getenv("USER"))));
    struct passwd *p=getpwuid(getuid());
    char *x=strchr(p->pw_gecos, ',');
    if(x) *x=0;
    const char *ed=getenv("EDITOR");
    config->set_variable("OUTBOX", lisp::object(new lisp::str("/dev/null")));
    config->set_variable("EDITOR", lisp::object(new lisp::str(ed?ed:"vi")));
    config->set_variable("REPLY-TO", lisp::object(new lisp::str("")));
    config->set_variable("ORGANIZATION", lisp::object(new lisp::str("")));
    config->set_variable("REAL-NAME",
			 lisp::object(new lisp::str(p->pw_gecos)));
    config->set_variable("SENDMAIL",
			 lisp::object(new lisp::str("/usr/sbin/sendmail -t")));
    config->set_variable("AUTO-EXPAND", lisp::t);
    string inbox(access("/var/mail", F_OK)?"/var/spool/mail/":"/var/mail/");
    inbox+=getenv("USER");
    config->set_variable("INBOX", lisp::object(new lisp::str(inbox.c_str())));
    string signature(getenv("HOME"));
    signature+="/.signature";
    config->set_variable("SIGNATURE",
			 lisp::object(new lisp::str(signature.c_str())));
    string maildir(getenv("HOME"));
    maildir+="/mail";
    config->set_variable("LOCAL-MAIL",
			 lisp::object(new lisp::str(maildir.c_str())));
    config->set_variable("ISPELL", lisp::object(new lisp::str("ispell -a")));
  }

  config->set_function("execute", config->get_function("EXECUTE"));
  config->set_function("EXECUTE", lisp::object
		       (new lisp::system_function(config_execute)));

  config_swapdir();
  umask(077);
  chdir(getenv("HOME"));
  if(chdir(".ttygrin")) {
    if(mkdir(".ttygrin", 0700)) {
      config_swapdir();
      tgws::error_message("Couldn't create ~/.ttygrin");
      return;
    }
    chdir(".ttygrin");
  }

  tgws::active_screen->push();
  if(!access("config", R_OK)) {
    try {
      config->load("config");
      lisp::object m=config->get_variable("LOCAL-MAIL");
      const char *maildir=as_str(m).get_text();
      if(access(maildir, F_OK)) mkdir(maildir, 0700);
    }
    catch(lisp::error e) {
      config_lisp_error(e);
    }
    catch(lisp::return_from r) {
      config_lisp_error(r);
    }
  }
  tgws::active_screen->pop();

  config_swapdir();
  lisp::gc();
}

const char *
config_get_string(const char *what) {
  try {
    const char *t=lisp::as_str(config->get_variable(what)).get_text();
    lisp::gc();
    return t;
  }
  catch(lisp::error e) {
    config_lisp_error(e);
    throw config_error(0);
  }
  catch(lisp::return_from r) {
    config_lisp_error(r);
    throw config_error(0);
  }
}

bool
config_get_bool(const char *what) {
  try {
    bool t=!config->get_variable(what).eq(lisp::nil);
    lisp::gc();
    return t;
  }
  catch(lisp::error e) {
    config_lisp_error(e);
    throw config_error(0);
  }
  catch(lisp::return_from r) {
    config_lisp_error(r);
    throw config_error(0);
  }
}

const char *
config_filename_to_mimetype(const char *filename) {
  try {
    return config->filename_to_mimetype(filename);
  }
  catch(lisp::error e) {
    config_lisp_error(e);
    throw config_error(0);
  }
  catch(lisp::return_from r) {
    config_lisp_error(r);
    throw config_error(0);
  }
}

bool
config_handle_attachment(const char *file, const char *type,
			 const char *name) {
  try {
    bool t=config->handle_attachment(file, type, name);
    lisp::gc();
    return t;
  }
  catch(lisp::error e) {
    config_lisp_error(e);
    throw config_error(0);
  }
  catch(lisp::return_from r) {
    config_lisp_error(r);
    throw config_error(0);
  }
  return false;
}
