// TTY-Grin
// Copyright (C) 2001 Daniel Beer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <dirent.h>
#include <fcntl.h>
#include <sys/types.h>
#include "addressbook.h"
#include "unix.h"
#include "tgws.h"

void
tty_grin_addressbook__edit(tgws::clist_node *ent, const char *tn) {
  tgws::active_screen->push();
  {
    tgws::title *t=new tgws::title(tn);
    t->size(1, COLS);
    t->position(0, 0);
    t->show();
    tgws::keyblock *k=new tgws::keyblock;
    k->add("^C", _("Cancel"));
    k->size(1, COLS);
    k->position(LINES-1, 0);
    k->show();
    for(int i=0;i<3;i++) {
      tgws::label *l=new tgws::label(ent->get_parent()->get_title(i));
      l->size(1, 15);
      l->position(i+2, 0);
      l->show();
    }
  }
  tgws::entry *e[3];
  for(int i=0;i<3;i++) {
    e[i]=new tgws::entry;
    e[i]->set_text(ent->get_text(i));
    e[i]->size(1, COLS-15);
    e[i]->position(i+2, 15);
    e[i]->show();
  }
  int state(0), key;
  do {
    key=e[state]->run();
    switch(key) {
    case KEY_UP: if(!(state--)) state=2; break;
    case KEY_DOWN: if((++state)>=3) state=0; break;
    }
  } while(key!='\n'&&key!=3);
  if(key=='\n')
    for(int i=0;i<3;i++) ent->set_text(i, e[i]->get_text());
  tgws::active_screen->pop();
}

void
tty_grin_addressbook(addressbook& abook, const char *name) {
  tgws::active_screen->push();
  {
    tgws::title *t=new tgws::title(name);
    t->size(1, COLS);
    t->position(0, 0);
    t->show();
    tgws::keyblock *k=new tgws::keyblock;
    k->add("<", _("Back"));
    k->add("^N", _("New"));
    k->add("^K", _("Delete"));
    k->add("^E", _("Edit"));
    k->size(1, COLS);
    k->position(LINES-1, 0);
    k->show();
  }
  tgws::clist *book=new tgws::clist(3);
  for(int i=0;i<abook.size();i++) {
    tgws::clist_node *n=book->insert(0);
    n->set_text(0, abook[i].get_key());
    n->set_text(1, abook[i].get_comment());
    n->set_text(2, abook[i].get_address());
  }
  book->set_width(0, COLS/5);
  book->set_width(1, COLS*2/5);
  book->set_width(2, COLS);
  book->set_title(0, _("Key"));
  book->set_title(1, _("Name"));
  book->set_title(2, _("Address"));
  book->set_headings(true);
  book->size(LINES-4, COLS);
  book->position(2, 0);
  book->show();

  int key;
  do {
    key=book->run();
    switch(key) {
    case 14:
      tty_grin_addressbook__edit(book->insert(0), _("New entry"));
      break;
    case 11:
      if(book->get_cursor()&&
	 tgws::yes_or_no(_("Delete this address (yes or no)?")))
	delete book->get_cursor();
      break;
    case 5:
      if(book->get_cursor())
	tty_grin_addressbook__edit(book->get_cursor(),
				   book->get_cursor()->get_text(0));
      break;
    }
  } while(key!='<'&&key!=','&&key!=KEY_LEFT);

  abook.clear();
  for(tgws::clist_node *n=book->get_first();n;n=n->get_next()) {
    addressbook_entry *e=&(abook.append());
    e->set_key(n->get_text(0));
    e->set_comment(n->get_text(1));
    e->set_address(n->get_text(2));
  }
  
  tgws::active_screen->pop();
}

void
tty_grin_addresses(void) {
  tgws::clist *books;
  {
    string config_dir(getenv("HOME"));
    config_dir+="/.ttygrin";
    DIR *d=opendir(config_dir.c_str());
    if(!d) {
      tgws::error_message(strerror(errno));
      return;
    }
    tgws::active_screen->push();
    books=new tgws::clist(1);
    struct dirent *ent;
    while((ent=readdir(d)))
      if(!strncasecmp(ent->d_name, "addresses:", 10))
	books->insert(0)->set_text(0, ent->d_name+10);
    books->set_width(0, COLS);
    books->size(LINES-4, COLS);
    books->position(2, 0);
    books->show();
    tgws::title *t=new tgws::title(_("Address books"));
    t->size(1, COLS);
    t->position(0, 0);
    t->show();
    tgws::keyblock *k=new tgws::keyblock;
    k->add("<", _("Back"));
    k->add("^N", _("New"));
    k->add("^K", _("Delete"));
    k->size(1, COLS);
    k->position(LINES-1, 0);
    k->show();
  }
  int key;
  do {
    key=books->run();
    switch(key) {
    case 14:
      {
	string name;
	if(tgws::get_line(_("Name:"), name)) {
	  string filename(getenv("HOME"));
	  filename+="/.ttygrin/addresses:";
	  filename+=name;
	  int fd=open(filename.c_str(), O_CREAT|O_WRONLY|O_EXCL, 0600);
	  if(fd<0) tgws::error_message(strerror(errno));
	  else {
	    close(fd);
	    books->insert(0)->set_text(0, name.c_str());
	  }
	}
      }
      break;
    case 11:
      if(books->get_cursor()&&
	 tgws::yes_or_no(_("Delete this address book (yes or no)?"))) {
	string filename(getenv("HOME"));
	filename+="/.ttygrin/addresses:";
	filename+=books->get_cursor()->get_text(0);
	unlink(filename.c_str());
	delete books->get_cursor();
      }
      break;
    case ' ':
    case KEY_RIGHT:
    case '\n':
      if(books->get_cursor()) {
	string filename(getenv("HOME"));
	filename+="/.ttygrin/addresses:";
	filename+=books->get_cursor()->get_text(0);
	addressbook book(filename.c_str());
	tty_grin_addressbook(book, books->get_cursor()->get_text(0));
      }
      break;
    }
  } while(key!='<'&&key!=','&&key!=KEY_LEFT);
  tgws::active_screen->pop();
}

bool
tty_grin_addresses_choose(string& result) {
  string src(getenv("HOME"));
  src+="/.ttygrin";
  DIR *d=opendir(src.c_str());
  if(!d) return false;
  tgws::menu m;
  struct dirent *ent;
  while((ent=readdir(d)))
    if(!strncmp(ent->d_name, "addresses:", 10))
      m.add_item(ent->d_name+10);
  closedir(d);
  if(!m.count()) return false;
  m.size(LINES/3, COLS/4);
  m.position(LINES/3, COLS/4);
  m.show();
  int key;
  do {
    key=m.run();
  } while(key!=3&&key!='\n');
  if(key==3) return false;
  src+="/addresses:";
  src+=m.get_cursor_name();
  addressbook book(src.c_str());
  tgws::menu sub;
  for(int i=0;i<book.size();i++) sub.add_item(book[i].get_comment());
  do {
    key=sub.run();
  } while(key!=3&&key!='\n');
  if(key==3) return false;
  addressbook_entry& e(book[sub.get_cursor_number()]);
  result=e.get_comment();
  result+=" <";
  result+=e.get_address();
  result+='>';
  return true;
}
