// TTY-Grin Widget Set
// Copyright (C) 2001 Daniel Beer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <ctype.h>
#include <regex.h>
#include "tgws.h"

void
tgws::pager::size(int h, int w) {
  input::size(h, w);
  rewrap();
}

void
tgws::pager::draw(bool running) {
  wbkgdset(win, look->get_attr(TEXT));
  werase(win);
  wrap_t::const_iterator i=scr_line;
  unsigned int begin=scr, lines;
  for(lines=0;lines<(unsigned)height&&i!=wrap.end();lines++) {
    unsigned int len=*i-1;

    if(begin+len<buffer.size()&&!isspace(buffer[begin+len])&&
       buffer[begin+len]!='\n') len++;

    while(len&&buffer[begin+len-1]=='\r') len--;
    draw_line(begin, len);
    begin+=*(i++);
  }
  mvwaddstr(win, height-1, 0, "");
  input::draw();
}

int
tgws::pager::run(void) {
  for(;;) {
    draw();
    doupdate();
    switch(int key=wgetch(win)) {
    case KEY_HOME: home(); break;
    case KEY_LL:
    case KEY_END: end(); break;
    case KEY_UP: up(); break;
    case KEY_DOWN: down(); break;
    case KEY_PPAGE: for(int i=0;i<height;i++) up(); break;
    case KEY_NPAGE: for(int i=0;i<height;i++) down(); break;
    case '/':
    case '?':
      {
	string pattern;
	if(tgws::get_line("Find:", pattern)) {
	  if(pattern.size()) last_pattern=pattern;
	  else pattern=last_pattern;

	  regex_t preg;
	  regcomp(&preg, pattern.c_str(), REG_EXTENDED|REG_ICASE|REG_NOSUB);

	  wrap_t::iterator find_line=scr_line;
	  unsigned int find=scr, find_line_number=scr_line_number;

	  do {
	    find+=*(find_line++);
	    find_line_number++;

	    if(find_line!=wrap.end()) {
	      char line[*find_line];
	      for(unsigned int i=0;i<*find_line;i++) line[i]=buffer[find+i];
	      line[*find_line-1]=0;
	      if(!regexec(&preg, line, 0, 0, 0)) break;
	    }
	  } while(find_line!=wrap.end());

	  if(find_line==wrap.end()) tgws::error_message("Pattern not found.");
	  else {
	    scr=find;
	    scr_line=find_line;
	    scr_line_number=find_line_number;
	  }
	}
      }
      break;
    default: return key;
    }
  }
}

void
tgws::pager::draw_line(unsigned int begin, unsigned int len) {
  for(unsigned int i=0;i<len;i++) waddch(win, (unsigned char)buffer[begin+i]);
  waddch(win, '\n');
}

void
tgws::pager::rewrap(void) {
  unsigned int max=(unsigned)width-1, begin=0, end;

  wrap.clear();
  while(begin<buffer.size()) {
    for(end=begin;end<buffer.size()&&buffer[end]!='\n'&&end-begin<max;end++);
    if(end<buffer.size()&&buffer[end]!='\n') {
      while(isspace(buffer[end])&&end>begin) end--;
      while(!isspace(buffer[end])&&end>begin) end--;
      if(begin==end) end=begin+max-1;
      else end++;
    } else end++;
    wrap.push_back(end-begin);
    begin=end;
  }
  scr_line=wrap.begin();
  scr=scr_line_number=0;
}

void
tgws::pager::up(void) {
  if(scr_line_number) {
    --scr_line_number;
    scr-=*(--scr_line);
  }
}

void
tgws::pager::down(void) {
  if(scr_line_number+height<wrap.size()) {
    ++scr_line_number;
    scr+=*(scr_line++);
  }
}

void
tgws::pager::home(void) {
  scr=scr_line_number=0;
  scr_line=wrap.begin();
}

void
tgws::pager::end(void) {
  scr_line=wrap.end();
  scr_line_number=wrap.size();
  scr=buffer.size();
  for(int i=0;i<height;i++) up();
}

void
tgws::pager::read_file(istream& in) {
  buffer="";
  char buf[4096];
  in.getline(buf, sizeof(buf));
  while(!in.fail()) {
    buffer+=buf;
    buffer+='\n';
    in.getline(buf, sizeof(buf));
  }
  rewrap();
}
