// TTY-Grin
// Copyright (C) 2001 Daniel Beer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <pfstream.h>
#include <fstream>
#include "composition.h"

char composition_attachment::basis_64[]=
"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

bool
composition_attachment::extract(ostream& out, const char *lend) const {
  out<<"Content-Type: "<<mime;
  if(name.size()) out<<"; name=\""<<name<<"\"";
  out<<lend;

  out<<"Content-Disposition: ";
  if(where==INLINE) out<<"inline";
  else out<<"attachment; filename=\""<<name<<"\"";
  out<<lend;

  if(how==BASE64) out<<"Content-Transfer-Encoding: base64"<<lend;

  out<<lend;

  ifstream in(filename.c_str());
  if(in.fail()) return false;

  if(how==PLAIN) {
    char buf[1024];

    in.getline(buf, sizeof(buf));
    while(!in.fail()) {
      if(buf[0]=='.'&&!buf[1]) out<<". "<<lend;
      else out<<buf<<lend;
      in.getline(buf, sizeof(buf));
    }
  } else {
    int chunks=0;
    do {
      int a=in.get();
      if(a>=0) {
	unsigned char o[5];
	o[0]=basis_64[(a&0xfc)>>2];
	int b=in.get();
	if(b<0) {
	  o[1]=basis_64[(a&0x03)<<4];
	  o[2]=o[3]='=';
	} else {
	  o[1]=basis_64[((a&0x03)<<4)|((b&0xf0)>>4)];
	  int c=in.get();
	  if(c<0) {
	    o[2]=basis_64[(b&0x0f)<<2];
	    o[3]='=';
	  } else {
	    o[2]=basis_64[((b&0x0f)<<2)|((c&0xc0)>>6)];
	    o[3]=basis_64[c&0x3f];
	  }
	}
	o[4]=0;
	out<<o;
      }
      if(++chunks==18) {
	chunks=0;
	out<<lend;
      }
    } while(!in.fail());
    if(chunks) out<<lend;
  }

  in.close();
  return true;
}

const char *
composition::get_header(const char *name) const {
  map<string, string, lt_string>::const_iterator i;

  if((i=headers.find(name))==headers.end()) return "";
  else return (*i).second.c_str();
}

void
composition::extract_content(ostream& out, const char *lend) const {
  out<<"Content-Type: text/plain"<<lend<<
    "Content-Disposition: inline"<<lend<<lend;
  const char *start=body.c_str();
  int end;

  while(*start) {
    if(*start=='>') {
      for(end=0;start[end]&&start[end]!='\n';end++);
    } else {
      for(end=0;end<72&&start[end]&&start[end]!='\n';end++);
      if(end>=72) {
	while(end&&!isspace(start[end])) end--;
	if(!end) end=72;
      }
    }
    for(int i=0;i<end;i++) out.put(start[i]);
    if(end==1&&*start=='.') out.put(' ');
    out<<lend;
    start+=end;
    while(*start!='\n'&&isspace(*start)) start++;
    if(*start=='\n') start++;
  }
}

void
composition::extract_headers(ostream& out, const char *lend) const {
  map<string, string, lt_string>::const_iterator h;
  
  for(h=headers.begin();h!=headers.end();h++) {
    const string& name=(*h).first;
    
    if(name!="To"&&name!="Cc"&&name!="Bcc")
      out<<name<<": "<<(*h).second<<lend;
  }
}

void
composition::extract(ostream& out, const char *lend) const {
  extract_headers(out, lend);
  if(attachments.empty()) extract_content(out, lend);
  else {
    char boundary[80];
    snprintf(boundary, sizeof(boundary), "%x.%x", rand(), time(NULL));
    out<<"Content-Type: multipart/mixed; boundary=\""<<boundary<<
      "\""<<lend<<lend;
    out<<"--"<<boundary<<lend;
    extract_content(out, lend);
    out<<lend;
    vector<composition_attachment>::const_iterator i;
    for(i=attachments.begin();i!=attachments.end();i++) {
      out<<"--"<<boundary<<lend;
      (*i).extract(out, lend);
      out<<lend;
    }
    out<<"--"<<boundary<<"--"<<lend;
  }
}

void
composition::attach(const char *filename, const char *mime, const char *name) {
  composition_attachment a(filename, mime, name,
			   composition_attachment::BASE64,
			   composition_attachment::ATTACHMENT);
  attachments.push_back(a);
}

void
composition::forward(const char *filename) {
  composition_attachment a(filename, "message/rfc822", "",
			   composition_attachment::PLAIN,
			   composition_attachment::INLINE);
  attachments.push_back(a);
}

void
composition::private_headers(ostream& out, const char *lend) const {
  static char *which[]={"To", "Cc", "Bcc", 0};
  int i;

  for(i=0;which[i];i++) {
    map<string, string, lt_string>::const_iterator h;

    if((h=headers.find(which[i]))!=headers.end())
      out<<which[i]<<": "<<(*h).second<<lend;
  }
}

bool
composition::sendmail(const char *command) const {
  string c("|");
  c+=command;
  opfstream mailer(c.c_str());

  if(mailer.fail()) return false;
  private_headers(mailer);
  extract(mailer);
  mailer.flush();
  mailer.close();
  return true;
}

bool
composition::news(const char *server) const {
  struct hostent *info;
  int port;

  {
    const char *div=strchr(server, ':');
    if(div) {
      info=gethostbyname(string(server, 0, div-server).c_str());
      port=atoi(div+1);
    } else {
      info=gethostbyname(server);
      port=119;
    }
    if(!info) return false;
  }

  int fd=socket(PF_INET, SOCK_STREAM, 0);
  struct sockaddr_in addr;
  addr.sin_family=AF_INET;
  addr.sin_port=htons(port);
  addr.sin_addr=*(struct in_addr *)info->h_addr;
  if(connect(fd, (struct sockaddr *)&addr, sizeof(addr))<0) {
    close(fd);
    return false;
  }
  ofstream out(fd);
  ifstream in(fd);
  char buf[1024];

  in.getline(buf, sizeof(buf));
  if(atoi(buf)!=200) {
    close(fd);
    return false;
  }
  out<<"POST\r\n";
  out.flush();
  in.getline(buf, sizeof(buf));
  if(atoi(buf)!=340) {
    close(fd);
    return false;
  }
  extract(out, "\r\n");
  out<<".\r\n";
  out.flush();
  in.getline(buf, sizeof(buf));
  if(atoi(buf)!=240) {
    close(fd);
    return false;
  }
  out<<"QUIT\r\n";
  out.flush();
  close(fd);
  return true;
}

void
composition::clear(void) {
  body="";
  headers.clear();
  attachments.clear();
}

void
composition::append(istream& in) {
  int x;

  while((x=in.get())>=0) body+=(char)x;
}

void
composition::outbox(ostream& out) const {
  time_t now=time(NULL);
  out<<"From "<<getenv("USER")<<' '<<ctime(&now);
  private_headers(out);
  extract(out);
}
