/* treewm - an X11 window manager.
 * Copyright (c) 2000 Thomas Jger <thehunter2000@web.de>
 * This code is released under the terms of the GNU GPL. See
 * the included file LICENSE for details.
 */

#include "desktop.h"
#include "clienttree.h"
#include "icon.h"
#include "sceme.h"
#include "action.h"
#include "manager.h"
#include "clientinfo.h"
#include "tile.h"
#include <signal.h>
#include <string.h>

unsigned long Desktop::GetRegionMask(int x, int y, Client * &t) {
  x -= x_root;
  y += THeight - y_root;
  if (0 <= y && y <= THeight && 0 <= x && x <= width) {
    if (!t)
      for (t=firstchild;t;t=t->next)
        if (t->mapped && t->o.HasTBEntry && t->TBEx <= x && x <= t->TBEy)
          break;
    if (!t && TBx <= x && x <= TBy)
      t=this;
    if (t)
      return ((x >= (t == this ? TBy : t->TBEy) - ((t->flags & CF_FIXEDSIZE) ? 1 : 2)*BUTTONWIDTH)
                ? (x >= (t == this ? TBy : t->TBEy) - BUTTONWIDTH) ? T_F2 : T_F1 :
                T_BAR);
  }
  return 0;
};

void Desktop::AutoClose() {
  if (!autoclose || !parent)
    return;
  for (Client *i = firstchild;i;i = i->next)
    if (i->mapped || i->icon)
      return;
  SendWMDelete(true);
}


Desktop::Desktop(Desktop *p, Options *opt, ClientInfo *cin, Window win, char *wname) : Client(p,opt) {
  flags = 0;
  firstchild = NULL;
  firsticon = NULL;
  PosX = 0;
  PosY = 0;
  for (int i=0; i!=3; ++i)
    LastRaised[i] = 0;
  focus = NULL;
  window = win;
  if (parent)
    if (wname)
      XStoreName(dpy,win,wname); else
      XStoreName(dpy,win,"");
    else
      ct->rootname = wname;
  wmname = new char[8];
  strcpy(wmname,"Desktop");
  ci = cin;
}

bool Desktop::Init() {
  VirtualX = (ci && (ci->flags & CI_VIRTUALX)) ? ci->VirtualX : (parent ? 1:3);
  VirtualY = (ci && (ci->flags & CI_VIRTUALY)) ? ci->VirtualY : (parent ? 1:2);
  if (ci && (ci->flags & CI_AUTOSCROLL))
    flags |= DF_AUTOSCROLL;
  if (!Client::Init())
    return false;
  if (window) {
    XMapWindow(dpy,window);
    XMapWindow(dpy,frame);
  }
  man->WantFocus = this;
  if (!parent)
    window = root;
  XSetWindowAttributes pattr;
  pattr.override_redirect = true;
  for (int i = 0; i!=3; ++i) {
    StackRef[i] = XCreateWindow(dpy, window,
                  0,0,1,1, 0,
                  DefaultDepth(dpy, screen), CopyFromParent, DefaultVisual(dpy, screen),
                  CWOverrideRedirect, &pattr);
    XRaiseWindow(dpy,StackRef[i]);
  }
  XRaiseWindow(dpy,title);
  if (parent || !ci || !(ci->flags & CI_NOBACKGROUND)) {
    pattr.background_pixmap = (ci && ci->bgpm) ? ci->bgpm->GetPixmap() : 0;
    if (!pattr.background_pixmap)
      pattr.background_pixmap = (o.Sc && o.Sc->bgpm) ? o.Sc->bgpm->GetPixmap() : 0;
    unsigned long Mask = 0;
    if (pattr.background_pixmap)
      Mask |= CWBackPixmap;
    if (!parent) {
      pattr.background_pixel = o.Sc->colors[C_BG].pixel;
      Mask |= CWBackPixel;
    }
    if (Mask) {
      XChangeWindowAttributes(dpy,window,Mask,&pattr);
      XClearWindow(dpy,window);
    }
  }
  autoclose = ci ? (ci->flags & CI_AUTOCLOSE) : false;
  for (ActionList *al = ci ? ci->actions :0;al;al=al->next) {
    if (al->a->flags & AC_AUTOSTART)
      al->a->Execute(); else
      AddIcon(0,al->a);
  }

  flags |= DF_ICONSRAISED;
  if (!parent)
    Raise(0);

  return true;
}


Client *Desktop::AddChild(Window win, Options *o, ClientInfo *ci) {
// Check if there's a default ClientInfo
  if (!ci)
    ci = (ClientInfo *)rman->GetInfo(SE_CLIENTINFO,"");
  Client *c = new Client(this,o,ci,win);
  if (firstchild) {
    Client *cc;
    for (cc=firstchild;cc->next;cc=cc->next);
    cc->next = c;
  } else {
    firstchild = c;
  }
  c->next = NULL;
  if (!c->Init()) {
    RemoveChild(c,R_WITHDRAW);
    return 0;
  };
  if (c->o.HasTBEntry)
    UpdateTB();
  if (ct->Focus == 0 || ct->Focus == this || ct->Focus == focus)
    man->WantFocus = c;
  if(flags & DF_TILE)
    Tile();

  return c;
}

Desktop *Desktop::AddDesktop(int x,int y, int w, int h,Options *o,ClientInfo *ci) {
  Window win;
  XSetWindowAttributes pattr;

// Check if there's a default ClientInfo
  if (!ci) ci = (ClientInfo *)rman->GetInfo(SE_CLIENTINFO,"");

  pattr.override_redirect = true;
  pattr.background_pixel = o->Sc->colors[C_BG].pixel;
  pattr.event_mask = ChildMask|ButtonMask|ExposureMask|EnterWindowMask|LeaveWindowMask;

  win = XCreateWindow(dpy, window,
      w ? x : 200, h ? y : 200, w ? w : 300, h ? h : 300, 0,
      DefaultDepth(dpy, screen), CopyFromParent, DefaultVisual(dpy, screen),
      CWOverrideRedirect|CWEventMask|CWBackPixel, &pattr);
  XMapWindow(dpy,win);
  Desktop *c = new Desktop(this,o,ci,win,0);
  if (firstchild) {
    Client *cc;
    for (cc=firstchild;cc->next;cc=cc->next);
    cc->next = c;
  } else {
    firstchild = c;
  }
  c->next = NULL;
  if (!c->Init()) {
    RemoveChild(c,R_WITHDRAW);
    return 0;
  }
  if (c->o.HasTBEntry)
    UpdateTB();
  if (!focus)
    man->WantFocus = c;
  return c;
}


Desktop *Desktop::DesktopAbove() {
  return this;
}


Client* Desktop::FindPointerClient() {
  Window w, dummyw;
  int d1,d2,root_x,root_y;
  unsigned int d3;
  Client *tb = 0;
  if (XQueryPointer(dpy,window,&dummyw,&w,&root_x,&root_y,&d1,&d2,&d3)) {
    for (Client *c = firstchild; c; c=c->next) {
      if (w==c->frame)
        return c->FindPointerClient();
    };
  };
  GetRegionMask(root_x-x_root,root_y-y_root+THeight,tb);

  return tb ? tb : this;
};

bool Desktop::GetWindowList(MenuItem *m,int &n,bool mouse,int mx,int my,int ind) {
  if (!Client::GetWindowList(m,n,mouse,mx,my,ind))
    return false;
  ++ind;
  for (Client *i=firstchild;i;i=i->next)
    i->GetWindowList(m,n,mouse,mx,my,ind);
  return true;
}


void Desktop::RemoveClientReferences(Client *c) {
  Client::RemoveClientReferences(c);
  for (int i = 0; i!=3; ++i)
    if (c == LastRaised[i])
      LastRaised[i] = 0;
  for (Client *cc=firstchild;cc;cc=cc->next) {
    cc->RemoveClientReferences(c);
  }
}


void Desktop::RemoveChild(Client *c, int mode) {
  c->GiveFocus();
  c->Remove(mode);
  if (c==firstchild)
    firstchild = c->next; else
    for (Client *cc=firstchild;cc->next;cc=cc->next)
      if (cc->next==c) {
        cc->next=c->next;
        break;
      }
  if (c->o.HasTBEntry)
    UpdateTB();
  delete c;
  if(flags & DF_TILE)
    Tile();
  AutoClose();
};

void Desktop::GiveAway(Client *c) {
  c->flags &= ~CF_ICONPOSKNOWN; // New Desktop may be smaller
  c->GiveFocus();

  if (c==firstchild)
    firstchild = c->next; else
    for (Client *cc=firstchild;cc->next;cc=cc->next)
      if (cc->next==c) {
        cc->next=c->next;
        break;
      }


  if (c->mapped && c->o.HasTBEntry)
    UpdateTB();

}


void Desktop::SendWMDelete(bool soft) {
// Show a dialog box here...
  if (!parent) {
    man->alive = false;
    return;
  }
  if (firstchild) {
    Client *cc; // if c is deleted we can't access c->next any more
    for (Client *c = firstchild; c; c=cc) {
      cc = c->next;
      if (c) {
        if (c->mapped || c->icon) {
          c->SendWMDelete(soft);
        } else {
          GiveAway(c);
          parent->Take(c,c->x,c->y);
        }
      }
    }
  }
  autoclose = true;
  if (!firstchild)
    man->AddDeleteClient(this);
};


void Desktop::Remove(int mode) {
  if (mode == R_REMAP)
    Goto(-PosX,-PosY,GOTO_RELATIVE);
  for (Client *c=firstchild; c; c=firstchild) {
    // That bug was hard to find, the destructor accesses this desktop in
    // RemoveClientReferences
    firstchild = c->next;
    c->Remove(mode);
    delete c;
  }

  Icon *savei;
  for (Icon *i = firsticon;i;i=savei) {
    savei = i->next;
    if (i->client) i->client->icon = 0;
    delete i;
  }
  firsticon = 0;

  Client::Remove(mode);
  XDestroyWindow(dpy, window); // If this window is still there, we'll have to help a bit
  if (ci && ci->bgpm) {
    ci->bgpm->FreePixmap();
  } else if (o.Sc && o.Sc->bgpm) {
    o.Sc->bgpm->FreePixmap();
  }
}

void Desktop::ChangeSize() {
  if (!parent)
    return;
  Client::ChangeSize();
  for (Client *c=firstchild;c;c=c->next)
    if (c->MaxW || c->MaxH)
      c->Maximize(MAX_REMAX);
  if (flags & DF_TILE)
    Tile();
}

// Translates coordinates when children want to move their windows

void Desktop::Translate(int *X,int *Y) {
  if (parent)
    return;
  if (X)
    *X += o.Sc->BW;
  if (Y)
    *Y += THeight + o.Sc->BW;
}

void Desktop::ReDraw() {
  if (!o.HasTitle || !visible) return;
  XDrawLine(dpy, title, o.Sc->border_gc,
    0, THeight - 1,
    width, THeight - 1);
  for (Client *t=firstchild;t;t=t->next)
    if (t->mapped && t->o.HasTBEntry)
      ReDrawTBEntry(t);
  ReDrawTBEntry(this);
};


void Desktop::UpdateTB() {
  if (!o.HasTitle)
    return;
  int i=0;
  int TBNum=1;
  for (Client *t=firstchild;t;t=t->next)
    if (t->mapped && t->o.HasTBEntry)
      ++TBNum;
  for (Client *t=firstchild;t;t=t->next)
    if (t->mapped && t->o.HasTBEntry) {
      t->TBEx=i*width/TBNum;
      ++i;
      t->TBEy=i*width/TBNum - 1;
  };
  TBx=(TBNum-1)*width/TBNum;
  TBy=width;
  Clear();

};

void Desktop::Raise(int mode) {
  if (!(mode & R_INDIRECT))
    if (parent) {
      ShowIcons(parent->LastRaised[stacking] == this);
    } else {
      ShowIcons(true);
    }
  Client::Raise(mode);
}

void Desktop::GrabButtons(bool) {
  if (!(o.Sc->flags & SC_RAISEONCLICK))
    return;
  for (int i = 0; i!=3; ++i)
    if (LastRaised[i])
      LastRaised[i]->GrabButtons();
}


void Desktop::ShowIcons(bool raise) {
  if (raise && !(flags & DF_ICONSRAISED)) {
    for (Icon *i = firsticon;i; i = i->next) {
      XRaiseWindow(dpy,i->window);
      if (i->iconwin)
        XRaiseWindow(dpy,i->iconwin);
    }
    flags |= DF_ICONSRAISED;
  } else
    if (flags & DF_ICONSRAISED) {
      for (Icon *i = firsticon;i; i = i->next) {
        XLowerWindow(dpy,i->window);
        if (i->iconwin)
          XLowerWindow(dpy,i->iconwin);
      }
      flags &= ~DF_ICONSRAISED;
    }
};


Icon *Desktop::AddIcon(Client *c, Action *a) {
  Icon *i = new Icon(this,c,a);
  Icon *ii = firsticon;
  if (ii) {
    for (;ii->next;ii = ii->next);
    ii->next = i;
  } else {
    firsticon = i;
  }
  i->next = 0;
  return i;
}

void Desktop::RemoveIcon(Icon *i) {
  if (i==firsticon)
    firsticon = i->next; else
    for (Icon *ii=firsticon;ii->next;ii=ii->next)
      if (ii->next==i) {
        ii->next=i->next;
        break;
      }
  i->client->icon = 0;
  delete i;
};



void Desktop::Take(Client *c, int x, int y) {
  c->parent=this;
  if (firstchild) {
    Client *cc;
    for (cc=firstchild;cc->next;cc=cc->next);
    cc->next = c;
  } else {
    firstchild = c;
    focus = c; // give first window on this Desktop the focus
    focus->UpdateName(false); // we must be very careful with this call
  }
  c->next = NULL;
  if (c->MaxW || c->MaxH) {
    if (c->MaxW == c->width && c->MaxH == c->height) {
      c->MaxW = 0;
      c->MaxH = 0;
    } else c->Maximize(MAX_UNMAX);

  }
  c->x = x;
  c->y = y;
  Translate(&x,&y);
  XReparentWindow(dpy,c->frame,window,x,y - c->THeight);
  SendConfig();
  c->Raise(0);

  if (c->mapped && c->o.HasTBEntry)
    UpdateTB();
  if (ct->Focus == 0 || ct->Focus == this || ct->Focus == focus)
      man->WantFocus = c;

}


void Desktop::RequestDesktop() {
  Desktop *d = AddDesktop(0,0,0,0,&o,0);
  if (!d)
    return;
  d->MoveResize(0,1,true,0,0);
  d->MoveResize(S_BOTTOM | S_RIGHT,0,true,0,0);
};


void Desktop::GetFocus() {
  if (!visible || ct->MaxWindow)
    return;
  ct->CurrentDesktop = this;
  if (parent) {
    if (parent->focus)
      parent->focus->Clear();;
    parent->focus=this;
    parent->Clear();
  };
  if (focus) {
//  The child will ask me to ReDraw() myself
    focus->GetFocus();
  } else {
    Client::GetFocus();
    Clear();
  }
  ct->Current = this;
};

Client *Desktop::Focus() {
  if (focus) return focus->Focus();
  return this;
}

// width - o.Sc->BW ???
bool Desktop::Goto(int x, int y, int mode) {
  int oldPosX = PosX;
  int oldPosY = PosY;
  int diff;
  // i hate this implemetation of the / - operator
  diff = (mode & GOTO_RELATIVE) ? x :
        ((x + width*VirtualX) / width - VirtualX) * width;
  if (diff) {
    PosX += diff;
    if (mode & GOTO_SNAP) {
      diff = (PosX + snap) % width;
      if (diff <= 2*snap)
        PosX += snap - diff;
    }
    if (PosX < 0)
      PosX = 0;
    if (PosX > width*(VirtualX - 1))
      PosX = width*(VirtualX - 1);
  }
  diff = (mode & GOTO_RELATIVE) ? y :
        ((y + height*VirtualY) / height - VirtualY) * height;
  if (diff) {
    PosY += diff;
    if (mode & GOTO_SNAP) {
      diff = (PosY + snap) % height;
      if (diff <= 2*snap)
        PosY += snap - diff;
    }
    if (PosY < 0)
      PosY = 0;
    if (PosY > height*(VirtualY - 1))
      PosY = height*(VirtualY - 1);
  }

  if (PosX == oldPosX && PosY == oldPosY)
    return false;
  for (Client *i= firstchild;i;i=i->next)
    if (i->mapped && !(i->flags & CF_STICKY)) {
      i->x -= PosX - oldPosX;
      i->y -= PosY - oldPosY;
      i->ChangePos();
    }
  if (mode & (GOTO_MOVEFULL | GOTO_MOVEHALF)) {
    oldPosX -= PosX;
    oldPosY -= PosY;
    if (mode & GOTO_MOVEHALF) {
      oldPosX /= 2;
      oldPosY /= 2;
    }
    XWarpPointer(dpy,root,0,0,0,0,0,oldPosX,oldPosY);
  }
  return true;
}

bool Desktop::Leave(int x,int y,bool half) {
  return !Goto(x - x_root,y - y_root,half ? GOTO_MOVEHALF : GOTO_MOVEFULL);
}


void Desktop::Tile() {
  Client *c;
  int n = 0;
  for (c = firstchild; c; c=c->next,++n)
    if (c->mapped)
      ++n;
  if (!n)
    return;
  RectRec rects[n];
  int idNo=0;
  for (c = firstchild; c; c=c->next)
    if (c->mapped) {
      rects[idNo].assign(c->width,c->height+c->THeight,idNo);
      ++idNo;
    }
  bool tilingOK=tileArea(width,height,idNo,rects);
  if(tilingOK){
    /*this going be nasty because tileArea has sorted the rects list by
      area so we need to search for the correct entry*/
    idNo=0;
    for (c = firstchild; c; c=c->next)
      if (c->mapped) {
        int k=0;
        while(rects[k].idNo!=idNo){
          ++k;
        }
        c->x=rects[k].x;
        c->y=rects[k].y+c->THeight;
        c->ChangePos();
        ++idNo;
      };
  }
}


void Desktop::SetWMState(){
  for (Client *c=firstchild;c;c=c->next)
    if (c->mapped) {
      c->visible = visible;
      c->SetWMState();
    }

}


void Desktop::SendConfig() {
  Client::SendConfig();
  for (Client *c = firstchild; c; c=c->next) {
    c->SendConfig();
  };
};


Desktop::~Desktop() {
  for (Client *c = firstchild; c; c=c->next){
    delete c;
  }
}
