/*
 *	Copyright (c) 1999 RISS-Telecom Networking Center
 *
 *	Copyright (c) 1993 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	BPFT $Id: traflog.c,v 1.6 1994/01/14 11:36:44 bob Exp $
 *
 *	$Log: traflog.c,v $
 * Revision 1.6  1994/01/14  11:36:44  bob
 * Bug fixed in display tcp/udp fragmented datagramms
 *
 * Revision 1.5  1994/01/06  15:31:54  bob
 * Support new format of daemon log files
 *
 * Revision 1.4  1993/11/18  14:53:06  bob
 * Fixed bug with fseek length
 *
 * Revision 1.3  1993/11/06  12:58:37  bob
 * Change default output on last record, add '-a' for all records
 *
 * Revision 1.2  1993/11/01  15:13:30  bob
 * Now support '-r' flag for output only number of kb or number of records
 *
 * Revision 1.1  1993/10/20  16:05:18  bob
 * Initial revision
 *
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 * Redistribution in binary form may occur without any restrictions.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

/*	traflog.c - tcp/udp data traffic log manager	*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <pcap.h>

#include "interface.h"
#include "addrtoname.h"
#include "traffic.h"
#include "traflog.h"

static int no = 0;
static int n_call = 0;
static int n_pat = 0;

static struct t_entry *sum = NULL;

static struct pat_mask {
	struct  in_addr in_ip, out_ip;
	struct  in_addr in_mask, out_mask;
	u_char  ip_protocol;
	u_short p_port;
} *pat = NULL;

static struct t_header {
	int    t_size;
	struct timeval start;
	struct timeval stop;
} s;

u_long
addrmask(addr)
	u_long addr;
{
	register u_long m = INADDR_NONE;

	if (addr)
		while (!(addr & IN_CLASSA_NET))
			addr <<= IN_CLASSC_NSHIFT, m >>= IN_CLASSC_NSHIFT;
	else
		m = INADDR_ANY;

	return m;
}

char *
hostnetstr(in, mask)
	struct in_addr in;
	u_long mask;
{
	char *cp = NULL;
	static char line[MAXHOSTNAMELEN + 1];
	register u_long i;

	i = ntohl(in.s_addr);
	NTOHL(mask);
	i &= mask;
	if (!i) {
		strcpy(line, "all");
		return line;
	}
	if (!(htonl(i) ^ in.s_addr) && addrmask(in.s_addr) == INADDR_NONE) {
		strcpy(line, ipaddr_string(&in));
		return line;
	}
	if (!nflag) {
		register u_long net = i;
		struct netent *np = NULL;

		while ((mask & 1) == 0)
			mask >>= 1, net >>= 1;
		if (np = getnetbyaddr(net, AF_INET))
			cp = np->n_name;
	}	
#define	C(x)	((x) & 0xff)
	if (cp)
		strncpy(line, cp, sizeof(line) - 1);
	else if ((i & 0xffffff) == 0)
		sprintf(line, "%u", C(i >> 24));
	else if ((i & 0xffff) == 0)
		sprintf(line, "%u.%u", C(i >> 24) , C(i >> 16));
	else if ((i & 0xff) == 0)
		sprintf(line, "%u.%u.%u", C(i >> 24), C(i >> 16), C(i >> 8));
	else
		sprintf(line, "%u.%u.%u.%u", C(i >> 24),
			C(i >> 16), C(i >> 8), C(i));
	return line;
}

int
pat_print()
{
	register int i;
	char port[16], proto[16];
	struct servent *svc;

	printf("%-24s %-24s %-10s %s\n", "From", "To", "Port", "Protocol");
	for (i = 0; i < n_pat; i++) {
		printf("%-24.24s ", hostnetstr(pat[i].in_ip,
					       pat[i].in_mask.s_addr));
		printf("%-24.24s ", hostnetstr(pat[i].out_ip,
					       pat[i].out_mask.s_addr));
		port[0] = proto[0] = 0;
		if (pat[i].p_port) {
			register p = htons(pat[i].p_port);
			if (pat[i].ip_protocol != IPPROTO_UDP)
				if (svc = getservbyport(p, "tcp")) {
					strcpy(port, svc->s_name);
					strcpy(proto, "tcp ");
				}
			if (pat[i].ip_protocol != IPPROTO_TCP)
				if (svc = getservbyport(p, "udp")) {
					if (!port[0])
						strcpy(port, svc->s_name);
					strcat(proto, "udp");
				}
			if (port[0] && !nflag)
				printf("%-10.10s ", port);
			else
				printf("%-10d ", pat[i].p_port);
		} else {
			printf("%-10.10s ", "all");
			if (pat[i].ip_protocol == IPPROTO_TCP)
				strcpy(proto, "tcp");
			if (pat[i].ip_protocol == IPPROTO_UDP)
				strcpy(proto, "udp");
		}
		printf("%s\n", proto[0] ? proto : "tcp udp");
	}
	return 0;
}

u_long
strhostnet(cmdarg, mask)
	char *cmdarg;
	u_long *mask;
{
	struct in_addr in;
	struct netent *np;
	struct hostent *hp;

	if ((in.s_addr = inet_network(cmdarg)) != INADDR_NONE) {
		net_mask(&in.s_addr);
		HTONL(in.s_addr);
		*mask = addrmask(in.s_addr);
		return in.s_addr;
	}
	if (!strcasecmp(cmdarg, "all")) {
		*mask = INADDR_ANY;
		return INADDR_ANY;
	}
	if (np = getnetbyname(cmdarg)) {
		in.s_addr = np->n_net;
		*mask = htonl(net_mask(&in.s_addr));
		return htonl(in.s_addr);
	}
	if (hp = gethostbyname(cmdarg)) {
		in.s_addr = *(u_long *)*hp->h_addr_list;
		*mask = addrmask(in.s_addr);
		return in.s_addr;
	}
	error("unknown host or network '%s'", cmdarg);
}

int
iscmd(token)
	register char *token;
{
	if (!strcasecmp(token, "from"))
		return FROM;
	if (!strcasecmp(token, "to"))
		return TO;
	if (!strcasecmp(token, "port"))
		return PORT;
	if (!strcasecmp(token, "proto"))
		return PROTO;
	if (!strcasecmp(token, "mask"))
		return MASK;
	return NOCMD;
}

void
alloc_pat()
{
	if ((pat = realloc(pat, ++n_pat * sizeof(struct pat_mask))) == NULL)
		error("can't reallocate memory");
	if (n_pat > 1)
		bcopy(pat + n_pat-2, pat + n_pat-1, sizeof(struct pat_mask));
	else
		bzero(pat, sizeof(struct pat_mask));
}

int
pattern(cmdbuf)
	char *cmdbuf;
{
	register int cmd, cmdp = MASK;
	register char *cmdarg;
	char *token;
	struct servent *svc;

	if (cmdbuf)
	while (token = strtok(cmdbuf, " \t\r\n")) {
		if (*token == '#' || *token == ';')
			continue;
		if ((cmd = iscmd(token)) == NOCMD)
			error("unknown keyword '%s'", token);
		if (!(cmdarg = strtok(NULL, " \t\r\n")))
			error("keyword '%s': missing argument", token);
		if (cmd <= cmdp)
			alloc_pat();
		switch (cmd) {
		case FROM:
			pat[n_pat-1].in_ip.s_addr = strhostnet(cmdarg,
						&pat[n_pat-1].in_mask.s_addr);
			break;
		case TO:
			pat[n_pat-1].out_ip.s_addr = strhostnet(cmdarg,
						&pat[n_pat-1].out_mask.s_addr);
			break;
		case MASK:
			switch(cmdp) {
			case FROM:
				pat[n_pat-1].in_mask.s_addr = inet_addr(cmdarg);
				break;
			case TO:
				pat[n_pat-1].out_mask.s_addr = inet_addr(cmdarg);
				break;
			default:
				error("inconsistent usage '%s %s'", token, cmdarg);
			}
			cmd = cmdp;
			break;
		case PORT:
			if (!(pat[n_pat-1].p_port = (u_short)atoi(cmdarg))) {
				if (!strcasecmp(cmdarg, "all")) {
					pat[n_pat-1].p_port = 0;
					break;
				}
				if (svc = getservbyname(cmdarg, "tcp"))
					pat[n_pat-1].p_port = ntohs(svc->s_port);
				if (svc = getservbyname(cmdarg, "udp"))
					pat[n_pat-1].p_port = ntohs(svc->s_port);
				if (!pat[n_pat-1].p_port)
					error("unknown port '%s'", cmdarg);
			}
			break;
		case PROTO:
			if (!strcasecmp(cmdarg, "udp")) {
				pat[n_pat-1].ip_protocol = IPPROTO_UDP;
				break;
			}
			if (!strcasecmp(cmdarg, "tcp")) {
				pat[n_pat-1].ip_protocol = IPPROTO_TCP;
				break;
			}
			error("unknown protocol '%s'", cmdarg);
		default:
			error("unknown keyword '%s'", token);
		}
		cmdp = cmd;
		cmdbuf = NULL;
	}
	if (!n_pat)
		alloc_pat();
	return;
}

int
getheader(fd, header)
	register FILE *fd;
	register struct t_header *header;
{
	if (fread((struct t_header *)header, sizeof(struct t_header), 1, fd) != 1) {
		if (!feof(fd))
			warning("fread: can't read header");
		return 0;
	}
	if (header->t_size > MAX_TO_SAVE) {
		warning("table too big to fit into memory");
		return 0;
	}
	if (!header->t_size)
		warning("empty table");
	return 1;
}

/*
 * Output contents list of daemon logfile.
 */
int
contents(fd, h)
	register FILE *fd;
	register struct t_header *h;
{
	register int i;
	register u_int64_t bytes = 0, psize = 0;
	struct t_entry t;

	if (!n_call && !rflag)
		printf("\
 #        Started            Dumped            Data         All     Recs\n");

	for (i = 0; i < h->t_size; i++) {
		if (fread((struct t_entry *)&t, sizeof(struct t_entry), 1, fd) != 1) {
			warning("fread: can't read entry");
			return 0;
		}
		bytes += t.n_bytes, psize += t.n_psize;
	}
	if (!rflag) {
		printf("%03d   %.15s", no, ctime((time_t *)&h->start) + 4);
		printf("%18.15s   %10qu   %10qu   %4d\n", ctime((time_t *)&h->stop)+4,
		       bytes, psize, h->t_size);
	}
	return 1;
}

int
patmatch(e)
	register struct t_entry *e;
{
	register int i = 0;

	if (e->in_ip.s_addr == 0)
		return 0;
	for (i = 0; i < n_pat; i++) {
		if ((e->in_ip.s_addr & pat[i].in_mask.s_addr) ^
		      pat[i].in_ip.s_addr)
			continue;
		if ((e->out_ip.s_addr & pat[i].out_mask.s_addr) ^
		    pat[i].out_ip.s_addr)
			continue;
		if (pat[i].ip_protocol && e->ip_protocol != pat[i].ip_protocol)
			continue;
		if (!pat[i].p_port || e->p_port == pat[i].p_port)
			break;
	}
	return (n_pat - i);
}

int
ioblock(fd, h)
	FILE *fd;
	register struct t_header *h;
{
	register int i, j;
	register struct t_entry *block;

	if (!sflag || !n_call) {
		bzero((struct t_header *)&s, sizeof(struct t_header));
		if (sum != NULL)
			free(sum);
		sum = NULL;
	}
	block = (struct t_entry *) alloca(h->t_size * sizeof(struct t_entry));
	if (block == NULL) {
		warning("alloca: can't allocate memory");
		return 0;
	}
	if (fread((struct t_entry *)block, sizeof(struct t_entry), h->t_size,
		  fd) != h->t_size) {
		warning("fread: can't read table");
		return 0;
	}
	for (i = 0; i < h->t_size; i++) {
		if (!patmatch(block + i))
			continue;
		for (j = 0; j < s.t_size; j++) {
			if (block[i].in_ip.s_addr == sum[j].in_ip.s_addr &&
			    block[i].out_ip.s_addr == sum[j].out_ip.s_addr &&
			    block[i].ip_protocol == sum[j].ip_protocol &&
			    block[i].who_srv == sum[j].who_srv &&
			    block[i].p_port == sum[j].p_port) {
				sum[j].n_bytes += block[i].n_bytes;
				sum[j].n_psize += block[i].n_psize;
				break;
			}
		}
		if (j < s.t_size)
			continue;
		sum = (struct t_entry *) realloc(sum, ++s.t_size *
						sizeof(struct t_entry));
		if (sum == NULL) {
			warning("realloc: can't reallocate memory");
			return 0;
		}
		bcopy(block + i, sum + j, sizeof(struct t_entry));
	}
	if (!s.start.tv_sec)
		s.start = h->start;
	s.stop = h->stop;
	return 1;
}

log_loop(fd, fdw)
	register FILE *fd;
	FILE *fdw;
{
	struct t_header h;
	int retval;

	while (getheader(fd, &h)) {
		no++;
		if (range(&h) == 2) {
			if (lflag) retval = contents(fd, &h);
			else retval = ioblock(fd, &h);
			if (retval) n_call++;
			else break;
			if (aflag && !sflag) {
				if (fdw) traf_write(fdw);
				else traf_print();
			}
		} else
			fseek(fd, h.t_size * sizeof(struct t_entry), SEEK_CUR);
	}
	if (aflag && !sflag)
		return;
	if (lflag) {
		if (rflag)
			printf("%d\n", n_call);
		return;
	}
	if (fdw) traf_write(fdw);
	else traf_print();

	return;
}

range(h)
	register struct t_header *h;
{
	register true = 0;

	if (tbflag) {
		if (tb.tv_sec <= h->start.tv_sec)
			++true;
	} else {
		if (begin <= no)
			++true;
	}
	if (teflag) {
		if (te.tv_sec >= h->stop.tv_sec)
			++true;
	} else {
		if (endin >= no)
			++true;
	}
	return true;
}

traf_write(fd)
	FILE *fd;
{
	if (fwrite((struct t_header *)&s, sizeof(struct t_header), 1, fd) != 1)
		error("write header error");
	if (fwrite((struct t_entry *)sum, sizeof(struct t_entry), s.t_size, fd) != s.t_size)
		error("write table error");
	return;
}

/*
int
sortbyfrom(left, right)
	register struct t_entry *left;
	register struct t_entry *right;
{
	if (htonl(left->in_ip.s_addr) < htonl(right->in_ip.s_addr))
		return LESS;
	if (htonl(left->in_ip.s_addr) > htonl(right->in_ip.s_addr))
		return GREATER;
	return EQUAL;
}

int
sortbyto(left, right)
	register struct t_entry *left;
	register struct t_entry *right;
{
	if (htonl(left->out_ip.s_addr) < htonl(right->out_ip.s_addr))
		return LESS;
	if (htonl(left->out_ip.s_addr) > htonl(right->out_ip.s_addr))
		return GREATER;
	return EQUAL;
}

int
sortbyport(left, right)
	register struct t_entry *left;
	register struct t_entry *right;
{
	if (htons(left->p_port) < htons(right->p_port))
		return LESS;
	if (htons(left->p_port) > htons(right->p_port))
		return GREATER;
	return EQUAL;
}
*/

int
sortbysize(left, right)
	register struct t_entry *left;
	register struct t_entry *right;
{
	if (left->n_psize < right->n_psize)
		return GREATER;
	if (left->n_psize > right->n_psize)
		return LESS;
	return EQUAL;
}

traf_print()
{
	register int i, j;
	u_int64_t abytes = 0, dbytes = 0;
	char buf[MAXHOSTNAMELEN + 1], *port, *user, *proto;

	for (i = 0; i < s.t_size; i++)
		abytes += sum[i].n_psize, dbytes += sum[i].n_bytes;
	if (rflag) {
		printf("%qu\n", abytes);
		return;
	}
	qsort(sum, s.t_size, sizeof(struct t_entry), sortbysize);
	if (!fvnum) {
		gethostname(buf, MAXHOSTNAMELEN);
		printf("\n (%s) %s at", device_name, buf);
		printf(" %.15s -", ctime((time_t *)&s.start) + 4);
		printf(" %.15s\n", ctime((time_t *)&s.stop) + 4);
		printf(" Summary: %qu data bytes, %qu all bytes, %u records\n",
		       dbytes, abytes, s.t_size);
		printf("\
     From           Port         To            Port  Proto     Data       All\n");
	}
	for (i = 0; i < s.t_size; i++) {
		port = "undef";
		switch(sum[i].ip_protocol) {
			case IPPROTO_TCP:
				if (sum[i].p_port)
					port = tcpport_string(sum[i].p_port);
				proto = "tcp";
				break;
			case IPPROTO_UDP:
				if (sum[i].p_port)
					port = udpport_string(sum[i].p_port);
				proto = "udp";
				break;
			case IPPROTO_ICMP:
				proto = "icmp";
				break;
			case IPPROTO_EGP:
				proto = "egp";
				break;
			case IPPROTO_OSPF:
				proto = "ospf";
				break;
			case IPPROTO_IGMP:
				proto = "igmp";
				break;
			default:
				proto = "unkn";
		}
		if (sum[i].p_port)	user = "client";
		else			user = "none";
		if (!fvnum) {
			printf("%-18.18s %-6.6s  ", ipaddr_string(&sum[i].in_ip),
			       (sum[i].who_srv & 1) ? port : user);
			printf("%-18.18s %-6.6s  ", ipaddr_string(&sum[i].out_ip),
			       (sum[i].who_srv & 2) ? port : user);
			printf("%-4.4s %9ld %10ld\n", proto, sum[i].n_bytes,
			       sum[i].n_psize);
		} else 
			for (j = 0; j < fvnum; j++)
				switch (fv[j].cmd) {
				case FROM:
					printf(fv[j].format, ipaddr_string(&sum[i].in_ip));
					break;
				case TO:
					printf(fv[j].format, ipaddr_string(&sum[i].out_ip));
					break;
				case SPORT:
					printf(fv[j].format,
					       (sum[i].who_srv & 1) ? port : user);
					break;
				case DPORT:
					printf(fv[j].format,
					       (sum[i].who_srv & 2) ? port : user);
					break;
				case PROTO:
					printf(fv[j].format, proto);
					break;
				case BYTES:
					printf(fv[j].format, sum[i].n_bytes);
					break;
				case PSIZE:
					printf(fv[j].format, sum[i].n_psize);
					break;
				case FTIME:
					strftime(buf, sizeof(buf), fv[j].format,
						 localtime((time_t *)&s.start));
					printf("%s", buf);
					break;
				case LTIME:
					strftime(buf, sizeof(buf), fv[j].format,
						 localtime((time_t *)&s.stop));
					printf("%s", buf);
					break;
				}
	}
}

