/*
 *	Copyright (c) 1999 RISS-Teleocm Networking Center
 *
 *	Copyright (c) 1993 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	BPFT $Id: traffic.c,v 1.4 1994/01/14 11:50:38 bob Exp $
 *
 *	$Log: traffic.c,v $
 * Revision 1.4  1994/01/14  11:50:38  bob
 * Now really logged tcp/udp fragmented datagramms, little syslog's corrections
 *
 * Revision 1.3  1994/01/06  15:00:41  bob
 * Modification for new hash table format, optimized for speed
 *
 * Revision 1.2  1993/11/01  15:03:22  bob
 * Improve interrupts queue processing
 *
 * Revision 1.1  1993/10/20  16:04:41  bob
 * Initial revision
 *
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 * Redistribution in binary form may occur without any restrictions.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

/*	traffic.c - collect network traffic	*/

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <syslog.h>
#ifdef USE_PIPE
#include <fcntl.h>
#include <signal.h>
#include <setjmp.h>
#else /* Unix Domain socket */
#include <sys/socket.h>
#include <sys/un.h>
#endif
#include <sys/types.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip_var.h>
#include <netinet/udp.h>
#include <netinet/udp_var.h>
#include <netinet/tcp.h>
#include <netinet/tcpip.h>
#include <netdb.h>
#include <pcap.h>

#include "interface.h"
#include "traffic.h"
#include "trafd.h"

#ifdef USE_PIPE
static jmp_buf after_pipe;
#endif

static int iplen;
static int dlen;
typedef struct t_entry *p_entry;
static struct t_entry t;
static struct timeval begin_time;
static u_char s_port_big[IPPORT_RESERVED];

/*
 * Initialize traffic collector.
 */
int
traf_init(restore)
	int restore;
{
	register int i;
	struct servent *sv;

	i = MAX_T_ENTRIES * sizeof(struct t_entry);
	if ((entries = (struct t_entry *) valloc(i)) == NULL)
		return (i);
	(void)bzero((struct t_entry *)entries, i);
	for(i = 0; i < IPPORT_RESERVED; i++)
		s_port_big[i] = 0;
	setservent(1);
	while (sv = getservent()) {
		NTOHS(sv->s_port);
		if (sv->s_port > IPPORT_RESERVED) {
			i = sv->s_port & (IPPORT_RESERVED-1);
			if (strcmp(sv->s_proto, "tcp") == 0)
				s_port_big[i] |= IPPROTO_TCP;
			else if (strcmp(sv->s_proto, "udp") == 0)
				s_port_big[i] |= IPPROTO_UDP;
		}
	}
	endservent();
	n_entry = 0;
	if (restore)
		traf_restore();
	else
		gettimeofday(&begin_time, (struct timezone *)NULL);
	return 0;
}

/*
 * Hash by in_ip.s_addr.
 */
static inline unsigned
hash( e )
	register p_entry e;
{
	return HASHFUNC(e->in_ip.s_addr);
}

/*
 * Find entry in hash table.
 */
int
findentry( e )
	register p_entry e;
/* return -1 if not found or -2 if not found & table full */
{
	register unsigned inx = hash(e);
	register unsigned oldinx = inx;
	do {
		if (!entries[inx].in_ip.s_addr)
			return -1;
		if (e->in_ip.s_addr == entries[inx].in_ip.s_addr &&
		    e->out_ip.s_addr == entries[inx].out_ip.s_addr &&
		    e->ip_protocol == entries[inx].ip_protocol &&
		    e->who_srv == entries[inx].who_srv &&
		    e->p_port == entries[inx].p_port)
			return inx;
		inx = (inx + E_OFFSET) % MAX_T_ENTRIES;
	} while (inx != oldinx);
	return -2;
}

/*
 * Insert entry.
 */
int
insertentry( e )
	register p_entry e;
/* return -1 if success digit if already in table or -2 if table full */ 
{
	register int ec = findentry(e);
	register unsigned inx;
	if (ec != -1)
		return ec;
	inx = hash(e);
	while (entries[inx].in_ip.s_addr)
		inx = (inx + E_OFFSET) % MAX_T_ENTRIES;
	entries[inx] = *e;
	return -1;
}

/*
 * Add new record to hash table or modify existed record.
 */
void
traf_add(t_idx, bytes, psize)
	register int t_idx;
	register u_long bytes;
	register u_long psize;
{
	if (n_entry > MAX_TO_SAVE || t_idx == -2) {
		(void)syslog(LOG_WARNING, "(%s) table full, autobackup",
			     device_name);
		if (fork() == 0) {
			(void)traf_save(file_backup, "a");
			exit(0);
		}
		(void)traf_clear();
		return;
	}
	if (t_idx >= 0)
		entries[t_idx].n_bytes += bytes,
		entries[t_idx].n_psize += psize;
	else
		n_entry++;
	return;
}

/*
 * Calculate data length and server port in tcp packets.
 */
void
traf_tcp(tp, ip)
	register struct tcphdr *tp;
	register struct ip *ip;
{
	struct t_entry t;

	if ((u_char *)(tp + 1) > snapend || dlen < sizeof(struct tcphdr))
		return;

	NTOHS(tp->th_sport);
	NTOHS(tp->th_dport);

	t.in_ip.s_addr = ip->ip_src.s_addr;
	t.out_ip.s_addr = ip->ip_dst.s_addr;
	t.ip_protocol = IPPROTO_TCP;
	t.n_psize = (u_long)iplen;
	t.n_bytes = (u_long)(dlen - tp->th_off * 4) & 2047L;

	if (tp->th_sport < tp->th_dport)
		t.p_port = tp->th_sport, t.who_srv = 1;
	else if (tp->th_sport > tp->th_dport)
		t.p_port = tp->th_dport, t.who_srv = 2;
	else if (tp->th_sport = tp->th_dport)
		t.p_port = tp->th_sport, t.who_srv = 3;
	if (t.p_port > IPPORT_RESERVED) {
		if (s_port_big[tp->th_sport & (IPPORT_RESERVED-1)] & IPPROTO_TCP) {
			t.p_port = tp->th_sport;
			t.who_srv = 1;
		} else if (s_port_big[tp->th_dport & (IPPORT_RESERVED-1)] & IPPROTO_TCP) {
				t.p_port = tp->th_dport;
				t.who_srv = 2;
			}
		if (tp->th_sport = tp->th_dport) t.who_srv = 3;
	}
	traf_add(insertentry(&t), t.n_bytes, t.n_psize);

	HTONS(tp->th_sport);
	HTONS(tp->th_dport);
	return;
}

/*
 * Calculate data length and server port in udp packets.
 */
void
traf_udp(up, ip)
	register struct udphdr *up;
	register struct ip *ip;
{
	register u_char *cp = (u_char *)(up + 1);
	struct t_entry t;

	if (cp > snapend || dlen < sizeof(struct udphdr))
		return;

	NTOHS(up->uh_sport);
	NTOHS(up->uh_dport);
	NTOHS(up->uh_ulen);

	t.in_ip.s_addr = ip->ip_src.s_addr;
	t.out_ip.s_addr = ip->ip_dst.s_addr;
	t.ip_protocol = IPPROTO_UDP;
	t.n_psize = (u_long)iplen;
	t.n_bytes = (u_long)(up->uh_ulen - sizeof(*up)) & 2047L;

	if (up->uh_sport < up->uh_dport)
		t.p_port = up->uh_sport, t.who_srv = 1;
	else if (up->uh_sport > up->uh_dport)
		t.p_port = up->uh_dport, t.who_srv = 2;
	else if (up->uh_sport = up->uh_dport)
		t.p_port = up->uh_sport, t.who_srv = 3;
	if (t.p_port > IPPORT_RESERVED) {
		if (s_port_big[up->uh_sport & (IPPORT_RESERVED-1)] & IPPROTO_UDP) {
			t.p_port = up->uh_sport;
			t.who_srv = 1;
		} else if (s_port_big[up->uh_dport & (IPPORT_RESERVED-1)] & IPPROTO_UDP) {
				t.p_port = up->uh_dport;
				t.who_srv = 2;
			}
		if (up->uh_sport = up->uh_dport) t.who_srv = 3;
	}
	traf_add(insertentry(&t), t.n_bytes, t.n_psize);

	HTONS(up->uh_sport);
	HTONS(up->uh_dport);
	HTONS(up->uh_ulen);
	return;
}

/*
 * All other ip packets also collect.
 */
void
traf_ip(ip)
	register struct ip *ip;
{
	struct t_entry t;

	t.in_ip.s_addr = ip->ip_src.s_addr;
	t.out_ip.s_addr = ip->ip_dst.s_addr;
	t.ip_protocol = ip->ip_p;
	t.n_psize = (u_long)iplen;
	t.n_bytes = (u_long)dlen;
	t.p_port = t.who_srv = 0;
	traf_add(insertentry(&t), t.n_bytes, t.n_psize);
	return;
}

/*
 * Drop data to disk.
 * If mode = "w" then dump full table, else drop only essential data.
 */
void
traf_save(fname, mode)
	char *fname;
	char *mode;
{
	register FILE *fd;
	register int i;
	struct timeval now;
	int c = 0, ess_entry = 0;
	char *msg;

	if ((fd = fopen(fname, mode)) == NULL) {
		(void)syslog(LOG_WARNING, "(%s) fopen %s: %m",
			     device_name, fname);
		return;
	}
	gettimeofday(&now, (struct timezone *)NULL);
	if (*mode == 'w') {
		int max = MAX_T_ENTRIES;

		c += fwrite((int *)&max, sizeof(max), 1, fd);
		c += fwrite((int *)&n_entry, sizeof(n_entry), 1, fd);
		ess_entry = n_entry;
		msg = "dumped";
	} else {
		for (i = 0; i < MAX_T_ENTRIES; i++)
			if (entries[i].in_ip.s_addr != 0 &&
			    entries[i].n_psize >= 2048L)
				ess_entry++;
		c += fwrite((int *)&ess_entry, sizeof(ess_entry), 1, fd);
		msg = "append";
	}
	c += fwrite((struct timeval *)&begin_time, sizeof(begin_time), 1, fd);
	c += fwrite((struct timeval *)&now, sizeof(now), 1, fd);
	if (*mode == 'w') {
		c += fwrite((struct t_entry *)entries, sizeof(struct t_entry),
			    MAX_T_ENTRIES, fd);
		(void)fwrite((int *)&c, sizeof(c), 1, fd);
	} else
		for (i = 0; i < MAX_T_ENTRIES; i++) {
			if (entries[i].in_ip.s_addr == 0 ||
			    entries[i].n_psize < 2048L)
				continue;
			fwrite((struct t_entry *)&entries[i],
			       sizeof(struct t_entry), 1, fd);
		}
	(void)fclose(fd);
	(void)syslog(LOG_INFO, "(%s) %d records %s to %s", device_name,
		     ess_entry, msg, fname);
	return;
}

/*
 * Restore data from disk into memory. Useful when need to resume collect,
 * for example, after shutdown.
 */
void
traf_restore()
{
	register FILE *fd;
	register int c = 0;
	struct timeval now;
	int max, count;

	if ((fd = fopen(file_out, "r")) == NULL) {
		gettimeofday(&begin_time, (struct timezone *)NULL);
		return;
	}
	c += fread((int *)&max, sizeof(max), 1, fd);
	c += fread((int *)&n_entry, sizeof(n_entry), 1, fd);
	c += fread((struct timeval *)&begin_time, sizeof(begin_time), 1, fd);
	c += fread((struct timeval *)&now, sizeof(now), 1, fd);
	c += fread((struct t_entry *)entries, sizeof(struct t_entry),
		   MAX_T_ENTRIES, fd);
	(void)fread((int *)&count, sizeof(count), 1, fd);
	(void)fclose(fd);
	unlink(file_out);
	if (c != count || max != MAX_T_ENTRIES) {
		(void)syslog(LOG_WARNING, "(%s) can't restore from %s",
			     device_name, file_out);
		n_entry = MAX_T_ENTRIES;
		(void)traf_clear();
	} else
		(void)syslog(LOG_NOTICE, "(%s) %d records restored from %s",
			     device_name, n_entry, file_out);
}

/*
 * Full clear hash table, start time and misc counters. 
 */
void
traf_clear()
{
	(void)bzero((struct t_entry *)entries,
		    sizeof(struct t_entry) * MAX_T_ENTRIES);
	gettimeofday(&begin_time, (struct timezone *)NULL);
	(void)syslog(LOG_INFO, "(%s) %d records cleared",
		     device_name, n_entry);
	n_entry = 0;
	return;
}

/*
 * Send data via ipc, some utility like trafstat may use it.
 */
void
traf_pipe()
{
	register int fd, i;
	struct timeval now;
	int c = 0, ess_entry =0;
#ifdef USE_PIPE
	void onpipe();
#else /* UNIX Domain socket */
	struct sockaddr_un server;
#endif
	/* Calculate number of records contained essential data */
	for (i = 0; i < MAX_T_ENTRIES; i++)
		if (entries[i].in_ip.s_addr != 0 &&
		    entries[i].n_psize >= 2048L)
			ess_entry++;

#ifdef	USE_PIPE
	/* Prepare reaction on SIGPIPE interrupt */
	if (setjmp(after_pipe)) {
		close(fd);
		(void)signal(SIGPIPE, SIG_IGN);
		return;
	}
	(void)signal(SIGPIPE, onpipe);

	/* Open named pipe */
	if ((fd = open(file_fifo, O_WRONLY /* |O_NONBLOCK */)) < 0) {
		(void)syslog(LOG_WARNING, "(%s) open %s: %m",
			     device_name, file_fifo);
		return;
	}
#else	/* Using UNIX IPC Domain */
	if ((fd = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
		(void)syslog(LOG_WARNING, "(%s) open socket: %m", device_name);
		return;
	}
	server.sun_family = AF_UNIX;
	strcpy(server.sun_path, file_fifo);

	if (connect(fd, (struct sockaddr *)&server,
		    sizeof(struct sockaddr_un)) < 0) {
		close(fd);
		(void)syslog(LOG_WARNING, "(%s) connect: %m", device_name);
		return;
	}
#endif /* USE_PIPE */
	gettimeofday(&now, (struct timezone *)NULL);
	c += write(fd, (int *)&ess_entry, sizeof(ess_entry));
	c += write(fd, (struct timeval *)&begin_time, sizeof(begin_time));
	c += write(fd, (struct timeval *)&now, sizeof(now));
	for (i = 0; i < MAX_T_ENTRIES; i++) {
		/*
		 * Don't send wholly memory table to ipc,
		 * only records contained essential data.
		 */
		if (entries[i].in_ip.s_addr == 0 ||
		    entries[i].n_psize < 2048L)
			continue;
		c += write(fd, (struct t_entry *)&entries[i],
			   sizeof(struct t_entry));
	}
	(void)write(fd, (int *)&c, sizeof(c));
	(void)close(fd);
#ifdef	USE_PIPE
	(void)syslog(LOG_INFO, "(%s) %d records sent to named pipe %s",
		     device_name, ess_entry, file_fifo);
	(void)signal(SIGPIPE, SIG_IGN);
#else
	(void)syslog(LOG_INFO, "(%s) %d records sent to socket %s",
		     device_name, ess_entry, file_fifo);
#endif
	return;
}

#ifdef	USE_PIPE
/*
 * Make clean return after SIGPIPE.
 */
void
onpipe()
{
	(void)syslog(LOG_WARNING, "(%s) broken pipe", device_name);
	longjmp(after_pipe, 1);
}
#endif

/*
 * This point will be called from bpf_readloop() per each packet.
 */
void
processing_ip(bp, length)
	register const u_char *bp;
	register uint length;
{
	register const struct ip *ip = (const struct ip*) bp;
	register int hlen;
	register unsigned char *cp;

	if ((u_char *)(ip + 1) > snapend)
		return;
	if (length < sizeof(struct ip)) {
		syslog(LOG_WARNING, "(%s) truncated-ip: discard %d bytes",
		       device_name, length);
		return;
	}
	iplen = (int)ntohs(ip->ip_len);
	if (length < iplen)
		syslog(LOG_INFO, "(%s) truncated-ip: %d bytes missing",
			device_name, iplen - length);
	hlen = ip->ip_hl * 4;
	dlen = iplen - hlen;
	/*
	 * If this fragment zero, hand it to the next higher level protocol,
	 * else it is fragmented datagram.
	 */
	if (!(ntohs(ip->ip_off) & 0x1fff)) {
		cp = (unsigned char *)ip + hlen;
		switch (ip->ip_p) {
		case IPPROTO_TCP:
			traf_tcp((struct tcphdr *)cp, ip);
			break;
		case IPPROTO_UDP:
			traf_udp((struct udphdr *)cp, ip);
			break;
		default:
			traf_ip(ip);
			break;
		}
	} else	traf_ip(ip);

	/* Check interrupt's queue */
	if (flag_hup) {
		alarm(0);
		flag_hup = 0;
		if (fork() == 0) {
			traf_save(file_out, "w");
			exit(0);
		}
	}
	if (flag_int) {
		alarm(0);
		flag_int = 0;
		if (fork() == 0) {
			traf_save(file_backup, "a");
			exit(0);
		}
		traf_clear();
	}
	if (flag_usr1) {
		flag_usr1 = 0;
		if (fork() == 0) {
			traf_pipe();
			exit(0);
		}
	}
	return;
}

/* The END */
