/****************************************************************************
 * Copyright (C) 1998 WIDE Project. All rights reserved.
 * Copyright (C) 1999,2000,2001,2002 University of Tromso. All rights reserved.
 * Copyright (C) 2002 Invenia Innovation AS. All rights reserved.
 *
 * Author: Feike W. Dillema, feico@pasta.cs.uit.no.
 *         based on newbie code by Yusuke DOI, Keio Univ. Murai Lab.
 ****************************************************************************/

/*
 * <$Id: totd.c,v 3.63 2002/03/06 14:56:03 dillema Exp $>
 */

#include "totd.h"

struct ToT T;
char *version = "Trick or Treat Daemon (totd) version 1.3";

void usage () {
	printf ("%s\n\
Usage: totd [-6|-no6|-4|-no4|-64|-dn|-v|-q|-p <prefix>|-c <filename>]\n\
\n\
-[no]6      : enable[disable] IPv6 service functionality\n\
-[no]4      : enable[disable] IPv4 service functionality\n\
-64         : alias to -6 -4\n\
-dn         : debug mode (no fork / loglevel = n)\n\
-v          : verbose\n\
-q          : quiet\n\
-p <prefix> : a prefix to use for totd tricks; more than one allowed\n\
-c <file>   : specify alternative totd configfile, default=%s\n\
\n\
default   : IPv6 is %s and IPv4 is %s\n\n\
totd %s use IPv6 because it was compiled %s USE_INET6 option in config.h\n\
totd %s use IPv4 because it was compiled %s USE_INET4 option in config.h\n",
		version,
		TOTCONF,
		(T.ip6) ? "enabled" : "disabled",
		(T.ip4) ? "enabled" : "disabled",
		(V4 (1) + 0) ? "can" : "can not",
		(V4 (1) + 0) ? "with" : "without",
		(V6 (1) + 0) ? "can" : "can not",
		(V6 (1) + 0) ? "with" : "without"
	);

	exit (1);
}

const char *hex = "0123456789abcdef";

int main (int argc, char **argv) {
	struct passwd *pwd_p;
	int i;

	/* initialize global totd structure */
	T.tot_uid = getuid();
	T.tot_gid = getgid();
	T.su = 0;
	T.totuser = NULL;
	if (!T.tot_uid) {
                /* got started as root user */
		T.su = 1;
#ifdef TOT_USER
		T.totuser = TOT_USER;
#endif
	}

	/* fill in default config values */
	T.ip4 = V4 (1) + 0;	/* bit ugly, but short */
	T.ip6 = V6 (1) + 0;
	T.use_mapped = 0;
	T.quiet = 0;
	T.debug = 0;
	T.prefixnum = 0;
	T.stf = 0;
	T.retry_interval = 300;
	T.port = PORT_SRV;
	T.wildcard = 1;
	T.Fwd_list = NULL;
	T.pidfile = TOT_PID_FILE;
	T.configfile = TOTCONF;
	T.current_fwd = NULL;

	/* make sure these start out empty */
	for (i = 0; i < MAXPREFIXES; i++) {
		T.tot_prefix[i][0] = '\0';
	}

	/* list of forwarders */
	T.Fwd_list = list_init ();
	if (!T.Fwd_list)
		exit(1);

	/* parse command line arguments */
	for (i = 1; i < argc; i++) {
		if (!strncmp (argv[i], "-d", 2) ) {
			/* debug option */
			T.debug = atoi(&argv[i][2]);
			syslog (LOG_INFO, "debug level %d enabled", T.debug);
#ifdef NDEBUG
			if (T.debug > 1) {
				syslog (LOG_INFO, "NDEBUG was defined at compile time, so not much more debug info available");
				syslog (LOG_INFO, "Comment NDEBUG in config.h and recompile to enable verbose debugging");
			}
#endif
		} else if (!strcmp (argv[i], "-6"))
			T.ip6 = 1;
		else if (!strcmp (argv[i], "-4"))
			T.ip4 = 1;
		else if (!strcmp (argv[i], "-46") || !strcmp (argv[i], "-64"))
			T.ip4 = 1, T.ip6 = 1;
		else if (!strcmp (argv[i], "-no4"))
			T.ip4 = 0;
		else if (!strcmp (argv[i], "-no6"))
			T.ip6 = 0;
		else if (!strcmp (argv[i], "-c")) {
			T.configfile = strdup(argv[++i]);
		} else if (!strcmp (argv[i], "-p")) {
			if (conv_trick_conf((u_char *)argv[++i])) {
			    syslog (LOG_ERR, "invalid prefix on command line: %s", argv[++i]);
			    usage(1);
			}
		} else if (!strcmp (argv[i], "-h"))
			usage (1);
		else if (!strcmp (argv[i], "-v"))
			T.quiet = -1;
		else if (!strcmp (argv[i], "-q"))
			T.quiet = 1;
		else {
			syslog (LOG_ERR, "unknown option %s", argv[i]);
			usage (1);
		}
	}

#ifndef LOG_PERROR
#define LOG_PERROR 0
#endif /* not LOG_PERROR */

	if (T.debug > 0)
	    openlog("totd", LOG_PID | LOG_NDELAY | LOG_CONS | LOG_PERROR, LOG_DAEMON);
	else
	    openlog ("totd", LOG_PID | LOG_NDELAY | LOG_CONS, LOG_DAEMON);

	syslog (LOG_NOTICE, "%s", version);

	if (T.debug) {
		if (T.quiet > 0)
			setlogmask (LOG_UPTO (LOG_ERR));
		else if (!T.quiet)
        		setlogmask (LOG_UPTO (LOG_INFO));
		else
        		setlogmask (LOG_UPTO (LOG_DEBUG));
	} else {
		if (T.quiet > 0)
			setlogmask (LOG_UPTO (LOG_ERR));
		else if (!T.quiet)
			setlogmask (LOG_UPTO (LOG_WARNING));
		else if (T.quiet < 0)
        		setlogmask (LOG_UPTO (LOG_INFO));
	}

	/*
	 * read in config file
	 * possibly overriding defaults and command line options
 	 */
	if (read_config (T.configfile)) {
	    syslog (LOG_ERR, "Configuration failure");
	    syslog (LOG_INFO, "Check log, or try -d option for debug mode...");
	    exit(1);
	}

	/*
	 * Let's do some (in)sanity config checking now
	 */
	if (T.port < IPPORT_RESERVED && T.tot_uid) {
		syslog (LOG_ERR, "Need root privileges to run at reserved port %d", T.port);
		syslog (LOG_INFO, "Start totd as root or configure it to use \
a non-reserved port");
		exit(1);
	}

	if (T.totuser)
		pwd_p = getpwnam (T.totuser);
	else
		pwd_p = getpwuid (T.tot_uid);

	if (pwd_p) {
		syslog (LOG_INFO, "Found user record of %s; uid: %d gid: %d",
			pwd_p->pw_name, pwd_p->pw_uid, pwd_p->pw_gid);
		if (T.tot_uid && T.tot_uid != pwd_p->pw_uid) {
			syslog (LOG_ERR, "Need root privileges to change user to: %s", T.totuser);
			syslog (LOG_INFO, "Check also `totuser' configfile option");
			exit(1);
		} else {
			T.totuser = strdup(pwd_p->pw_name);
			T.tot_uid = pwd_p->pw_uid;
			T.tot_gid = pwd_p->pw_gid;
			/* T.su = 1; */
		}
	} else {
		syslog (LOG_ERR, "can't find user record of %s", T.totuser);
		exit(1);
	}

	if (T.pidfile) {
		FILE *pid_fp;
		pid_t pid;

		/* check if pid file exists */
		pid_fp = fopen (T.pidfile, "r");
		if (pid_fp) {
			if (fscanf (pid_fp, "%d", &pid) != 1) {
				syslog (LOG_NOTICE, "Removing bogus lockfile");
				unlink (T.pidfile);
			} else if (kill (pid, 0) == -1 && !unlink (T.pidfile)) {
				syslog (LOG_NOTICE, "Removed stale lockfile");
			} else {
				syslog (LOG_ERR, "PID file %s already exists",
					T.pidfile);
				syslog (LOG_INFO, "There can be another totd \
running. Please make sure there's no totd already running. Then delete the \
file %s and try again.", T.pidfile);
				exit (1);
			}
			fclose (pid_fp);
		}
	}

	if (!T.ip4 && T.ip6) {
		/* If user specifies we should *not* accept IPv4
		 * requests on Linux, we will bail out with an error.
		 * totd does not filter incoming requests on their source
		 * address. It is not totd's task to do so IMHO, and I will
		 * probably never implement that.
		 */
#ifdef NEEDSV4MAPPED
		syslog(LOG_ERR, "Cannot disable IPv4 when IPv6 is enabled on\
this OS, due to IPv4 mapped addresses");
		syslog(LOG_INFO, "Will always accept IPv4 *and* IPv6 requests.\
Bailing out, so that you can explicitly tell me to do so");
		exit(1);
#endif
	}

	if (T.ip4 && T.ip6) {
#ifdef WILDCARDONLY
		if (!T.wildcard) {
			syslog(LOG_ERR, "On this OS we only support \
wildcard binding when IPv6 is enabled.");
			syslog(LOG_INFO, "Please remove `interfaces' \
specification in your config file.");
			exit(1);
		}
#endif
#ifdef NEEDSV4MAPPED
		syslog(LOG_DEBUG, "IPv6 wildcard socket with IPv4 mapped, \
will not bind to wildcard IPv4 socket.");
		T.use_mapped = 1;
#endif
	}

	if (T.ip4) {
#ifdef USE_INET4
		syslog (LOG_INFO, "IPv4 activated");
#else
		syslog (LOG_ERR, "IPv4 support is not compiled in");
		exit (1);
#endif
	}

	if (T.ip6) {
#ifdef USE_INET6
		syslog (LOG_INFO, "IPv6 activated");
#else
		syslog (LOG_ERR, "IPv6 support is not compiled in");
		exit (1);
#endif
	}

	if (!T.ip4 && !T.ip6) {
		syslog (LOG_ERR, "all supported protocols are deactivated; \
what do you want me to do then?");
		exit (1);
	}

#ifdef SCOPED_REWRITE
	if (T.wildcard && T.scoped_prefixes) {
		syslog (LOG_ERR, "Scoped address rewriting currently not \
implemented when wildcard sockets are used. Please use `interfaces' keyword \
in your config file or remove `scoped' keyword");
		exit (1);
	}
#endif

	fwd_init();
	fwd_select();
	if (!T.current_fwd) {
	    syslog (LOG_ERR, "no forwarder available, what should we do then?");
	    return -1;
	}

	if (!T.debug) {
		/* purge tty */
		if (fork())
			exit (0); /* let parent exit */
		/* child goes into background */
		close (0);
		close (1);
		close (2);
		open ("/dev/null", O_RDONLY);
		dup2 (0, 1);
		dup2 (0, 2);
		i = open ("/dev/tty", O_RDWR);
		if (i >= 0) {
			ioctl (i, TIOCNOTTY, (char *) 0);
			close (i);
		}
	}

	/* as detached child, we can now write pid file */
	if (T.pidfile) {
		FILE *pid_fp;
		pid_fp = fopen (T.pidfile, "w");
		if (!pid_fp) {
			syslog (LOG_ERR, "can't open pid file");
			exit(1);
		}
		fprintf (pid_fp, "%d", getpid ());
		fclose (pid_fp);
	}

	/*
	 * let's give up root privileges (if we have them),
	 * only go back to them when needed.
	 */
	ROOTPRIV_DISABLE ("Startup successful.");

	syslog (LOG_INFO, "totd started");
	totd_eventloop ();
	exit (0);
}

void totd_eventloop (void) {
	const char *fn = "totd_eventloop()";
	struct timeval tv_out, *tvp;
	fd_set fd_read, fd_write;
	int max_fd, fdnum, i;
	time_t next_timeout;

#ifdef DBMALLOC
	unsigned long histid1, histid2, orig_size, current_size;
#endif

	/* initialize each event routine */
	ev_dup_init ();
	if (ev_signal_init () < 0) {
		syslog (LOG_ERR, "Signal event handling  initialize failed");
		totd_exit (-1);
	}
	if (ev_to_init () < 0) {
		syslog (LOG_ERR, "Timeout event handling initialize failed");
		totd_exit (-1);
	}
	if (T.port < IPPORT_RESERVED) {
		ROOTPRIV_ENABLE ("Event: initialize reserve sockets");
	}

	if (ev_tcp_conn_in_init () < 0) {
		syslog (LOG_ERR, "TCP connection initialize failed");
		totd_exit (-1);
	}

	if (net_init_socketlist(T.port) < 0) {
		syslog (LOG_ERR, "Init list of sockets failed");
		totd_exit (-1);
	}

	if (net_bind_socketlist() < 0) {
		syslog (LOG_ERR, "Binding list of sockets failed");
		totd_exit (-1);
	}

	if (ev_udp_in_init () < 0) {
		syslog (LOG_ERR, "UDP initialize failed");
		totd_exit (-1);
	}
	if (ev_tcp_out_init () < 0) {
		syslog (LOG_ERR, "TCP output routine initialize failed");
		totd_exit (-1);
	}
	if (T.port < IPPORT_RESERVED) {
		ROOTPRIV_DISABLE ("Event: initialize reserved sockets");
	}

#ifdef DBMALLOC
	/* Malloc debugging */
	if (T.debug) 
		malloc_dump (2);
	orig_size = malloc_inuse (&histid1);
	syslog (LOG_DEBUG, "Malloc Size: %ld", orig_size);
#endif

	while (1) {		/* main loop */
		if (T.port < IPPORT_RESERVED) {
			ROOTPRIV_DISABLE ("Event: mainloop");
		}

		if (T.debug > 2)
			syslog (LOG_DEBUG, "main loop: start");

		/* pick a proper forwarder */
		fwd_select ();

#ifdef DBMALLOC
		/* Malloc debugging */
		current_size = malloc_inuse (&histid2);

		if (current_size != orig_size) {
			syslog (LOG_DEBUG, "Malloc Size: %ld", current_size);
			if (T.debug) 
				malloc_list (2, histid1, histid2);
			orig_size = current_size;
			/* histid1 = histid2; */
		}
#endif

		/* signal event */
		ev_signal_process ();

		/* timeout event */
		next_timeout = ev_timeout_process ();
		if (!next_timeout) {
			if (T.debug > 2)
				syslog (LOG_DEBUG, "no timeouts at present");
			tvp = NULL;
		} else {
			tv_out.tv_usec = 0;
			tv_out.tv_sec = next_timeout - time (NULL);
			tvp = &tv_out;
			if (T.debug > 2)
				syslog (LOG_DEBUG, "next timeout after %ld s.",
					tv_out.tv_sec);
		}

		/* get FD_SET for now */
		max_fd = 0;
		FD_ZERO (&fd_read);
		FD_ZERO (&fd_write);

		if (T.debug > 3)
			syslog (LOG_DEBUG, "check for UDP fds...");

		nia_fds_set (&fd_read, &max_fd);

		if (T.debug > 3)
			syslog (LOG_DEBUG, "check for TCP-in fds...");
		i = ev_tcp_conn_in_fds (&fd_read);
		max_fd = MAXNUM (i, max_fd);

		if (T.debug > 3)
			syslog (LOG_DEBUG, "check for TCP-out fds...");
		i = ev_tcp_out_fds (&fd_write);
		max_fd = MAXNUM (i, max_fd);

		if (T.debug > 3)
			syslog (LOG_DEBUG, "%s: max_fd = %d", fn, max_fd);

		/* select */
		if (T.debug > 2)
			syslog (LOG_DEBUG, "main loop: select");

		fdnum = select (max_fd + 1, &fd_read, &fd_write, NULL, tvp);
		if (fdnum < 0) {
			if (errno == EINTR) {
				syslog (LOG_DEBUG, "%s: select() interrupted",
					fn);
				continue;	/* while(1) */
			} else {
				syslog (LOG_ERR, "%s: select(): %m", fn);
				if (net_reinit_socketlist (T.port, 1) < 0)
					totd_exit(-1);
				sleep (1);
			}
		} else {
			int sock;

			switch (nia_fds_isset (&fd_read, &sock)) {
			case 0: /* UDP */
				if (ev_udp_in_read (sock) < 0)
					syslog (LOG_INFO, "udp service error");
				continue;
			case 1: /* TCP */
				if (ev_tcp_srv_accept (sock) < 0)
					syslog (LOG_INFO, "tcp service error");
				continue;
			default: /* not found */
				if (ev_tcp_out_fd_check (&fd_write) < 0)
					syslog (LOG_INFO, "tcp output failed");

				if (ev_tcp_conn_in_fd_check (&fd_read) < 0)
					syslog (LOG_INFO, "tcp input failed");
			}
		}		/* if(...select...) */
	}			/* while(1) */
}

void totd_exit (int status) {

	ROOTPRIV_ENABLE ("exit");

	/* finish in reverse order of initialize */
	ev_tcp_out_finish ();
	ev_udp_in_finish ();
	ev_tcp_conn_in_finish ();
	/* ev_tcp_srv_in_finish() -- no such func */
	ev_to_finish ();
	ev_signal_finish ();
        if (T.Fwd_list)
                list_destroy (T.Fwd_list, fwd_freev);

	if (T.pidfile)
		unlink (T.pidfile);

	ROOTPRIV_DISABLE ("exit");
	setlogmask (LOG_UPTO (LOG_INFO));
	syslog (LOG_INFO, "terminated. %s euid = %d",
		(T.su) ? " (in su mode)" : "", geteuid ());

	exit (status);
}
