/****************************************************************************
 * Copyright (C) 1998 WIDE Project.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    This product includes software developed by WIDE Project and
 *    and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 ****************************************************************************/

/****************************************************************************
 * Copyright (C) 1999 University of Tromso.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    This product includes software developed by the University of Tromso
 *    and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 ****************************************************************************/

/*
 * <$Id: cons_tcp_request.c,v 3.28 2000/12/13 21:29:25 dillema Exp $>
 */

#include "tot.h"

COPYRIGHT1(
"@(#) Copyright (C) 1998 WIDE Project.  All rights reserved.\n");

COPYRIGHT2(
"@(#) Copyright (C) 1999 The University of Tromso.  All rights reserved.\n");

COPYRIGHT3("Written by: F. W. Dillema. feico@pasta.cs.uit.no.\n
based on code by: Yusuke DOI, Keio Univ. Murai Lab.");

CVSID("$Id: cons_tcp_request.c,v 3.28 2000/12/13 21:29:25 dillema Exp $");

int cos_tcp_request_start (struct context *parent, u_char * qname,
		           int qname_len, u_int16_t r_class, u_int16_t r_type,
			   struct sockaddr *sa) {
	const char *fn = "cos_tcp_request_start()";
	Context *cont;
        const int on = 1;
	int sock;

	if (T.debug > 1) {
		DX (syslog (LOG_DEBUG, "%s: start", fn));
	}

	cont = context_create(CTYPE_TCP_REQUEST);
	if (!cont) 
		return (cos_tcp_request_abort (cont, -1));

	cont->parent = parent;
	cont->parent->child = cont;
	cont->current_state = COS_TCP_REQUEST_START;
	cont->param.request->timeout = SEARCH_REMOTE_TIMEOUT;

	if (qname_len > MAX_DNAME) {
		qname_len = MAX_DNAME;
		syslog (LOG_WARNING, "%s: domainname longer than %d, truncating it", fn, MAX_DNAME);
	}
	memcpy (cont->qname, qname, qname_len);

	cont->qname_len = qname_len;
	cont->q_class = r_class;
	cont->q_type = r_type;
	cont->q_id = mesg_id ();
	cont->param.request->conn_sock = -1;
	cont->process = cos_tcp_request_wait_connect_process;
	cont->retry = cos_tcp_request_wait_connect_retry;

	/* make up query */
	cont->mesg_len = mesg_make_query (qname, r_type, r_class, cont->q_id, 1, Cos_TCP_Buf, MAX_STREAM);
	if (cont->mesg_len > 0 && (cont->mesg.p = malloc (cont->mesg_len))) {
		memcpy (cont->mesg.p, Cos_TCP_Buf, cont->mesg_len);
		cont->wp = cont->mesg.p;

       		/* create socket */
        	sock = socket (sa->sa_family, SOCK_STREAM, 0);
        	if (sock != -1 && ioctl (sock, FIONBIO, (char *) &on) != -1) {
        		if (!connect (sock, sa, SOCKADDR_SIZEOF(*sa)) || errno == EINPROGRESS) {
				cont->param.request->conn_sock = sock;
				if (T.debug > 1) {
					DX (syslog (LOG_DEBUG, "%s: connected socket = %d", fn, cont->param.request->conn_sock));
				}
				/* put me to input list */
				if (!ev_tcp_out_register (cont->param.request->conn_sock, cont)) {
					/* put me to timeout list */
					if (!context_timeout_register (cont, cont->param.request->timeout)) {
						if (T.debug > 1) {
	    						DX (syslog (LOG_DEBUG, "%s: Query will time out in %d seconds.\n", fn, cont->param.request->timeout));
						}
						cont->current_state = COS_TCP_REQUEST_WAIT_CONNECT;

						if (T.debug > 1) {
							DX (syslog (LOG_DEBUG, "%s: return success", fn));
						}
						/* SUCCESS */
						return 0;
					}
				}
			} else
					syslog (LOG_WARNING, "%s: can't connect: %m", fn);
		} else
               		syslog (LOG_WARNING, "%s: can't open socket: %m", fn);
	} else
		syslog (LOG_ERR, "%s: failed to make up query in %d bytes.", fn, MAX_STREAM);

	if (T.debug > 1) {
		DX (syslog (LOG_DEBUG, "%s: return failure", fn));
	}
	/* FAILURE */
	return (cos_tcp_request_abort (cont, -1));
}

int cos_tcp_request_wait_connect_process (Context * cont) {
	DX (char *fn = "cos_tcp_request_wait_connect_process()";)
	int sockerr;
	socklen_t errlen;

	if (T.debug > 1) {
		DX (syslog (LOG_DEBUG, "%s: start", fn));
	}

	/* renew timeout */
	if (cont->tout)
		cont->tout->handler = NULL;

	if (!context_timeout_register (cont, cont->param.request->timeout)) {
		/* check whether connect() actually succeeded or not. */
		errlen = sizeof (sockerr);
		if (!getsockopt (cont->param.request->conn_sock, SOL_SOCKET, SO_ERROR, (void *) &sockerr, &errlen)) {
			switch (sockerr) {
			case 0:
				if (T.debug > 1) {
					DX(syslog (LOG_DEBUG, "%s: TCP connect succeeded.", fn));
				}
				/* change state to writing */
				cont->process = cos_tcp_request_writing_process;
				cont->retry = cos_tcp_request_writing_retry;
				cont->current_state = COS_TCP_REQUEST_WRITING;
		
				/* reset wp to make sure writing process begins from first byte */
				cont->wp = NULL;
				/* SUCCESS */
				return 0;
			case ETIMEDOUT:
				if (T.debug > 1) {
					syslog (LOG_DEBUG, "%s: TCP forwarder connect timed out.", fn);
				}
				cos_tcp_request_wait_connect_retry(cont);
				/* SUCCESS */
				return 0;
			case ECONNREFUSED:
				syslog (LOG_ERR, "%s: forwarder refused connection.", fn);
				cos_tcp_request_wait_connect_retry(cont);
				/* SUCCESS */
				return 0;
			case ENETUNREACH:
				syslog (LOG_WARNING, "%s: TCP forwarder unreachable.", fn);
				cos_tcp_request_wait_connect_retry(cont);
				/* SUCCESS */
				return 0;
			default:
				/* FAILURE */
			}
		}
	}

	if (T.debug > 1) {
		DX (syslog (LOG_DEBUG, "%s: abort request", fn));
	}
	/* FAILURE */
	return (cos_tcp_request_abort (cont, -1));
}

int cos_tcp_request_wait_connect_retry (Context * cont) {
	DX (char *fn = "cos_tcp_request_wait_connect_retry";)
	struct sockaddr *sa;
        const int on = 1;
	int sa_len, sock;

	if (T.debug > 1) {
		DX (syslog (LOG_DEBUG, "%s: start", fn));
	}

	/* remove old I/O List */
	if (cont->param.request->conn_sock) {
		ev_tcp_out_remove (cont->param.request->conn_sock);
		/* stop current connection */
		close (cont->param.request->conn_sock);
		cont->param.request->conn_sock = -1;
	}

	do {
	        sa = ((Fwd *) T.current_fwd->list_data)->sa;
	        sa_len = ((Fwd *) T.current_fwd->list_data)->sa_len;
		if (!sa) /* FAILURE */
			return (cos_tcp_request_abort (cont, -1));

		cont->param.request->retry++;
		cont->param.request->timeout = cont->param.request->timeout * 2;
		if (cont->param.request->retry >= SEARCH_REMOTE_RETRY) {
		    /* pick another forwarder, if no backup keep on trying for while */
		    if (fwd_select_next() && cont->param.request->retry >= SEARCH_REMOTE_RETRY*2) {
	            	syslog (LOG_INFO, "%s: Retry limit exceeded, and out of backup forwarders", fn);
			/* FAILURE */
			return (cos_tcp_request_abort (cont, 1));
		    }
		    /* Note that we keep timout time high, even when we switched to backup */
		}

		if (T.debug > 1) {
  			DX(syslog(LOG_DEBUG, "%s: Retry %d of %d+%d will time out in %d seconds", 
				 fn, cont->param.request->retry + 1, SEARCH_REMOTE_RETRY,
				 SEARCH_REMOTE_RETRY, cont->param.request->timeout));
		}

		/* create tcp socket */
		sock = socket (sa->sa_family, SOCK_STREAM, 0);
		if (sock < 0) {
			syslog (LOG_WARNING, "%s: can't open socket: %m", fn);
			continue;
		}
		if (ioctl (sock, FIONBIO, (char *) &on) < 0) {
			syslog (LOG_WARNING, "can't ioctl on service socket: %m");
			continue;
		}
		/* start connection */
		if (connect (sock, sa, sa_len)) {
			if (errno != EINPROGRESS) {
				syslog (LOG_WARNING, "can't connect: %m");
				continue;
			}
		}

		cont->param.request->conn_sock = sock;
		break; /* Got a socket, break out of loop */
	} while (1);

	/* We successfully opened connection to a forwarder */

	/* put me to outgoing list */
	if (!ev_tcp_out_register (cont->param.request->conn_sock, cont)) {
		/* put me to timeout list */
		if (!context_timeout_register (cont, cont->param.request->timeout)) {
			if (T.debug > 1) {
				DX (syslog (LOG_DEBUG, "%s: return", fn));
			}
			/* SUCCESS */
			return 0;
		}
	}

	if (T.debug > 1) {
		DX (syslog (LOG_DEBUG, "%s: abort", fn));
	}

	/* FAILURE */
	return (cos_tcp_request_abort (cont, -1));
}

int cos_tcp_request_writing_process (Context * cont) {
        DX (char *fn = "cos_tcp_request_writing_process";)

	if (T.debug > 1) {
		DX (syslog (LOG_DEBUG, "%s: start", fn));
	}

	/* renew timeout */
	if (cont->tout)
		cont->tout->handler = NULL;

	if (!context_timeout_register (cont, cont->param.request->timeout)) {
		switch (cos_common_tcp_writemesg (cont, cont->param.request->conn_sock)) {
		case 0:
			/* continue this process */
			DX (syslog (LOG_DEBUG, "%s: return, continue writing", fn));
			return 0;
		case 1:
			/* all data written */

			/* add to new I/O list */
			if (!ev_tcp_conn_in_register (cont->param.request->conn_sock, cont)) {
				/* delete old I/O list */
				ev_tcp_out_remove (cont->param.request->conn_sock);

				/* go to readlen context */
				cont->process = cos_tcp_request_readlen_process;
				cont->retry = cos_tcp_request_readlen_retry;
				cont->current_state = COS_TCP_REQUEST_READLEN;

				if (T.debug > 1) {
					DX (syslog (LOG_DEBUG, "%s: return, start reading", fn));
				}

				/* SUCCESS */
				return 0;
			}
			/* FAILURE */
			break;
		default:
			/* FAILURE */
			break;
		}
	}

	if (T.debug > 1) {
		DX (syslog (LOG_DEBUG, "%s: abort", fn));
	}
	return (cos_tcp_request_abort (cont, -1));

}

int cos_tcp_request_writing_retry (Context *cont) {
	syslog (LOG_NOTICE, "cos_tcp_request_writing_retry(): connection not responding.... closed");
	return (cos_tcp_request_abort (cont, -1));
}

int cos_tcp_request_readlen_process (Context * cont) {
	char *fn = "cos_tcp_request_readlen_process()";
	u_int16_t length_buf;

	if (T.debug > 1) {
		DX (syslog (LOG_DEBUG, "%s: start", fn));
	}

	/* renew timeout */
	if (cont->tout)
		cont->tout->handler = NULL;

	if (context_timeout_register (cont, cont->param.request->timeout) < 0)
		return (cos_tcp_request_abort (cont, -1));

	/* read length buffer */
	if (read (cont->param.request->conn_sock, (u_char *) (&length_buf), sizeof (u_int16_t)) < sizeof (u_int16_t)) {
		syslog (LOG_NOTICE, "failed to read the length buffer on TCP connection.");
		return (cos_tcp_request_abort (cont, -1));
	}
	length_buf = ntohs (length_buf);

	if (T.debug > 1) {
		DX (syslog (LOG_DEBUG, "%s: data length = %d", fn, length_buf));
	}

	free (cont->mesg.p);
	cont->mesg_len = 0;
	cont->mesg.p = malloc (length_buf);
	if (!cont->mesg.p) {
		syslog (EM_F (EM_F_MEMEX), fn);
		return (cos_tcp_request_abort (cont, -1));
	}
	cont->mesg_len = length_buf;
	cont->wp = cont->mesg.p;

	/* next state */
	cont->process = cos_tcp_request_reading_process;
	cont->retry = cos_tcp_request_reading_retry;
	cont->current_state = COS_TCP_REQUEST_READING;

	if (T.debug > 1) {
		DX (syslog (LOG_DEBUG, "%s: return 0", fn));
	}
	return 0;
}

int cos_tcp_request_readlen_retry (Context * cont) {
	syslog (LOG_NOTICE, "cos_tcp_request_readlen_retry(): connection not respond.... closed");
	return (cos_tcp_request_abort (cont, -1));
}

int cos_tcp_request_reading_process (Context * cont) {
	const char *fn = "cos_tcp_request_reading_process()";

	if (T.debug > 1) {
		DX (syslog (LOG_DEBUG, "%s: start", fn));
	}

	/* renew timeout */
	if (cont->tout)
		cont->tout->handler = NULL;

	if (!context_timeout_register (cont, cont->param.request->timeout)) {
		int len;

		len = read (cont->param.request->conn_sock, cont->wp, cont->mesg_len - (cont->wp - cont->mesg.p));
		if (len <= 0)
			syslog (LOG_NOTICE, "%s: Read failed on TCP connection: %m", fn);
		else {
			cont->wp += len;

			if (cont->wp < (cont->mesg.p + cont->mesg_len)) {
				/* continue this process */
				if (T.debug > 1) {
					DX (syslog (LOG_DEBUG, "%s: return, %d bytes left to read", fn, (cont->mesg.p + cont->mesg_len) - cont->wp));
				}
				return 0;
			} else {
				if (T.debug > 1) {
					DX (syslog (LOG_DEBUG, "%s: return, finish", fn));
				}
				return cos_tcp_request_finish (cont);
			}
		}
	}

	if (T.debug > 1) {
		DX (syslog (LOG_DEBUG, "%s: abort", fn));
	}

	/* FAILURE */
	return (cos_tcp_request_abort (cont, -1));
}

int cos_tcp_request_reading_retry (Context *cont) {
	syslog (LOG_NOTICE, "cos_tcp_request_reading_retry(): connection not respond... closed");
	return (cos_tcp_request_abort (cont, -1));
}

/* destroy them all! */
void cos_tcp_request_destroy (Context *cont) {
	const char *fn = "cos_tcp_request_reading_destroy()";

	if (T.debug > 1) {
		DX (syslog (LOG_DEBUG, "%s: start", fn));
	}

	if (cont) {
		if (cont->param.request) {
			if (cont->param.request->conn_sock >= 0) {
				shutdown (cont->param.request->conn_sock, SHUT_RDWR);
				close (cont->param.request->conn_sock);
				switch (cont->current_state) {
				case COS_TCP_REQUEST_START:
				case COS_TCP_REQUEST_WAIT_CONNECT:
				case COS_TCP_REQUEST_WRITING:
					ev_tcp_out_remove (cont->param.request->conn_sock);
					break;
				case COS_TCP_REQUEST_READLEN:
				case COS_TCP_REQUEST_READING:
					ev_tcp_conn_in_remove (cont->param.request->conn_sock);
					break;
				default:
					syslog (LOG_ERR, "%s: Unknown state on destruction.", fn);
					break;
				}	/* switch */
				free (cont->param.request);
			}	/* if(...conn_sock > 0) */
		}		/* if(param.request) */
	}			/* if(cont) */
	return;
}

int cos_tcp_request_abort (Context * cont, int status) {

	/* if parent exists, process parent */
	if (cont->current_state != COS_TCP_REQUEST_START && cont->parent) {
		cont->parent->child = NULL;	/* indicates child failure */
		cont->parent->process (cont->parent);
	}
	/* free all resources */
	context_destroy (cont);

	return (status);
}

int cos_tcp_request_finish (Context * cont) {
	const char *fn = "cos_tcp_request_finish";

	if (T.debug > 1) {
		DX (syslog (LOG_DEBUG, "%s: start", fn));
	}

	/* parse answers */

	list_destroy (cont->an_list, rrset_freev);
	list_destroy (cont->ns_list, rrset_freev);
	list_destroy (cont->ar_list, rrset_freev);
	cont->an_list = list_init ();
	cont->ns_list = list_init ();
	cont->ar_list = list_init ();
        if (!cont->an_list || !cont->ns_list || !cont->ar_list) {
                syslog (LOG_ERR, "%s: can't recreate lists for parse answer data.", fn);
                return(cos_tcp_request_abort (cont, -1));
        }

        if (mesg_parse (cont->mesg.p, cont->mesg_len, cont->an_list, cont->ns_list, cont->ar_list)) {
                syslog (LOG_WARNING, "%s: can't parse answer data.", fn);
                return (cos_tcp_request_abort (cont, -1));
        }

        /* if parent exists (it should always), process parent */
        if (cont->parent) {
		if (T.debug > 1) {
                	DX (syslog (LOG_DEBUG, "%s: process parent context", fn));
		}
                cont->parent->process (cont->parent);
        }
        context_destroy (cont);

	if (T.debug > 1) {
	        DX (syslog (LOG_DEBUG, "%s: return success", fn));
	}
        return 0; /* we're happy */
}
