/*
 *   tosha.c
 *
 *   Oliver Fromme  <olli@fromme.com>
 *
 *   Copyright (C) 1997,1998,1999
 *        Oliver Fromme.  All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *   1. Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *   3. Neither the name of the author nor the names of any co-contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY OLIVER FROMME AND CONTRIBUTORS ``AS IS'' AND
 *   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *   ARE DISCLAIMED.  IN NO EVENT SHALL OLIVER FROMME OR CONTRIBUTORS BE LIABLE
 *   FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 *   OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 *   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *   OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 *   SUCH DAMAGE.
 *
 *   @(#)$Id: tosha.c,v 1.7 1999/01/01 23:32:04 olli Exp $
 */

static const char cvsid[]
    = "@(#)$Id: tosha.c,v 1.7 1999/01/01 23:32:04 olli Exp $";

#include "utils.h"
#include "getlopt.h"
#include "global.h"
#include "toscsi.h"
#include "toform.h"
#include "toconf.h"

const char *Vnumber = "0.6";
const char *Vdate   = "01-Jan-1999";

/*
 *   In case someone is actually reading this source:
 *   Every CD DA audio sector consists of 98 frames.
 *   Thus:
 *
 *      1 sample = 4 bytes    (16 bits left, 16 bits right)
 *      1 frame  = 6 samples  (plus 1 byte subchannel info,
 *                             but we don't need that)
 *      1 sector = 98 frames  (= 2352 bytes = 1/75 second)
 *      1 second = 75 sectors (= 44,100 samples = 176,400 bytes)
 *
 *   CD-XA mode2/form2 sectors (used for Video-CD) also consist
 *   of 2352 bytes, 2324 of which contain the actual MPEG system
 *   stream data.  Video-CDs use the same data rate as audio
 *   CDs (1 sector = 1/75 second).
 */

int readcmd = 0x28;	/* read command to use for DA reading */
int byteswap = FALSE;	/* do we have to swap byte order? */
toscsi_handle *handle;	/* SCSI layer access handle */

byte buf[READBUFSIZE];

/*
 *   Get the current density code and sector size.
 */

void
get_density (int *density, int *secsize)
{
	toscsi_request (handle, 12, TOSCSI_READ, "1A 0 1 0 C 0");
	*density = buf[4];
	*secsize = buf[10] << 8 | buf[11];
}

/*
 *   Set the density code and sector size.
 */

void
set_density (int density, int secsize)
{
	memset (buf, 0, 12);
	buf[3] = 8;
	buf[4] = density;
	buf[10] = secsize >> 8;
	buf[11] = secsize & 0xff;
	toscsi_request (handle, 12, TOSCSI_WRITE, "15 10 0 0 C 0");
}

/*
 *   Calculate minutes/seconds/sectors from a number of sectors.
 */

void
sectors2msf (ulong sectors, int *min, int *sec, int *frm)
{
	*min = sectors / (60 * 75);
	*sec = (sectors - *min * (60 * 75)) / 75;
	*frm = sectors - 75 * (*min * 60 + *sec);
}

/*
 *   Variables and defaults for command line options.
 *   Refer to the manual page tosha(1) for more information.
 */

char default_tracklstr[] = "1-100";
char default_outname[]   = "track%02d.%s";
char default_device[]    = "/dev/cd0c";
#define DEFAULT_KBPS 128
const formspec *default_formatspec = format;	/* also see formatspec! */

char *tracklstr	 = default_tracklstr;
char *outname 	 = default_outname;
char *device	 = default_device;
int do_reset	 = FALSE;
int indexonly	 = FALSE;
int quiet	 = FALSE;
int verbose	 = 0;
int startsec	 = -1;
int endsec	 = -1;
int kbps	 = DEFAULT_KBPS;
int sectorsperbuf= -1;
int userjitter	 = -1;
const formspec *formatspec = format;	/* 1st format == raw/pcm */

void
tracklist_usage ()
{
	fprintf (stderr, "Track list format:  a | a,b | [a]-[b] ...\n");
	fprintf (stderr, "Example:  -3,5,8-10,12-\n");
	exit (1);
}

void
tracklist_syntax ()
{
	fprintf (stderr, "%s: Syntax error in track list.\n", me);
	tracklist_usage();
}

long long starttime;		/* for estimating the remaining time */
ulong totalsectors = 0;		/* number of sectors we're going to read */
int tracklistsize = 0;		/* number of tracks to read */
int tracklist[100];		/* tracks numbers we're going to read */
ulong trackstart[101];		/* start offsets of the CD's tracks */
unsigned char trackctl[101];	/* type/control codes of the CD's tracks */
int numtracks;			/* number of tracks on the CD */
toconf_entry *devconfig;
int modechange, denschange, jitter;

/*
 *   Add the tracks from x to y to our tracklist.
 */

void
addtracks (int x, int y)
{
	while (x <= y && tracklistsize < 100)
		tracklist[tracklistsize++] = x++ - 1;
	if (x <= y) {
		fprintf (stderr, "%s: Too many tracks specified (maximum is 100).\n", me);
		exit (1);
	}
}

/*
 *   Convert the track list specification (char *tracklstr)
 *   to a sequence of track numbers (int tracklist[]).
 */

void
xlate_tracklist ()
{
	int a = 0, b = 1;
	int listopen = FALSE;
	char *cptr = tracklstr;

	while (*cptr)
		switch (*cptr) {
			case ',':
				if (listopen)
					addtracks (a, b);
				else
					if (b > 0)
						addtracks (b, b);
					else
						tracklist_syntax();
				listopen = FALSE;
				a = 0;
				b = 1;
				cptr++;
				break;
			case '-':
				if (listopen)
					tracklist_syntax();
				a = b;
				b = 100;
				listopen = TRUE;
				cptr++;
				break;
		default:
			if (*cptr >= '0' && *cptr <= '9') {
				b = 0;
				do {
					b = b * 10 + (*cptr++ - '0');
				} while (*cptr >= '0' && *cptr <= '9');
				
			}
			else {
				fprintf (stderr, "%s: Illegal character \"%c\" in track list.\n", me, *cptr);
				tracklist_usage();
			}
		}
	if (listopen)
		addtracks (a, b);
	else
		if (b > 0)
			addtracks (b, b);
		else
			tracklist_syntax();
}

char *
resolve_extension (char *template)
{
	char *cptr, *tmpstr;
	int prefix;

	if (!(cptr = strstr(template, "%s")))
		return template;
	if (!(tmpstr = malloc(strlen(template) + strlen(formatspec->ext) - 1)))
		out_of_memory();
	if ((prefix = cptr - template))
		strncpy (tmpstr, template, prefix);
	strcat (tmpstr + prefix, formatspec->ext);
	prefix += strlen(formatspec->ext);
	strcat (tmpstr + prefix, cptr + 2);
	return tmpstr;
}

int
version (char *dummy)
{
	fprintf (stderr, "tosha v%s, %s\n", Vnumber, Vdate);
	fprintf (stderr, "Copyright (c) 1997-1999 by Oliver Fromme <olli@fromme.com>\n");
	exit (1);
}

int
usage (char *dummy)
{
	int i;

	fprintf (stderr, "Usage:  %s [options]\n", me);
	fprintf (stderr, "Options [defaults]:\n");
	fprintf (stderr, "   -t track(s)   list of tracks to read   [%s]\n", default_tracklstr);
	fprintf (stderr, "   -o file       output file name (\"-\" = stdout)   [%s]\n", default_outname);
	fprintf (stderr, "   -f format     output format:");
	for (i = 0; format[i].name1; i++)
		fprintf (stderr, "%s %s", i ? "," : "", format[i].name1);
	fprintf (stderr, "   [%s]\n", default_formatspec->name1);
	fprintf (stderr, "   -d device     CD device   [%s]\n", default_device);
	fprintf (stderr, "   -s sector     start sector (overrides -t) [none]\n");
	fprintf (stderr, "   -e sector     end sector (overrides -t) [none]\n");
	fprintf (stderr, "   -k kbps       kbps for mp3 size estimates [%d]\n", DEFAULT_KBPS);
	fprintf (stderr, "   -b size       size of read buffer in sectors [device-dependant]\n");
	fprintf (stderr, "   -j size       size of jitter corr. overlap (0 = no JC) [device-dependant]\n");
	fprintf (stderr, "   -i            display track index only (-iq = compact format for scripts)\n");
	fprintf (stderr, "   -r            try to reset CD-ROM density and sector size, then exit\n");
	fprintf (stderr, "   -q            quiet (don't display anything except for errors)\n");
	fprintf (stderr, "   -v            verbose (display additional information, -vv = even more)\n");
	fprintf (stderr, "   -V            print the version number, then exit\n");
	exit (1);
}

int
setfmt (char *fmtname)
{
	int i;

	for (i = 0; format[i].name1; i++)
		if (strcasecmp(format[i].name1, fmtname) == 0 ||
		    strcasecmp(format[i].name1, fmtname) == 0) {
			formatspec = format + i;
			return 0;
		}
	usage (NULL);
	return -1;
}

int
incverb (char *dummy)
{
	verbose++;
	return 0;
}

/*
 *   Definition of command line options.
 *   See getlopt.h for more information if you're interested.
 */

topt opts[] = {
	{'t', "track",   GLO_CHAR, 0,       &tracklstr,    0},
	{'o', "output",  GLO_CHAR, 0,       &outname,      0},
	{'f', "format",  GLO_CHAR, setfmt,  0,             0},
	{'d', "device",  GLO_CHAR, 0,       &device,       0},
	{'s', "start",   GLO_NUM,  0,       &startsec,     0},
	{'e', "end",     GLO_NUM,  0,       &endsec,       0},
	{'k', "kbps",    GLO_NUM,  0,       &kbps,         0},
	{'b', "buffer",  GLO_NUM,  0,       &sectorsperbuf,0},
	{'j', "jitter",  GLO_NUM,  0,       &userjitter,   0},
	{'i', "index",   0,        0,       &indexonly,    TRUE},
	{'r', "reset",   0,        0,       &do_reset,     TRUE},
	{'q', "quiet",   0,        0,       &quiet,        TRUE},
	{'v', "verbose", 0,        incverb, 0,             0},
	{'V', "version", 0,        version, 0,             0},
	{'h', "help",    0,        usage,   0,             0},
	{0, 0, 0, 0, 0, 0}
};

void
print_separator (void)
{
	fprintf (stderr, "---------------------------------------------------------------------\n");
}

void
print_head (void)
{
	fprintf (stderr, "\n");
	fprintf (stderr, " track   playing  start    end     raw size  mp3 size   # of   track\n");
	fprintf (stderr, " number   time    sector  sector   in bytes  %3d kbps  frames  type\n", kbps);
	print_separator();
}

/*
 *   Return the track mode:
 *      -1   data
 *       0   video
 *       1   audio
 *
 *   We're declaring all mode2 tracks as "video" tracks...
 *   This is certainly not correct, but good enough for tosha.
 */

int
is_audio (int track, ulong start)
{
	int i, frmctl;

	frmctl = -1;
	if (track > 0)
		frmctl = trackctl[track - 1];
	else {
		for (i = 0; i < numtracks; i++)
			if (start >= trackstart[i]
			    && start < trackstart[i + 1]) {
				frmctl = trackctl[i];
				break;
			}
		if (frmctl < 0) /* Ugh! */
			if (numtracks >= 1 && start >= trackstart[1])
				frmctl = trackctl[numtracks - 1];
			else
				frmctl = trackctl[0];
	}
	if ((frmctl & 4) == 0)
		return 1;
	else if ((frmctl & 3) == 2)
		return 0;
	else
		return -1;
}

void
print_trackinfo (int track, int tracks, ulong start, ulong end1, int audio)
{
	int min, sec, frm;
	int audiosectors, timesectors;
	static int audiototal = 0;
	static int timetotal = 0;

	if (track > 0 && tracks == 1)
		fprintf (stderr, "  %3d  ", track);
	if (track == -2)
		timesectors = timetotal;
	else
		timesectors = (audio >= 0) ? end1 - start : 0;
	if (timesectors > 0) {
		sectors2msf (timesectors, &min, &sec, &frm);
		fprintf (stderr, "%3d:%02d'%02d", min, sec, frm);
	}
	else
		fprintf (stderr, "        -");
	if (track == -2) {
		fprintf (stderr, " %7ld sectors", end1 - start);
		audiosectors = audiototal;
	}
	else {
		fprintf (stderr, " %7ld %7ld", start, end1 - 1);
		audiosectors = (audio > 0) ? end1 - start : 0;
		audiototal += audiosectors;
	}
	fprintf (stderr, " %10ld", (long) (end1 - start) * 2352);
	if (audiosectors > 0)
		fprintf (stderr, " %9ld %7ld",
		    (audiosectors * (long) kbps * 5) / 3
			/* simple heuristic: add two mp3 frames per track */
			+ (kbps * (long)320 * tracks) / 49,
		    ((long) audiosectors * 49 + 95) / 96);
			/* This one doesn't work for layer-1 streams,
			   but who uses them anyway ...  */
	else
		fprintf (stderr, "         -       -");
	if (track >= -1)
		if (audio >= 0) {
			fprintf (stderr, "  %s\n",
			    (audio > 0) ? "audio" : "video");
			timetotal += end1 - start;
		}
		else
			fprintf (stderr, "  data\n");
	else
		fprintf (stderr, "\n");
}

void
get_time (long long *tm)
{
	struct timeval tv;

	gettimeofday (&tv, NULL);
	*tm = (long long)tv.tv_sec * 1000000 + tv.tv_usec;
}

/*
int orig_density, orig_sectorsize;
*/

/*
 *   Get the original density code from the drive.
 */

void
get_orig_density (void)
{
	/*
	get_density (&orig_density, &orig_sectorsize);
#ifdef DEBUG
	fprintf (stderr, "Original density: code 0x%02x, sector size %d.\n",
	    density, sectorsize);
#endif
	*/
}

/*
 *   Reset the density code to the original value.
 */

void
reset_density (void)
{
	/*
	set_density (orig_density, orig_sectorsize);
	*/
	set_density (0, 2048);
}

/*
 *   Set CDDA density code and sector size,
 *   if the drive wants this.
 */

void
init_density (bool audio)
{
	static int last_audio = 42;
	static bool mode_changed = FALSE;

	if (last_audio == audio)
		return;
	last_audio = audio;
	if (audio) {
		readcmd = devconfig->readcmd;
		modechange = devconfig->mdchng;
		denschange = devconfig->density;
		byteswap = devconfig->swab;
		if (userjitter >= 0)
			jitter = userjitter;
		else
			jitter = devconfig->jitter;
	}
	else {
		readcmd = 0x28;
		modechange = 1;
		denschange = 0x00;
		byteswap = FALSE;	/* Don't byteswap VCD data! */
		if (userjitter >= 0) {
			jitter = userjitter;
			fprintf (stderr, "%s: Warning: Using non-zero "
			    "overlap size of %d sectors.\n", me, jitter);
		}
		else
			jitter = 0;	/* ignore default jitter for VCDs */
	}
	if (modechange) {
		set_density (denschange, 2352);
		mode_changed = TRUE;
	}
	else if (mode_changed) {
		reset_density();
		mode_changed = FALSE;
	}
}

/*
 *   Read the table of contents (TOC) from the CD.
 */

void
read_toc (void)
{
	int i;
	int trackspp;	/* number of tracks incl. lead-out track */
	bool needmod2;

	set_density (0, 2048);
	if (toscsi_request (handle, 2048, TOSCSI_READ,
	    "43 0 0 0 0 0 1 8 0 0") < 0) {
		fprintf (stderr, "%s: Can't read table of contents.\n", me);
		exit (1);
	}
	trackspp = ((buf[0] << 8 | buf[1]) - 2) / 8;
	if (trackspp > 101)
		trackspp = 101;	/* max. 100 + lead-out track */
	else if (trackspp < 1)
		trackspp = 1;
	numtracks = trackspp - 1;
	for (i = 0; i < trackspp; i++) {
		trackctl[i] = buf[5 + 8 * i];
		trackstart[i] =
		    buf[8 + 8 * i] << 24 | buf[9 + 8 * i] << 16 |
		    buf[10 + 8 * i] << 8 | buf[11 + 8 * i];
	}
	needmod2 = TRUE;
	for (i = 0; i < numtracks; i++)
		if ((trackctl[i] & 4) != 0) {
			if (needmod2) {
				set_density (0, 2352);
				needmod2 = FALSE;
			}
			trackctl[i] = 4
			    | toscsi_readmode(handle, trackstart[i]);
		}
	reset_density();
}

/*
 *   This is the ugly long main function ...
 */

int
main (int argc, char *argv[])
{
	int pcmfd = 1;	/* output file descriptor, default = 1 = stdout */
	long long endtime;
	int i;
	
	char *vendor, *product, *versid;
	char *ofname;	/* output file name */
	int index;
	int singlefile = FALSE;
	int isaudio;

	utils_init (argv[0]);
#ifdef DEBUG
	fprintf (stderr, "%s: DEBUG mode is ON.\n", me);
#endif
	if (argc <= 1)
		usage (NULL);
	toconf_readconfig();
	parselopts (argc, argv, opts, me);
	if (loptind < argc)
		usage (NULL);
	if (startsec >= 0 || endsec >= 0) {
		if (startsec < 0 || endsec < 0) {
			fprintf (stderr,
			    "%s: -s and -e must be used together.\n", me);
			exit (1);
		}
		if (startsec > endsec) {
			fprintf (stderr, "%s: end sector must not be "
			    "smaller than start sector.\n", me);
			exit (1);
		}
		endsec++;
		tracklistsize = 1;
	}
	else
		xlate_tracklist();

	/*
	 *   Open SCSI device and initialize SCSI request structure.
	 */

	handle = toscsi_open(device);
	if (do_reset) {
		if (!quiet)
			fprintf (stderr, "Trying to reset CD-ROM "
			    "density and sector size...\n");
		set_density (0, 2048);
		toscsi_close (handle);
		if (!quiet)
			fprintf (stderr, "Done.\n");
		exit (0);
	}

	/*
	 *   Get vendor & product IDs.
	 */

	toscsi_request (handle, 64, TOSCSI_READ, "12 0 0 0 40 0");
	vendor = justify(strndup((char *) buf + 8, 8));
	product = justify(strndup((char *) buf + 16, 16));
	versid = justify(strndup((char *) buf + 32, 4));
	if (!quiet)
		fprintf (stderr, "Device: %s -- \"%s\" \"%s\" \"%s\"\n",
			device, vendor, product, versid);
	devconfig = toconf_searchentry(vendor, product, versid);
	if (sectorsperbuf <= 0)
		sectorsperbuf = devconfig->blocks;
	else
		fprintf (stderr, "%s: Warning: Overriding read size "
		    "of %d sectors.\n", me, devconfig->blocks);
	if (userjitter >= 0)
		fprintf (stderr, "%s: Warning: Overriding overlap size "
		    "of %d sectors.\n", me, devconfig->jitter);
	if (sectorsperbuf < 1 || sectorsperbuf > MAX_SECTORSPERBUF) {
		fprintf (stderr, "%s: Error: Number of sectors per "
		    "read access must be <= %d.\n",
		    me, MAX_SECTORSPERBUF);
		exit (1);
	}
	if (userjitter >= sectorsperbuf) {
		fprintf (stderr, "%s: Error: Overlapping sectors "
		    "must be < sectors per read access (%d).\n",
		    me, sectorsperbuf);
		exit (1);
	}
	if (userjitter > 0) {
		fprintf (stderr, "%s: Error: Jitter correction is not "
		    "enabled in this version of tosha.\n", me);
		exit (1);
	}

	/*
	 *   Read the table of contents (track index).
	 */

	get_orig_density();
	read_toc();

	/*
	 *   Calculate total number of sectors that we're going to read.
	 */

	if (startsec >= 0)
		totalsectors = endsec - startsec;
	else {
		totalsectors = 0;
		for (index = 0; index < tracklistsize; index++) {
			if ((i = tracklist[index]) >= numtracks)
				continue;
			totalsectors += trackstart[i+1] - trackstart[i];
		}
	}

	/*
	 *   Initialize variables for the desired output format.
	 */

	if (formatspec->swapbytes)
		byteswap = !byteswap;
	outname = resolve_extension(outname);

	/*
	 *   Now get us the stuff!
	 */

	if (!indexonly && (singlefile = startsec >= 0 || !strchr(outname, '%')))
		if (!strcmp(outname, "-"))
			pcmfd = 1;
		else {
			if (!quiet)
				fprintf (stderr,
					"Output file: %s\n",
					outname);
			if ((pcmfd = open(outname, O_WRONLY | O_CREAT |
					O_TRUNC, 0644)) < 0)
				die ("open(output file)");
		}
	if (!quiet)
		print_head();
	get_time (&starttime);
	if (startsec >= 0) {
		isaudio = is_audio(-1, startsec);
		if (!quiet || indexonly) {
			fprintf (stderr, "    -  ");
			print_trackinfo (-1, 1, startsec, endsec, isaudio);
		}
		if (!indexonly) {
			if (!quiet)
				fprintf (stderr, "  Reading ...\r");
			formatspec->writeheader (
			    (endsec - startsec) * SECTORSIZE, pcmfd);
			init_density (isaudio > 0);
			toscsi_readsectors (handle, startsec, endsec, pcmfd);
			reset_density();
		}
	}
	else {
		if (singlefile && !indexonly)
			formatspec->writeheader (totalsectors * SECTORSIZE,
			    pcmfd);
		for (index = 0; index < tracklistsize; index++) {
			if ((i = tracklist[index]) >= numtracks)
				continue;
			isaudio = is_audio(i + 1, trackstart[i]);
			if (!quiet || indexonly)
				print_trackinfo (i + 1, 1, trackstart[i],
				    trackstart[i+1], isaudio);
			if (indexonly)
				continue;
			if (singlefile) {
				if (!quiet)
					fprintf (stderr, "  Reading ...\r");
			}
			else {
				asprintf (&ofname, outname, i+1);
				if (!quiet)
					fprintf (stderr, verbose >= 2 ?
						"  (output file: %s)\n" :
						"  Reading to %s ...\r",
						ofname);
				if ((pcmfd = open(ofname, O_WRONLY | O_CREAT |
						O_TRUNC, 0644)) < 0)
					die ("open(output file)");
				formatspec->writeheader ((trackstart[i+1] -
				    trackstart[i]) * SECTORSIZE, pcmfd);
			}
			init_density (isaudio > 0);
			toscsi_readsectors (handle,
			    trackstart[i], trackstart[i + 1], pcmfd);
			if (!singlefile)
				close (pcmfd);
		}
		reset_density();
		if (!quiet && tracklistsize > 1) {
			print_separator();
			fprintf (stderr, " total ");
			print_trackinfo (-2, tracklistsize, 0, totalsectors,
			    TRUE);
		}
	}
	if (!indexonly && singlefile && pcmfd != 1)
		close (pcmfd);
	get_time (&endtime);
	if (endtime <= starttime)
		endtime = starttime + 1;
	if ((!quiet || verbose) && !indexonly) {
		ulong tdiff = (endtime - starttime + 500) / 1000;
		float Kb = (float)(totalsectors * 147) / 64;
		float kbs = (Kb * 128) / (tdiff < 4 ? 1 : ((tdiff + 4) / 8));
		float speed = kbs / 176.4;
		if (!quiet)
			fprintf (stderr, "%62s\n", "");
		fprintf (stderr,
			"Transferred %.3f Mb in %.1f seconds, %.1f kb/s, speed: %.1f.\n",
			Kb / 1024, (float)tdiff / 1000, kbs, speed);
	}

	/*
	 *   We're done.  Close the device.
	 */

	toscsi_close (handle);
	exit (0);
}

/* EOF */
