/*
 *   toconf.c
 *
 *   Configuration file support for tosha.
 *
 *   Oliver Fromme  <olli@fromme.com>
 *
 *   Copyright (C) 1997,1998,1999
 *        Oliver Fromme.  All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *   1. Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *   3. Neither the name of the author nor the names of any co-contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY OLIVER FROMME AND CONTRIBUTORS ``AS IS'' AND
 *   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *   ARE DISCLAIMED.  IN NO EVENT SHALL OLIVER FROMME OR CONTRIBUTORS BE LIABLE
 *   FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 *   OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 *   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *   OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 *   SUCH DAMAGE.
 *
 *   @(#)$Id: toconf.c,v 1.4 1999/01/01 23:31:56 olli Exp $
 */

static const char cvsid[]
    = "@(#)$Id: toconf.c,v 1.4 1999/01/01 23:31:56 olli Exp $";

#include <stdlib.h>
#include <limits.h>

#include "utils.h"
#include "global.h"
#include "toconf.h"

toconf_entry *toconf = NULL;
int toconf_num = 0;

#ifndef GLOBAL_ETC_DIR
#	define GLOBAL_ETC_DIR "/usr/local/etc"
#endif

const char *configfile[] = {
#ifndef NO_HOME_TOSHARC
#	ifdef LOCAL_TOSHARC
		LOCAL_TOSHARC
#	else
		"~/.tosharc",
#	endif
#endif
#ifdef ADDITIONAL_TOSHARC
	ADDITIONAL_TOSHARC ,
#endif
	GLOBAL_ETC_DIR "/tosharc",
	NULL
};

static bool
whitespace (char c)
{
	return c == ' ' || c == '\t' || c == '\r' || c == '\n';
}

static void
skipwhitespace (char **c2ptr)
{
	while (**c2ptr && whitespace(**c2ptr))
		(*c2ptr)++;
}

typedef struct {
	const char *name;
	char *buf;
	int line;
} filestate;

static void
file_error (filestate *fs, const char *msg, const char *par, const char *pos)
{
	fprintf (stderr, "%s:  Error in %s line %d column %d:\n   ",
	    me, fs->name, fs->line, pos - fs->buf);
	fprintf (stderr, msg, par);
	fprintf (stderr, "\n");
	exit (1);
}

static char *
readstring (char **c2ptr, const char *field, filestate *fs)
{
	char *cptr, *qptr, *s;

	cptr = *c2ptr;
	if (*cptr++ != '\"')
		file_error (fs, "Expected quoted %s string.", field, cptr);
	if (!(qptr = strchr(cptr, '\"')))
		file_error (fs, "Unterminated %s string.", field, cptr);
	if (!(s = (char *) malloc(1 + qptr - cptr)))
		out_of_memory();
	strncpy (s, cptr, qptr - cptr);
	s[qptr - cptr] = 0;
	*c2ptr = qptr + 1;
	skipwhitespace (c2ptr);
	return s;
}

static bool
readbool (char **c2ptr, const char *field, filestate *fs)
{
	char *cptr;
	bool result;

	cptr = *c2ptr;
	result = FALSE;
	if (*cptr == '1')
		result = TRUE;
	else if (*cptr != '0')
		file_error (fs, "Expected 0 or 1 for \"%s\".", field, cptr);
	*c2ptr = cptr + 1;
	skipwhitespace (c2ptr);
	return result;
}

static int
readint (char **c2ptr, int min, int max, const char *field, filestate *fs)
{
	char *cptr, *eptr;
	int result;

	cptr = *c2ptr;
	if (!*cptr)
		file_error (fs, "Missing \"%s\" field.", field, cptr);
	result = strtol(cptr, &eptr, 0);
	if (eptr == cptr || (*eptr && !whitespace(*eptr)))
		file_error (fs, "Illegal characters in \"%s\" field.",
		    field, eptr);
	if (result < min || result > max)
		file_error (fs, "\"%s\" field out of range.", field, cptr);
	*c2ptr = eptr;
	skipwhitespace (c2ptr);
	return result;
}

static int
readconfigfile (const char *filename)
{
	FILE *f;
	char buf[8192];
	char *cptr, *jptr;
	int cn;
	filestate fs;

	if (!(f = fopen(filename, "r")))
		return 0;
	fs.name = filename;
	fs.buf = buf;
	fs.line = 0;
	while ((cptr = fgets(buf, 8192, f))) {
		fs.line++;
		skipwhitespace (&cptr);
		if (!*cptr || *cptr == '#')
			continue;
		cn = toconf_num++;
		if (!(toconf = (toconf_entry *) realloc(toconf,
		    sizeof(toconf_entry) * toconf_num)))
			out_of_memory();
		toconf[cn].vendor  = readstring(&cptr, "vendor", &fs);
		toconf[cn].product = readstring(&cptr, "product", &fs);
		toconf[cn].version = readstring(&cptr, "version", &fs);
		toconf[cn].readcmd = readint(&cptr, 0, 255, "readcmd", &fs);
		toconf[cn].mdchng  = readbool(&cptr, "mdchng", &fs);
		toconf[cn].density = readint(&cptr, 0, 255, "density", &fs);
		toconf[cn].swab    = readbool(&cptr, "swap bytes", &fs);
		toconf[cn].blocks  = readint(&cptr, 1, MAX_SECTORSPERBUF,
		    "blocks", &fs);
		jptr = cptr;
		toconf[cn].jitter  = readint(&cptr, 0, MAX_SECTORSPERBUF,
		    "jitter", &fs);
		if (toconf[cn].jitter >= toconf[cn].blocks)
			file_error (&fs, "\"jitter\" value must be less "
			    "than \"blocks\" value.", "", jptr);
		if (*cptr)
			file_error (&fs, "Invalid characters at the end "
			    "of the line.", "", cptr);
	}
	fclose (f);
	return 1;
}

static int
getfilename (int number, char **fn)
{
	const char *filename, *home;

	if (!(filename = configfile[number]))
		return -1;	/* end of list */
	if (*filename == '~') {
		if (!(home = getenv("HOME")) || *home != '/') {
			fprintf (stderr,
			    "%s: Warning: $HOME not set or invalid.\n", me);
			return 0;	/* skip */
		}
		if (!(*fn = (char *) malloc(strlen(home) + strlen(filename))))
			out_of_memory();
		strcpy (*fn, home);
		strcat (*fn, filename + 1);
	}
	else {
		if (!(*fn = strdup(filename)))
			out_of_memory();
	}
	return 1;	/* success */
}

/*
 *   toconf_readconfig()
 *   reads all files in the list configfile[] (see above).
 *   At least one of those _must_ exist and contain at least
 *   one CD-ROM drive configuration entry!
 */

int
toconf_readconfig (void)
{
	int result, i;
	char *fn;

	i = 0;
	for (i = 0; (result = getfilename(i, &fn)) >= 0; i++)
		if (result) {
			readconfigfile (fn);
			free (fn);
		}
	if (!toconf_num) {
		fprintf (stderr, "%s:  No configuration entries found!  "
		    "Searched at the following places:\n", me);
		for (i = 0; (result = getfilename(i, &fn)) >= 0; i++)
			if (result) {
				fprintf (stderr, "   %s\n", fn);
				free (fn);
			}
		exit (1);
	}
	return toconf_num;
}

toconf_entry *
toconf_searchentry
    (const char *vendor, const char *product, const char *version)
{
	toconf_entry *te;
	int i, l1, l2, l3;

	for (i = 0; i < toconf_num; i++) {
		te = toconf + i;
		l1 = strlen(te->vendor);
		l2 = strlen(te->product);
		l3 = strlen(te->version);
		if (!l1 && !l2 && !l3) {
			fprintf (stderr, "%s: Warning: Unknown drive, "
			    "using default configuration!\n", me);
			return te;
		}
		if ((!l1 || !strncmp(te->vendor, vendor, l1)) &&
		    (!l2 || !strncmp(te->product, product, l2)) &&
		    (!l3 || !strncmp(te->version, version, l3)))
			return te;
	}
	fprintf (stderr,
	    "%s: Error: Unknown drive and missing default entry!\n", me);
	exit (1);
}

/* EOF */
