# vfolderdef.tcl -
#
# Create and edit vfolders
#
# Each virtual folder is described by one entry in the vFolderDef array.
# Each entry in the array is a list. The three first elements in this list
# are the same for all folder types:
#  {NAME TYPE FLAGS TYPE_SPECIFIC_ELEMENTS}
# Flags is a list of flags and their values, suitable for the "array set" cmd
# The following is a list of all the different folder types and their elems
#
# file:		{name file flags filename}
# mh:		{name mh flags path_to_mh_dir}
# dbase:	{name dbase flags keywords extype exdate}
# imap:		{name imap flags {{{host:port}folder}} user}
# pop3:		{name pop3 flags {{{host/pop3}}} user}
# dynamic	{name dynamic flags path_to_dir policy}
# disconnected:	{name dis flags {{{host:port}folder}} user}
#
#  TkRat software and its included text is Copyright 1996-1999 by
#  by Martin Forssn
#
#  The full text of the legal notice is contained in the file called
#  COPYRIGHT, included with this distribution.


# VFolderDef --
#
# Create the vfolderdef window
#
# Arguments:

proc VFolderDef {} {
    global t b vFolderChanged vf

    set vFolderChanged 0

    # Create identifier
    set id vfolderdef
    set w .$id
    if [winfo exists $w] {
	wm deiconify $w
	raise $w
	return
    }
    upvar #0 $id hd
    set hd(w) $w
    set hd(done) 0
    set hd(dragging) 0
    set hd(oldfocus) [focus]
    set vf(drag_after) {}

    # Create toplevel
    toplevel $w -class TkRat
    wm title $w $t(vfolderdef)

    # Populate window
    FindAccelerators a {window folder import}
    frame $w.mbar -relief raised -bd 1
    set m $w.mbar.w.m
    menubutton $w.mbar.w -menu $m -text $t(window) -underline $a(window)
    menu $m
    $m add command -label $t(collapse_all) -command "VFolderSetVFState closed"
    set b($m,[$m index end]) vd_collapse
    $m add command -label $t(expand_all) -command "VFolderSetVFState open"
    set b($m,[$m index end]) vd_expand
    $m add separator
    $m add command -label $t(close) -command "VFolderWinClose $id"
    set b($m,[$m index end]) dismiss

    set m $w.mbar.folder.m
    menubutton $w.mbar.folder -menu $m -text $t(folder) -underline $a(folder)
    set b($w.mbar.folder) vd_foldermenu
    menu $m
    $m add command -label $t(create_submenu)... \
	    -command "set vFolderCurrent {0 end}; VFolderCreateMenu $w"
    $m add command -label $t(create_simple)... \
	    -command "set vFolderCurrent {0 end}; VFolderCreateSimple $w"
    $m add command -label $t(create_advanced)... \
	    -command "set vFolderCurrent {0 end}; VFolderCreateAdv $w"

    set m $w.mbar.import.m
    menubutton $w.mbar.import -menu $m -text $t(import) -underline $a(import)
    set b($w.mbar.import) vd_importmenu
    menu $m
    $m add command -label $t(directory)... \
	    -command "set vFolderCurrent {0 end}; VFolderImport dir $w"
    set b($m,[$m index end]) vd_importdir
    $m add command -label "MH $t(directory)..." \
	    -command "set vFolderCurrent {0 end}; VFolderImport mh $w"
    set b($m,[$m index end]) vd_importmh
    $m add command -label $t(import_IMAP)... \
	    -command "set vFolderCurrent {0 end}; VFolderImport imap $w"
    set b($m,[$m index end]) vd_importimap
    $m add command -label $t(import_DIS)... \
	    -command "set vFolderCurrent {0 end}; VFolderImport dis $w"
    set b($m,[$m index end]) vd_importdis

    pack $w.mbar.w \
	 $w.mbar.folder \
	 $w.mbar.import -side left -padx 5
    scrollbar $w.scrolly -relief raised -command "$w.canvas yview" \
	    -highlightthickness 0 -bd 1
    scrollbar $w.scrollx -relief raised -command "$w.canvas xview" \
	    -highlightthickness 0 -orient horizontal -bd 1
    canvas $w.canvas -yscrollcommand "$w.scrolly set" \
	    -xscrollcommand "$w.scrollx set" -relief raised -bd 1 \
	    -highlightthickness 0 -selectborderwidth 0
    set b($w.canvas) vd_canvas
    bind $w.canvas <B1-Leave> "VFolderLeave $w.canvas %x %y"
    bind $w.canvas <B1-Enter> VFolderCancelRepeat
    bind $w.canvas <ButtonRelease-1> VFolderCancelRepeat
    Size $w.canvas vFolderDef

    grid $w.mbar -sticky ew -columnspan 2
    grid $w.canvas -sticky nsew
    grid $w.scrolly -column 1 -row 1 -sticky ns
    grid $w.scrollx -sticky ew
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 1 -weight 1

    menu $w.m -tearoff 0
    $w.m add command -label $t(edit)... -command "EditVFolderEntry $w"
    set b($w.m,[$w.m index end]) vd_edit
    $w.m add command -label $t(delete)... -command "VFolderDelete $w"
    set b($w.m,[$w.m index end]) vd_delete
    $w.m add command -label $t(create_submenu)... \
	    -command "VFolderCreateMenu $w"
    $w.m add command -label $t(create_simple)... \
	    -command "VFolderCreateSimple $w"
    $w.m add command -label $t(create_advanced)... \
	    -command "VFolderCreateAdv $w"

    # Create images (images lent from tk sources)
    if {![info exists vf(folder)]} {
        set vf(folder) [image create photo -data {
R0lGODlhEAAMAKEAAAD//wAAAPD/gAAAACH5BAEAAAAALAAAAAAQAAwAAAIghINhyycvVFsB
QtmS3rjaH1Hg141WaT5ouprt2HHcUgAAOw==}]
        set vf(file)   [image create photo -data {
R0lGODlhDAAMAKEAALLA3AAAAP//8wAAACH5BAEAAAAALAAAAAAMAAwAAAIgRI4Ha+IfWHsO
rSASvJTGhnhcV3EJlo3kh53ltF5nAhQAOw==}]
    }

    # Find height of text
    set i [$w.canvas create text 0 0 -anchor n -text F]
    set vf(dy) [expr [lindex [$w.canvas bbox $i] 3]+2]
    $w.canvas delete $i

    bind $w.mbar <Destroy> VFolderWindowDestroy
    wm protocol $w WM_DELETE_WINDOW "VFolderWinClose $id"

    RedrawVFolderList
    Place $w vFolderDef
}

# VFolderWinClose --
#
# Close a vfolderdef window
#
# Arguments:
# handler - The handler identifying the window

proc VFolderWinClose {handler} {
    upvar #0 $handler hd
    global b

    RecordPos $hd(w) vFolderDef
    RecordSize $hd(w).canvas vFolderDef
    catch {focus $hd(oldfocus)}
    destroy $hd(w)
    foreach bn [array names b $hd(w).*] {unset b($bn)}
    unset hd
}

# VFolderSetVFState --
#
# Set the state of all vfolder structs
#
# Arguments:
# state	- New state

proc VFolderSetVFState {state} {
    global vf

    foreach m [array names vf state,*] {
	set vf($m) $state
    }
    RedrawVFolderList
}


# RedrawVFolderList --
#
# Redraws the list of vfolders. This is done by first clearing the canvas
# and then call DrawVFolderStruct which does the actual drawing.
#
# Arguments:

proc RedrawVFolderList {} {
    global vf

    set c .vfolderdef.canvas
    $c delete all
    set vf(y) 0
    set vf(x) 0
    set vf(dx0) 16
    set vf(dx1) 8

    set vf(ids) {}
    set vf(types) {}
    DrawVFolderStruct 0 10 {}
    $c raise sel

    $c configure -scrollregion [list 0 0 [expr $vf(x)+10] [expr $vf(y)+10]]
}

# DrawVFolderStruct --
#
# Draws a given struct, and its children by calling itself recursively.
#
# Arguments:
# sIdent  -	The ident of the struct to be drawn
# x       -	X position of left line
# parents -	List of parents to this struct

proc DrawVFolderStruct {sIdent x parents} {
    global option vf vFolderStruct vFolderInbox vFolderSave t vFolderDef

    set c .vfolderdef.canvas
    set y0 $vf(y)
    set id 0
    set element ""
    foreach e $vFolderStruct($sIdent) {
	set x1 [expr $x+$vf(dx0)]
	set x2 [expr $x1+$vf(dx1)]
	set y [incr vf(y) $vf(dy)]
	lappend vf(ids) [list $sIdent $id $parents]
	lappend vf(types) [lindex $e 0]
	set element [lindex $e 0]
	switch $element {
	    struct {
		set i0 [$c create rectangle  [expr $x-4] [expr $y-4] \
					    [expr $x+4] [expr $y+4] \
					    -fill white -tags sel]
		set sm [lindex $e 1]
		if ![info exists vf(state,$sm)] {
		    set vf(state,$sm) closed
		}
		switch $vf(state,$sm) {
		    open {
			set i1 [$c create line [expr $x-2] $y \
					       [expr $x+2] $y \
					       -tags sel]
			DrawVFolderStruct [lindex $e 1] $x1 \
				[concat $parents [lindex $e 1]]
		    }
		    closed {
			set i1 [$c create line [expr $x-2] $y \
					       [expr $x+2] $y \
					       $x $y \
					       $x [expr $y-2] \
					       $x [expr $y+3] \
					       -tags sel]
		    }
		}
		set text [lindex $e 2]
		set image $vf(folder)
		$c bind $i0 <1> "VFolderToggle $sm"
		$c bind $i1 <1> "VFolderToggle $sm"
	    }
	    vfolder {
		catch {unset f}
		array set f [lindex $vFolderDef([lindex $e 1]) 2]
		set ta {}
		if ![string compare $vFolderInbox [lindex $e 1]] {
		    lappend ta INBOX
		}
		if ![string compare $vFolderSave [lindex $e 1]] {
		    lappend ta $t(save_outgoing)
		}
		if {[info exists f(monitor)] && $f(monitor)} {
		    lappend ta $t(monitored)
		}
		if {[info exists f(watch)] && $f(watch)} {
		    lappend ta $t(watched)
		}
		set text "[lindex $e 2]        [join $ta ,]"
		set image $vf(file)
	    }
	    import {
		set text [lindex $e 2]
		set image $vf(folder)
	    }
	}
	$c create line $x $y $x1 $y
	$c create image $x1 $y -image $image -tags ${sIdent}_$id
	$c create text $x2 $y -anchor w -text $text -tags ${sIdent}_$id
	$c bind ${sIdent}_$id <3> "tk_popup .vfolderdef.m %x %Y"
	$c bind ${sIdent}_$id <3> "VFolderPostMenu %X %Y $sIdent $id"
	$c bind ${sIdent}_$id <1> "VFolderStartDrag $c %x %y $sIdent $id"
	$c bind ${sIdent}_$id <B1-Motion> "VFolderDrag $c %x %y"
	$c bind ${sIdent}_$id <ButtonRelease-1> "VFolderStopDrag $c"

	set extr [lindex [$c bbox ${sIdent}_$id] 2]
	if {$extr > $vf(x)} {
	    set vf(x) $extr
	}
	incr id
    }
    if { $element == "struct" } {
	$c create line $x $y0 $x $y
    }  else {
	$c create line $x $y0 $x $vf(y) 
    }
}

# VFolderToggle --
#
# Toggles a directory button
#
# Arguments:
# ident -	Identifiwer of menu to modify

proc VFolderToggle {ident} {
    global vf

    switch $vf(state,$ident) {
	open {set vf(state,$ident) closed}
	closed {set vf(state,$ident) open}
    }
    RedrawVFolderList
}

# VFolderPostMenu:
# Post a the menu for a given entry. The arguments are: the root window
# coordinates, the listbox, the ident of the vFolderStruct and the index
# of the active member.
#
# Post the menu for the given entry.
#
# Arguments:
# x, y   -	Position of pointer, this is where we will popup the menu
# sIdent -	Identifier of the vFolderStruct the entry belongs to
# id     -	Index in the above mentioned vFolderStruct

proc VFolderPostMenu {x y sIdent id} {
    global vFolderCurrent vFolderStruct vFolderInbox vFolderSave t

    set vFolderCurrent [list $sIdent $id]
    tk_popup .vfolderdef.m $x $y
}

# VFolderStartDrag --
#
# Invoked when the users starts to drag an item (or at least wants to).
#
# Arguments:
# c      -	Widget command name
# x, y   -	Place from which the user wants to pickup an object,
#		relative to the listbox nw corner.
# sIdent -	Ident of the vfolderstruct which contains the object
# id	 -	The itemId of the listbox in the canvas

proc VFolderStartDrag {c x y sIdent id} {
    global vf

    set vf(drag_origx) [expr round([$c canvasx $x])]
    set vf(drag_origy) [expr round([$c canvasy $y])]
    set vf(drag_x) $vf(drag_origx)
    set vf(drag_y) $vf(drag_origy)
    set vf(drag_tag) ${sIdent}_$id
    set vf(drag_sIdent) $sIdent
    set vf(drag_id) $id
    set vf(drag_mark) [list $sIdent $id]
    set vf(drag_left) 0
    set vf(drag_after) {}
    set bb [$c bbox $vf(drag_tag)]
    set vf(drag_bb) $bb
    $c create line [lindex $bb 0] [lindex $bb 3] [lindex $bb 2] [lindex $bb 3] \
	    -tags dragline
}

# VFolderDrag --
#
# Does the acutal dragging.
#
# Arguments:
# c    -	Canvas widget command name
# x, y -	Place to drag the item to

proc VFolderDrag {c wx wy} {
    global vf

    # Dragging
    set x [$c canvasx $wx]
    set y [$c canvasy $wy]
    $c move $vf(drag_tag) [expr $x-$vf(drag_x)] [expr $y-$vf(drag_y)]
    set vf(drag_x) $x
    set vf(drag_y) $y

    # Finding object under cursor
    set i [expr round($y-$vf(dy)/2)/$vf(dy)]
    set m [lindex $vf(ids) $i]
    if {"" == $m} {
	return
    }
    set t [lindex $m 0]_[lindex $m 1]
    if {"$t" == "$vf(drag_tag)"} {
	set bb $vf(drag_bb)
    } else {
	set bb [$c bbox $t]
    }
    set left [expr $x < [lindex $bb 0]]

    # Mark object if needed
    if { "$m" != "$vf(drag_mark)" || $left != $vf(drag_left)} {
	set x1 [expr [lindex $bb 0] - 1]
	set y1 [lindex $bb 1]
	set x2 [lindex $bb 2]
	set y2 [lindex $bb 3]
	if {"struct" == [lindex $vf(types) $i] && !$left} {
	    $c coords dragline $x1 $y1 $x2 $y1 $x2 $y2 $x1 $y2 $x1 $y1
	} else {
	    $c coords dragline $x1 $y2 $x2 $y2
	}
	set vf(drag_mark) $m
	set vf(drag_left) $left
    }
}

# VFolderLeave --
# Implement scanning when the mouse leaves the window
#
# Arguments:
# c    -	Canvas widget command name
# x, y -	Place to drag the item to

proc VFolderLeave {c x y} {
    global vf

    if {![winfo exists $c]} return
    if {$y >= [winfo height $c]} {
	$c yview scroll 1 units
    } elseif {$y < 0} {
	$c yview scroll -1 units
    } elseif {$x >= [winfo width $c]} {
	$c xview scroll 1 units
    } elseif {$x < 0} {
	$c xview scroll -1 units
    } else {
	return
    }
    set vf(drag_after) [after 50 VFolderLeave $c $x $y]
}

# VFolderCancelRepeat --
#
# Cancel the auto scrolling

proc VFolderCancelRepeat {} {
    global vf

    after cancel $vf(drag_after)
    set vf(drag_after) {}
}

# VFolderStopDrag --
#
# The user is tired of dragging this heavy object around and wants to drop
# it.
#
# Arguments:
# c    -	Canvas widget command name

proc VFolderStopDrag {c} {
    global vf vFolderStruct vFolderChanged

    $c delete dragline

    # Move data in structures
    set i [expr round($vf(drag_y)-$vf(dy)/2)/$vf(dy)]
    set m [lindex $vf(ids) $i]
    if {"" == $m} {
	set m [lindex $vf(ids) end]
    }
    set sIdent [lindex $m 0]
    set id [lindex $m 1]
    if {"$vf(drag_sIdent)" == $sIdent && "$vf(drag_id)" == $id} {
	# Sortcut if we return to the same position
	RedrawVFolderList
	return
    } 
    if {"struct" == [lindex $vf(types) $i] && 0 == $vf(drag_left)} {
	set sIdent [lindex [lindex $vFolderStruct($sIdent) $id] 1]
	set id 0
    } else {
	incr id
    }
    set item [lindex $vFolderStruct($vf(drag_sIdent)) $vf(drag_id)]
    if {"struct" == [lindex $item 0]
	    && -1 != [lsearch [lindex $m 2] [lindex $item 1]]} {
	# Do not allow a struct to be placed below itself
	RedrawVFolderList
	return
    }
    set vFolderStruct($vf(drag_sIdent)) \
	[lreplace $vFolderStruct($vf(drag_sIdent)) $vf(drag_id) $vf(drag_id)]
    if {"$sIdent" == $vf(drag_sIdent) && $vf(drag_id) < $id} {
	incr id -1
    }
    set vFolderStruct($sIdent) [linsert $vFolderStruct($sIdent) $id $item]
    incr vFolderChanged
    RedrawVFolderList
}

# VFolderCreateMenu --
# Asks the user for a name and then creates a new empty submenu with
# that name. The new submenu is inserted into the structure at the place
# indicated by the vFolderCurrent global variable.
#
# Arguments:

proc VFolderCreateMenu {} {
    global vFolderCurrent t b idCnt

    # Create identifier
    set id cf[incr idCnt]
    set w .$id
    upvar #0 $id hd
    set hd(done) 0

    # Create toplevel
    toplevel $w -class TkRat
    wm title $w $t(create_submenu)

    # Folder name
    frame $w.name
    label $w.name.label -text $t(name):
    set b($w.name) vd_menuname
    entry $w.name.entry -textvariable ${id}(name) -width 20
    pack $w.name.entry \
	 $w.name.label -side right

    OkButtons $w $t(ok) $t(cancel) "set ${id}(done)"
    pack $w.name \
	 $w.buttons -side top -fill both -padx 5 -pady 5

    Place $w vFolderCreateMenu
    focus $w.name.entry
    Moda| $w
    tkwait variable ${id}(done)
    RecordPos $w vFolderCreateMenu
    unset b($w.name)
    destroy $w

    if {1 == $hd(done)} {
	global vFolderStruct vFolderStructIdent vFolderChanged
	incr vFolderChanged
	incr vFolderStructIdent
	set sIdent [lindex $vFolderCurrent 0]
	set index [lindex $vFolderCurrent 1]
	set vFolderStruct($sIdent) [linsert $vFolderStruct($sIdent) $index \
		[list struct $vFolderStructIdent $hd(name)]]
	set vFolderStruct($vFolderStructIdent) {}
	RedrawVFolderList
    }
    unset hd
}

# VFolderDelete --
#
# This procedure asks the user if it is really sure it wants to delete
# the specified item. The item is specified via the vFolderCurrent
# global variable.
#
# Arguments:
# parent -	Parent of window

proc VFolderDelete {parent} {
    global t vFolderCurrent vFolderDef vFolderStruct vFolderChanged \
	   vFolderInbox vFolderSave

    set sIdent [lindex $vFolderCurrent 0]
    set index [lindex $vFolderCurrent 1]

    set elem [lindex $vFolderStruct($sIdent) $index]
    if ![string compare [lindex $elem 0] struct] {
	if [llength $vFolderStruct([lindex $elem 1])] {
	    if [RatDialog $parent ! $t(submenu_not_empty) {} 0 $t(delete) \
		    $t(cancel)] {
		return
	    }
	    set imaps [VFolderDeleteStruct -1 [lindex $elem 1] $parent]
	}
	unset vFolderStruct([lindex $elem 1])
    } else {
	set vd $vFolderDef([lindex $elem 1])
	if {![string compare imap [lindex $vd 1]]
		|| ![string compare dis [lindex $vd 1]]} {
	    set imaps [list [list [lindex $vd 3] [lindex $vd 4]]]
	}
	unset vFolderDef([lindex $elem 1])
    }
    set vFolderStruct($sIdent) [lreplace $vFolderStruct($sIdent) $index $index]
    if ![info exists vFolderDef($vFolderInbox)] {
	set vFolderInbox [lindex [lsort -integer [array names vFolderDef]] 0]
    }
    if ![info exists vFolderDef($vFolderInbox)] {
	set vFolderSave {}
    }
    if [llength $imaps] {
	set ret [RatDialog $parent ! $t(delete_imap) {} 0 $t(delete) \
		$t(dont_delete)]
	if {0 == $ret} {
	    foreach i $imaps {
		if [catch {RatImapDeleteFolder [lindex $i 0] [lindex $i 1]}] {
		    Popup $t(delete_failed) $parent
		}
	    }
	}
    }
    incr vFolderChanged
    RedrawVFolderList
}

# VFolderDeleteStruct --
#
# Remove a struct and all contained folders (as well ass all children
#
# Arguments:
# op	 - Operation to perform on IMAP-folders (-1=dont_know 0=delete 1=keep)
# sIdent - id of struct
# parent -	Parent of window

proc VFolderDeleteStruct {op sIdent parent} {
    global vFolderStruct vFolderDef t
    set imaps {}

    foreach elem $vFolderStruct($sIdent) {
	set im {}
	if ![string compare [lindex $elem 0] struct] {
	    set im [VFolderDeleteStruct $op [lindex $elem 1] $parent]
	    unset vFolderStruct([lindex $elem 1])
	} else {
	    set vd $vFolderDef([lindex $elem 1])
	    if {(![string compare imap [lindex $vd 1]]
		    || ![string compare dis [lindex $vd 1]]) && 1 != $op} {
		set im [list [list [lindex $vd 3] [lindex $vd 4]]]
	    }
	    unset vFolderDef([lindex $elem 1])
	}
	if [llength $im] {
	    set imaps [concat $imaps $im]
	}
    }
    return $imaps
}

# VFolderCreateSimple --
#
# Create a simple folder.
#
# Arguments:
# parent - parent window

proc VFolderCreateSimple {parent} {
    global vFolderCurrent vFolderDefIdent vFolderStruct vFolderChanged \
	    vFolderDef t b idCnt option

    # Create identifier
    set id cf[incr idCnt]
    set w .$id
    upvar #0 $id hd
    set hd(done) 0

    # Create toplevel
    toplevel $w -class TkRat
    wm title $w $t(create_simple)

    # Folder name
    frame $w.name
    label $w.name.label -text $t(name):
    entry $w.name.entry -textvariable ${id}(name) -width 20
    set b($w.name.entry) vd_name
    pack $w.name.entry \
	 $w.name.label -side right

    OkButtons $w $t(ok) $t(cancel) "set ${id}(done)"
    pack $w.name \
	 $w.buttons -side top -fill both -padx 5 -pady 5

    Place $w vFolderCreateMenu
    Modal $w $parent
    focus $w.name.entry
    tkwait variable ${id}(done)
    RecordPos $w vFolderCreateMenu
    unset b($w.name.entry)
    destroy $w

    if {1 == $hd(done)} {
	set f [list sort default browse 0]
	switch $option(simple_prot) {
	    file {
		    set def [list $hd(name) file $f \
				  $option(simle_data)/$hd(name)]
		}
	    mh {
		    set def [list $hd(name) mh $f $hd(name)]
		}
	    imap {
		    set def [list $hd(name) imap $f \
				  \{$option(simle_data)\}$hd(name) $env(USER)]
		}
	    dis {
		    set def [list $hd(name) dis $f \
				  \{$option(simle_data)\}$hd(name) $env(USER)]
		}
	}
	set vFolderDef($vFolderDefIdent) $def
	set sIdent [lindex $vFolderCurrent 0]
	set index [lindex $vFolderCurrent 1]
	set vFolderStruct($sIdent) [linsert $vFolderStruct($sIdent) $index \
		[list vfolder $vFolderDefIdent $hd(name)]]
	incr vFolderDefIdent
	incr vFolderChanged
	RedrawVFolderList
    }
    unset hd
}

# VFolderCreateAdv --
#
# Create a new advanced vfolder
#
# Arguments:
# parent - parent window

proc VFolderCreateAdv {parent} {
    global vFolderCurrent vFolderDefIdent vFolderStruct vFolderChanged \
	    vFolderDef

    set vFolderDef($vFolderDefIdent) {{} file {} {}}
    set vFolderName [EditVFolder $vFolderDefIdent $parent]
    if ![string length $vFolderName] {
	unset vFolderDef($vFolderDefIdent)
	return
    }
    set sIdent [lindex $vFolderCurrent 0]
    set index [lindex $vFolderCurrent 1]
    set vFolderStruct($sIdent) [linsert $vFolderStruct($sIdent) $index \
	    [list vfolder $vFolderDefIdent $vFolderName]]
    incr vFolderDefIdent
    RedrawVFolderList
}

# EditVFolderEntry --
# 
# Edit an entry in the vfolder menu
# 
# Arguments:
# parent - parent window
  
proc EditVFolderEntry {parent} {
    global t b idCnt vFolderCurrent vFolderStruct vFolderChanged vFolderDef
  
    set sIdent [lindex $vFolderCurrent 0]
    set index [lindex $vFolderCurrent 1]
    set elem [lindex $vFolderStruct($sIdent) $index]
    switch [lindex $elem 0] {
    struct  {
            set ident [lindex $elem 1]
            set id cf[incr idCnt]
            set w .$id
            upvar #0 $id hd
            set hd(done) 0

            # Create toplevel
            toplevel $w -class TkRat
            wm title $w $t(name)

            # Folder name
            set hd(name) [lindex $elem 2]
            frame $w.name
            label $w.name.label -text $t(name): -width 15 -anchor e
            entry $w.name.entry -textvariable ${id}(name) -width 20
            pack $w.name.label $w.name.entry -side left
            pack $w.name -side top -expand 1 -padx 5 -pady 5
            set b($w.name.entry) vd_menuname
            OkButtons $w $t(ok) $t(cancel) "set ${id}(done)"
            pack $w.buttons -fill both -padx 5 -pady 5

            Place $w editVfolderEntry
	    Modal $w $parent
            focus $w.name.entry
            while 1 {
                tkwait variable ${id}(done)
                if { 1 == $hd(done) } {
                    if ![string length $hd(name)] {
                        Popup $t(need_name) $w
                        continue
                    }
                    incr vFolderChanged
                    set vFolderStruct($sIdent) \
                            [lreplace $vFolderStruct($sIdent) $index $index \
                             [list struct $ident $hd(name)]]
                }
                break
            }
            RecordPos $w editVfolderEntry
            destroy $w
            unset b($w.name.entry)
            unset hd
        }
    vfolder {
            EditVFolder [lindex $elem 1] $parent
            set vFolderStruct($sIdent) \
                    [lreplace $vFolderStruct($sIdent) $index $index \
                          [list vfolder \
                                [lindex $elem 1] \
                                [lindex $vFolderDef([lindex $elem 1]) 0]]]
        }   
    import {
            set nd [VFolderImportEdit $vFolderDef([lindex $elem 1]) $parent]
	    array set f [lindex $nd 2]
	    if {"" != $nd} {
		if {[info exists f(trace)] && $f(trace)} {
		    set top [list import [lindex $elem 1] [lindex $nd 0]]
		    set vFolderDef([lindex $elem 1]) $nd
		} else {
		    set top [list struct [lindex $elem 1] [lindex $nd 0]]
		    unset vFolderDef([lindex $elem 1])
		    VFolderImportNow [lindex $elem 1] [RatImport $nd]
		}
		set vFolderStruct($sIdent) \
			[lreplace $vFolderStruct($sIdent) $index $index $top]
	    }
        }   
    }
    RedrawVFolderList
}


# EditVFolder --
#
# Edit an advanced vfolder
#
# Arguments:
# ident	- Identifier of folder
# parent - parent window

proc EditVFolder {ident parent} {
    global t b idCnt vFolderChanged vFolderDef option vFolderInbox vFolderSave \
	   env vFolderMonitorFH vFolderMonitorID folderExists \
	   vFolderWatch
    upvar #0 vFolderDef($ident) def

    # Create identifier
    set id cf[incr idCnt]
    set w .$id
    upvar #0 $id hd
    set hd(done) 0
    set hd(w) $w

    # Populate variables
    set hd(extype) $option(def_extype)
    set hd(exdate) $option(def_exdate)
    set hd(user) $option(remote_user)
    set hd(host) $option(remote_host)
    set hd(imapport) $option(imap_port)
    set hd(pop3port) $option(pop3_port)
    set hd(name) [lindex $def 0]
    set hd(sort) default
    set hd(monitor) 0
    set hd(watch) 0
    set hd(old) {}
    array set hd [lindex $def 2]
    set hd(had_monitor) $hd(monitor)
    set hd(sort_l) $t(sort_$hd(sort))
    switch -regexp [lindex $def 1] {
	file|mh|dynamic {
		set hd(entry) [lindex $def 3]
	    }
	dbase {
		set hd(keywords) [lindex $def 3]
		set hd(extype) [lindex $def 4]
		set hd(exdate) [lindex $def 5]
	    }
	dis|imap {
		set h [lindex $def 3]
		set hd(old) $h
		regsub {(/|\}).*} [string trim $h \{] {} host
		set hlist [split $host :]
		set hd(host) [lindex $hlist 0]
		if {2 == [llength $hlist]} {
		    set hd(imapport) [lindex $hlist 1]
		}
		regsub {.+\}} $h {} hd(mbox)  
		set hd(user) [lindex $def 4]
		set hd(oldmbox) $hd(mbox)
	    }
	pop3 {
		set h [lindex $def 3]
		regsub {(/|\}).*} [string trim $h \{] {} host
		set hlist [split $host :]
		set hd(host) [lindex $hlist 0]
		if {2 == [llength $hlist]} {
		    set hd(pop3port) [lindex $hlist 1]
		}
		set hd(user) [lindex $def 4]
	    }
    }
    if {$vFolderInbox == $ident} {
	set hd(inbox) 1
    } else {
	set hd(inbox) 0
    }
    if {$vFolderSave == $ident} {
	set hd(save_outgoing) 1
    } else {
	set hd(save_outgoing) 0
    }

    # Create window
    toplevel $w -class TkRat
    wm title $w "$t(vfolderedit)  ([lindex $def 1])"
    rat_tabbed::create $w.t
    set p [rat_tabbed::page $w.t $t(options)]
    label $p.name_label -text $t(name): -anchor e
    entry $p.name_entry -textvariable ${id}(name)
    set b($p.name_entry) vd_name
    label $p.sort_label -text $t(sort_order): -anchor e
    set m $p.sort_m.m
    menubutton $p.sort_m -textvariable ${id}(sort_l) \
	    -relief raised -indicatoron 1 -menu $m
    set b($p.sort_m) vd_sort
    menu $m -tearoff 0
    foreach o {default threaded folder reverseFolder date reverseDate subject \
	       subjectonly} {
	$m add command -label $t(sort_$o) \
		-command "set ${id}(sort) $o; \
			  set ${id}(sort_l) [list $t(sort_$o)]"
    }
    checkbutton $p.browse -text $t(browse_mode) -variable ${id}(browse)
    set b($p.browse) browse_mode
    checkbutton $p.inbox -text $t(incom_mbox) -variable ${id}(inbox)
    set b($p.inbox) vd_setinbox
    checkbutton $p.monitor -text $t(monitor_mbox) -variable ${id}(monitor) \
	    -command "if !\$${id}(monitor) {set ${id}(watch) 0}"
    set b($p.monitor) vd_monitor
    checkbutton $p.watch -text "    $t(watch_mbox)" -variable ${id}(watch) \
	    -command "if \$${id}(watch) {set ${id}(monitor) 1}"
    set b($p.watch) vd_watch
    checkbutton $p.save -text $t(save_outgoing) -variable ${id}(save_outgoing)
    set b($p.save) vd_setsaveout
    grid $p.name_label $p.name_entry -sticky ew
    grid $p.sort_label $p.sort_m -sticky ew
    grid x $p.browse -sticky w
    grid x $p.monitor -sticky w
    grid x $p.watch -sticky w
    grid x $p.inbox -sticky w
    grid x $p.save -sticky w
    grid columnconfigure $p 1 -weight 1
    set p [rat_tabbed::page $w.t $t(definitions)]
    set p [rat_tabbed::create $p.f]
    pack $p -side top
    foreach m {file mh dbase imap dis pop3 dynamic} {
	set mp [rat_tabbed::page $p $t(folder_$m) "set ${id}(type) $m"]
	switch -regexp $m {
	    file {
		    label $mp.label -text $t(pathname): -anchor w
		    entry $mp.entry -textvariable ${id}(entry) -width 40
		    set b($mp.entry) vd_pathname
		    button $mp.browse -text $t(browse)... -anchor e \
			    -command "Browse $hd(w) ${id}(entry) any"
		    set b($mp.browse) file_browse
		    pack $mp.label -side top -anchor w -padx 5
		    pack $mp.entry -side top -fill x -expand 1 
		    pack $mp.browse -anchor e -pady 3 -padx 5
		}
	    mh|dynamic {
		    label $mp.label -text $t(pathname): -anchor w
		    entry $mp.entry -textvariable ${id}(entry) -width 40
		    set b($mp.entry) vd_pathname
		    button $mp.browse -text $t(browse)... -anchor e \
			    -command "Browse $hd(w) ${id}(entry) dirok"
		    set b($mp.browse) file_browse
		    pack $mp.label -side top -anchor w -padx 5
		    pack $mp.entry -side top -fill x -expand 1 
		    pack $mp.browse -anchor e -pady 3 -padx 5
		}
	    dbase {
		    label $mp.kw_lab -text $t(keywords): -anchor e
		    entry $mp.kw_entry -textvariable ${id}(keywords) -width 20
		    set b($mp.kw_entry) keywords
		    label $mp.extype_lab -text $t(extype): -anchor e
		    frame $mp.extype
		    foreach et {none remove incoming backup} {
			radiobutton $mp.extype.$et -anchor w -text $t($et) \
				-variable ${id}(extype) -value $et
			set b($mp.extype.$et) exp_$et
			grid $mp.extype.$et -sticky w
		    }
		    label $mp.exdate_lab -text $t(exdate): -anchor e
		    entry $mp.exdate_entry -textvariable ${id}(exdate)
		    set b($mp.exdate_entry) exp_date
		    grid $mp.kw_lab $mp.kw_entry -sticky we
		    grid $mp.extype_lab $mp.extype -sticky wen
		    grid $mp.exdate_lab $mp.exdate_entry -sticky we
		}
	    imap|dis {
		    foreach e {host user mbox imapport} {
			label $mp.${e}_lab -text $t($e): -anchor e -width 15
			entry $mp.${e}_entry -textvariable ${id}($e)
			grid $mp.${e}_lab $mp.${e}_entry -sticky ew
			set b($w.${e}_entry) vd_$e
		    }
		    grid columnconfigure $mp 1 -weight 1
		}
	    pop3 {
		    foreach e {host user pop3port} {
			label $mp.${e}_lab -text $t($e): -anchor e -width 15
			entry $mp.${e}_entry -textvariable ${id}($e)
			grid $mp.${e}_lab $mp.${e}_entry -sticky ew
			set b($w.${e}_entry) vd_$e
		    }
		    grid columnconfigure $mp 1 -weight 1
		}
	}
    }
    OkButtons $w $t(ok) $t(cancel) "set ${id}(done)"
    pack $w.t -side top -fill both
    pack $w.buttons -fill both -padx 5 -pady 5

    Place $w editVfolder
    rat_tabbed::display $w.t $t(options)
    rat_tabbed::display $p $t(folder_[lindex $def 1])
    Modal $w $parent
    while 1 {
	tkwait variable ${id}(done)
	if { 0 == $hd(done) } {
	    break
	}
	if ![string length $hd(name)] {
	    Popup $t(need_name) $w
	    continue
	}
	incr vFolderChanged
	set flags [list sort $hd(sort) browse $hd(browse) monitor $hd(monitor) \
		        watch $hd(watch)]
	switch -regexp $hd(type) {
	    file {
		    if {"" == $hd(entry)} {
			Popup "$t(illegal_file_spec): $hd(entry)" $w
			continue
		    }
		    regsub {/$} $env(HOME) {} home
		    regsub ^~/ $hd(entry) $home/ path
		    set def [list $hd(name) file $flags $path]
		}
	    mh {
		    if {"" == $hd(entry)} {
			Popup $t(need_mh_name) $w
			continue
		    }
		    set def [list $hd(name) mh $flags $hd(entry)]
		}
	    dbase {
		    if {"" == $hd(keywords)} {
			Popup $t(need_keyword) $w
			continue
		    }
		    set def [list $hd(name) dbase $flags $hd(keywords) \
				  $hd(extype) $hd(exdate)]
		}
	    imap|dis {
		    if {"" == $hd(host) || "" == $hd(user)} {
			Popup $t(need_host_and_user) $w
			continue
		    }
		    if {"" != $hd(imapport)} {
			set hp \{$hd(host):$hd(imapport)\}$hd(mbox)
		    } else {
			set hp \{$hd(host)\}$hd(mbox)
		    }
		    set def [list $hd(name) $hd(type) $flags $hp $hd(user)]
		    if {![info exists h] || "$hp" != $h} {
			catch {RatImapCreateFolder $hp $hd(user)}
		    }
		    if {"dis" == $hd(type)
			    && (![info exists hd(oldmbox)]
				|| "$hd(oldmbox)" !=  "$hd(mbox)")} {
			set f [RatOpenFolder dis {} $hd(name) $hp $hd(user) imap]
			$f close
		    }
		}
	    pop3 {
		    if {"" == $hd(host) || "" == $hd(user)} {
			Popup $t(need_host_and_user) $w
			continue
		    }
		    if {"" != $hd(pop3port) && 110 != $hd(pop3port)} {
			set hp \{$hd(host):$hd(pop3port)/pop3\}
		    } else {
			set hp \{$hd(host)/pop3\}
		    }
		    set def [list $hd(name) pop3 $flags $hp $hd(user)]
		}
	    dynamic {
		    if {![file isdirectory $hd(entry)]} {
			Popup "$t(illegal_file_spec): $hd(entry)" $w
			continue
		    }
		    set def [list $hd(name) dynamic $flags $hd(entry) \
				  sender]
		}
	}
	if {$hd(monitor) != $hd(had_monitor)} {
	    if $hd(monitor) {
		if ![catch [VFolderOpenCmd $def] nhd] {
		    set vFolderMonitorFH($ident) $nhd
		    set vFolderMonitorID($nhd) $ident
		    set folderExists($nhd) $folderExists($nhd)
		}
	    } else {
		catch {$vFolderMonitorFH($ident) close}
		catch {unset vFolderMonitorID($vFolderMonitorFH)}
		catch {unset vFolderMonitorFH($ident)}
	    }
	}
	if {$hd(inbox)} {
	    set vFolderInbox $ident
	} elseif {$vFolderInbox == $ident} {
	    set vFolderInbox {}
	}
	if {$hd(save_outgoing)} {
	    set vFolderSave $ident
	} elseif {$vFolderInbox == $ident} {
	    set vFolderSave {}
	}
	break
    }
    RecordPos $w editVfolder
    destroy $w
    foreach bn [array names b $w*] {unset b($bn)}
    set name $hd(name)
    unset hd
    return $name
}


# VFolderWrite --
#
# Writes the list of vfolders to disk
#
# Arguments:

proc VFolderWrite {} {
    global option vFolderDef vFolderStruct vFolderDefIdent vFolderStructIdent \
	   vFolderVersion vFolderInbox vFolderSave

    # Do nothing on errors
    if [catch {open $option(ratatosk_dir)/vfolderlist w} fh] {
	return
    }
    puts $fh "set vFolderVersion $vFolderVersion"
    puts $fh "set vFolderStructIdent $vFolderStructIdent"
    puts $fh "set vFolderDefIdent $vFolderDefIdent"
    puts $fh "set vFolderInbox $vFolderInbox"
    puts $fh "set vFolderSave [list $vFolderSave]"
    foreach str [array names vFolderStruct] {
	set vs {}
	foreach e $vFolderStruct($str) {
	    switch [lindex $e 0] {
	    struct  {
		    set keep [info exists vFolderStruct([lindex $e 1])]
		}
	    vfolder {
		    set keep [info exists vFolderDef([lindex $e 1])]
		}
	    import {
		    set keep 1
		}
	    }
	    if $keep {
		lappend vs $e
	    } else {
		puts "Warning, removing {$e}"
	    }
	}
	puts $fh "set vFolderStruct($str) [list $vs]"
    }
    foreach elem [array names vFolderDef] {
	puts $fh "set vFolderDef($elem) [list $vFolderDef($elem)]"
    }
    close $fh
}

# VFolderWindowDestroy --
#
# This procedure is called when the vfolder window is destroyed. It
# writes any changes to the vfolder structre to disk, if needed.
#
# Arguments:

proc VFolderWindowDestroy {} {
    global vFolderChanged
    if $vFolderChanged {
	VFolderWrite
    }
}

# VFolderImport --
#
# Create the template for an import window and
#
# Arguments:
# type - Type of item to import
# parent - parent window

proc VFolderImport {type parent} {
    global vFolderCurrent vFolderDefIdent vFolderStruct vFolderChanged \
	    vFolderDef option

    switch $type {
	dir {set def [list Import dir {} {sort default} * {}]}
	mh {set def [list Import mh {} {sort default} * {}]}
	imap {set def [list Import imap {} {sort default} * {} {} {} \
			    $option(imap_port)]}
	dis {set def [list Import dis {} {sort default} * {} {} {} \
			    $option(imap_port)]}
    }
    set def [VFolderImportEdit $def $parent]
    if {"" == $def} {
	return
    }
    set id $vFolderDefIdent
    incr vFolderDefIdent
    incr vFolderChanged
    array set f [lindex $def 2]
    if {[info exists f(trace)] && $f(trace)} {
	set top [list import $id [lindex $def 0]]
	set vFolderDef($id) $def
    } else {
	set top [list struct $id [lindex $def 0]]
	VFolderImportNow $id [RatImport $def]
    }
    set sIdent [lindex $vFolderCurrent 0]
    set index [lindex $vFolderCurrent 1]
    set vFolderStruct($sIdent) [linsert $vFolderStruct($sIdent) $index $top]
    RedrawVFolderList
}

# VFolderImportNow --
#
# Do the actual import now
#
# Arguments:
# id	- Id of struct to build
# if	- Imported folders

proc VFolderImportNow {id if} {
    global vFolderDefIdent vFolderStruct vFolderDef option

    set vFolderStruct($id) {}
    foreach v $if {
	set d [lindex $v 1]
	if {"v" == [lindex $v 0]} {
	    set vFolderDef($vFolderDefIdent) $d
	    lappend vFolderStruct($id) \
		    [list vfolder $vFolderDefIdent [lindex $d 0]]
	    incr vFolderDefIdent
	} else {
	    set i $vFolderDefIdent
	    incr vFolderDefIdent
	    set old $vFolderStruct($id)
	    lappend vFolderStruct($id) [list struct $i $d]
	    VFolderImportNow $i [lindex $v 2]
	}
    }
}

# VFolderImportEdit --
#
# Edit an imported folder
#
# Arguments:
# def	- Folder to edit
# parent - parent window

proc VFolderImportEdit {def parent} {
    global t b idCnt

    # Create identifier
    set id cf[incr idCnt]
    set w .$id
    upvar #0 $id hd
    set hd(done) 0
    set hd(name) [lindex $def 0]
    set hd(sort) default
    array set hd [lindex $def 2]
    set hd(sort_l) $t(sort_$hd(sort))
    set hd(pattern) [lindex $def 4]
    set hd(w) $w

    # Create toplevel
    toplevel $w -bd 5 -class TkRat
    wm title $w $t(import)

    label $w.nl -text $t(name): -anchor e
    entry $w.ne -textvariable ${id}(name)
    set b($w.ne) vd_menuname
    grid $w.nl $w.ne -sticky we

    switch -regexp [lindex $def 1] {
	dir|mh {
		set hd(entry) [lindex $def 5]
		label $w.el -text $t(pathname):
		entry $w.ee -textvariable ${id}(entry) -width 30
		set b($w.ee) vd_pathname
		button $w.eb -text $t(browse)... \
			-command "Browse $hd(w) ${id}(entry) dirok"
		set b($w.eb) file_browse
		grid $w.el $w.ee $w.eb -sticky ew
	    }
	imap|dis {
		set hd(host) [lindex $def 5]
		set hd(user) [lindex $def 6]
		set hd(mbox) [lindex $def 7]
		set hd(imapport) [lindex $def 8]
		foreach e {host imapport user mbox} {
		    label $w.${e}_lab -text $t($e): -anchor e
		    entry $w.${e}_entry -textvariable ${id}($e) -width 30
		    grid $w.${e}_lab $w.${e}_entry -sticky ew
		    set b($w.$e.entry) vd_$e
		}
	    }
    }
    # Pattern and flags
    label $w.pl -text $t(pattern): -anchor e
    entry $w.pe -textvariable ${id}(pattern)
    set b($w.pe) vd_pattern
    checkbutton $w.sub -text $t(subscribed_only) -variable ${id}(subscribed)
    set b($w.sub) vd_subscribed
    checkbutton $w.trace -text $t(trace_changes) -variable ${id}(trace)
    set b($w.trace) vd_trace
    label $w.sort_label -text $t(sort_order): -anchor e
    set m $w.sort_m.m
    menubutton $w.sort_m -textvariable ${id}(sort_l) \
	    -relief raised -indicatoron 1 -menu $m
    set b($w.sort_m) vd_sort
    menu $m -tearoff 0
    foreach o {default threaded folder reverseFolder date reverseDate subject
	       subjectonly} {
	$m add command -label $t(sort_$o) \
		-command "set ${id}(sort) $o; \
			  set ${id}(sort_l) [list $t(sort_$o)]"
    }
    checkbutton $w.browse -text $t(browse_mode) -variable ${id}(browse)
    set b($w.browse) browse_mode
    OkButtons $w $t(ok) $t(cancel) "set ${id}(done)"

    grid $w.pl $w.pe -sticky we
    grid x $w.sub -sticky w
    grid x $w.trace -sticky w
    grid $w.sort_label $w.sort_m -sticky we
    grid x $w.browse -sticky w
    grid $w.buttons - - -pady 5 -sticky ew

    Place $w editImport
    Modal $w $parent
    focus $w.ne

    while 1 {
	tkwait variable ${id}(done)
	if { 0 == $hd(done) } {
	    set def {}
	    break
	}
	if ![string length $hd(name)] {
	    Popup $t(need_name) $w
	    continue
	}
	set flags {}
	if $hd(trace) {
	   lappend flags trace
	   lappend flags 1
	}
	if $hd(subscribed) {
	   lappend flags subscribed
	   lappend flags 1
	}
	set opt [list sort $hd(sort) browse $hd(browse)]
	switch -regexp [lindex $def 1] {
	    dir|mh {
		    set def [list $hd(name) [lindex $def 1] $flags $opt \
				  $hd(pattern) $hd(entry)]
		}
	    imap|dis {
		    set def [list $hd(name) [lindex $def 1] $flags $opt \
				  $hd(pattern) $hd(host) $hd(user) $hd(mbox) \
				  $hd(imapport)]
		}
	}
	break
    }
    RecordPos $w editImport
    destroy $w
    foreach bn [array names b $w.*] {unset b($bn)}
    unset hd
    return $def
}
