/*
 * ratMessage.c --
 *
 *	This file contains code which implements the message entities.
 *
 * TkRat software and its included text is Copyright 1996-1999 by
 * Martin Forssn
 *
 * The full text of the legal notice is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

#include "ratFolder.h"
#include "ratStdFolder.h"
#include "ratPGP.h"
#include <signal.h>

/*
 * An array of commands. It contains one entry for each internal message
 * type (as defined by RatMessageType).
 */
static MessageProcInfo *messageProcInfo;

/*
 * The number of replies created. This is used to create new unique
 * message handlers.
 */
static int numReplies = 0;

/*
 * The number of message entities created. This is used to create new
 * unique command names.
 */
static int numBodies = 0;

static void RatBodyDelete(Tcl_Interp *interp, BodyInfo *bodyInfoPtr);
static BodyInfo *RatFindFirstText(BodyInfo *bodyInfoPtr);
static char *RatGetCitation(Tcl_Interp *interp, MessageInfo *msgPtr);
static void RatCiteMessage(Tcl_Interp *interp, Tcl_Obj *dstObjPtr, 
	char *src, char *myCitation);

extern long unix_create (MAILSTREAM *stream,char *mailbox);


/*
 *----------------------------------------------------------------------
 *
 * RatInitMessages --
 *
 *      Initialize the message data structures.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The messageProcInfo array is allocated and initialized.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatInitMessages()
{
    messageProcInfo = (MessageProcInfo*)ckalloc(3*sizeof(MessageProcInfo));
    RatStdMessagesInit(&messageProcInfo[RAT_CCLIENT_MESSAGE]);
    RatDbMessagesInit(&messageProcInfo[RAT_DBASE_MESSAGE]);
    RatFrMessagesInit(&messageProcInfo[RAT_FREE_MESSAGE]);
}


/*
 *----------------------------------------------------------------------
 *
 * RatMessageCmd --
 *
 *      Main std mail entity procedure. This routine implements the mail
 *	commands mentioned in ../INTERFACE.
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *	many.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatMessageCmd(ClientData clientData,Tcl_Interp *interp, int objc,
	Tcl_Obj *CONST objv[])
{
    MessageInfo *msgPtr = (MessageInfo*) clientData;
    int length;
    char *cmdPtr;

    if (objc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		Tcl_GetString(objv[0]), " option ?arg?\"", (char *) NULL);
	return TCL_ERROR;
    }
    cmdPtr = Tcl_GetString(objv[1]);
    length = strlen(cmdPtr);
    if ((cmdPtr[0] == 'h') && (strncmp(cmdPtr, "headers", length) == 0)
	    && (length > 1)) {
	if (objc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    Tcl_GetString(objv[0]), " headers\"", (char *) NULL);
	    return TCL_ERROR;
	}
	return RatMessageGetHeader(interp,
		(*messageProcInfo[msgPtr->type].getHeadersProc)(interp,msgPtr));

    } else if ((cmdPtr[0] == 'b') && (strncmp(cmdPtr, "body", length) == 0)
	    && (length > 1)) {
	if (!msgPtr->bodyInfoPtr) {
	    msgPtr->bodyInfoPtr =
		    (*messageProcInfo[msgPtr->type].createBodyProc)(interp,
		    msgPtr);
	    RatPGPBodyCheck(interp, messageProcInfo, &msgPtr->bodyInfoPtr);
	    Tcl_CreateCommand(interp, msgPtr->bodyInfoPtr->cmdName,
		    RatBodyCmd, (ClientData) msgPtr->bodyInfoPtr, NULL);
	}
	Tcl_SetResult(interp, msgPtr->bodyInfoPtr->cmdName, TCL_STATIC);
	return TCL_OK;
    
    } else if ((cmdPtr[0] == 'r') && (strncmp(cmdPtr, "rawText", length) == 0)
	    && (length > 1)) {
	Tcl_DString ds;
	char *buf;
	MESSAGE *messagePtr;

	Tcl_DStringInit(&ds);
	Tcl_DStringAppend(&ds,
		(*messageProcInfo[msgPtr->type].getHeadersProc)(interp, msgPtr),
		-1);
	Tcl_DStringAppend(&ds, "\n", 1);
	Tcl_DStringAppend(&ds,
		(*messageProcInfo[msgPtr->type].fetchTextProc)(interp, msgPtr),
		-1);
	messagePtr = RatParseMsg(interp, Tcl_DStringValue(&ds));
	Tcl_DStringSetLength(&ds, 0);
	buf = (char*)ckalloc(RatHeaderSize(messagePtr->env, messagePtr->body));
	rfc822_output(buf, messagePtr->env, messagePtr->body,
		RatStringPuts, &ds, 0);
	mail_free_envelope(&messagePtr->env);
	mail_free_body(&messagePtr->body);
	ckfree(messagePtr);
	ckfree(buf);

	Tcl_DStringResult(interp, &ds);
	return TCL_OK;

    } else if ((cmdPtr[0] == 'g') && (strncmp(cmdPtr, "get", length) == 0)
	    && (length > 1)) {
	ENVELOPE *env = (*messageProcInfo[msgPtr->type].envelopeProc)(msgPtr);
	int i;
	if (objc < 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    Tcl_GetString(objv[0]), " get fields\"", (char *) NULL);
	    return TCL_ERROR;
	}
	for (i=2; i<objc; i++) {
	    if (!strcasecmp(Tcl_GetString(objv[i]), "return_path")) {
		RatInitAddresses(interp, env->return_path);
	    } else if (!strcasecmp(Tcl_GetString(objv[i]), "from")) {
		RatInitAddresses(interp, env->from);
	    } else if (!strcasecmp(Tcl_GetString(objv[i]), "sender")) {
		RatInitAddresses(interp, env->sender);
	    } else if (!strcasecmp(Tcl_GetString(objv[i]), "reply_to")) {
		RatInitAddresses(interp, env->reply_to);
	    } else if (!strcasecmp(Tcl_GetString(objv[i]), "to")) {
		RatInitAddresses(interp, env->to);
	    } else if (!strcasecmp(Tcl_GetString(objv[i]), "cc")) {
		RatInitAddresses(interp, env->cc);
	    } else if (!strcasecmp(Tcl_GetString(objv[i]), "bcc")) {
		RatInitAddresses(interp, env->bcc);
	    } else {
		Tcl_ResetResult(interp);
		Tcl_AppendResult(interp, "bad field \"", Tcl_GetString(objv[i]),
			"\": must be one of return_path, from, sender, ",
			"reply_to, to, cc or bcc", (char*)NULL);
		return TCL_ERROR;
	    }
	}
	return TCL_OK;

    } else if ((cmdPtr[0] == 'r') && (strncmp(cmdPtr, "reply", length) == 0)
	    && (length > 1)) {
	/*
	 * Construct a reply to a message. We should really handle
	 * different character sets here. /MaF
	 */
	ENVELOPE *env = (*messageProcInfo[msgPtr->type].envelopeProc)(msgPtr);
	char handler[32], buf[1024], *cPtr;
	BodyInfo *bodyInfoPtr;
	unsigned long bodylength;
	ADDRESS *adrPtr;
	char *dataPtr;
	Tcl_DString ds;
	Tcl_Obj *oPtr;

	Tcl_DStringInit(&ds);

	if (objc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    Tcl_GetString(objv[0]), " reply to\"", (char *) NULL);
	    return TCL_ERROR;
	}
	sprintf(handler, "reply%d", numReplies++);
	if (!strcasecmp(Tcl_GetString(objv[2]), "sender")) {
	    /*
	     * We should construct a reply which should go only to one person.
	     * We look for the address in the following fields:
	     *  Reply-To:, From:. Sender:, From
	     * As sson as an address is found the search stops.
	     */
	    if (env->reply_to) {
		adrPtr = env->reply_to;
	    } else if (env->from) {
		adrPtr = env->from;
	    } else if (env->sender) {
		adrPtr = env->sender;
	    } else {
		adrPtr = env->return_path;
	    }
	    for (;adrPtr; adrPtr = adrPtr->next) {
		RatAddressTranslate(interp, adrPtr);
		if (adrPtr->mailbox) {
		    if (Tcl_DStringLength(&ds)) {
			Tcl_DStringAppend(&ds, ", ", -1);
		    }
		    Tcl_DStringAppend(&ds, RatAddressMail(adrPtr), -1);
		}
	    }
	    Tcl_SetVar2(interp, handler, "to", Tcl_DStringValue(&ds),
		    TCL_GLOBAL_ONLY);
	} else {
	    /*
	     * We should construct a reply which goes to everybody who has
	     * recieved this message. This is done by first collecting all
	     * addresses found in: Reply-To:, From:, Sender:, To: and Cc:.
	     * Then go though this list and eliminate myself and any
	     * duplicates. Now we use the first element of the list as To:
	     * and the rest as Cc:.
	     */
	    ADDRESS **recipientPtrPtr = (ADDRESS**)ckalloc(16*sizeof(ADDRESS*));
	    int numAllocated = 16;
	    int numRecipients = 0;
	    int inList = 0;
	    ADDRESS *to = NULL;
	    int i, j;

#define SCANLIST(x) for (adrPtr = (x); adrPtr; adrPtr = adrPtr->next) { \
			if (numRecipients == numAllocated) { \
			    numAllocated += 16; \
			    recipientPtrPtr = (ADDRESS**)ckrealloc( \
				    recipientPtrPtr, \
				    numAllocated*sizeof(ADDRESS*)); \
			} \
			recipientPtrPtr[numRecipients++] = adrPtr; \
	    	    }

	    SCANLIST(env->reply_to);
	    if (NULL == env->reply_to) {
		SCANLIST(env->from);
	    }
	    SCANLIST(env->to);
	    SCANLIST(env->cc);

	    for (i=0; i<numRecipients; i++) {
		adrPtr = recipientPtrPtr[i];
		if (!adrPtr->host) {
		    inList = (inList)? 0 : 1;
		    continue;
		}
		if (RatAddressIsMe(interp, adrPtr, 1)) {
		    continue;
		}
		RatAddressTranslate(interp, adrPtr);
		for (j=0; j<i; j++) {
		    if (!RatAddressCompare(adrPtr, recipientPtrPtr[j])) {
			break;
		    }
		}
		if (j < i) {
		    continue;
		}
		if (!to) {
		    to = adrPtr;
		} else {
		    if (Tcl_DStringLength(&ds)) {
			Tcl_DStringAppend(&ds, ", ", 2);
		    }
		    Tcl_DStringAppend(&ds, RatAddressMail(adrPtr), -1);
		}
	    }
	    if (Tcl_DStringLength(&ds)) {
		Tcl_SetVar2(interp, handler, "cc", Tcl_DStringValue(&ds),
			TCL_GLOBAL_ONLY);
	    }
	    if (!to && numRecipients) {
		to = recipientPtrPtr[0];
	    }
	    if (to) {
		if (strcmp(to->host, currentHost)) {
		    snprintf(buf, sizeof(buf), "%s@%s", to->mailbox, to->host);
		} else {
		    snprintf(buf, sizeof(buf), "%s", to->mailbox);
		}
		Tcl_SetVar2(interp, handler, "to", buf, TCL_GLOBAL_ONLY);
	    }
	    ckfree(recipientPtrPtr);
	}
	if (env->subject) {
	    int match;

	    cPtr = RatDecodeHeader(interp, env->subject, 0);
	    Tcl_DStringSetLength(&ds, 0);
	    Tcl_DStringAppendElement(&ds, "regexp");
	    Tcl_DStringAppendElement(&ds, "-nocase");
	    Tcl_DStringAppendElement(&ds,
		    Tcl_GetVar2(interp, "option", "re_regexp",TCL_GLOBAL_ONLY));
	    Tcl_DStringAppendElement(&ds, cPtr);
	    Tcl_DStringAppendElement(&ds, "reply_match");
	    if (TCL_OK == Tcl_EvalEx(interp, Tcl_DStringValue(&ds), -1,
				     TCL_EVAL_DIRECT)
		    && TCL_OK == Tcl_GetBoolean(interp,
						Tcl_GetStringResult(interp),
						&match)
		    && match) {
		char *s = Tcl_GetVar(interp, "reply_match", 0);

		if (!strncmp(s, cPtr, strlen(s))) {
		    cPtr += strlen(s);
		    while (isspace(*cPtr)) cPtr++;
		}
	    }
	    oPtr = Tcl_NewStringObj("Re: ", 4);
	    Tcl_AppendToObj(oPtr, cPtr, -1);
	    Tcl_SetVar2Ex(interp, handler,"subject", oPtr,TCL_GLOBAL_ONLY);
	} else {
	    Tcl_SetVar2(interp, handler, "subject",
		    Tcl_GetVar2(interp, "option", "no_subject",TCL_GLOBAL_ONLY),
		    TCL_GLOBAL_ONLY);
	}
	if (env->message_id) {
	    Tcl_SetVar2(interp, handler, "in_reply_to", env->message_id,
		    TCL_GLOBAL_ONLY);
	}
	bodyInfoPtr = RatFindFirstText(msgPtr->bodyInfoPtr);
	if (bodyInfoPtr && (NULL != (dataPtr =
		(*messageProcInfo[bodyInfoPtr->msgPtr->type].fetchBodyProc)
		(bodyInfoPtr, &bodylength)))) {
	    char *citation, *attrFormat, *alias,
		 *charset = "us-ascii";
	    ListExpression *exprPtr;
	    PARAMETER *parameter;
	    Tcl_DString *decBPtr;
	    BODY *bodyPtr;
	    int wrap;

	    bodyPtr = bodyInfoPtr->bodyPtr;

	    for (parameter = bodyPtr->parameter; parameter;
		    parameter = parameter->next) {
		if ( 0 == strcasecmp("charset", parameter->attribute)) {
		    charset = parameter->value;
		}
	    }
	    if ((alias = Tcl_GetVar2(interp, "charsetAlias", charset,
		    TCL_GLOBAL_ONLY))) {
		charset = alias;
	    }
	    decBPtr = RatDecode(interp, bodyPtr->encoding, dataPtr, bodylength,
		    charset);

	    attrFormat = Tcl_GetVar2(interp, "option", "attribution",
		    TCL_GLOBAL_ONLY);
	    if (attrFormat && strlen(attrFormat)
		    && (exprPtr = RatParseList(attrFormat))) {
		oPtr = RatDoList(interp, exprPtr,
			messageProcInfo[msgPtr->type].getInfoProc,
			(ClientData)msgPtr, 0);
		RatFreeListExpression(exprPtr);
		Tcl_AppendToObj(oPtr, "\n", 1);
	    } else {
		oPtr = Tcl_NewObj();
	    }

	    citation = RatGetCitation(interp, msgPtr);
	    RatCiteMessage(interp, oPtr, Tcl_DStringValue(decBPtr), citation);
	    Tcl_DStringFree(decBPtr);
	    Tcl_GetBoolean(interp, Tcl_GetVar2(interp, "option", "wrap_cited",
		    TCL_GLOBAL_ONLY), &wrap);
	    if (wrap) {
		Tcl_Obj *nPtr;
		nPtr = RatWrapMessage(interp, oPtr);
		Tcl_DecrRefCount(oPtr);
		oPtr = nPtr;
	    }
	    Tcl_SetVar2Ex(interp, handler, "data", oPtr, TCL_GLOBAL_ONLY);
	}
	Tcl_SetResult(interp, handler, TCL_VOLATILE);
	Tcl_DStringFree(&ds);
	return TCL_OK;

    } else if ((cmdPtr[0] == 'c') && (strncmp(cmdPtr, "copy", length) == 0)
	    && (length > 1)) {
	char *cPtr, flags[128], date[128];
	Tcl_Obj **listObjv;
	MAILSTREAM *stream = NULL;
	struct stat sbuf;
	Tcl_DString ds, def;
	STRING string;
	int listObjc, result, i, freeListObjv = 0;
	RatFolderInfo *infoPtr;

	if (objc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    Tcl_GetString(objv[0])," copy vfolder_def\"", (char *)NULL);
	    return TCL_ERROR;
	}
	Tcl_ListObjGetElements(interp, objv[2], &listObjc, &listObjv);

	/*
	 * If the destination is dbase then we call RatInsert
	 */
	if (!strcmp("dbase", Tcl_GetString(listObjv[1]))) {
	    result = RatInsertMsg(interp, msgPtr, Tcl_GetString(listObjv[3]),
		    Tcl_GetString(listObjv[5]), Tcl_GetString(listObjv[4]));
	    return result;
	}

	/*
	 * If the destination is a dynamic folder then we have to do some
	 * magic.
	 */
	if (!strcmp("dynamic", Tcl_GetString(listObjv[1]))) {
	    char *name;
	    Tcl_Obj *oPtr, **newObjv;

	    oPtr = (*messageProcInfo[msgPtr->type].getInfoProc)(interp,
		    (ClientData)msgPtr, RAT_FOLDER_MAIL_REAL, 0);
	    if (!oPtr) {
		name = currentMailboxName;
	    } else {
		name = Tcl_GetString(oPtr);
		if (!strlen(name)) {
		    name = currentMailboxName;
		}
	    }
	    newObjv = (Tcl_Obj**)ckalloc(4*sizeof(Tcl_Obj*));
	    newObjv[listObjc=0] = listObjv[0];
	    newObjv[++listObjc] = Tcl_NewStringObj("file", 4);
	    newObjv[++listObjc] = Tcl_NewStringObj("", 0);
	    newObjv[++listObjc] = Tcl_DuplicateObj(listObjv[3]);
	    Tcl_AppendToObj(newObjv[listObjc], "/", 1);
	    for (i=0; name[i] && name[i] != '@'; i++);
	    Tcl_AppendToObj(newObjv[listObjc], name, i);
	    listObjv = newObjv;
	    for (i=0; i<listObjc; i++) {
		Tcl_IncrRefCount(newObjv[i]);
	    }
	    freeListObjv = 1;
	}

	/*
	 * If the destination is a disconnected folder then we have to do some
	 * magic.
	 */
	if (!strcmp("dis", Tcl_GetString(listObjv[1]))) {
	    char *dir;
	    Tcl_Obj **newObjv;

	    dir = RatDisFolderDir(interp, Tcl_GetString(listObjv[3]),
		    Tcl_GetString(listObjv[4]), "imap");
	    newObjv = (Tcl_Obj**)ckalloc(4*sizeof(Tcl_Obj*));
	    newObjv[listObjc=0] = listObjv[0];
	    newObjv[++listObjc] = Tcl_NewStringObj("dis", 4);
	    newObjv[++listObjc] = Tcl_NewStringObj("", 0);
	    newObjv[++listObjc] = Tcl_NewStringObj(dir, -1);
	    Tcl_AppendToObj(newObjv[listObjc], "/folder", 7);
	    listObjv = newObjv;
	    for (i=0; i<listObjc; i++) {
		Tcl_IncrRefCount(newObjv[i]);
	    }
	    freeListObjv = 1;
	}

	/*
	 * Try to create nonexisting files
	 */
	if (!strcmp("file", Tcl_GetString(listObjv[1]))
		&& stat(Tcl_GetString(listObjv[3]), &sbuf)) {
	    struct stat sbuf;
	    if (0 != stat(Tcl_GetString(listObjv[3]), &sbuf)) {
		unix_create(NIL, Tcl_GetString(listObjv[3]));
	    }
	}

	/*
	 * Try the simple (but common) case where both source
	 * and destination are c-client messages. 
	 */
	if (RAT_CCLIENT_MESSAGE == msgPtr->type
		&& RatStdEasyCopyingOK(msgPtr, Tcl_GetString(listObjv[1]),
				       Tcl_GetString(listObjv[3]),
			(listObjc > 4 ? Tcl_GetString(listObjv[4]) : NULL))) {
	    result =RatStdMessageCopy(interp,msgPtr,Tcl_GetString(listObjv[3]));
	    if (freeListObjv) {
		for (i=0; i<listObjc; i++) {
		    Tcl_DecrRefCount(listObjv[i]);
		}
		ckfree(listObjv);
	    }
	    return result;
	}

	/*
	 * Open a folder and get the stream
	 */
	Tcl_DStringInit(&def);
	Tcl_DStringAppend(&def, "std ", 4);
	Tcl_DStringAppend(&def, Tcl_GetString(listObjv[3]), -1);
	if (listObjc > 4) {
	    Tcl_DStringAppend(&def, " ", 1);
	    Tcl_DStringAppend(&def, Tcl_GetString(listObjv[1]), -1);
	    Tcl_DStringAppend(&def, " ", 1);
	    Tcl_DStringAppend(&def, Tcl_GetString(listObjv[4]), -1);
	}
	if ((infoPtr = RatGetOpenFolder(Tcl_DStringValue(&def)))) {
	    if (TCL_OK != Tcl_VarEval(interp, infoPtr->cmdName, " insert ", msgPtr->name, NULL)) {
		fprintf(stderr, "error: %s\n", Tcl_GetStringResult(interp));
	    }
	} else {
	    stream = OpenStdFolder(interp, Tcl_GetString(listObjv[3]),
			       Tcl_GetString(listObjv[1]),
			       (listObjc>4 ? Tcl_GetString(listObjv[4]) : NULL),
			       NULL);
	}
	Tcl_DStringFree(&def);
	if (stream) {
	    Tcl_DStringInit(&ds);
	    RatMessageGet(interp, msgPtr, &ds, flags, sizeof(flags),
		    date, sizeof(date));
	    INIT(&string, mail_string, Tcl_DStringValue(&ds),
		    Tcl_DStringLength(&ds));

	    if (NULL != (cPtr = strstr(flags, RAT_FLAGGED_STR))) {
		i = strlen(RAT_FLAGGED_STR);
		if (flags == cPtr) {
		    if (' ' == cPtr[i]) {
			i++;
		    }
		} else {
		    cPtr--;
		    i++;
		}
		strcpy(cPtr, cPtr+i);
	    }
	    if (NULL != (cPtr = strstr(flags, RAT_DELETED_STR))) {
		i = strlen(RAT_DELETED_STR);
		if (flags == cPtr) {
		    if (' ' == cPtr[i]) {
			i++;
		    }
		} else {
		    cPtr--;
		    i++;
		}
		strcpy(cPtr, cPtr+i);
	    }
	    if (!mail_append_full(stream, Tcl_GetString(listObjv[3]),
				  flags, date, &string)){
		CloseStdFolder(interp, stream);
		Tcl_SetResult(interp, "mail_append failed", TCL_STATIC);
		if (freeListObjv) {
		    for (i=0; i<listObjc; i++) {
			Tcl_DecrRefCount(listObjv[i]);
		    }
		    ckfree(listObjv);
		}
		Tcl_DStringFree(&ds);
		return TCL_ERROR;
	    }
	    if (infoPtr) {
		RatFolderClose(interp, infoPtr, 0);
	    } else {
		CloseStdFolder(interp, stream);
	    }
	    Tcl_DStringFree(&ds);
	}
	if (freeListObjv) {
	    for (i=0; i<listObjc; i++) {
		Tcl_DecrRefCount(listObjv[i]);
	    }
	    ckfree(listObjv);
	}
	return TCL_OK;

    } else if ((cmdPtr[0] == 'l') && (strncmp(cmdPtr, "list", length) == 0)
	    && (length > 1)) {
	ListExpression *exprPtr;
	Tcl_Obj *oPtr;

	if (objc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    Tcl_GetString(objv[0]), " list format\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (NULL == (exprPtr = RatParseList(Tcl_GetString(objv[2])))) {
	    Tcl_SetResult(interp, "Illegal list format", TCL_STATIC);
	    return TCL_ERROR;
	}
	oPtr = RatDoList(interp, exprPtr,
		messageProcInfo[msgPtr->type].getInfoProc,
		(ClientData)msgPtr, 0);
	Tcl_SetObjResult(interp, oPtr);
	RatFreeListExpression(exprPtr);
	return TCL_OK;

    } else {
	Tcl_AppendResult(interp, "bad option \"", Tcl_GetString(objv[1]),
		"\": must be one of header, body, rawText reply, or get",
		(char*)NULL);
	return TCL_ERROR;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RatMessageGetHeader --
 *
 *      Gets the header of a message
 *
 * Results:
 *	The header is returned as a list in the result area.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */
int
RatMessageGetHeader(Tcl_Interp *interp, char *srcHeader)
{
    char *header, *listArgv[2];
    char *dstPtr, *srcPtr = srcHeader, *cPtr, *tPtr;
    Tcl_Obj *oPtr = Tcl_NewObj(), *fPtr[2];
    int adr;

    if (!srcHeader) {
	RatLog(interp, RAT_FATAL, Tcl_GetStringResult(interp), RATLOG_TIME);
	exit(1);
    }
    header = (char*) ckalloc (strlen(srcHeader)+1);
    if (!strncmp("From ", srcPtr, 5)) {
	while ('\n' != *srcPtr) {
	    srcPtr++;
	}
	if ('\r' == *(++srcPtr)) {
	    srcPtr++;
	}
    }
    while (*srcPtr) {
	dstPtr = header;
	listArgv[0] = dstPtr = header;
	while (*srcPtr && ':' != *srcPtr && ' ' != *srcPtr) {
	    *dstPtr++ = *srcPtr++;
	}
	*dstPtr = '\0';
	fPtr[0] = Tcl_NewStringObj(header, -1);
	cPtr = ++dstPtr;
	do {
	    srcPtr++;
	} while (' ' == *srcPtr || '\t' == *srcPtr);
	do {
	    for (; *srcPtr && '\n' != *srcPtr; srcPtr++) {
		if ('\r' != *srcPtr) {
		    *dstPtr++ = *srcPtr;
		}
	    }
	    while ('\n' == *srcPtr || '\r' == *srcPtr) {
		srcPtr++;
	    }
	} while (*srcPtr && (' ' == *srcPtr || '\t' == *srcPtr));
	*dstPtr = '\0';
	tPtr = cPtr;
	if (0 == strncasecmp("resent-", tPtr, 7)) {
	    tPtr += 7;
	} 
	if (!strcasecmp(tPtr, "to")
		|| !strcasecmp(tPtr, "cc")
		|| !strcasecmp(tPtr, "bcc")
		|| !strcasecmp(tPtr, "from")
		|| !strcasecmp(tPtr, "sender")
		|| !strcasecmp(tPtr, "reply-to")) {
	    adr = 1;
	} else {
	    adr = 0;
	}
	fPtr[1] = Tcl_NewStringObj(RatDecodeHeader(interp, cPtr, adr), -1);
	Tcl_ListObjAppendElement(interp, oPtr, Tcl_NewListObj(2, fPtr));
    }
    ckfree(header);
    Tcl_SetObjResult(interp, oPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatMessageDelete --
 *
 *      Deletes the given message.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The message and all its bodyparts are deleted from the interpreter
 *	and all the structures are freed.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatMessageDelete(Tcl_Interp *interp, char *msgName)
{
    Tcl_CmdInfo cmdInfo;
    MessageInfo *msgPtr;
    char buf[256];
    int i;

    if (0 == Tcl_GetCommandInfo(interp, msgName, &cmdInfo)) {
	Tcl_AppendResult(interp, "No such message: ", msgName, NULL);
	return TCL_ERROR;
    }
    msgPtr = (MessageInfo*)cmdInfo.objClientData;

    (*messageProcInfo[msgPtr->type].msgDeleteProc)(msgPtr);
    if (msgPtr->bodyInfoPtr) {
	if (msgPtr->bodyInfoPtr->altPtr) {
	    RatBodyDelete(interp, msgPtr->bodyInfoPtr->altPtr);
	}
	if (msgPtr->bodyInfoPtr->decodedTextPtr) {
	    Tcl_DStringFree(msgPtr->bodyInfoPtr->decodedTextPtr);
	    ckfree(msgPtr->bodyInfoPtr->decodedTextPtr);
	}
	if (msgPtr->bodyInfoPtr->secPtr) {
	    RatBodyDelete(interp, msgPtr->bodyInfoPtr->secPtr);
	} else {
	    RatBodyDelete(interp, msgPtr->bodyInfoPtr);
	}
    }
    snprintf(buf, sizeof(buf), "msgInfo_%s", msgPtr->name);
    Tcl_UnsetVar(interp, buf, TCL_GLOBAL_ONLY);
    Tcl_DeleteCommand(interp, msgName);
    for (i=0; i<sizeof(msgPtr->info)/sizeof(*msgPtr->info); i++) {
	if (msgPtr->info[i]) {
	    Tcl_DecrRefCount(msgPtr->info[i]);
	}
    }
    ckfree(msgPtr);

    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * CreateBodyInfo --
 *
 *      Create and somewhat initialize a BodyInfo structure.
 *
 * Results:
 *	A pointer to a BodyInfo structure.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */
BodyInfo*
CreateBodyInfo(MessageInfo *msgPtr)
{
    BodyInfo *bodyInfoPtr;
    int pad = sizeof(char*) - sizeof(BodyInfo)%sizeof(char*);

    if (sizeof(char*) == pad) {
	pad = 0;
    }

    bodyInfoPtr = (BodyInfo*)ckalloc(sizeof(BodyInfo)+pad+16);

    bodyInfoPtr->cmdName = (char*)bodyInfoPtr + pad + sizeof(BodyInfo);

    sprintf(bodyInfoPtr->cmdName, "RatBody%d", numBodies++);
    bodyInfoPtr->firstbornPtr = NULL;
    bodyInfoPtr->nextPtr = NULL;
    bodyInfoPtr->containedEntity = NULL;
    bodyInfoPtr->type = msgPtr->type;
    bodyInfoPtr->msgPtr = msgPtr;
    bodyInfoPtr->secPtr = NULL;
    bodyInfoPtr->altPtr = NULL;
    bodyInfoPtr->decodedTextPtr = NULL;
    bodyInfoPtr->encoded = 0;
    bodyInfoPtr->sigStatus = RAT_UNSIGNED;
    bodyInfoPtr->pgpOutput = NULL;
    return bodyInfoPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * RatBodyCmd --
 *
 *      Main bodypart entity procedure. This routine implements the
 *	bodypart commands mentioned in ../INTERFACE.
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *	many.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatBodyCmd(ClientData clientData,Tcl_Interp *interp,int argc,char *argv[])
{
    BodyInfo *bodyInfoPtr = (BodyInfo*) clientData;
    BODY *bodyPtr = bodyInfoPtr->bodyPtr;
    unsigned long length;
    char c;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" option ?arg?\"", (char *) NULL);
	return TCL_ERROR;
    }
    c = argv[1][0];
    length = strlen(argv[1]);
    if ((c == 'c') && (strncmp(argv[1], "children", length) == 0)
	    && (length > 2)) {
	BodyInfo *partInfoPtr;
	Tcl_DString resultDS;

	if (TYPEMULTIPART != bodyPtr->type) {
	    return TCL_OK;
	}

	if (!bodyInfoPtr->firstbornPtr) {
	    (*messageProcInfo[bodyInfoPtr->type].makeChildrenProc)
		    (interp, bodyInfoPtr);
	    for (partInfoPtr = bodyInfoPtr->firstbornPtr; partInfoPtr;
		    partInfoPtr = partInfoPtr->nextPtr) {
		RatPGPBodyCheck(interp, messageProcInfo, &partInfoPtr);
		Tcl_CreateCommand(interp, partInfoPtr->cmdName, RatBodyCmd,
			(ClientData) partInfoPtr, NULL);
	    }
	}
	Tcl_DStringInit(&resultDS);
	for (partInfoPtr = bodyInfoPtr->firstbornPtr; partInfoPtr;
		partInfoPtr = partInfoPtr->nextPtr) {
	    Tcl_DStringAppendElement(&resultDS, partInfoPtr->cmdName);
	}
	Tcl_DStringResult(interp, &resultDS);
	return TCL_OK;

    } else if ((c == 'm') && (strncmp(argv[1], "message", length) == 0)
	    && (length > 1)) {
	char *body;

	if (!bodyInfoPtr->containedEntity) {
	    if (TYPEMESSAGE != bodyPtr->type &&
		    !strcasecmp(bodyPtr->subtype, "rfc822")) {
		Tcl_SetResult(interp, "Not an message/rfc822 bodypart",
			TCL_STATIC);
		return TCL_ERROR;
	    }
	    body = (*messageProcInfo[bodyInfoPtr->type].fetchBodyProc)
		    (bodyInfoPtr, &length);
	    if (body && *body) {
		bodyInfoPtr->containedEntity =
			RatFrMessageCreate(interp, body, length, NULL);
		Tcl_SetResult(interp, bodyInfoPtr->containedEntity, TCL_STATIC);
	    } else {
		Tcl_SetResult(interp, "mail_fetchbody failed", TCL_STATIC);
		return TCL_ERROR;
	    }
	} else {
	    Tcl_SetResult(interp, bodyInfoPtr->containedEntity, TCL_STATIC);
	}
	return TCL_OK;

    } else if ((c == 't') && (strncmp(argv[1], "type", length) == 0)
	    && (length > 1)) {
	Tcl_SetObjResult(interp, RatBodyType(bodyInfoPtr));
	return TCL_OK;

    } else if ((c == 'p') && (strncmp(argv[1], "params", length) == 0)) {
	PARAMETER *parameter;
	Tcl_DString result;

	Tcl_DStringInit(&result);
	for (parameter = bodyPtr->parameter; parameter;
		parameter = parameter->next) {
	    Tcl_DStringStartSublist(&result);
	    Tcl_DStringAppendElement(&result, parameter->attribute);
	    Tcl_DStringAppendElement(&result, parameter->value);
	    Tcl_DStringEndSublist(&result);
	}
	Tcl_DStringResult(interp, &result);
	return TCL_OK;

    } else if ((c == 'p') && (strncmp(argv[1], "parameter", length) == 0)) {
	PARAMETER *parameter;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " parameter attribute\"", (char *) NULL);
	    return TCL_ERROR;
	}
	for (parameter = bodyPtr->parameter; parameter;
		parameter = parameter->next) {
	    if ( 0 == strcasecmp(argv[2], parameter->attribute)) {
		Tcl_SetResult(interp, parameter->value, TCL_VOLATILE);
		break;
	    }
	}
	return TCL_OK;

    } else if ((c == 'i') && (strncmp(argv[1], "id", length) == 0)
	    && (length > 1)) {
	if (bodyPtr->id) {
	    Tcl_SetResult(interp, bodyPtr->id, TCL_VOLATILE);
	}
	return TCL_OK;

    } else if ((c == 'd') && (strncmp(argv[1], "description", length) == 0)
	    && (length > 1)) {
	if (bodyPtr->description) {
	    Tcl_SetResult(interp, bodyPtr->description, TCL_VOLATILE);
	}
	return TCL_OK;

    } else if ((c == 's') && (strncmp(argv[1], "size", length) == 0)
	    && (length > 1)) {
	Tcl_SetObjResult(interp, Tcl_NewIntObj(bodyPtr->size.bytes));
	return TCL_OK;

    } else if ((c == 'l') && (strncmp(argv[1], "lines", length) == 0)
	    && (length > 1)) {
	Tcl_SetObjResult(interp, Tcl_NewIntObj(bodyPtr->size.lines));
	return TCL_OK;

    } else if ((c == 'e') && (strncmp(argv[1], "encoding", length) == 0)
	    && (length > 1)) {
	char *enc;

	switch(bodyPtr->encoding) {
	case ENC7BIT:		enc = "7bit"; break;
	case ENC8BIT:		enc = "8bit"; break;
	case ENCBINARY:		enc = "binary"; break;
	case ENCBASE64:		enc = "base64"; break;
	case ENCQUOTEDPRINTABLE:enc = "quoted-printable"; break;
	default:		enc = "unkown"; break;
	}
	Tcl_SetResult(interp, enc, TCL_STATIC);
	return TCL_OK;

    } else if ((c == 'i') && (strncmp(argv[1], "isGoodCharset", length) == 0)
	    && (length > 1)) {
	PARAMETER *parameter;
	char *charset = "us-ascii", *alias;

	for (parameter = bodyPtr->parameter; parameter;
		parameter = parameter->next) {
	    if ( 0 == strcasecmp("charset", parameter->attribute)) {
		charset = parameter->value;
		break;
	    }
	}
	if ((alias = Tcl_GetVar2(interp, "charsetAlias", charset,
		TCL_GLOBAL_ONLY))) {
	    charset = alias;
	}
	if (RatGetEncoding(interp, charset)) {
	    Tcl_SetResult(interp, "1", TCL_STATIC);
	} else {
	    Tcl_SetResult(interp, "0", TCL_STATIC);
	}
	return TCL_OK;

    } else if ((c == 'd') && (strncmp(argv[1], "data", length) == 0)
	    && (length > 1)) {
	char *isCharset;
	int encoded;

	if (3 != argc && 4 != argc) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " data encoded [charset]\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (TCL_OK != Tcl_GetBoolean(interp, argv[2], &encoded)) {
	    return TCL_ERROR;
	}
	if (argc == 4) {
	    isCharset = argv[3];
	} else {
	    isCharset = NULL;
	}
	Tcl_SetObjResult(interp,
		RatBodyData(interp, bodyInfoPtr, encoded, isCharset));
	return TCL_OK;


    } else if ((c == 's') && (strncmp(argv[1], "saveData", length) == 0)
	    && (length > 1)) {
	int encoded, convertNL;
	Tcl_Channel channel;

	if (5 != argc) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " saveData fileId encoded convertNL\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (NULL == (channel = Tcl_GetChannel(interp, argv[2], NULL))) {
	    return TCL_ERROR;
	}
	if (TCL_OK != Tcl_GetBoolean(interp, argv[3], &encoded)) {
	    return TCL_ERROR;
	}
	if (TCL_OK != Tcl_GetBoolean(interp, argv[4], &convertNL)) {
	    return TCL_ERROR;
	}
	return RatBodySave(interp, channel, bodyInfoPtr, encoded, convertNL);

    } else if ((c == 'd') && (strncmp(argv[1], "dsn", length) == 0)
	    && (length > 1)) {
	char *body;
	Tcl_Obj *oPtr;
	int r;

	if (TYPEMESSAGE != bodyPtr->type &&
		!strcasecmp(bodyPtr->subtype, "delivery-status")) {
	    Tcl_SetResult(interp, "Not an message/delivery-status bodypart",
		    TCL_STATIC);
	    return TCL_ERROR;
	}
	body = (*messageProcInfo[bodyInfoPtr->type].fetchBodyProc)
		(bodyInfoPtr, &length);
	if (*body) {
	    oPtr = Tcl_NewStringObj(body, length);
	    r = RatDSNExtract(interp, oPtr);
	    Tcl_DecrRefCount(oPtr);
	    return r;
	} else {
	    Tcl_SetResult(interp, "No body", TCL_STATIC);
	    return TCL_ERROR;
	}

    } else if ((c == 'g') && (strncmp(argv[1], "getShowCharset", length) == 0)
	    && (length > 7)) {
	char *charset = "us-ascii", *alias;
	PARAMETER *parmPtr;

	if (TYPETEXT != bodyPtr->type) {
	    Tcl_AppendElement(interp, "good");
	    Tcl_AppendElement(interp, "us-ascii");
	    return TCL_OK;
	}
	for (parmPtr = bodyPtr->parameter; parmPtr; parmPtr = parmPtr->next) {
	    if (!strcasecmp(parmPtr->attribute, "charset")) {
		charset = parmPtr->value;
		break;
	    }
	}

	charset = cpystr(charset);
	lcase(charset);

	/*
	 * - See if this charset is an alias and resolve that if so.
	 * - Check if this is a charset we do have a font for
	 *   return good and the charset in that case.
	 * - Check if this is a character set we know about and convert.
	 *   return lose if that is the case.
	 * - return none.
	 */
	if ((alias=Tcl_GetVar2(interp,"charsetAlias",charset,TCL_GLOBAL_ONLY))){
	    ckfree(charset);
	    charset = cpystr(alias);
	}
	if (Tcl_GetVar2(interp, "fontEncoding", charset, TCL_GLOBAL_ONLY)) {
	    Tcl_ResetResult(interp);
	    Tcl_AppendElement(interp, "good");
	    Tcl_AppendElement(interp, charset);
	    ckfree(charset);
	    return TCL_OK;
	}
	/*
	 * This converting part is not implemented yet
	 */
	Tcl_ResetResult(interp);
	Tcl_AppendElement(interp, "none");
	Tcl_AppendElement(interp, "");
	ckfree(charset);
	return TCL_OK;

    } else if ((c == 'f') && (strncmp(argv[1], "findShowCommand", length) == 0)
	    && (length > 1)) {
	return RatMcapFindCmd(interp, bodyInfoPtr);

    } else if ((c == 'f') && (strncmp(argv[1], "filename", length) == 0)
	    && (length > 1)) {
	PARAMETER *parmPtr;
	char *filename = NULL, *delim;

	for (parmPtr = bodyPtr->disposition.parameter; parmPtr;
		parmPtr = parmPtr->next) {
	    if (!strcasecmp(parmPtr->attribute, "filename")
		    || !strcasecmp(parmPtr->attribute, "name")) {
		filename = parmPtr->value;
		break;
	    }
	}
	for (parmPtr = bodyPtr->parameter; !filename && parmPtr;
		parmPtr = parmPtr->next) {
	    if (!strcasecmp(parmPtr->attribute, "filename")
		    || !strcasecmp(parmPtr->attribute, "name")) {
		filename = parmPtr->value;
		break;
	    }
	}
	if (!filename && bodyPtr->description
		&& !strchr(bodyPtr->description, ' ')) {
	    filename = bodyPtr->description;
	}
	if (filename) {
	    delim = strrchr(filename, '/');
	    if (delim) {
		Tcl_SetResult(interp, delim+1, TCL_VOLATILE);
	    } else {
		Tcl_SetResult(interp, filename, TCL_VOLATILE);
	    }
	}
	return TCL_OK;

    } else if ((c == 'e') && (strncmp(argv[1], "encoded", length) == 0)
	    && (length > 1)) {
	Tcl_SetObjResult(interp, Tcl_NewIntObj(bodyInfoPtr->encoded));
	return TCL_OK;

    } else if ((c == 's') && (strncmp(argv[1], "sigstatus", length) == 0)
	    && (length > 1)) {
	char *status = NULL;

	switch (bodyInfoPtr->sigStatus) {
	case RAT_UNSIGNED:	status = "pgp_none"; break;
	case RAT_UNCHECKED:	status = "pgp_unchecked"; break;
	case RAT_SIG_GOOD:	status = "pgp_good"; break;
	case RAT_SIG_BAD:	status = "pgp_bad"; break;
	}
	Tcl_SetResult(interp, status, TCL_STATIC);
	return TCL_OK;

    } else if ((c == 'c') && (strncmp(argv[1], "checksig", length) == 0)
	    && (length > 2)) {
	RatPGPChecksig(interp, messageProcInfo, bodyInfoPtr);
	return TCL_OK;

    } else if ((c == 'g') && (strncmp(argv[1], "getPGPOutput", length) == 0)
	    && (length > 3)) {
	if (bodyInfoPtr->pgpOutput) {
	    Tcl_SetResult(interp, Tcl_DStringValue(bodyInfoPtr->pgpOutput),
		    TCL_VOLATILE);
	} else {
	    Tcl_ResetResult(interp);
	}
	return TCL_OK;

    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[1],
		"\": must be one of children, message, type, parameter, id",
		", description, data, saveData, dsn, getShowCharset",
		", getShowCommand, filename, encoded, sigstatus, checksig",
		" or getPGPOutput", NULL);
	return TCL_ERROR;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RatBodyDelete --
 *
 *      Deletes the given body.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The bodypart and all its siblings are deleted from the interpreter
 *	and all the structures are freed.
 *
 *
 *----------------------------------------------------------------------
 */

static void
RatBodyDelete(Tcl_Interp *interp, BodyInfo *bodyInfoPtr)
{
    BodyInfo *siblingInfoPtr, *nextSiblingInfoPtr;
    Tcl_DeleteCommand(interp, bodyInfoPtr->cmdName);
    siblingInfoPtr = bodyInfoPtr->firstbornPtr;
    (*messageProcInfo[bodyInfoPtr->type].bodyDeleteProc)(bodyInfoPtr);
    while (siblingInfoPtr) {
	nextSiblingInfoPtr = siblingInfoPtr->nextPtr;
	RatBodyDelete(interp, siblingInfoPtr);
	siblingInfoPtr = nextSiblingInfoPtr;
    }
    if (bodyInfoPtr->containedEntity) {
	RatMessageDelete(interp, bodyInfoPtr->containedEntity);
    }
    if (bodyInfoPtr->pgpOutput) {
	ckfree(bodyInfoPtr->pgpOutput);
    }
    ckfree(bodyInfoPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * RatMessageGet --
 *
 *      Retrieves a message in textual form. The text is placed in the
 *	supplied Tcl_DString.
 *
 * Results:
 *	No result.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatMessageGet(Tcl_Interp *interp, MessageInfo *msgPtr, Tcl_DString *ds,
	      char *flags, size_t flaglen, char *date, size_t datelen)
{
    char *data;
    int seen;
    Tcl_Obj *oPtr;

    data = (*messageProcInfo[msgPtr->type].getHeadersProc)(interp, msgPtr);
    Tcl_DStringAppend(ds, data, strlen(data));
    if (msgPtr->folderInfoPtr) {
	seen  = (*msgPtr->folderInfoPtr->getFlagProc)(msgPtr->folderInfoPtr,
		interp, msgPtr->msgNo, RAT_SEEN);
    } else {
	seen = 1;
    }
    Tcl_DStringAppend(ds, "\n", 1);
    data = (*messageProcInfo[msgPtr->type].fetchTextProc)(interp, msgPtr);
    Tcl_DStringAppend(ds, data, strlen(data));
    if (!seen) {
        (*msgPtr->folderInfoPtr->setFlagProc)(msgPtr->folderInfoPtr,
	    interp, msgPtr->msgNo, RAT_SEEN, 0);
    }
    if (flags) {
	oPtr = (*messageProcInfo[msgPtr->type].getInfoProc)(interp,
		(ClientData)msgPtr, RAT_FOLDER_FLAGS, 0);
	RatStrNCpy(flags, Tcl_GetString(oPtr), flaglen);
	oPtr = (*messageProcInfo[msgPtr->type].getInfoProc)(interp,
		(ClientData)msgPtr, RAT_FOLDER_DATE_IMAP4, 0);
	RatStrNCpy(date, Tcl_GetString(oPtr), datelen);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RatInsertCmd --
 *
 *      Inserts the given message into the database
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatInsertCmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
    Tcl_CmdInfo cmdInfo;
    MessageInfo *msgPtr;

    if (argc != 5) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" msgId keywords exDate exType\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (0 == Tcl_GetCommandInfo(interp, argv[1], &cmdInfo)) {
	Tcl_AppendResult(interp, "No such message: ", argv[1], NULL);
	return TCL_ERROR;
    }
    msgPtr = (MessageInfo*)cmdInfo.objClientData;
    return RatInsertMsg(interp, msgPtr, argv[2], argv[3], argv[4]);
}


/*
 *----------------------------------------------------------------------
 *
 * RatInsertMsg --
 *
 *      Inserts the given message into the database
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatInsertMsg (Tcl_Interp *interp, MessageInfo *msgPtr, char *keywords,
	char *exDate, char *exType)
{
    char *to, *from, *cc, *subject, *msgid, *ref, *flags, **listArgv,
	    **elemArgv;
    MESSAGECACHE elt;
    int listArgc, elemArgc;
    char *eFrom, *header, *body, *s, *e, *d;
    Tcl_DString dString;
    int result, i;
    struct tm tm;
    time_t date = 0, exTime;
    Tcl_Obj *oPtr;

    to = from = cc = subject = msgid = ref = flags = NULL;
    if (TCL_OK != RatMessageGetHeader(interp,
	    (*messageProcInfo[msgPtr->type].getHeadersProc)(interp, msgPtr))) {
	return TCL_ERROR;
    }
    Tcl_SplitList(interp, Tcl_GetStringResult(interp), &listArgc, &listArgv);
    for (i=0; i<listArgc; i++) {
	Tcl_SplitList(interp, listArgv[i], &elemArgc, &elemArgv);
	if (!strcasecmp(elemArgv[0], "to")) {
	    to = cpystr(elemArgv[1]);
	} else if (!strcasecmp(elemArgv[0], "from")) {
	    from = cpystr(elemArgv[1]);
	} else if (!strcasecmp(elemArgv[0], "cc")) {
	    cc = cpystr(elemArgv[1]);
	} else if (!strcasecmp(elemArgv[0], "subject")) {
	    subject = cpystr(elemArgv[1]);
	} else if (!strcasecmp(elemArgv[0], "message-id")) {
	    msgid = cpystr(elemArgv[1]);
	} else if (!strcasecmp(elemArgv[0], "references")
		&& !ref
		&& (s = strchr(elemArgv[1], '<'))
		&& (e = strchr(s, '>'))) {
	    ref = (char*)ckalloc(e-s+1);
	    RatStrNCpy(ref, s, e-s+1);
	} else if (!strcasecmp(elemArgv[0], "in-reply-to")
		&& (s = strchr(elemArgv[1], '<'))
		&& (e = strchr(s, '>'))) {
	    if (ref) {
		ckfree(ref);
	    }
	    ref = (char*)ckalloc(e-s+1);
	    RatStrNCpy(ref, s, e-s+1);
	    ref = cpystr(elemArgv[1]);
	} else if (!strcasecmp(elemArgv[0], "status") ||
		   !strcasecmp(elemArgv[0], "x-status")) {
	    if (flags) {
		flags = (char*)ckrealloc(flags,
			strlen(flags)+strlen(elemArgv[1])+1);
		strcpy(&flags[strlen(flags)], elemArgv[1]);
	    } else {
		flags = cpystr(elemArgv[1]);
	    }
	} else if (!strcasecmp(elemArgv[0], "date")) {
	    if (T == mail_parse_date(&elt, elemArgv[1])) {
		tm.tm_sec = elt.seconds;
		tm.tm_min = elt.minutes;
		tm.tm_hour = elt.hours;
		tm.tm_mday = elt.day;
		tm.tm_mon = elt.month - 1;
		tm.tm_year = elt.year+70;
		tm.tm_wday = 0;
		tm.tm_yday = 0;
		tm.tm_isdst = -1;
		date = (int)mktime(&tm);
	    } else {
		date = 0;
	    }
	}
	ckfree(elemArgv);
    }
    ckfree(listArgv);
    if (flags) {
	for (s = d = flags; *s; s++) {
	    if ('D' != *s && 'F' != *s) {
		*d++ = *s;
	    }
	}
	*d = '\0';
    } else {
	oPtr = (*messageProcInfo[msgPtr->type].getInfoProc)(interp,
		(ClientData)msgPtr, RAT_FOLDER_UNIXFLAGS, 0);
	flags = Tcl_GetString(oPtr);
    }
    if (0 == date) {
	oPtr = (*messageProcInfo[msgPtr->type].getInfoProc)(interp,
		(ClientData)msgPtr, RAT_FOLDER_DATE_N, 0);
	Tcl_GetLongFromObj(interp, oPtr, &date);
    }
    Tcl_DStringInit(&dString);
    eFrom = (*messageProcInfo[msgPtr->type].getEnvelopeProc)(interp, msgPtr);
    header = (*messageProcInfo[msgPtr->type].getHeadersProc)(interp, msgPtr);
    Tcl_DStringAppend(&dString, header, strlen(header));
    body = (*messageProcInfo[msgPtr->type].fetchTextProc)(interp, msgPtr);
    Tcl_DStringAppend(&dString, body, strlen(body));
    Tcl_ResetResult(interp);
    exTime = atol(exDate);
    if (!strcmp("none", exType)) {
	exTime = 0;
    }
    result = RatDbInsert(interp, to, from, cc, msgid, ref, subject, date,
	    flags, keywords, exTime, exType, eFrom, Tcl_DStringValue(&dString),
	    Tcl_DStringLength(&dString));
    Tcl_DStringFree(&dString);
    if (to) ckfree(to);
    if (from) ckfree(from);
    if (cc) ckfree(cc);
    if (msgid) ckfree(msgid);
    if (ref) ckfree(ref);
    if (subject) ckfree(subject);
    if (flags) ckfree(flags);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * RatParseList --
 *
 *      Parse a list expression (almost like a printf format string)
 *
 * Results:
 *	A structure representing the parsed expression, or null if
 *	there is a syntax error in the format string.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

ListExpression*
RatParseList(char *format)
{
    ListExpression *expPtr;
    int i, w, expIndex, bufLen, num;
    char buf[1024];

    for(i=num=0; '\0' != format[i]; i++) {
	if ('%' == format[i] && format[i+1] && '%' != format[i+1]) {
	    while (format[++i] && ('-' == format[i]
		    || isdigit((unsigned char)format[i])));
	    if (!strchr("snmrRbBdDSit", format[i])) {
		return NULL;
	    }
	    num++;
	}
    }
    expPtr = (ListExpression*)ckalloc(sizeof(ListExpression));
    expPtr->size = num;
    expPtr->preString = (char**)ckalloc(num*sizeof(char*));
    expPtr->typeList=(RatFolderInfoType*)ckalloc(num*sizeof(RatFolderInfoType));
    expPtr->fieldWidth = (int*)ckalloc(num*sizeof(int));
    expPtr->leftJust = (int*)ckalloc(num*sizeof(int));
    for (i = expIndex = bufLen = 0; format[i]; i++) {
	if ('%' == format[i]) {
	    if ('%' == format[++i]) {
		buf[bufLen++] = format[i];
		continue;
	    }
	    buf[bufLen] = '\0';
	    expPtr->preString[expIndex] = cpystr(buf);
	    if ('-' == format[i]) {
		expPtr->leftJust[expIndex] = 1;
		i++;
	    } else {
		expPtr->leftJust[expIndex] = 0;
	    }
	    w=0;
	    while (isdigit((unsigned char)format[i])) {
		w = w*10+format[i++]-'0';
	    }
	    expPtr->fieldWidth[expIndex] = w;
	    switch(format[i]) {
	    case 's': expPtr->typeList[expIndex++] = RAT_FOLDER_SUBJECT; break;
	    case 'n': expPtr->typeList[expIndex++] = RAT_FOLDER_NAME; break;
	    case 'm': expPtr->typeList[expIndex++] = RAT_FOLDER_MAIL; break;
	    case 'r': expPtr->typeList[expIndex++] = RAT_FOLDER_NAME_RECIPIENT;
		    break;
	    case 'R': expPtr->typeList[expIndex++] = RAT_FOLDER_MAIL_RECIPIENT;
		    break;
	    case 'b': expPtr->typeList[expIndex++] = RAT_FOLDER_SIZE; break;
	    case 'B': expPtr->typeList[expIndex++] = RAT_FOLDER_SIZE_F; break;
	    case 'd': expPtr->typeList[expIndex++] = RAT_FOLDER_DATE_F; break;
	    case 'D': expPtr->typeList[expIndex++] = RAT_FOLDER_DATE_N; break;
	    case 'S': expPtr->typeList[expIndex++] = RAT_FOLDER_STATUS; break;
	    case 'i': expPtr->typeList[expIndex++] = RAT_FOLDER_INDEX; break;
	    case 't': expPtr->typeList[expIndex++] = RAT_FOLDER_THREADING;break;
	    }
	    bufLen = 0;
	} else {
	    buf[bufLen++] = format[i];
	}
    }
    if (bufLen) {
	buf[bufLen] = '\0';
	expPtr->postString = cpystr(buf);
    } else {
	expPtr->postString = NULL;
    }
    return expPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * RatFreeListExpression --
 *
 *      Frees all memory associated with a list expression.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Some memory is freed.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatFreeListExpression(ListExpression *exPtr)
{
    int i;

    for (i=0; i<exPtr->size; i++) {
	ckfree(exPtr->preString[i]);
    }
    ckfree(exPtr->preString);
    ckfree(exPtr->typeList);
    ckfree(exPtr->fieldWidth);
    ckfree(exPtr->leftJust);
    if (exPtr->postString) {
	ckfree(exPtr->postString);
    }
    ckfree(exPtr);
}


/*
 *----------------------------------------------------------------------
 *
 * RatDoList --
 *
 *      Print the list information about a message.
 *
 * Results:
 *	A tcl object
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

Tcl_Obj*
RatDoList(Tcl_Interp *interp, ListExpression *exprPtr, RatInfoProc *infoProc,
	ClientData clientData, int index)
{
    Tcl_Obj *oPtr = Tcl_NewObj(), *iPtr;
    char *str;
    int i, j, slen, length;

    for (i=0; i<exprPtr->size; i++) {
	if (exprPtr->preString[i]) {
	    Tcl_AppendToObj(oPtr, exprPtr->preString[i], -1);
	}
	iPtr = (*infoProc)(interp, clientData, exprPtr->typeList[i], index);
	if (!iPtr) {
	    for (j=0; j<exprPtr->fieldWidth[i]; j++) {
		Tcl_AppendToObj(oPtr, " ", 1);
	    }
	    continue;
	}
	if (exprPtr->fieldWidth[i]) {
	    str = Tcl_GetStringFromObj(iPtr, &slen);
	    length = Tcl_NumUtfChars(str, slen);
	    if (length > exprPtr->fieldWidth[i]) {
		j = Tcl_UtfAtIndex(str, exprPtr->fieldWidth[i]) - str;
		Tcl_AppendToObj(oPtr, str, j);
	    } else {
		if (exprPtr->leftJust[i]) {
		    Tcl_AppendObjToObj(oPtr, iPtr);
		    for (j=length; j<exprPtr->fieldWidth[i]; j++) {
			Tcl_AppendToObj(oPtr, " ", 1);
		    }
		} else {
		    for (j=length; j<exprPtr->fieldWidth[i]; j++) {
			Tcl_AppendToObj(oPtr, " ", 1);
		    }
		    Tcl_AppendObjToObj(oPtr, iPtr);
		}
	    }
	} else {
	    Tcl_AppendObjToObj(oPtr, iPtr);
	}
    }
    if (exprPtr->postString) {
	Tcl_AppendToObj(oPtr, exprPtr->postString, -1);
    }
    return oPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * RatMsgInfo --
 *
 *      get information about a message
 *
 * Results:
 *	A tcl object
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

Tcl_Obj*
RatMsgInfo(Tcl_Interp *interp, MessageInfo *msgPtr, RatFolderInfoType type)
{
    return (*messageProcInfo[msgPtr->type].getInfoProc)(interp,
	    (ClientData)msgPtr, type, 0);
}


/*
 *----------------------------------------------------------------------
 *
 * RatBodySave --
 *
 *      Save a bodypart to an open channel.
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatBodySave(Tcl_Interp *interp,Tcl_Channel channel, BodyInfo *bodyInfoPtr,
	    int encoded, int convertNL)
{
    BODY *bodyPtr = bodyInfoPtr->bodyPtr;
    char *body;
    int result = 0, i;
    unsigned long length;
    Tcl_DString *dsPtr = NULL;

    if (NULL == (body = (*messageProcInfo[bodyInfoPtr->type].fetchBodyProc)
	    (bodyInfoPtr, &length))) {
	Tcl_SetResult(interp, "[Body not available]\n", TCL_STATIC);
	return TCL_OK;
    }
    if (!encoded) {
	dsPtr = RatDecode(interp, bodyPtr->encoding, body, length, NULL);
	body =Tcl_DStringValue(dsPtr);
	length = Tcl_DStringLength(dsPtr);
    }
    if (convertNL) {
	/* 
	 * This isn't really elegant but since the channel is buffered
	 * we shouldn't suffer too badly.
	 */
	for (i=0; i<length && -1 != result; i++) {
	    if ('\r' == body[i] && '\n' == body[i+1]) {
		i++;
	    }
	    result = Tcl_Write(channel, &body[i], 1);
	}
    } else {
	result = Tcl_Write(channel, body, length);
    }
    if (!encoded) {
	Tcl_DStringFree(dsPtr);
	ckfree(dsPtr);
    }
    if (-1 == result) {
	Tcl_AppendResult(interp, "error writing : ",
		Tcl_PosixError(interp), (char *) NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * RatFindFirstText --
 *
 *      Finds the first text part of a message
 *
 * Results:
 *	A pointer to the BodyInfo of the first text part, or NULL if none
 *	are found.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

static BodyInfo*
RatFindFirstText(BodyInfo *bodyInfoPtr)
{
    BodyInfo *b2Ptr;

    for (; NULL != bodyInfoPtr; bodyInfoPtr = bodyInfoPtr->nextPtr) {
	if (TYPETEXT == bodyInfoPtr->bodyPtr->type) {
	    return bodyInfoPtr;
	}
	if (bodyInfoPtr->firstbornPtr && (NULL !=
		(b2Ptr = RatFindFirstText(bodyInfoPtr->firstbornPtr)))) {
	    return b2Ptr;
	}
    }
    return NULL;
}


/*
 *----------------------------------------------------------------------
 *
 * RatGetCitation --
 *
 *      Get the citation to use
 *
 * Results:
 *	A pointer to a citation string to use. This pointer will
 *	remain valid until the next call to this function.
 *	are found.
 *
 * Side effects:
 *	May call userproc.
 *
 *
 *----------------------------------------------------------------------
 */

static char*
RatGetCitation(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    Tcl_CmdInfo cmdInfo;
    static char citation[80];

    if (0 != Tcl_GetCommandInfo(interp, "RatUP_Citation", &cmdInfo)) {
	if (TCL_OK != Tcl_VarEval(interp,"RatUP_Citation ",msgPtr->name,NULL)) {
	    RatLog(interp, RAT_ERROR, Tcl_GetStringResult(interp),
		    RATLOG_EXPLICIT);
	    return "";
	}
	if (79 < strlen(Tcl_GetStringResult(interp))) {
	    RatLog(interp, RAT_ERROR, "Too long citation", RATLOG_EXPLICIT);
	    return "";
	}
	RatStrNCpy(citation, Tcl_GetStringResult(interp), sizeof(citation));
	return citation;
    }
    return Tcl_GetVar2(interp, "option", "reply_lead", TCL_GLOBAL_ONLY);
}

/*
 *----------------------------------------------------------------------
 *
 * RatCiteMessage --
 *
 *      Copy a message and add citation.
 *
 * Results:
 *	The cited text is appended to dstObjPtr
 *
 * Side effects:
 *	Modifies *dstObjPtr
 *
 *----------------------------------------------------------------------
 */

static void
RatCiteMessage(Tcl_Interp *interp, Tcl_Obj *dstObjPtr, char *src,
	char *myCitation)
{
    int i, skipSig, addCitBlank, myCitLength;
    char *srcPtr;

    /*
     * Initialize and find out desired behaviour
     */
    myCitLength = strlen(myCitation);
    if (' ' == myCitation[myCitLength-1]) {
	addCitBlank = 1;
	myCitLength--;
    } else {
	addCitBlank = 0;
    }
    Tcl_GetBoolean(interp, Tcl_GetVar2(interp, "option", "skip_sig",
	    TCL_GLOBAL_ONLY), &skipSig);

    /*
     * Go over the text and add citation
     */
    for (srcPtr = src; *srcPtr;) {
	/*
	 * Stop when encoutering signature (if that option is true)
	 */
	if (skipSig && '-'== srcPtr[0] && '-' == srcPtr[1] && ' ' == srcPtr[2]
		&& '\n' == srcPtr[3]) {
	    break;
	}

	Tcl_AppendToObj(dstObjPtr, myCitation, myCitLength);
	if ('>' != *srcPtr && addCitBlank) {
	    Tcl_AppendToObj(dstObjPtr, " ", 1);
	}
	for (i=0; '\n' != srcPtr[i] && srcPtr[i]; i++);
	Tcl_AppendToObj(dstObjPtr, srcPtr, i);
	srcPtr += i;
	if ('\n' == *srcPtr) {
	    Tcl_AppendToObj(dstObjPtr, "\n", 1);
	    srcPtr++;
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RatWrapMessage --
 *
 *      Wraps the text of a message
 *
 * Results:
 *	An opject containing the wrapped text is returned.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

Tcl_Obj*
RatWrapMessage(Tcl_Interp *interp, Tcl_Obj *oPtr)
{
    int wrapLength, l, citLength, overflow;
    char *s, *e, *cPtr, *lineStartPtr, *startPtr, *citPtr = NULL;
    Tcl_RegExp citexp;
    Tcl_Obj *nPtr = Tcl_NewObj();

    Tcl_GetInt(interp, Tcl_GetVar2(interp, "option", "wrap_length",
	    TCL_GLOBAL_ONLY), &wrapLength);
    s = Tcl_GetVar2(interp, "option", "citexp", TCL_GLOBAL_ONLY);
    citexp = Tcl_RegExpCompile(interp, s);
    if (NULL == citexp) {
	RatLogF(interp, RAT_ERROR, "illegal_regexp", RATLOG_EXPLICIT,
		Tcl_GetStringResult(interp));
    }

    for (cPtr = Tcl_GetString(oPtr); *cPtr;) {
	/*
	 * Check if this line needs to be wrapped
	 */
	startPtr = cPtr;
	for (l=0; l < wrapLength && '\n' != *cPtr && *cPtr;
		l++, cPtr = Tcl_UtfNext(cPtr));
	if (l < wrapLength) {
	    Tcl_AppendToObj(nPtr, startPtr, cPtr-startPtr);
	    if ('\n' == *cPtr) {
		Tcl_AppendToObj(nPtr, "\n", 1);
		cPtr++;
	    }
	    continue;
	}

	/*
	 * It should be wrapped, find citation
	 */
	if (citexp && Tcl_RegExpExec(interp, citexp, startPtr, startPtr)) {
	    Tcl_RegExpRange(citexp, 0, &s, &e);
	    if (s == startPtr) {
		citLength = e-s;
		citPtr = startPtr;
	    } else {
		citLength = 0;
	    }
	} else {
	    citLength = 0;
	}

	/*
	 * Add first part of line, linebreak and citation
	 */
	for (; !isspace(*cPtr) && cPtr > startPtr; cPtr--);
	Tcl_AppendToObj(nPtr, startPtr, cPtr-startPtr);
	Tcl_AppendToObj(nPtr, "\n", 1);
	Tcl_AppendToObj(nPtr, citPtr, citLength);

	/*
	 * Continue adding
	 */
	lineStartPtr = startPtr = ++cPtr;
	l = citLength;
	while (*cPtr) {
	    if ('\n' == *cPtr) {
		for (e = cPtr; isspace(*e) && e > startPtr; e--);
		if (e > startPtr) e++;
		Tcl_AppendToObj(nPtr, startPtr, e-startPtr);
		cPtr++;
		if (!strncmp(cPtr, citPtr, citLength)) {
		    for (s=cPtr+citLength; isspace(*s) && '\n' != *s && *s;s++);
		    if (*s != '\n') {
			Tcl_AppendToObj(nPtr, " ", 1);
			l++;
			cPtr += citLength;
			startPtr = cPtr;
			continue;
		    }
		}
		Tcl_AppendToObj(nPtr, "\n", 1);
		l = 0;
		break;
	    } else if (l >= wrapLength) {
		for (; !isspace(*cPtr) && cPtr > startPtr; cPtr--);
		l = overflow = 0;
		if (cPtr == startPtr && startPtr == lineStartPtr) {
		    while (!isspace(*cPtr)) cPtr++;
		    overflow = 1;
		}
		Tcl_AppendToObj(nPtr, startPtr, cPtr-startPtr);
		Tcl_AppendToObj(nPtr, "\n", 1);
		if (startPtr != cPtr) {
		    cPtr++;
		}
		lineStartPtr = startPtr = cPtr;
		if (overflow) break;
		Tcl_AppendToObj(nPtr, citPtr, citLength);
		l += citLength;
	    } else {
		l++;
		cPtr = Tcl_UtfNext(cPtr);
	    }
	}
    }

    return nPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * RatBodyType --
 *
 *      Gets the types of a bodypart
 *
 * Results:
 *	A list object containing two strings, the first is the major
 *	type and the second is the subtype.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
Tcl_Obj*
RatBodyType(BodyInfo *bodyInfoPtr)
{
    BODY *bodyPtr = bodyInfoPtr->bodyPtr;
    Tcl_Obj *oPtr[2];

    oPtr[0] = Tcl_NewStringObj(body_types[bodyPtr->type], -1);
    if (bodyPtr->subtype) {
	oPtr[1] = Tcl_NewStringObj(bodyPtr->subtype, -1);
    } else {
	oPtr[1] = Tcl_NewStringObj("", 0);
    }
    return Tcl_NewListObj(2, oPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * RatBodyData --
 *
 *      Gets the content of a bodypart
 *
 * Results:
 *	An object containing the data.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
Tcl_Obj*
RatBodyData(Tcl_Interp *interp, BodyInfo *bodyInfoPtr, int encoded,
	char *charset)
{
    BODY *bodyPtr = bodyInfoPtr->bodyPtr;
    Tcl_Obj *oPtr;
    char *body, *isCharset = NULL, *alias;
    PARAMETER *parameter;
    unsigned long length;

    if (charset) {
	isCharset = charset;
    } else if (TYPETEXT == bodyPtr->type){
	isCharset = "us-ascii";
	for (parameter = bodyPtr->parameter; parameter;
		parameter = parameter->next) {
	    if ( 0 == strcasecmp("charset", parameter->attribute)) {
		isCharset = parameter->value;
	    }
	}
	if ((alias = Tcl_GetVar2(interp, "charsetAlias", isCharset,
		TCL_GLOBAL_ONLY))) {
	    isCharset = alias;
	}
    }

    body = (*messageProcInfo[bodyInfoPtr->type].fetchBodyProc)
	    (bodyInfoPtr, &length);
    if (body) {
	if (encoded) {
	    Tcl_Encoding enc;
	    Tcl_DString ds;

	    Tcl_DStringInit(&ds);
	    if (ENC8BIT == bodyPtr->encoding) {
		enc = RatGetEncoding(interp, isCharset);
		Tcl_ExternalToUtfDString(enc, body, length, &ds);
	    } else {
		 Tcl_DStringAppend(&ds, body, length);
	    }
	    oPtr = Tcl_NewStringObj(Tcl_DStringValue(&ds),
				    Tcl_DStringLength(&ds));
	    Tcl_DStringFree(&ds);
	} else {
	    Tcl_DString *dsPtr = RatDecode(interp, bodyPtr->encoding,
		    body, length, isCharset);
	    oPtr = Tcl_NewStringObj(Tcl_DStringValue(dsPtr),
				    Tcl_DStringLength(dsPtr));
	    Tcl_DStringFree(dsPtr);
	    ckfree(dsPtr);
	}
    } else {
	oPtr = Tcl_NewStringObj("[Body not available]\n", -1);
    }
    return oPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * RatMessageInternalDate --
 *
 *      Gets the internal date of a message
 *
 * Results:
 *	A pointer to a MESSAGECACHE entry where only the date-fields may
 *	be used. May return NULL on errors.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
MESSAGECACHE*
RatMessageInternalDate(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    return (*messageProcInfo[msgPtr->type].getInternalDateProc)(interp, msgPtr);
}

#ifdef MEM_DEBUG
void ratMessageCleanup()
{
    ckfree(messageProcInfo);
}
#endif /* MEM_DEBUG */
