/*
 * ratAddress.c --
 *
 *	This file contains basic support for handling addresses.
 *
 * TkRat software and its included text is Copyright 1996-1999 by
 * Martin Forssn
 *
 * The full text of the legal notice is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

#include "rat.h"

#define PADDED(n)	(n+(4-((int)(n)%4))%4)

/*
 * This struct defines an alias.
 */
typedef struct {
    Tcl_Obj *book;	/* Address book this alias comes from */
    Tcl_Obj *fullname;	/* Long name of alias (phrase part) */
    Tcl_Obj *content;	/* Content that alias expands to */
    Tcl_Obj *comment;	/* Comment to alias */
    unsigned int flags;	/* Option flags */
    unsigned long mark;	/* Id of last use */
} AliasInfo;

#define ALIAS_FLAG_ISLIST	(1<<0)
#define ALIAS_FLAG_NOFULLNAME	(1<<1)

/*
 * This table contains all the aliases.
 */
Tcl_HashTable aliasTable;

/*
 * The number of address entities created. This is used to create new
 * unique command names.
 */
static int numAddresses = 0;

/*
 * A mark used to prevent loops when resolving aliases
 */
static unsigned long aliasMark = 0;

/*
 * Internal functions
 */
static int AddressClean(Tcl_Obj *aPtr);

#ifdef MEM_DEBUG
static char *mem_store;
#endif /* MEM_DEBUG */


/*
 *----------------------------------------------------------------------
 *
 * AddressClean --
 *
 *      Clean an address list by removing all whitespace around addresses
 *
 * Results:
 *	The number of addresses contained in the object
 *
 * Side effects:
 *	Modifies the given object
 *
 *
 *----------------------------------------------------------------------
 */

static int
AddressClean(Tcl_Obj *aPtr)
{
    char *mark, *dst, *src, *new;
    int quoted = 0;
    int skip = 1, length, num = 1;

    src = Tcl_GetStringFromObj(aPtr, &length);
    new = dst = mark = (char*)ckalloc(length);
    for (; *src; src++) {
	if ('\\' == *src) {
	    *dst++ = *src++;
	} else if ('"' == *src) {
	    skip = 0;
	    if (quoted) {
		quoted = 0;
	    } else {
		quoted = 1;
	    }
	} else if (!quoted) {
	    if (',' == *src) {
		num++;
		dst = mark;
		skip = 1;
	    } else if (isspace((unsigned char)*src)) {
		if (skip) {
		    continue;
		}
	    } else {
		mark = dst+1;
		skip = 0;
	    }
	}
	*dst++ = *src;
    }
    if (0 == mark-new) {
	num = 0;
    }
    Tcl_SetStringObj(aPtr, new, mark-new);
    ckfree(new);
    return num;
}



/*
 *----------------------------------------------------------------------
 *
 * RatCreateAddressCmd --
 *
 *      This routine creates an address command by an address given
 *	as argument
 *
 * Results:
 *	A list of address entity names is appended to the result
 *
 * Side effects:
 *	New address entities are created,
 *
 *
 *----------------------------------------------------------------------
 */

int
RatCreateAddressCmd(ClientData clientData, Tcl_Interp *interp, int objc,
	Tcl_Obj *CONST objv[])
{
    ADDRESS *adrPtr = NULL;
    char *s;

    if (objc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		Tcl_GetString(objv[0]), " address\"", (char *) NULL);
	return TCL_ERROR;
    }

    s = cpystr(Tcl_GetString(objv[1]));
    rfc822_parse_adrlist(&adrPtr, s, currentHost);
    ckfree(s);
    RatEncodeAddresses(interp, adrPtr);
    RatInitAddresses(interp, adrPtr);
    mail_free_address(&adrPtr);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * RatInitAddresses --
 *
 *      This routine takes an address list as argument and constructs a
 *	list of address entities of it.
 *
 * Results:
 *	A list of address entity names is appended to the result
 *
 * Side effects:
 *	New address entities are created,
 *
 *
 *----------------------------------------------------------------------
 */

void
RatInitAddresses(Tcl_Interp *interp, ADDRESS *addressPtr)
{
    ADDRESS *adrPtr, *newPtr;
    Tcl_DString result;
    char name[32];

    Tcl_DStringInit(&result);
    Tcl_DStringGetResult(interp, &result);
    for (adrPtr = addressPtr; adrPtr; adrPtr = adrPtr->next) {
	newPtr = mail_newaddr();
	if (adrPtr->personal)	{
	    newPtr->personal =
		    cpystr(RatDecodeHeader(interp, adrPtr->personal, 0));
	}
	if (adrPtr->adl)	newPtr->adl = cpystr(adrPtr->adl);
	if (adrPtr->mailbox)	newPtr->mailbox = cpystr(adrPtr->mailbox);
	if (adrPtr->host)	newPtr->host = cpystr(adrPtr->host);
	if (adrPtr->error)	newPtr->error = cpystr(adrPtr->error);
	sprintf(name, "RatAddress%d", numAddresses++);
	Tcl_CreateCommand(interp, name, RatAddress, (ClientData) newPtr,
		RatDeleteAddress);
	Tcl_DStringAppendElement(&result, name);
    }
    Tcl_DStringResult(interp, &result);
}

/*
 *----------------------------------------------------------------------
 *
 * RatAddress --
 *
 *      This routine handles the address entity commands. See ../doc/interface
 *	for a documentation of them.
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *	May be some
 *
 *
 *----------------------------------------------------------------------
 */

int
RatAddress(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
    ADDRESS *adrPtr = (ADDRESS*)clientData;
    Tcl_Obj *oPtr;
    int useup;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" option ?arg?\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (!strcmp(argv[1], "isMe")) {
	if (3 == argc) {
	    Tcl_GetBoolean(interp, argv[2], &useup);
	} else {
	    useup = 1;
	}
	if (RatAddressIsMe(interp, adrPtr, useup)) {
	    Tcl_SetResult(interp, "1", TCL_STATIC);
	} else {
	    Tcl_SetResult(interp, "0", TCL_STATIC);
	}
	return TCL_OK;
	
    } else if (!strcmp(argv[1], "compare")) {
	Tcl_CmdInfo info;
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " compare address\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (0 == Tcl_GetCommandInfo(interp, argv[2], &info)) {
	    Tcl_AppendResult(interp, "there is no address entity \"",
		    argv[2], "\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (RatAddressCompare(adrPtr, (ADDRESS*)info.clientData)) {
	    Tcl_SetResult(interp, "1", TCL_STATIC);
	} else {
	    Tcl_SetResult(interp, "0", TCL_STATIC);
	}
	return TCL_OK;

    } else if (!strcmp(argv[1], "set")) {
	if (argc != 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " set personal name host\"", (char *) NULL);
	    return TCL_ERROR;
	}
	ckfree(adrPtr->mailbox);
	if (adrPtr->personal) {
	    ckfree(adrPtr->personal);
	}
	if (adrPtr->host) {
	    ckfree(adrPtr->host);
	}
	adrPtr->personal = cpystr(argv[2]);
	adrPtr->mailbox = cpystr(argv[3]);
	adrPtr->host = cpystr(argv[4]);
	return TCL_OK;

    } else if (!strcmp(argv[1], "get")) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " get form\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (!strcasecmp(argv[2], "rfc822")) {
	    if (adrPtr->personal) {
		char *personal;

		oPtr = Tcl_NewStringObj(adrPtr->personal, -1);
		personal = RatEncodeHeaderLine(interp, oPtr, 0);
		Tcl_DecrRefCount(oPtr);
		oPtr = Tcl_NewObj();
		Tcl_AppendStringsToObj(oPtr, personal, " <", NULL);
		Tcl_AppendToObj(oPtr, RatAddressMail(adrPtr), -1);
		Tcl_AppendToObj(oPtr, ">", 1);
		Tcl_SetObjResult(interp, oPtr);
		ckfree(personal);
	    } else {
		Tcl_SetResult(interp, RatAddressMail(adrPtr), TCL_VOLATILE);
	    }
	    return TCL_OK;

	} else if (!strcasecmp(argv[2], "mail")) {
	    Tcl_SetResult(interp, RatAddressMail(adrPtr), TCL_VOLATILE);
	    return TCL_OK;

	} else if (!strcasecmp(argv[2], "name")) {
	    if (adrPtr->personal) {
		Tcl_SetResult(interp, adrPtr->personal, TCL_VOLATILE);
	    }
	    return TCL_OK;

	} else {
	    Tcl_AppendResult(interp, "bad form \"", argv[2],
		    "\": must be one of rfc822, mail or name", (char *) NULL);
	    return TCL_ERROR;
	}
    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[1],
		"\": must be one of isMe, compare, set or get", (char *) NULL);
	return TCL_ERROR;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RatDeleteAddress --
 *
 *      Frees the client data of an address entity.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatDeleteAddress(ClientData clientData)
{
    ADDRESS *adrPtr = (ADDRESS*)clientData;
    if (adrPtr->personal)	ckfree(adrPtr->personal);
    if (adrPtr->adl)		ckfree(adrPtr->adl);
    if (adrPtr->mailbox)	ckfree(adrPtr->mailbox);
    if (adrPtr->host)		ckfree(adrPtr->host);
    if (adrPtr->error)		ckfree(adrPtr->error);
    ckfree(adrPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * RatAddressIsMe --
 *
 *      Checks if the address points to me.
 *
 * Results:
 *	If it is then non zero is returned otherwise zero.
 *
 * Side effects:
 *	May initialize the current* variables
 *
 *
 *----------------------------------------------------------------------
 */

int
RatAddressIsMe(Tcl_Interp *interp, ADDRESS *adrPtr, int useUP)
{
    char *from;

    if (adrPtr == NULL) {
	return 0;
    }
    if (adrPtr->mailbox && !strcasecmp(adrPtr->mailbox, currentMailboxName)
	    && (adrPtr->host && !strcasecmp(adrPtr->host, currentHost))) {
	return 1;
    }
    if (NULL != (from = Tcl_GetVar2(interp,"option","from",TCL_GLOBAL_ONLY))
	    && *from != '\0') {
	ADDRESS *a = NULL;
	char *s = cpystr(from);
	rfc822_parse_adrlist(&a, s, currentHost);
	ckfree(s);
	if (a && adrPtr->mailbox && adrPtr->host
		&& !strcasecmp(a->mailbox, adrPtr->mailbox)
		&& !strcasecmp(a->host, adrPtr->host)) {
	    mail_free_address(&a);
	    return 1;
	}
	mail_free_address(&a);
    }
    if (useUP) {
	Tcl_CmdInfo cmdInfo;
	if (Tcl_GetCommandInfo(interp, "RatUP_IsMe", &cmdInfo)) {
	    Tcl_DString cmd;
	    int isMe;
	    Tcl_Obj *oPtr;

	    Tcl_DStringInit(&cmd);
	    Tcl_DStringAppendElement(&cmd, "RatUP_IsMe");
	    Tcl_DStringAppendElement(&cmd,adrPtr->mailbox?adrPtr->mailbox:"");
	    Tcl_DStringAppendElement(&cmd,adrPtr->host?adrPtr->host:"");
	    Tcl_DStringAppendElement(&cmd,adrPtr->personal?adrPtr->personal:"");
	    Tcl_DStringAppendElement(&cmd,adrPtr->adl?adrPtr->adl:"");
	    if (TCL_OK == Tcl_Eval(interp, Tcl_DStringValue(&cmd))
	    	    && (oPtr = Tcl_GetObjResult(interp))
		    && TCL_OK == Tcl_GetBooleanFromObj(interp, oPtr, &isMe)) {
		Tcl_DStringFree(&cmd);
		return isMe;
	    }
	    Tcl_DStringFree(&cmd);
	}
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * RatAddressCompare --
 *
 *      Check if two addresses are equal.
 *
 * Results:
 *	If they are then zero is returned otherwise non zero.
 *
 * Side effects:
 *	May initialize the current* variables
 *
 *
 *----------------------------------------------------------------------
 */

int
RatAddressCompare(ADDRESS *adr1Ptr, ADDRESS *adr2Ptr)
{
    if (((adr1Ptr->mailbox && adr2Ptr->mailbox
		&& !strcasecmp(adr1Ptr->mailbox, adr2Ptr->mailbox))
  	      || adr1Ptr->mailbox == adr2Ptr->mailbox)
   	    && ((adr1Ptr->host && adr2Ptr->host
		&& !strcasecmp(adr1Ptr->host, adr2Ptr->host))
	      || adr1Ptr->host == adr2Ptr->host)) {
	return 0;
    } else {
	return 1;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RatAddressTranslate --
 *
 *      Let the user do their translation of this address.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The address may be affected.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatAddressTranslate(Tcl_Interp *interp, ADDRESS *adrPtr)
{
    Tcl_CmdInfo cmdInfo;
    Tcl_DString cmd;
    Tcl_Obj *oPtr, *lPtr;
    char **destPtrPtr = NULL, *s;
    int argc, i;

    if (!Tcl_GetCommandInfo(interp, "RatUP_Translate", &cmdInfo)) {
	return;
    }
    Tcl_DStringInit(&cmd);
    Tcl_DStringAppendElement(&cmd, "RatUP_Translate");
    Tcl_DStringAppendElement(&cmd,adrPtr->mailbox?adrPtr->mailbox:"");
    Tcl_DStringAppendElement(&cmd,adrPtr->host?adrPtr->host:"");
    Tcl_DStringAppendElement(&cmd,adrPtr->personal?adrPtr->personal:"");
    Tcl_DStringAppendElement(&cmd,adrPtr->adl?adrPtr->adl:"");
    if (TCL_OK != Tcl_Eval(interp, Tcl_DStringValue(&cmd))
	    || !(lPtr = Tcl_GetObjResult(interp))
    	    || TCL_OK != Tcl_ListObjLength(interp, lPtr, &argc)
	    || 4 != argc) {
	RatLogF(interp, RAT_ERROR, "translate_error", RATLOG_TIME,
		Tcl_DStringValue(&cmd));
    } else {
	for (i=0; i<4; i++) {
	    switch(i) {
		case 0: destPtrPtr = &adrPtr->mailbox; break;
		case 1: destPtrPtr = &adrPtr->host; break;
		case 2: destPtrPtr = &adrPtr->personal; break;
		case 3: destPtrPtr = &adrPtr->adl; break;
	    }
	    Tcl_ListObjIndex(interp, lPtr, i, &oPtr);
	    s = Tcl_GetString(oPtr);
	    if (   (*s && (!(*destPtrPtr) || strcmp(s,*destPtrPtr)))
		|| (!*s && *destPtrPtr)) {
		if (*destPtrPtr) {
		    ckfree(*destPtrPtr);
		}
		if (*s) {
		    *destPtrPtr = cpystr(s);
		} else {
		    *destPtrPtr = NULL;
		}
	    }
	}
    }
    Tcl_DStringFree(&cmd);
}

/*
 *----------------------------------------------------------------------
 *
 * RatAliasCmd --
 *
 *      Implements the RatAlias command as per ../doc/interface
 *
 * Results:
 *	Probably.
 *
 * Side effects:
 *	Probably.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatAliasCmd(ClientData dummy, Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
    if (objc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		Tcl_GetString(objv[0]), " option ?arg?\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (!strcmp(Tcl_GetString(objv[1]), "add")) {
	AliasInfo *aliasPtr;
	Tcl_HashEntry *entryPtr;
	char *key;
	int new, s;

	if (objc < 6 || objc > 8) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    Tcl_GetString(objv[0]),
		    " add book name fullname content comment options\"",
		    (char *) NULL);
	    return TCL_ERROR;
	}
	aliasPtr = (AliasInfo*)ckalloc(sizeof(AliasInfo));
	if (6 == objc) {
	    s = 2;
	    aliasPtr->book = Tcl_NewStringObj("Personal", -1);
	    aliasPtr->comment = Tcl_NewObj();
	    aliasPtr->flags = 0;
	} else {
	    s = 3;
	    aliasPtr->book = objv[2];
	    aliasPtr->comment = objv[6];
	    if (8 == objc && !strcmp(Tcl_GetString(objv[7]), "nofullname")) {
		aliasPtr->flags = ALIAS_FLAG_NOFULLNAME;
	    } else {
		aliasPtr->flags = 0;
	    }
	}
	key = Tcl_GetString(objv[s]);
	if (!key || !*key) {
	    ckfree(aliasPtr);
	    Tcl_SetResult(interp, "The name can not be an empty string",
		    TCL_STATIC);
	    return TCL_OK;
	}
	aliasPtr->mark = 0;
	aliasPtr->fullname = objv[s+1];
	if (Tcl_IsShared(objv[s+2])) {
	    aliasPtr->content = Tcl_DuplicateObj(objv[s+2]);
	} else {
	    aliasPtr->content = objv[s+2];
	}
	if (1 < AddressClean(aliasPtr->content)) {
	    aliasPtr->flags |= ALIAS_FLAG_ISLIST;
	}
	Tcl_IncrRefCount(aliasPtr->book);
	Tcl_IncrRefCount(aliasPtr->fullname);
	Tcl_IncrRefCount(aliasPtr->content);
	Tcl_IncrRefCount(aliasPtr->comment);
	entryPtr = Tcl_CreateHashEntry(&aliasTable, key, &new);
	Tcl_SetHashValue(entryPtr, (ClientData)aliasPtr);
	return TCL_OK;
	
    } else if (!strcmp(Tcl_GetString(objv[1]), "delete")) {
	Tcl_HashEntry *entryPtr;
	AliasInfo *aliasPtr;
	int i;

	for (i=2; i<objc; i++) {
	    if ((entryPtr = Tcl_FindHashEntry(&aliasTable,
		    Tcl_GetString(objv[i])))) {
		aliasPtr = (AliasInfo*)Tcl_GetHashValue(entryPtr);
		Tcl_DecrRefCount(aliasPtr->book);
		Tcl_DecrRefCount(aliasPtr->fullname);
		Tcl_DecrRefCount(aliasPtr->content);
		Tcl_DecrRefCount(aliasPtr->comment);
		ckfree(aliasPtr);
		Tcl_DeleteHashEntry(entryPtr);
	    }
	}
	return TCL_OK;

    } else if (!strcmp(Tcl_GetString(objv[1]), "get")) {
	Tcl_HashEntry *entryPtr;
	AliasInfo *aliasPtr;
	Tcl_Obj *oPtr;

	if (objc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    Tcl_GetString(objv[0]), " get alias\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (!(entryPtr=Tcl_FindHashEntry(&aliasTable, Tcl_GetString(objv[2])))){
	    Tcl_SetResult(interp, "Illegal alias", TCL_STATIC);
	    return TCL_ERROR;
	}
	aliasPtr = (AliasInfo*)Tcl_GetHashValue(entryPtr);
	oPtr = Tcl_NewObj();
	Tcl_ListObjAppendElement(interp, oPtr, aliasPtr->book);
	Tcl_ListObjAppendElement(interp, oPtr, aliasPtr->fullname);
	Tcl_ListObjAppendElement(interp, oPtr, aliasPtr->content);
	Tcl_ListObjAppendElement(interp, oPtr, aliasPtr->comment);
	if (aliasPtr->flags & ALIAS_FLAG_NOFULLNAME) {
	    Tcl_ListObjAppendElement(interp, oPtr,
		    Tcl_NewStringObj("nofullname", -1));
	} else {
	    Tcl_ListObjAppendElement(interp, oPtr, Tcl_NewObj());
	}
	Tcl_SetObjResult(interp, oPtr);
	return TCL_OK;

    } else if (!strcmp(Tcl_GetString(objv[1]), "list")) {
	Tcl_HashEntry *entryPtr;
	Tcl_HashSearch search;
	AliasInfo *aliasPtr;
	Tcl_Obj *oPtr;

	if (objc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    Tcl_GetString(objv[0]), " list var\"", (char *) NULL);
	    return TCL_ERROR;
	}

	for (entryPtr = Tcl_FirstHashEntry(&aliasTable, &search);
		entryPtr; entryPtr = Tcl_NextHashEntry(&search)) {
	    aliasPtr = (AliasInfo*) Tcl_GetHashValue(entryPtr);
	    oPtr = Tcl_NewObj();
	    Tcl_ListObjAppendElement(interp, oPtr, aliasPtr->book);
	    Tcl_ListObjAppendElement(interp, oPtr, aliasPtr->fullname);
	    Tcl_ListObjAppendElement(interp, oPtr, aliasPtr->content);
	    Tcl_ListObjAppendElement(interp, oPtr, aliasPtr->comment);
	    if (aliasPtr->flags & ALIAS_FLAG_NOFULLNAME) {
		Tcl_ListObjAppendElement(interp, oPtr,
			Tcl_NewStringObj("nofullname", -1));
	    } else {
		Tcl_ListObjAppendElement(interp, oPtr, Tcl_NewObj());
	    }
	    Tcl_SetVar2Ex(interp, Tcl_GetString(objv[2]),
		    Tcl_GetHashKey(&aliasTable, entryPtr), oPtr, 0);
	}
	return TCL_OK;
	
    } else if (!strcmp(Tcl_GetString(objv[1]), "read")) {
	Tcl_Channel channel;
	Tcl_Obj *oPtr = Tcl_NewObj();

	if (objc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    Tcl_GetString(objv[0]), " read filename\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (NULL == (channel = Tcl_OpenFileChannel(interp,
		Tcl_GetString(objv[2]), "r", 0))) {
	    return TCL_ERROR;
	}
	Tcl_SetChannelOption(interp, channel, "-encoding", "utf-8");
	while (0 <= Tcl_GetsObj(channel, oPtr) && !Tcl_Eof(channel)) {
	    Tcl_AppendToObj(oPtr, ";", 1);
	}
	return Tcl_EvalObjEx(interp, oPtr,  TCL_EVAL_DIRECT);
	
    } else if (!strcmp(Tcl_GetString(objv[1]), "save")) {
	Tcl_HashEntry *entryPtr;
	Tcl_HashSearch search;
	AliasInfo *aliasPtr;
	Tcl_Channel channel;
	Tcl_Obj *lPtr;
	int perm;

	if (objc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    Tcl_GetString(objv[0])," save book filename\"",(char*)NULL);
	    return TCL_ERROR;
	}

	Tcl_GetInt(interp, Tcl_GetVar2(interp, "option", "permissions",
		TCL_GLOBAL_ONLY), &perm);
	if (NULL == (channel = Tcl_OpenFileChannel(interp,
		Tcl_GetString(objv[3]), "w", perm))) {
	    return TCL_ERROR;
	}

	for (entryPtr = Tcl_FirstHashEntry(&aliasTable, &search);
		entryPtr; entryPtr = Tcl_NextHashEntry(&search)) {
	    aliasPtr = (AliasInfo*) Tcl_GetHashValue(entryPtr);
	    if (strcmp(Tcl_GetString(objv[2]), Tcl_GetString(aliasPtr->book))) {
		continue;
	    }
	    lPtr = Tcl_NewObj();
	    Tcl_ListObjAppendElement(interp, lPtr, aliasPtr->book);
	    Tcl_ListObjAppendElement(interp, lPtr,
		    Tcl_NewStringObj(Tcl_GetHashKey(&aliasTable, entryPtr),-1));
	    Tcl_ListObjAppendElement(interp, lPtr, aliasPtr->fullname);
	    Tcl_ListObjAppendElement(interp, lPtr, aliasPtr->content);
	    Tcl_ListObjAppendElement(interp, lPtr, aliasPtr->comment);
	    if (aliasPtr->flags & ALIAS_FLAG_NOFULLNAME) {
		Tcl_ListObjAppendElement(interp, lPtr,
			Tcl_NewStringObj("nofullname", -1));
	    } else {
		Tcl_ListObjAppendElement(interp, lPtr, Tcl_NewObj());
	    }
	    Tcl_WriteChars(channel, "RatAlias add ", -1);
	    Tcl_WriteObj(channel, lPtr);
	    Tcl_DecrRefCount(lPtr);
	    Tcl_WriteChars(channel, "\n", 1);
	}
	return Tcl_Close(interp, channel);
	
    } else if (!strcmp(Tcl_GetString(objv[1]), "expand1")
	    || !strcmp(Tcl_GetString(objv[1]), "expand2")) {
	ADDRESS *adrPtr, *baseAdrPtr = NULL;
	struct passwd *pwPtr;
	Tcl_HashEntry *entryPtr;
	Tcl_DString newList;
	AliasInfo *aliasPtr;
	char buf[1024], *sPtr, *cPtr, *userPtr;
	int level, rescan, useTable, inGroup, length;

	if (objc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    Tcl_GetString(objv[0]), " expand[12] adrlist\"",
		    (char *) NULL);
	    return TCL_ERROR;
	}

	/*
	 * Ignore empty addresses
	 */
	for (sPtr = Tcl_GetString(objv[2]);
		    *sPtr && isspace((unsigned char)*sPtr); sPtr++);
	if (!*sPtr) {
	    return TCL_OK;
	}

	/*
	 * Create unique mark
	 */
	if (0 == ++aliasMark) {
	    Tcl_HashSearch search;
	    aliasMark++;

	    for (entryPtr = Tcl_FirstHashEntry(&aliasTable, &search);
		    entryPtr; entryPtr = Tcl_NextHashEntry(&search)) {
		aliasPtr = (AliasInfo*) Tcl_GetHashValue(entryPtr);
		aliasPtr->mark = 0;
	    }
	}

	Tcl_GetBoolean(interp, Tcl_GetVar2(interp, "option", "lookup_name",
			               TCL_GLOBAL_ONLY), &useTable);

	if (!strcmp(Tcl_GetString(objv[1]), "expand2")) {
	    level = 2;
	} else {
	    Tcl_GetIntFromObj(interp, Tcl_GetVar2Ex(interp, "option",
		    "alias_expand", TCL_GLOBAL_ONLY), &level);
	}
	Tcl_DStringInit(&newList);
	Tcl_DStringAppend(&newList, sPtr, -1);
	for (cPtr = Tcl_DStringValue(&newList); *cPtr; cPtr++) {
	    if (isspace((unsigned char)*cPtr)) {
		*cPtr = ' ';
	    }
	}
	do {
	    inGroup = rescan = 0;
	    rfc822_parse_adrlist(&baseAdrPtr, Tcl_DStringValue(&newList), "");
	    Tcl_DStringSetLength(&newList, 0);
	    for (adrPtr = baseAdrPtr; adrPtr; adrPtr = adrPtr->next) {
		if (!adrPtr->mailbox) {
		    if (inGroup) {
			Tcl_DStringAppend(&newList, ";", 1);
			inGroup = 0;
		    }
		    if (adrPtr->next && adrPtr->next->mailbox) {
			Tcl_DStringAppend(&newList, ", ", 2);
		    }
		    continue;
		}
		if (!inGroup && !adrPtr->host) {
		    Tcl_DStringAppend(&newList, adrPtr->mailbox, -1);
		    Tcl_DStringAppend(&newList, ":", 1);
		    inGroup = 1;
		    continue;
		}
		/*
		 * Check if this matches an alias. If the host is an empty
		 * string we just try the mailbox name as key, otherwise we
		 * try mailbox@host.
		 */
		if (adrPtr->error || (adrPtr->host && adrPtr->host[0] == '.')){
		    Tcl_DStringFree(&newList);
		    if (adrPtr->error) {
			Tcl_SetResult(interp, adrPtr->error, TCL_VOLATILE);
		    } else {
			Tcl_SetResult(interp, adrPtr->host, TCL_VOLATILE);
		    }
		    mail_free_address(&baseAdrPtr);
		    return TCL_ERROR;
		}
		entryPtr = NULL;
		userPtr = NULL;
		if (adrPtr->host && *adrPtr->host) {
		    snprintf(buf, sizeof(buf), "%s@%s",
			    adrPtr->mailbox, adrPtr->host);
		} else {
		    RatStrNCpy(buf, adrPtr->mailbox, sizeof(buf));
		}
		if (!(entryPtr = Tcl_FindHashEntry(&aliasTable, buf))
			&& useTable) {
		    if ((pwPtr = getpwnam(buf))) {
			RatStrNCpy(buf, pwPtr->pw_gecos, sizeof(buf));
			if ((cPtr = strchr(buf, ','))) {
			    *cPtr = '\0';
			}
			userPtr = buf;
		    }
		}
		if (entryPtr) {
		    aliasPtr = (AliasInfo*) Tcl_GetHashValue(entryPtr);
		    if (aliasPtr->mark == aliasMark) {
			continue;
		    }
		    switch (level) {
		    case 0:
			Tcl_DStringAppend(&newList, buf, -1);
			break;
		    case 1:
			Tcl_DStringAppend(&newList, RatAddressMail(adrPtr), -1);
			cPtr = Tcl_GetString(aliasPtr->fullname);
			if (cPtr) {
			    sprintf(buf, " (%.1020s)", cPtr);
			    Tcl_DStringAppend(&newList, buf, strlen(buf));
			}
			break;
		    case 2:
			rescan = 1;
			cPtr = Tcl_GetStringFromObj(aliasPtr->content, &length);
			Tcl_DStringAppend(&newList, cPtr, length);
			if (!(aliasPtr->flags & ALIAS_FLAG_ISLIST)
				&& !(aliasPtr->flags & ALIAS_FLAG_NOFULLNAME)) {
			    Tcl_DStringAppend(&newList, " (", 2);
			    cPtr = Tcl_GetStringFromObj(aliasPtr->fullname,
				    &length);
			    Tcl_DStringAppend(&newList, cPtr, length);
			    Tcl_DStringAppend(&newList, ")", 1);
			}
			break;
		    }
		    aliasPtr->mark = aliasMark;
		} else if ((!adrPtr->host || !*adrPtr->host) &&
			   !adrPtr->personal && level > 0 && userPtr) {
		    Tcl_DStringAppend(&newList, RatAddressMail(adrPtr), -1);
		    Tcl_DStringAppend(&newList, " (", 2);
		    Tcl_DStringAppend(&newList, userPtr, -1);
		    Tcl_DStringAppend(&newList, ")", 1);

		} else {
		    Tcl_DStringAppend(&newList, RatAddressMail(adrPtr), -1);
		    if (adrPtr->personal && strlen(adrPtr->personal)) {
			sprintf(buf, " (%.1020s)", adrPtr->personal);
			Tcl_DStringAppend(&newList, buf, -1);
		    }
		}
		if (adrPtr->next && adrPtr->next->mailbox) {
		    Tcl_DStringAppend(&newList, ", ", 2);
		}
	    }
	    mail_free_address(&baseAdrPtr);
	} while(rescan);
	Tcl_DStringResult(interp, &newList);
	return TCL_OK;
	
    } else {
	Tcl_AppendResult(interp, "bad option \"", Tcl_GetString(objv[1]),
		"\": must be one of add, delete, get, list, read, save,",
		" expand1 or expand2",
		(char *) NULL);
	return TCL_ERROR;
    }

}

/*
 *----------------------------------------------------------------------
 *
 * RatAddressMail --
 *
 *      Prints the mail address in rfc822 format of an ADDRESS entry.
 *	Only one address is printed and there is NO fullname.
 *
 * Results:
 *	Pointer to a static storage area where the string is stored.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

char*
RatAddressMail(ADDRESS *adrPtr)
{
    static char *store = NULL;
    static int length = 0;
    int quote = 0, grow, l, i;
    char *cPtr;

    /*
     * We start by checking if the mailbox part contains any special
     * chracters (actually specials, SPACE and CTLS). While we do that we
     * also calculate how much it will grow when converted to qtext.
     * If we must quote it then we do so.
     * After that we append the host part if any.
     */
    for (cPtr = adrPtr->mailbox, grow=0; *cPtr; cPtr++) {
	if (strchr("\"\\\n", *cPtr)) {
	    quote = 1;
	    grow++;
	}
	if (*cPtr <= 32 || strchr("()<>@,;:[]", *cPtr)) {
	    quote = 1;
	}
    }
    if (quote) {
	l = 2+strlen(adrPtr->mailbox)+grow
		+(adrPtr->host?strlen(adrPtr->host)+1:0)+1;
	if (l > length) {
	    length = l;
	    store = ckrealloc(store, length);
	}
	i = 0;
	store[i++] = '"';
	for (cPtr = adrPtr->mailbox, grow=0; *cPtr; cPtr++) {
	    if (strchr("\"\\\n", *cPtr)) {
		store[i++] = '\\';
	    }
	    store[i++] = *cPtr;
	}
	store[i++] = '"';
    } else {
	l = strlen(adrPtr->mailbox)+(adrPtr->host?strlen(adrPtr->host)+1:0)+1;
	if (l > length) {
	    length = l;
	    store = ckrealloc(store, length);
	}
	strcpy(store, adrPtr->mailbox);
	i = strlen(store);
    }
    if (adrPtr->host && *adrPtr->host) {
	store[i++] = '@';
	RatStrNCpy(&store[i], adrPtr->host, length-i);
    } else {
	store[i++] = '\0';
    }
#ifdef MEM_DEBUG
    mem_store = store;
#endif /* MEM_DEBUG */
    return store;
}

/*
 *----------------------------------------------------------------------
 *
 * RatSplitAddresses --
 *
 *	This routine takes an address list as argument and splits it.
 *
 * Results:
 *	A list of addresses contained in the argument
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */
 
int
RatSplitAddresses(ClientData clientData, Tcl_Interp *interp, int argc,
	char *argv[])
{
    char *srcPtr, *dstPtr, *cPtr, *adr;
    Tcl_DString result;
    int inq = 0, inc = 0;
 
    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" addresslist\"", (char *) NULL);
        return TCL_ERROR;
    }
 
    adr = (char*)ckalloc(strlen(argv[1])+1);
    Tcl_DStringInit(&result);
 
    for (srcPtr = argv[1], dstPtr = adr; *srcPtr; srcPtr++) {
	switch (*srcPtr) {
	case '\\':
	    if (srcPtr[1]) {
		*dstPtr++ = *srcPtr++;
	    }
	    *dstPtr++ = *srcPtr;
	    break;
	case '"': 
	    if (inq) {
		inq = 0;
	    } else {
		inq = 1;
	    }
	    *dstPtr++ = *srcPtr;
	    break;
	case '(':
	    inc = 1;
	    *dstPtr++ = *srcPtr;
	    break;
	case ')':
	    inc = 0;
	    *dstPtr++ = *srcPtr;
	    break;
	case ',':
	    if (!inq && !inc) {
		for (dstPtr--;
			dstPtr >= adr && isspace((unsigned char)*dstPtr);
			dstPtr--);
		if (++dstPtr != adr) {
		    *dstPtr = '\0';
		    for (cPtr = adr;
			    *cPtr && isspace((unsigned char)*cPtr); cPtr++);
		    if (cPtr) {
			Tcl_DStringAppendElement(&result, cPtr);
		    }
		    dstPtr = adr;
		}
		break;
	    }
	    /* fallthrough */
	default:
	    *dstPtr++ = *srcPtr;
	    break;
	}
    }
    if (dstPtr != adr) {
	*dstPtr = '\0';
	Tcl_DStringAppendElement(&result, adr);
    }
 
    Tcl_DStringResult(interp, &result);
    ckfree(adr);
    return TCL_OK;
}

#ifdef MEM_DEBUG
void ratAddressCleanup()
{
    Tcl_HashEntry *e;
    Tcl_HashSearch s;

    for (e = Tcl_FirstHashEntry(&aliasTable, &s); e; e = Tcl_NextHashEntry(&s)){
	ckfree(Tcl_GetHashValue(e));
    }
    Tcl_DeleteHashTable(&aliasTable);

    if (mem_store) {
	ckfree(mem_store);
    }
}
#endif /* MEM_DEBUG */


/*
 *----------------------------------------------------------------------
 *
 * RatAddressSize --
 *
 *	Calculate the maximum size of and address list (or single address)
 *
 * Results:
 *	The maximum length of the address
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

size_t
RatAddressSize(ADDRESS *adrPtr, int all)
{
    ADDRESS *a,tadr;
    char tmp[MAILTMPLEN];
    size_t len, t;

    tadr.next = NULL;
    for (len = 0, a = adrPtr; a; a = a->next) {
        t = (tadr.mailbox = a->mailbox) ? 2*strlen (a->mailbox) : 3;
        if ((tadr.personal = a->personal)) t += 3 + 2*strlen (a->personal);
        if ((tadr.adl = a->adl)) t += 1 + 2*strlen (a->adl);
        if ((tadr.host = a->host)) t += 1 + 2*strlen (a->host);
        if (t < MAILTMPLEN) {     /* ignore ridiculous addresses */
	    tmp[0] = '\0';
	    rfc822_write_address (tmp,&tadr); 
	    t = strlen(tmp);
	}
	len += t+2;
	if (!all) break;
    }
    return len;
}

