/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * ti-wnd-main.c:
 * Main Window - Main application window.
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "ti-wnd-main.h"
#include "ti-dlg-about.h"
#include "ti-dlg-preferences.h"
#include "ti-dlg-request-conn.h"
#include "ti-conn-manager.h"
#include "ti-wnd-connection.h"
#include "ti-page-active-conns.h"
#include "ti-config.h"
#include "ti-util.h"

#include <glade/glade.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <string.h>
#include <gdk/gdkkeysyms.h>

struct _TIWndMainClass {
    GObjectClass parent;
};

G_DEFINE_TYPE (TIWndMain, ti_wnd_main, G_TYPE_OBJECT);

/* Function prototypes */
static void _ti_wnd_main_setup_conn_man_page (TIWndMain* wnd_main);
static void _ti_wnd_main_query_conn_man (TIWndMain* wnd_main);
static void _ti_wnd_main_build_protocols_treeview (TIWndMain* wnd_main);
static void _ti_wnd_main_fill_protocols_list (TIWndMain* wnd_main);
static void _ti_wnd_main_build_prot_params_treeview (TIWndMain* wnd_main);
static void _ti_wnd_main_fill_prot_params_list (TIWndMain* wnd_main, const gchar* protocol_name);
static void _ti_wnd_main_conn_manager_entry_changed (TIWndMain* wnd_main);
static gboolean _ti_wnd_main_conn_manager_entry_key_pressed (TIWndMain* self, GdkEventKey* event);
static void _ti_wnd_main_request_connection (TIWndMain* wnd_main);
static gchar* _ti_wnd_main_get_selected_protocol_name (TIWndMain* wnd_main);
static void _ti_wnd_main_on_conn_window_closed (TIWndMain* wnd_main, TIWndConnection* wnd_connection);
static void _ti_wnd_main_add_prot_param (TIWndMain* wnd_main, const gchar* name, guint flags,
                                         const gchar* type_signature, const GValue* default_value);
static void _ti_wnd_main_protocol_selection_changed (TIWndMain* wnd_main);
static void _ti_wnd_main_show_about_dialog (TIWndMain* wnd_main);
static void _ti_wnd_main_show_preferences_dialog (TIWndMain* wnd_main);
static void _ti_wnd_main_find_installed_conn_managers (TIWndMain* self);


/**
 * Instance private data.
 */
struct _TIWndMainPrivate {
    gboolean disposed;

    TIConnManager* conn_manager;

    GladeXML* glade_xml;

    GtkWidget* window;
    GtkNotebook* notebook;

    TIPageActiveConns* page_active_conns;

    TIDlgAbout* dlg_about;
    TIDlgRequestConn* dlg_request_conn;
    TIDlgPreferences* dlg_preferences;

    GtkListStore* protocols_list;
    GtkListStore* prot_params_list;
    GtkListStore* conn_managers_list;

    GtkWidget* comboentry_conn_man;
    GtkWidget* button_query;
    GtkWidget* button_request_conn;

    GtkTreeSelection* protocol_selection;

    /* An array of TIWndConnection*
       containing all connection windows currently open. */
    GPtrArray* connection_wnds;

};
typedef struct _TIWndMainPrivate TIWndMainPrivate;

#define TI_WND_MAIN_GET_PRIVATE(object)  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_WND_MAIN, TIWndMainPrivate))

/**
 * Drop all references to other objects.
 */
static void
ti_wnd_main_dispose (GObject *object)
{
    TIWndMain *wnd_main = TI_WND_MAIN (object);
    TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
    guint i;
    GObject* obj = NULL;

    if (priv->disposed)
    {
        return;
    }
    else
    {
        priv->disposed = TRUE;
    }

    if (priv->conn_manager != NULL)
    {
        g_object_unref (priv->conn_manager);
        priv->conn_manager = NULL;
    }

    if (priv->glade_xml != NULL)
    {
        g_object_unref (priv->glade_xml);
        priv->glade_xml = NULL;
    }

    if (priv->page_active_conns != NULL)
    {
        g_object_unref (priv->page_active_conns);
        priv->page_active_conns = NULL;
    }

    if (priv->dlg_about != NULL)
    {
        g_object_unref (priv->dlg_about);
        priv->dlg_about = NULL;
    }

    if (priv->dlg_request_conn != NULL)
    {
        g_object_unref (priv->dlg_request_conn);
        priv->dlg_request_conn = NULL;
    }

    if (priv->dlg_preferences != NULL)
    {
        g_object_unref (priv->dlg_preferences);
        priv->dlg_preferences = NULL;
    }

    if (priv->connection_wnds != NULL)
    {
        for (i = 0; i < priv->connection_wnds->len; i++)
        {
            obj = G_OBJECT (g_ptr_array_index(priv->connection_wnds, i));
            g_object_unref (obj);
        }
        g_ptr_array_free (priv->connection_wnds, TRUE);
        priv->connection_wnds = NULL;
    }

    G_OBJECT_CLASS (ti_wnd_main_parent_class)->dispose (object);
}

/**
 * Class initialization.
 */
static void
ti_wnd_main_class_init (TIWndMainClass *ti_wnd_main_class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (ti_wnd_main_class);

	/* override base object methods */ 
	gobject_class->dispose = ti_wnd_main_dispose;
		
	/* Add private */
	g_type_class_add_private (ti_wnd_main_class, sizeof (TIWndMainPrivate));
}

/**
 * Instance initialization.
 */
static void
ti_wnd_main_init (TIWndMain *ti_wnd_main)
{
    TIWndMainPrivate* priv = TI_WND_MAIN_GET_PRIVATE (ti_wnd_main);

    priv->disposed = FALSE;
    priv->glade_xml = NULL;
    priv->conn_manager = NULL;
    priv->dlg_about = NULL;
    priv->dlg_request_conn = NULL;
    priv->dlg_preferences = NULL;
}

/**
 * Returns a new instance.
 */
TIWndMain*
ti_wnd_main_new ()
{
    TIWndMain* wnd_main = NULL;
    TIWndMainPrivate* priv = NULL;
    gchar* glade_file_path = NULL;

	wnd_main = g_object_new (TI_TYPE_WND_MAIN, NULL);
    priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);


    // Setup windows, dialogs, etc.

    glade_file_path = g_strdup_printf ("%s%s", TI_DATA_DIR_PREFIX, "wnd-main.xml");
    priv->glade_xml = glade_xml_new (glade_file_path, NULL, NULL);
    if (priv->glade_xml == NULL)
    {
        g_critical ("Error loading glade file \"%s\".", glade_file_path);
        g_object_unref (wnd_main);
        wnd_main = NULL;
        goto CLEAN_UP;
    }

    priv->window = glade_xml_get_widget(priv->glade_xml, "main_window");
    g_assert (GTK_IS_WINDOW (priv->window));

    priv->notebook = GTK_NOTEBOOK (glade_xml_get_widget (priv->glade_xml, "notebook"));
    g_assert (GTK_IS_NOTEBOOK (priv->notebook));

    priv->page_active_conns = ti_page_active_conns_new (GTK_WINDOW (priv->window), priv->notebook);

    priv->dlg_about = ti_dlg_about_new (GTK_WINDOW (priv->window));
    priv->dlg_request_conn = ti_dlg_request_conn_new (GTK_WINDOW (priv->window));
    priv->dlg_preferences = ti_dlg_preferences_new (GTK_WINDOW (priv->window));
    
    priv->connection_wnds = g_ptr_array_sized_new (5 /* Five is magic number! */);

    _ti_wnd_main_setup_conn_man_page (wnd_main);

    CLEAN_UP:
    g_free (glade_file_path);

    return wnd_main;
}

/**
 * Show
 */
void
ti_wnd_main_show (TIWndMain* wnd_main)
{
    TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);

    gtk_widget_show_all (priv->window);
}

/**
 * Setup Connection Managers Page
 */
static void
_ti_wnd_main_setup_conn_man_page (TIWndMain* wnd_main)
{
    TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
    GtkWidget* widget = NULL;

    // Setup page label
    widget = gtk_notebook_get_nth_page (priv->notebook, 0);
    gtk_notebook_set_tab_label_text (priv->notebook, widget, "Connection Managers");

    // "Query" button.
    priv->button_query = glade_xml_get_widget (priv->glade_xml, "button_query");
    g_assert (GTK_IS_BUTTON (priv->button_query));
    g_signal_connect_swapped (priv->button_query, "clicked", G_CALLBACK (_ti_wnd_main_query_conn_man), wnd_main);

    // ComboEntry "Connection Manager"
    priv->comboentry_conn_man = glade_xml_get_widget(priv->glade_xml, "comboentry_conn_man");
    g_assert (GTK_IS_COMBO_BOX_ENTRY (priv->comboentry_conn_man));
    g_signal_connect_swapped (GTK_COMBO_BOX (priv->comboentry_conn_man), "changed", G_CALLBACK (_ti_wnd_main_conn_manager_entry_changed), wnd_main);

    widget = gtk_bin_get_child (GTK_BIN (priv->comboentry_conn_man));
    g_assert (widget != NULL && GTK_IS_ENTRY (widget));
    g_signal_connect_swapped (widget, "key-press-event", G_CALLBACK (_ti_wnd_main_conn_manager_entry_key_pressed), wnd_main);

    priv->conn_managers_list = gtk_list_store_new (1, G_TYPE_STRING);
    gtk_combo_box_set_model (GTK_COMBO_BOX (priv->comboentry_conn_man), GTK_TREE_MODEL (priv->conn_managers_list) );
    gtk_combo_box_entry_set_text_column (GTK_COMBO_BOX_ENTRY (priv->comboentry_conn_man), 0);

    _ti_wnd_main_find_installed_conn_managers (wnd_main);

    // Main Window
    glade_xml_signal_connect (priv->glade_xml, "delete_main_window", G_CALLBACK(gtk_main_quit));

    // Protocols' list
    _ti_wnd_main_build_protocols_treeview (wnd_main);

    // Protocol parameters' list
    _ti_wnd_main_build_prot_params_treeview (wnd_main);

    // "Request Connection" button
    priv->button_request_conn = glade_xml_get_widget (priv->glade_xml, "button_request_conn");
    g_assert (GTK_IS_BUTTON (priv->button_request_conn));
    g_signal_connect_swapped (priv->button_request_conn, "clicked", G_CALLBACK (_ti_wnd_main_request_connection), wnd_main);

    // "About" menu item
    widget = glade_xml_get_widget (priv->glade_xml, "menuitem_about");
    g_assert (GTK_IS_MENU_ITEM (widget));
    g_signal_connect_swapped (widget, "activate", G_CALLBACK (_ti_wnd_main_show_about_dialog), wnd_main);

    // "Preferences" menu item
    widget = glade_xml_get_widget (priv->glade_xml, "menuitem_preferences");
    g_assert (GTK_IS_MENU_ITEM (widget));
    g_signal_connect_swapped (widget, "activate", G_CALLBACK (_ti_wnd_main_show_preferences_dialog), wnd_main);
}

/**
 * Query Connection Manager
 *
 * Called when the user clicks the "Query" button.
 * It queries the specified connection manager (e.g.: ask its protocols).
 */
static void
_ti_wnd_main_query_conn_man (TIWndMain* wnd_main)
{
    TIWndMainPrivate* priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
    gchar* active_entry = NULL;

    active_entry = gtk_combo_box_get_active_text (GTK_COMBO_BOX(priv->comboentry_conn_man));
    if (active_entry == NULL)
        goto CLEAN_UP;

    priv->conn_manager = ti_conn_manager_new (active_entry);
    if (priv->conn_manager == NULL)
        goto CLEAN_UP;

    _ti_wnd_main_fill_protocols_list (wnd_main);

    CLEAN_UP:
    g_free (active_entry);
}

/**
 * Fill Protocols' List
 */
static void
_ti_wnd_main_fill_protocols_list (TIWndMain* wnd_main)
{
    TIWndMainPrivate* priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
    char** str_list;
    char** str_list_ptr;
    GtkTreeIter iter;

    gtk_list_store_clear (priv->protocols_list);

    str_list = ti_conn_manager_list_protocols (priv->conn_manager);
    g_return_if_fail (str_list != NULL);

    /* Fill the list. */
    for (str_list_ptr = str_list; *str_list_ptr != NULL; str_list_ptr++)
    {
        gtk_list_store_append (priv->protocols_list, &iter);
        gtk_list_store_set (priv->protocols_list, &iter, 0, *str_list_ptr, -1);
    }

    g_strfreev (str_list);
}

/**
 * Build Protocols Treeview
 */
static void
_ti_wnd_main_build_protocols_treeview (TIWndMain* wnd_main)
{
    TIWndMainPrivate* priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
    GtkWidget* treeview_protocols;
    GtkCellRenderer *renderer;
    GtkTreeViewColumn *column;

    treeview_protocols = glade_xml_get_widget (priv->glade_xml, "treeview_protocols");
    g_assert (GTK_IS_TREE_VIEW (treeview_protocols));

    priv->protocols_list = gtk_list_store_new (1, G_TYPE_STRING);
    gtk_tree_view_set_model (GTK_TREE_VIEW (treeview_protocols), GTK_TREE_MODEL (priv->protocols_list));

    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes ("Protocol",
                                                       renderer,
                                                       "text", 0,
                                                       NULL);

    gtk_tree_view_append_column (GTK_TREE_VIEW (treeview_protocols), column);


    priv->protocol_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview_protocols));
    gtk_tree_selection_set_mode (priv->protocol_selection, GTK_SELECTION_SINGLE);

    g_signal_connect_swapped (priv->protocol_selection, "changed",
                              G_CALLBACK (_ti_wnd_main_protocol_selection_changed),
                              wnd_main);
}

/**
 * Build the GtkTreeView for the parameters of the selected protocol.
 */
static void
_ti_wnd_main_build_prot_params_treeview (TIWndMain* wnd_main)
{
    TIWndMainPrivate* priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
    GtkWidget* treeview_prot_params;
    GtkCellRenderer* renderer;
    GtkTreeViewColumn* column;
    GtkTreeSelection* tree_selection;

    treeview_prot_params = glade_xml_get_widget(priv->glade_xml, "treeview_prot_params");

    priv->prot_params_list = gtk_list_store_new (4, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);
    gtk_tree_view_set_model (GTK_TREE_VIEW(treeview_prot_params), GTK_TREE_MODEL(priv->prot_params_list));

    renderer = gtk_cell_renderer_text_new ();

    column = gtk_tree_view_column_new_with_attributes ("Name",
                                                       renderer,
                                                       "text", 0,
                                                       NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (treeview_prot_params), column);

    column = gtk_tree_view_column_new_with_attributes ("Flags",
                                                       renderer,
                                                       "text", 1,
                                                       NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (treeview_prot_params), column);

    column = gtk_tree_view_column_new_with_attributes ("D-Bus Type",
                                                       renderer,
                                                       "text", 2,
                                                       NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (treeview_prot_params), column);

    column = gtk_tree_view_column_new_with_attributes ("Default Value",
                                                       renderer,
                                                       "text", 3,
                                                       NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (treeview_prot_params), column);

    tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview_prot_params));
    gtk_tree_selection_set_mode (tree_selection, GTK_SELECTION_NONE);
}

/**
 * Fill protocol parameters list for the given protocol name.
 */
static void
_ti_wnd_main_fill_prot_params_list (TIWndMain* wnd_main, const gchar* protocol_name)
{
    TIWndMainPrivate* priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
    GPtrArray* parameters;
    GValueArray* param;
    guint i;

    // Parameter fields
    const gchar* name;
    guint flags;
    const gchar* type_signature;
    const GValue* default_value;

    gtk_list_store_clear (priv->prot_params_list);

    parameters = ti_conn_manager_get_parameters (priv->conn_manager, protocol_name);
    g_return_if_fail (parameters != NULL);

    for (i = 0; i < parameters->len; i++)
    {
        param = g_ptr_array_index (parameters, i);

        name           =           g_value_get_string (&(param->values[0]));
        flags          =           g_value_get_uint   (&(param->values[1]));
        type_signature =           g_value_get_string (&(param->values[2]));
        default_value  = (GValue*) g_value_get_boxed  (&(param->values[3]));

        _ti_wnd_main_add_prot_param (wnd_main, name, flags, type_signature, default_value);

        g_value_array_free (param);
    }

    g_ptr_array_free (parameters, TRUE);
}

/**
 * Connection Manager Entry Changed
 * Called when the connection manager entry is changed.
 */
static void
_ti_wnd_main_conn_manager_entry_changed (TIWndMain* wnd_main)
{
    TIWndMainPrivate* priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
    gchar* active_text;

    active_text = gtk_combo_box_get_active_text (GTK_COMBO_BOX (priv->comboentry_conn_man));

    /* Disable the query button if the connection manager entry is empty. */
    if (active_text == NULL || strlen(active_text) == 0)
    {
        gtk_widget_set_sensitive (priv->button_query, FALSE);
    }
    else
    {
        gtk_widget_set_sensitive (priv->button_query, TRUE);
    }

    g_free (active_text);
}

/**
 * Connection Manager Entry - Key Pressed
 * Called when the connection manager entry has focus and a key is pressed.
 */
static gboolean
_ti_wnd_main_conn_manager_entry_key_pressed (TIWndMain* self, GdkEventKey* event)
{
    if (event->keyval == GDK_Return && event->type == GDK_KEY_PRESS)
    {
        // The user pressed <Enter>, therefore we should query the
        // connection manager whose name he just wrote.
        _ti_wnd_main_query_conn_man (self);
    }

    return FALSE; // propagate the event further
}

/**
 * Request Connection
 */
static void
_ti_wnd_main_request_connection (TIWndMain* wnd_main)
{
    TIWndMainPrivate* priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
    gchar* protocol_name = NULL;
    GHashTable* param_values = NULL;
    TIConnection* connection = NULL;
    gboolean ok;
    TIWndConnection* connection_window = NULL;

    protocol_name = _ti_wnd_main_get_selected_protocol_name (wnd_main);

    ok = ti_dlg_request_conn_run (priv->dlg_request_conn, protocol_name,
                                  GTK_TREE_MODEL(priv->prot_params_list),
                                  &param_values);
    if (!ok) goto CLEAN_UP;


    connection = ti_conn_manager_request_connection (priv->conn_manager, protocol_name, param_values);
    if (connection == NULL) goto CLEAN_UP;

    connection_window = ti_wnd_connection_new_with_conn (GTK_WINDOW (priv->window), connection);
    if (connection_window == NULL) goto CLEAN_UP;

    g_ptr_array_add (priv->connection_wnds, connection_window);
    g_signal_connect_swapped (connection_window, "closed", G_CALLBACK (_ti_wnd_main_on_conn_window_closed), wnd_main);
    ti_wnd_connection_show (connection_window);

    CLEAN_UP:

    if (param_values != NULL)
    {
        g_hash_table_destroy (param_values);
    }

    g_free (protocol_name);

    if (connection != NULL)
        g_object_unref (connection);
}

/**
 * Returns the name of the currently selected protocol or NULL if no protocol
 * is being selected.
 *
 * The returned string must be freed after use.
 */
static gchar*
_ti_wnd_main_get_selected_protocol_name (TIWndMain* wnd_main)
{
    TIWndMainPrivate* priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
    gboolean ok;
    GtkTreeModel* tree_model;
    GtkTreeIter iter;
    GValue value = {0, };
    const gchar* protocol_name;
    gchar* ret_val = NULL;

    ok = gtk_tree_selection_get_selected (priv->protocol_selection, &tree_model, &iter);
    if (!ok)
        goto CLEAN_UP;

    // Get the selected protocol name.
    gtk_tree_model_get_value (tree_model, &iter, 0, &value);
    g_assert (G_VALUE_HOLDS(&value, G_TYPE_STRING));
    if (!G_VALUE_HOLDS(&value, G_TYPE_STRING))
        goto CLEAN_UP;

    protocol_name = g_value_get_string (&value);
    ret_val = g_strdup (protocol_name);

    CLEAN_UP:
    g_value_unset(&value);

    return ret_val;
}

/**
 * On Connection Window Closed
 * Called when a connection window is closed, i.e., when a TIWndConnection emits a "closed" signal.
 */
static void
_ti_wnd_main_on_conn_window_closed (TIWndMain* wnd_main, TIWndConnection* wnd_connection)
{
    TIWndMainPrivate* priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
    gboolean ok;

    ok = g_ptr_array_remove (priv->connection_wnds, wnd_connection);
    g_assert (ok);

    g_object_unref (wnd_connection);
}

/**
 * Add Protocol Parameter
 * Adds a protocol parameter to protocol parameters' list.
 */
static void
_ti_wnd_main_add_prot_param (TIWndMain* wnd_main, const gchar* name, guint flags,
                             const gchar* type_signature, const GValue* default_value)
{
    TIWndMainPrivate* priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
    GtkTreeIter iter;
    gchar* flags_str;
    gchar* def_val_str;

    flags_str = g_strdup_printf ("%u", flags);

    def_val_str = ti_value_to_string (default_value);

    gtk_list_store_append (priv->prot_params_list, &iter);
    gtk_list_store_set (priv->prot_params_list, &iter,
                        0, name,
                        1, flags_str,
                        2, type_signature,
                        3, def_val_str,
                        -1);

    g_free (flags_str);
    g_free (def_val_str);
}

/**
 * Protocol Selection Changed
 */
static void
_ti_wnd_main_protocol_selection_changed (TIWndMain* wnd_main)
{
    TIWndMainPrivate* priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
    gchar* protocol_name = NULL;

    protocol_name = _ti_wnd_main_get_selected_protocol_name (wnd_main);

    // You can connect only if a protocol has been selected.
    if (protocol_name == NULL)
    {
        gtk_widget_set_sensitive (priv->button_request_conn, FALSE);
        goto CLEAN_UP; // There's nothing else to do.
    }

    gtk_widget_set_sensitive (priv->button_request_conn, TRUE);

    // Fill the parameters list according to the selected protocol.
    _ti_wnd_main_fill_prot_params_list (wnd_main, protocol_name);

    CLEAN_UP:
    g_free (protocol_name);
}

/**
 * Show Preferences Dialog
 */
static void
_ti_wnd_main_show_preferences_dialog (TIWndMain* wnd_main)
{
    TIWndMainPrivate* priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);

    ti_dlg_preferences_run (priv->dlg_preferences);
}

/**
 * Show About Dialog
 */
static void
_ti_wnd_main_show_about_dialog (TIWndMain* wnd_main)
{
    TIWndMainPrivate* priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);

    ti_dlg_about_run (priv->dlg_about);
}

/**
 * Find Installed Connection Managers
 */
static void
_ti_wnd_main_find_installed_conn_managers (TIWndMain* self)
{
    TIWndMainPrivate* priv = TI_WND_MAIN_GET_PRIVATE (self);
    GDir* dir = NULL;
    const gchar* file = NULL;
    gchar* manager_name = NULL;
    GtkTreeIter iter;

    dir = g_dir_open (TI_TELEPATHY_MANAGERS_DIR, 0, NULL);
    if (dir == NULL)
        return;

    while ((file = g_dir_read_name (dir)) != NULL)
    {
        if (!g_str_has_suffix (file, ".manager"))
            continue;

        manager_name = g_strndup (file, strlen (file) - strlen (".manager"));

        gtk_list_store_append (priv->conn_managers_list, &iter);
        gtk_list_store_set (priv->conn_managers_list, &iter,
                            0, manager_name,
                            -1);

        g_free (manager_name);
    }

    g_dir_close (dir);
}
