/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * ti-util.h:
 * General purpose utility functions.
 * 
 * Copyright (C) 2006, 2007 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#ifndef __TI_UTIL_H__
#define __TI_UTIL_H__

#include <glib.h>
#include <glib-object.h>
#include <gtk/gtk.h>

/**
 * Parses a string containing a comma-separated uint list.
 * e.g.:  "134,591,343,937,7869"
 *
 * @return An array of guint or NULL if an error occurred. Must be freed after use.
 */
GArray* ti_parse_uint_list (const gchar* uint_list_str);

/**
 * Contents of a hash table entry.
 */
struct _TIHashEntry {
    gpointer key;
    gpointer value;
};
typedef struct _TIHashEntry TIHashEntry;

/**
 * Transforms a hash table into an array of TIHashEntry elements. Making life easier
 * (and cleaner) when one wants to iterate a GHashTable.
 *
 * PS: It actually leaves the original hash table untouched, so this is not really a "transformation" afterall.
 *
 * @return An array of TIHashEntry. Must be freed after use.
 */
GArray* ti_hash_table_to_array (GHashTable* hash_table);

/**
 * Returns a string representation of a GValue.
 *
 * @return A string representation of the given GValue. Must be freed after use.
 */
gchar* ti_value_to_string (const GValue* value);

/**
 * Sets an uninitialized GValue with the value specified by a string and a D-Bus signature (its type).
 *
 * Examples:
 *     ti_string_to_value (&value, "b", "True");
 *     ti_string_to_value (&value, "b", "true");
 *     ti_string_to_value (&value, "u", "123");
 *     ti_string_to_value (&value, "s", "foo");
 */
void ti_string_to_value (GValue* value, const gchar* dbus_sig, const gchar* value_str);

/**
 * Destroys a GValue. Useful as a destroy function for g_hash_table_new_full().
 *
 * @param data Pointer to a GValue to be destroyed, that is, unset and freed.
 */
void ti_value_destroy (gpointer data);

/**
 * Remove Selected Elements
 *
 * Removes the selected elelements from their tree model.
 */
void ti_remove_selected_elements (GtkTreeSelection* tree_selection);

/**
 * Get Selected Elements
 *
 * Returns an array with the selected elements (from a specific column)
 *
 * @return A newly created array. Must be freed after use (including its elements, if applicable).
 */
GArray* ti_get_selected_elements (GtkTreeSelection* tree_selection, guint column, GType type);

/**
 * Get Tree Model Elements
 *
 * Returns an array with the tree model elements (from a specific column)
 *
 * @return A newly created array. Must be freed after use (including its elements, if applicable).
 */
GArray* ti_get_tree_model_elements (GtkTreeModel* tree_model, guint column, GType type);

#define TI_OBJ_UNREF(obj) if((obj)!=NULL){g_object_unref(obj);obj=NULL;}

#endif
