/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * ti-dlg-advertise-caps.c:
 * Dialog to expose AdvertiseCapabilities() method on interface
 * org.freedesktop.Telepathy.Connection.Interface.Capabilities
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "ti-dlg-advertise-caps.h"
#include "ti-util.h"
#include "ti-config.h"

#include <glade/glade.h>

G_DEFINE_TYPE (TIDlgAdvertiseCaps, ti_dlg_advertise_caps, G_TYPE_OBJECT);

/**
 * Instance private data.
 */
struct _TIDlgAdvertiseCapsPrivate {
    GtkWindow* parent;

    GladeXML* glade_xml;

    GtkWidget* dialog;

    GtkTreeView* treeview_add;
    GtkTreeView* treeview_remove;

    GtkListStore* liststore_add;
    GtkListStore* liststore_remove;

    GtkComboBoxEntry* comboentry_channel_type;
    GtkEntry* entry_specific_flags;
};
typedef struct _TIDlgAdvertiseCapsPrivate TIDlgAdvertiseCapsPrivate;

#define TI_DLG_ADVERTISE_CAPS_GET_PRIVATE(object)  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_DLG_ADVERTISE_CAPS, TIDlgAdvertiseCapsPrivate))

/* Function prototypes */
static void _button_ok_clicked (GtkButton *button, gpointer user_data);
static void _button_cancel_clicked (GtkButton *button, gpointer user_data);
static void _ti_dlg_advertise_caps_setup (TIDlgAdvertiseCaps* self);
static void _ti_dlg_advertise_caps_build_treeview_add (TIDlgAdvertiseCaps* self);
static void _ti_dlg_advertise_caps_build_treeview_remove (TIDlgAdvertiseCaps* self);
static void _ti_page_capabilities_to_add_list (TIDlgAdvertiseCaps* self);
static void _ti_page_capabilities_to_remove_list (TIDlgAdvertiseCaps* self);
static void _ti_dlg_advertise_caps_get_add (TIDlgAdvertiseCaps* self, GPtrArray** add_caps);
static void _ti_dlg_advertise_caps_get_remove (TIDlgAdvertiseCaps* self, gchar*** remove_caps);


/**
 * Drop all references to other objects.
 */
static void
ti_dlg_advertise_caps_dispose (GObject *object)
{
    TIDlgAdvertiseCaps *dlg_advertise_caps = TI_DLG_ADVERTISE_CAPS (object);
    TIDlgAdvertiseCapsPrivate *priv = TI_DLG_ADVERTISE_CAPS_GET_PRIVATE (dlg_advertise_caps);

    TI_OBJ_UNREF (priv->glade_xml);

    G_OBJECT_CLASS (ti_dlg_advertise_caps_parent_class)->dispose (object);
}

/**
 * Class initialization.
 */
static void
ti_dlg_advertise_caps_class_init (TIDlgAdvertiseCapsClass *ti_dlg_advertise_caps_class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (ti_dlg_advertise_caps_class);

	/* override base object methods */ 
	gobject_class->dispose = ti_dlg_advertise_caps_dispose;
		
	/* Add private */
	g_type_class_add_private (ti_dlg_advertise_caps_class, sizeof (TIDlgAdvertiseCapsPrivate));
}

/**
 * Instance initialization.
 */
static void
ti_dlg_advertise_caps_init (TIDlgAdvertiseCaps *ti_dlg_advertise_caps)
{
    TIDlgAdvertiseCapsPrivate* priv = TI_DLG_ADVERTISE_CAPS_GET_PRIVATE (ti_dlg_advertise_caps);

    priv->glade_xml = NULL;
}

/**
 * Returns a new instance.
 */
TIDlgAdvertiseCaps*
ti_dlg_advertise_caps_new (GtkWindow* parent)
{
    TIDlgAdvertiseCaps* dialog = NULL;
    TIDlgAdvertiseCapsPrivate* priv = NULL;
    gchar* glade_file_path = NULL;

	dialog = g_object_new (TI_TYPE_DLG_ADVERTISE_CAPS, NULL);

    priv = TI_DLG_ADVERTISE_CAPS_GET_PRIVATE (dialog);
    priv->parent = parent;
    
    glade_file_path = g_strdup_printf ("%s%s", TI_DATA_DIR_PREFIX, "dlg-advertise-caps.xml");
    priv->glade_xml = glade_xml_new (glade_file_path, NULL, NULL);
    if (priv->glade_xml == NULL)
    {
        g_critical ("Error loading glade file \"%s\".", glade_file_path);
        g_object_unref (dialog);
        dialog = NULL;
        goto CLEAN_UP;
    }

    _ti_dlg_advertise_caps_setup (dialog);

    gtk_window_set_transient_for (GTK_WINDOW (priv->dialog), GTK_WINDOW (priv->parent));
    gtk_window_set_position (GTK_WINDOW (priv->dialog), GTK_WIN_POS_CENTER_ON_PARENT);

    glade_xml_signal_connect_data (priv->glade_xml, "button_ok_clicked", G_CALLBACK(_button_ok_clicked), priv->dialog);
    glade_xml_signal_connect_data (priv->glade_xml, "button_cancel_clicked", G_CALLBACK(_button_cancel_clicked), priv->dialog);

    CLEAN_UP:
    g_free (glade_file_path);

    return dialog;
}

/**
 * Run
 */
gboolean
ti_dlg_advertise_caps_run (TIDlgAdvertiseCaps* self, GPtrArray** add_caps, gchar*** remove_caps)
{
    TIDlgAdvertiseCapsPrivate *priv = TI_DLG_ADVERTISE_CAPS_GET_PRIVATE (self);
    gint result;

    gtk_widget_show_all (priv->dialog);
    result = gtk_dialog_run (GTK_DIALOG (priv->dialog));
    gtk_widget_hide_all (priv->dialog);

    _ti_dlg_advertise_caps_get_add (self, add_caps);
    _ti_dlg_advertise_caps_get_remove (self, remove_caps);

    return result == GTK_RESPONSE_OK;
}

/**
 * Called when OK button is clicked.
 */
static void
_button_ok_clicked (GtkButton *button, gpointer user_data)
{
    GtkDialog* dialog = GTK_DIALOG (user_data);
    gtk_dialog_response (dialog, GTK_RESPONSE_OK);
}

/**
 * Called when "Cancel" button is clicked.
 */
static void
_button_cancel_clicked (GtkButton *button, gpointer user_data)
{
    GtkDialog* dialog = GTK_DIALOG (user_data);
    gtk_dialog_response (dialog, GTK_RESPONSE_CANCEL);
}

/**
 * Setup
 */
static void
_ti_dlg_advertise_caps_setup (TIDlgAdvertiseCaps* self)
{
    TIDlgAdvertiseCapsPrivate* priv = TI_DLG_ADVERTISE_CAPS_GET_PRIVATE (self);
    GtkWidget* widget;

    priv->dialog = glade_xml_get_widget (priv->glade_xml, "dialog_advertise_caps");
    g_assert (GTK_IS_DIALOG (priv->dialog));

    // Tree View Add
    priv->treeview_add = GTK_TREE_VIEW (glade_xml_get_widget (priv->glade_xml, "treeview_add"));
    g_assert (priv->treeview_add != NULL && GTK_IS_TREE_VIEW (priv->treeview_add));
    _ti_dlg_advertise_caps_build_treeview_add (self);

    // Tree View Remove
    priv->treeview_remove = GTK_TREE_VIEW (glade_xml_get_widget (priv->glade_xml, "treeview_remove"));
    g_assert (priv->treeview_remove != NULL && GTK_IS_TREE_VIEW (priv->treeview_remove));
    _ti_dlg_advertise_caps_build_treeview_remove (self);

    // Button "To Add List"
    widget = glade_xml_get_widget (priv->glade_xml, "button_to_add_list");
    g_assert (widget != NULL && GTK_IS_BUTTON (widget));
    g_signal_connect_swapped (widget, "clicked", G_CALLBACK (_ti_page_capabilities_to_add_list), self);

    // Button "To Remove List"
    widget = glade_xml_get_widget (priv->glade_xml, "button_to_remove_list");
    g_assert (widget != NULL && GTK_IS_BUTTON (widget));
    g_signal_connect_swapped (widget, "clicked", G_CALLBACK (_ti_page_capabilities_to_remove_list), self);

    // ComboEntry Channel Type
    priv->comboentry_channel_type = GTK_COMBO_BOX_ENTRY (glade_xml_get_widget (priv->glade_xml, "comboentry_channel_type"));
    g_assert (priv->comboentry_channel_type != NULL && GTK_IS_COMBO_BOX_ENTRY (priv->comboentry_channel_type));

    // Entry Specific Flags
    priv->entry_specific_flags = GTK_ENTRY (glade_xml_get_widget (priv->glade_xml, "entry_specific_flags"));
    g_assert (priv->entry_specific_flags != NULL && GTK_IS_ENTRY (priv->entry_specific_flags));
}

/**
 * Build Treeview Add
 */
static void
_ti_dlg_advertise_caps_build_treeview_add (TIDlgAdvertiseCaps* self)
{
    TIDlgAdvertiseCapsPrivate* priv = TI_DLG_ADVERTISE_CAPS_GET_PRIVATE (self);
    GtkCellRenderer* renderer;
    GtkTreeViewColumn* column;
    GtkTreeSelection* selection;

    priv->liststore_add = gtk_list_store_new (2,
                                              G_TYPE_STRING, // channel type
                                              G_TYPE_UINT);  // specific flags
    
    gtk_tree_view_set_model (priv->treeview_add, GTK_TREE_MODEL (priv->liststore_add));

    renderer = gtk_cell_renderer_text_new ();

    column = gtk_tree_view_column_new_with_attributes ("Channel Type",
                                                        renderer,
                                                        "text", 0,
                                                        NULL);
    gtk_tree_view_append_column (priv->treeview_add, column);

    column = gtk_tree_view_column_new_with_attributes ("Flags",
                                                       renderer,
                                                       "text", 1,
                                                       NULL);
    gtk_tree_view_append_column (priv->treeview_add, column);


    selection = gtk_tree_view_get_selection (priv->treeview_add);
    gtk_tree_selection_set_mode (selection, GTK_SELECTION_NONE);
}

/**
 * Build Treeview Remove
 */
static void
_ti_dlg_advertise_caps_build_treeview_remove (TIDlgAdvertiseCaps* self)
{
    TIDlgAdvertiseCapsPrivate* priv = TI_DLG_ADVERTISE_CAPS_GET_PRIVATE (self);
    GtkCellRenderer* renderer;
    GtkTreeViewColumn* column;
    GtkTreeSelection* selection;

    priv->liststore_remove = gtk_list_store_new (1,
                                                 G_TYPE_STRING // channel type
                                                 );

    gtk_tree_view_set_model (priv->treeview_remove, GTK_TREE_MODEL (priv->liststore_remove));

    renderer = gtk_cell_renderer_text_new ();

    column = gtk_tree_view_column_new_with_attributes ("Channel Type",
                                                       renderer,
                                                       "text", 0,
                                                       NULL);
    gtk_tree_view_append_column (priv->treeview_remove, column);

    selection = gtk_tree_view_get_selection (priv->treeview_remove);
    gtk_tree_selection_set_mode (selection, GTK_SELECTION_NONE);
}

/**
 * To Add List
 */
static void
_ti_page_capabilities_to_add_list (TIDlgAdvertiseCaps* self)
{
    TIDlgAdvertiseCapsPrivate* priv = TI_DLG_ADVERTISE_CAPS_GET_PRIVATE (self);
    gchar* channel_type = NULL;
    GtkTreeIter iter;
    guint flags;
    const gchar* flags_str;

    channel_type = gtk_combo_box_get_active_text (GTK_COMBO_BOX (priv->comboentry_channel_type));
    flags_str = gtk_entry_get_text (priv->entry_specific_flags);

    sscanf (flags_str, "%u", &flags);

    gtk_list_store_append (priv->liststore_add, &iter);
    gtk_list_store_set (priv->liststore_add, &iter,
                        0, channel_type,
                        1, flags,
                        -1);

    // Clean up
    g_free (channel_type);
}

/**
 * To Remove List
 */
static void _ti_page_capabilities_to_remove_list (TIDlgAdvertiseCaps* self)
{
    TIDlgAdvertiseCapsPrivate* priv = TI_DLG_ADVERTISE_CAPS_GET_PRIVATE (self);
    gchar* channel_type = NULL;
    GtkTreeIter iter;

    channel_type = gtk_combo_box_get_active_text (GTK_COMBO_BOX (priv->comboentry_channel_type));

    gtk_list_store_append (priv->liststore_remove, &iter);
    gtk_list_store_set (priv->liststore_remove, &iter,
                        0, channel_type,
                        -1);

    // Clean up
    g_free (channel_type);
}

/**
 * Get Add
 */
static void
_ti_dlg_advertise_caps_get_add (TIDlgAdvertiseCaps* self, GPtrArray** add_caps)
{
    TIDlgAdvertiseCapsPrivate* priv = TI_DLG_ADVERTISE_CAPS_GET_PRIVATE (self);
    GtkTreeIter iter;
    gboolean ok;
    gchar* channel_type = NULL;
    guint specific_flags;
    GValueArray* capability;

    *add_caps = g_ptr_array_new ();

    ok = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (priv->liststore_add), &iter);
    while (ok)
    {
        gtk_tree_model_get (GTK_TREE_MODEL (priv->liststore_add), &iter,
                            0, &channel_type,
                            1, &specific_flags,
                            -1);

        capability = g_value_array_new (2);

        g_value_array_append (capability, NULL);
        g_value_init (g_value_array_get_nth (capability, 0), G_TYPE_STRING);
        g_value_set_string (g_value_array_get_nth (capability, 0), channel_type);

        g_value_array_append (capability, NULL);
        g_value_init (g_value_array_get_nth (capability, 1), G_TYPE_UINT);
        g_value_set_uint (g_value_array_get_nth (capability, 1),specific_flags);

        g_ptr_array_add (*add_caps, capability);

        g_free (channel_type);

        ok = gtk_tree_model_iter_next (GTK_TREE_MODEL (priv->liststore_add), &iter);
    }
}

/**
 * Get Remove
 */
static void
_ti_dlg_advertise_caps_get_remove (TIDlgAdvertiseCaps* self, gchar*** remove_caps)
{
    TIDlgAdvertiseCapsPrivate* priv = TI_DLG_ADVERTISE_CAPS_GET_PRIVATE (self);
    GtkTreeIter iter;
    gboolean ok;
    gchar* channel_type = NULL;
    GArray* remove_caps_array;

    remove_caps_array = g_array_new (TRUE, TRUE, sizeof (gchar*));

    ok = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (priv->liststore_remove), &iter);
    while (ok)
    {
        gtk_tree_model_get (GTK_TREE_MODEL (priv->liststore_remove), &iter,
                            0, &channel_type,
                            -1);

        g_array_append_val (remove_caps_array, channel_type);

        ok = gtk_tree_model_iter_next (GTK_TREE_MODEL (priv->liststore_remove), &iter);
    }

    *remove_caps = (gchar**) g_array_free (remove_caps_array, FALSE);
}
