# telepathy-butterfly - an MSN connection manager for Telepathy
#
# Copyright (C) 2006 Ali Sabil <ali.sabil@gmail.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import logging
import telepathy
import gobject
import dbus

from connection import MsnConnection

logger = logging.getLogger('telepathy-butterfly:connmgr')

class MsnConnectionManager(telepathy.server.ConnectionManager):
    """Msn connection manager
    implements the org.freedesktop.Telepathy.ConnectionManager interface
    """

    def __init__(self, idle_callback):
        """
        Initialise the connection manager.
        """
        telepathy.server.ConnectionManager.__init__(self, 'butterfly')
        logger.info("Connection manager created")
        self._protos['msn'] = MsnConnection
        self.__idle_callback = idle_callback

    def GetParameters(self, proto):
        """
        Returns the mandatory and optional parameters for the given proto.
        """
        if proto in self._protos:
            conn = self._protos[proto]
            ret = []
            for param_name, param_type in conn._mandatory_parameters.iteritems():
                param = (param_name,
                        telepathy.CONN_MGR_PARAM_FLAG_REQUIRED,
                        param_type,
                        '')
                ret.append(param)

            for param_name, param_type in conn._optional_parameters.iteritems():
                if param_name in conn._parameter_defaults:
                    default_value = conn._parameter_defaults[param_name]
                    param = (param_name,
                            telepathy.CONN_MGR_PARAM_FLAG_HAS_DEFAULT,
                            param_type,
                            default_value)
                else:
                    param = (param_name, 0, param_type, '')
                ret.append(param)
            return ret
        else:
            raise telepathy.NotImplemented('unknown protocol %s' % proto)
    
    def disconnected(self, conn):
        result = telepathy.server.ConnectionManager.disconnected(self, conn)
        gobject.timeout_add(5000, self.idle_callback)

    def idle_callback(self):
        if self.__idle_callback is not None and \
                len(self._connections) == 0:
            self.__idle_callback()
        return False

    def quit(self):
        """
        Terminates all connections. Must be called upon quit
        """
        for c in self._connections:
            c.Disconnect()

