/******************************************************************************
*   "Gif-Lib" - Yet another gif library.				      *
*									      *
* Written by:  Gershon Elber				Ver 1.1, Aug. 1990    *
*******************************************************************************
* The kernel of the GIF Encoding process can be found here.		      *
*******************************************************************************
* History:								      *
* 14 Jun 89 - Version 1.0 by Gershon Elber.				      *
*  3 Sep 90 - Version 1.1 by Gershon Elber (Support for Gif89, Unique names). *
* 26 Jun 96 - Version 3.0 by Eric S. Raymond (Full GIF89 support)
******************************************************************************/

#   include	<sys/types.h>
#   include	<sys/stat.h>

#   include	<fcntl.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<string.h>
#   include	<unistd.h>
#   include	"bm_gif_lib.h"
#   include	"bm_gif_lib_private.h"

#   include	<appDebugon.h>
#   include	<sioBlocked.h>
#   include	<sioLzw.h>
#   include	<sioEndian.h>

static char GifVersionPrefix[GIF_STAMP_LEN + 1] = GIF87_STAMP;

static int EGifSetupCompress(GifFileType *GifFile);

/************************************************************************/
/*									*/
/*  Write Wrapper.							*/
/*									*/
/************************************************************************/

static int GifWrite(		GifFileType *		gft,
				const unsigned char *	buf,
				int			len )
    {
    GifFilePrivateType *	gfpt= (GifFilePrivateType *)gft->Private;

    return sioOutWriteBytes( gfpt->gfptSos, buf, len );
    }

/************************************************************************/
/*									*/
/*  Initialize GIF writing to an OutputStream.				*/
/*									*/
/*  _GifError is cleared if succesfull.					*/
/*									*/
/************************************************************************/

GifFileType *EGifOpenFileHandle(	SimpleOutputStream *	sos )
{
    GifFileType *		GifFile;
    GifFilePrivateType *	gfpt;

    if ((GifFile = (GifFileType *) malloc(sizeof(GifFileType))) == NULL) {
        _GifError = E_GIF_ERR_NOT_ENOUGH_MEM;
        return NULL;
    }

    memset(GifFile, '\0', sizeof(GifFileType));

    if ((gfpt = (GifFilePrivateType *)
                   malloc(sizeof(GifFilePrivateType))) == NULL) {
        free(GifFile);
        _GifError = E_GIF_ERR_NOT_ENOUGH_MEM;
        return NULL;
    }

    GifFile->Private = (VoidPtr) gfpt;
    gfpt->gfptSos= sos;
    gfpt->gfptSosBlocked= (SimpleOutputStream *)0;
    gfpt->gfptSosLzw= (SimpleOutputStream *)0;
    gfpt->FileState = FILE_STATE_WRITE;
    
    _GifError = 0;

    return GifFile;
}

/******************************************************************************
*   Routine to set current GIF version. All files open for write will be      *
* using this version until next call to this routine. Version consists of     *
* 3 characters as "87a" or "89a". No test is made to validate the version.    *
******************************************************************************/
void EGifSetGifVersion(const char *Version)
{
    strncpy(GifVersionPrefix + GIF_VERSION_POS, Version, 3);
}

/************************************************************************/
/*									*/
/*  Write the Screen descriptor to the output.				*/
/*									*/
/*  This routine should be called before any other EGif calls, I.E	*/
/*  immediately after the GIF file openning.				*/
/*									*/
/*  1)  First write the version prefix into the file.			*/
/*  2)  Put the screen descriptor into the file				*/
/*  3)  If we have Global color map - dump it also			*/
/*									*/
/************************************************************************/

int EGifPutScreenDesc(	GifFileType *		GifFile,
			int			Width,
			int			Height,
			int			ColorRes,
			int			BackGround,
			const ColorMapObject *	ColorMap )
{
    int				i;
    GifByteType			Buf[3];
    GifFilePrivateType *	gfpt= (GifFilePrivateType *)GifFile->Private;

    if (gfpt->FileState & FILE_STATE_SCREEN) {
	/* If already has screen descriptor - something is wrong! */
	_GifError = E_GIF_ERR_HAS_SCRN_DSCR;
	return GIF_ERROR;
    }
    if (!IS_WRITEABLE(gfpt)) {
	/* This file was NOT open for writing: */
	_GifError = E_GIF_ERR_NOT_WRITEABLE;
	return GIF_ERROR;
    }

    /*  1  */
    sioOutPutString( GifVersionPrefix, gfpt->gfptSos );

    GifFile->SWidth = Width;
    GifFile->SHeight = Height;
    GifFile->SColorResolution = ColorRes;
    GifFile->SBackGroundColor = BackGround;
    if(ColorMap)
      GifFile->SColorMap=MakeMapObject(ColorMap->ColorCount,ColorMap->Colors);
    else
      GifFile->SColorMap=NULL;

    /*  2  */
    sioEndianPutLeInt16( Width, gfpt->gfptSos );
    sioEndianPutLeInt16( Height, gfpt->gfptSos );

    Buf[0] = (ColorMap ? 0x80 : 0x00) |
	     ((ColorRes - 1) << 4) |
	     (ColorMap->BitsPerPixel - 1);

    sioOutPutCharacter( Buf[0], gfpt->gfptSos );
    sioOutPutCharacter( BackGround, gfpt->gfptSos );
    sioOutPutCharacter( 0, gfpt->gfptSos );

    /*  3  */
    if  ( ColorMap != NULL )
	{
	for ( i= 0; i < ColorMap->ColorCount; i++ )
	    {
	    sioOutPutCharacter( ColorMap->Colors[i].Red, gfpt->gfptSos );
	    sioOutPutCharacter( ColorMap->Colors[i].Green, gfpt->gfptSos );
	    sioOutPutCharacter( ColorMap->Colors[i].Blue, gfpt->gfptSos );
	    }
	}

    /* Mark this file as has screen descriptor, and no pixel written yet: */
    gfpt->FileState |= FILE_STATE_SCREEN;

    return GIF_OK;
}

/******************************************************************************
*   This routine should be called before any attemp to dump an image - any    *
* call to any of the pixel dump routines.				      *
******************************************************************************/

int EGifPutImageDesc(	GifFileType *		GifFile,
			int			Left,
			int			Top,
			int			Width,
			int			Height,
			int			Interlace,
			const ColorMapObject *	ColorMap )
{
    int				i;
    GifByteType			Buf[3];
    GifFilePrivateType *	gfpt= (GifFilePrivateType *)GifFile->Private;

    if (gfpt->FileState & FILE_STATE_IMAGE &&
#if defined(__MSDOS__) || defined(__GNUC__)
	gfpt->PixelCount > 0xffff0000UL) {
#else
	gfpt->PixelCount > 0xffff0000) {
#endif /* __MSDOS__ */
	/* If already has active image descriptor - something is wrong! */
	_GifError = E_GIF_ERR_HAS_IMAG_DSCR;
	return GIF_ERROR;
    }
    if (!IS_WRITEABLE(gfpt)) {
	/* This file was NOT open for writing: */
	_GifError = E_GIF_ERR_NOT_WRITEABLE;
	return GIF_ERROR;
    }
    GifFile->Image.Left = Left;
    GifFile->Image.Top = Top;
    GifFile->Image.Width = Width;
    GifFile->Image.Height = Height;
    GifFile->Image.Interlace = Interlace;
    if(ColorMap)
      GifFile->Image.ColorMap =MakeMapObject(ColorMap->ColorCount,ColorMap->Colors);
    else
      GifFile->Image.ColorMap = NULL;

    /* Put the image descriptor into the file: */
    Buf[0] = ',';			       /* Image seperator character. */
    GifWrite(GifFile, Buf, 1);
    sioEndianPutLeInt16( Left, gfpt->gfptSos );
    sioEndianPutLeInt16( Top, gfpt->gfptSos );
    sioEndianPutLeInt16( Width, gfpt->gfptSos );
    sioEndianPutLeInt16( Height, gfpt->gfptSos );
    Buf[0] = (ColorMap ? 0x80 : 0x00) |
	  (Interlace ? 0x40 : 0x00) |
	  (ColorMap ? ColorMap->BitsPerPixel - 1 : 0);
    GifWrite(GifFile, Buf, 1);

    /* If we have Global color map - dump it also: */
    if (ColorMap != NULL)
	for (i = 0; i < ColorMap->ColorCount; i++) {
	    /* Put the ColorMap out also: */
	    Buf[0] = ColorMap->Colors[i].Red;
	    Buf[1] = ColorMap->Colors[i].Green;
	    Buf[2] = ColorMap->Colors[i].Blue;
	    if (GifWrite(GifFile, Buf, 3) != 3) {
	        _GifError = E_GIF_ERR_WRITE_FAILED;
		return GIF_ERROR;
	    }
	}
    if (GifFile->SColorMap == NULL && GifFile->Image.ColorMap == NULL)
    {
	_GifError = E_GIF_ERR_NO_COLOR_MAP;
	return GIF_ERROR;
    }

    /* Mark this file as has screen descriptor: */
    gfpt->FileState |= FILE_STATE_IMAGE;
    gfpt->PixelCount = (long) Width * (long) Height;

    EGifSetupCompress(GifFile);      /* Reset compress algorithm parameters. */

    return GIF_OK;
}

/******************************************************************************
*  Put one full scanned line (Line) of length LineLen into GIF file.	      *
******************************************************************************/

int EGifPutLine(		GifFileType *		GifFile,
				const GifPixelType *	Line,
				int			LineLen )
{
    GifFilePrivateType *	gfpt= (GifFilePrivateType *)GifFile->Private;

    if (!IS_WRITEABLE(gfpt)) {
	/* This file was NOT open for writing: */
	_GifError = E_GIF_ERR_NOT_WRITEABLE;
	return GIF_ERROR;
    }

    if (!LineLen)
      LineLen = GifFile->Image.Width;
    if (gfpt->PixelCount < (unsigned)LineLen) {
	_GifError = E_GIF_ERR_DATA_TOO_BIG;
	return GIF_ERROR;
    }
    gfpt->PixelCount -= LineLen;

    if  ( sioOutWriteBytes( gfpt->gfptSosLzw, Line, LineLen ) != LineLen )
	{
	LDEB(LineLen);
	_GifError = E_GIF_ERR_DISK_IS_FULL; return GIF_ERROR;
	}

    if (gfpt->PixelCount == 0)
    {
	if  ( sioOutClose( gfpt->gfptSosLzw ) )
	    { LDEB(1); return GIF_ERROR;	}
	gfpt->gfptSosLzw= (SimpleOutputStream *)0;

	if  ( sioOutClose( gfpt->gfptSosBlocked ) )
	    { LDEB(1); return GIF_ERROR;	}
	gfpt->gfptSosBlocked= (SimpleOutputStream *)0;
    }

    return GIF_OK;
}

/******************************************************************************
* Put a comment into GIF file using the GIF89 comment extension block.        *
******************************************************************************/
int EGifPutComment(GifFileType *GifFile, const char *Comment)
{
    return EGifPutExtension(GifFile, COMMENT_EXT_FUNC_CODE, strlen(Comment),
								Comment);
}

/******************************************************************************
*   Put a first extension block (see GIF manual) into gif file.  Here more    *
* extensions can be dumped using EGifPutExtensionMid until		      *
* EGifPutExtensionLast is invoked.					      *
******************************************************************************/
int EGifPutExtensionFirst(GifFileType *GifFile, int ExtCode, int ExtLen,
							const VoidPtr Extension)
{
    GifByteType			Buf[3];
    GifFilePrivateType *	Private = (GifFilePrivateType *) GifFile->Private;
    if (!IS_WRITEABLE(Private)) {
	/* This file was NOT open for writing: */
	_GifError = E_GIF_ERR_NOT_WRITEABLE;
	return GIF_ERROR;
    }

    if (ExtCode == 0)
    {
	Buf[0] = ExtLen;
	GifWrite(GifFile,Buf, 1);
    }
    else
    {
	Buf[0] = '!';
	Buf[1] = ExtCode;
	Buf[2] = ExtLen;
	GifWrite(GifFile,Buf, 3);
    }
    GifWrite(GifFile,Extension, ExtLen);

    return GIF_OK;
}

/******************************************************************************
*   Put a middle extension block (see GIF manual) into gif file.	      *
******************************************************************************/
int EGifPutExtensionNext(GifFileType *GifFile, int ExtCode, int ExtLen,
							const VoidPtr Extension)
{
    GifByteType Buf;
    GifFilePrivateType *Private = (GifFilePrivateType *) GifFile->Private;

    if (!IS_WRITEABLE(Private)) {
	/* This file was NOT open for writing: */
	_GifError = E_GIF_ERR_NOT_WRITEABLE;
	return GIF_ERROR;
    }

    Buf = ExtLen;
    GifWrite(GifFile,&Buf, 1);
    GifWrite(GifFile,Extension, ExtLen);

    return GIF_OK;
}

/******************************************************************************
*   Put a last extension block (see GIF manual) into gif file.		      *
******************************************************************************/

int EGifPutExtensionLast(	GifFileType *	GifFile,
				int		ExtCode,
				int		ExtLen,
				const VoidPtr	Extension )
{
    GifByteType Buf;
    GifFilePrivateType *Private = (GifFilePrivateType *) GifFile->Private;

    if (!IS_WRITEABLE(Private)) {
	/* This file was NOT open for writing: */
	_GifError = E_GIF_ERR_NOT_WRITEABLE;
	return GIF_ERROR;
    }

    Buf = ExtLen;
    GifWrite( GifFile, &Buf, 1 );
    GifWrite( GifFile, Extension, ExtLen );

    Buf = 0;
    GifWrite( GifFile, &Buf, 1 );

    return GIF_OK;
}

/******************************************************************************
*   Put an extension block (see GIF manual) into gif file.		      *
******************************************************************************/

int EGifPutExtension(GifFileType *GifFile, int ExtCode, int ExtLen,
						  const VoidPtr Extension)
{
    GifByteType Buf[3];
    GifFilePrivateType *Private = (GifFilePrivateType *) GifFile->Private;

    if (!IS_WRITEABLE(Private)) {
	    /* This file was NOT open for writing: */
	    _GifError = E_GIF_ERR_NOT_WRITEABLE;
	    return GIF_ERROR;
    }

    if (ExtCode == 0)
        GifWrite(GifFile, (GifByteType*)&ExtLen, 1);
    else {
	    Buf[0] = '!';
	    Buf[1] = ExtCode;
	    Buf[2] = ExtLen;
        GifWrite(GifFile, Buf, 3);
    }
    GifWrite(GifFile, Extension, ExtLen);
    Buf[0] = 0;
    GifWrite(GifFile, Buf, 1);

    return GIF_OK;
}

/******************************************************************************
*   This routine should be called last, to close GIF file.		      *
******************************************************************************/

int EGifCloseFile(GifFileType *GifFile)
{
    GifByteType			Buf;
    GifFilePrivateType *	Private;

    if (GifFile == NULL) return GIF_ERROR;

    Private = (GifFilePrivateType *) GifFile->Private;
    if (!IS_WRITEABLE(Private)) {
	/* This file was NOT open for writing: */
	_GifError = E_GIF_ERR_NOT_WRITEABLE;
	return GIF_ERROR;
    }

    Buf = ';';
    GifWrite(GifFile, &Buf, 1);

    if (GifFile->Image.ColorMap)
	FreeMapObject(GifFile->Image.ColorMap);
    if (GifFile->SColorMap)
	FreeMapObject(GifFile->SColorMap);
    if (Private) {
	free((char *) Private);
    }
    free(GifFile);

    return GIF_OK;
}

/************************************************************************/
/*									*/
/*  Setup the LZ compression for this image:				*/
/*									*/
/*  2)  Find out what color map is relevant and determine BitsPerPixel.	*/
/*  3)  Determine code size and insert it in the output stream.		*/
/*  4)  Open a blocked stream to divide the compressed output in	*/
/*	packets. (Push it on the unstructured one.)			*/
/*  5)  Open an LZW compressed stream, push it on the blocked one.	*/
/*									*/
/************************************************************************/

static int EGifSetupCompress(	GifFileType *		GifFile )
{
    int				codeSize;
    GifFilePrivateType *	gfpt= (GifFilePrivateType *)GifFile->Private;

    /*  2  */
    if  ( GifFile->Image.ColorMap )
	{ codeSize = GifFile->Image.ColorMap->BitsPerPixel;	}
    else{
	if  ( GifFile->SColorMap )
	    { codeSize = GifFile->SColorMap->BitsPerPixel;	}
	else{
	    _GifError = E_GIF_ERR_NO_COLOR_MAP;
	    return GIF_ERROR;
	    }
	}

    /*  3  */
    codeSize= ( codeSize < 2 ? 2 : codeSize );
    sioOutPutCharacter( codeSize, gfpt->gfptSos );

    /*  4  */
    if  ( gfpt->gfptSosBlocked )
	{ XDEB(gfpt->gfptSosBlocked);	}

    gfpt->gfptSosBlocked= sioOutBlockedOpen( gfpt->gfptSos );
    if  ( ! gfpt->gfptSosBlocked )
	{ XDEB(gfpt->gfptSosBlocked); return GIF_ERROR;	}

    /*  5  */
    if  ( gfpt->gfptSosLzw )
	{ XDEB(gfpt->gfptSosLzw);	}

    gfpt->gfptSosLzw= sioOutLzwOpen( gfpt->gfptSosBlocked, codeSize );
    if  ( ! gfpt->gfptSosLzw )
	{ XDEB(gfpt->gfptSosLzw); return GIF_ERROR;	}

    return GIF_OK;
}

/******************************************************************************
* This routine writes to disk an in-core representation of a GIF previously   *
* created by DGifSlurp().						      *
******************************************************************************/
int EGifSpew(GifFileType *GifFileOut)
{
    int	i, j, gif89 = FALSE;
    char SavedStamp[GIF_STAMP_LEN + 1];

    for (i = 0; i < GifFileOut->ImageCount; i++)
    {
        for (j = 0; j < GifFileOut->SavedImages[i].ExtensionBlockCount; j++) {
            int function=GifFileOut->SavedImages[i].ExtensionBlocks[j].Function;

	        if (function == COMMENT_EXT_FUNC_CODE
                || function == GRAPHICS_EXT_FUNC_CODE
                || function == PLAINTEXT_EXT_FUNC_CODE
                || function == APPLICATION_EXT_FUNC_CODE)
            gif89 = TRUE;
        }
    }

    strncpy(SavedStamp, GifVersionPrefix, GIF_STAMP_LEN);
    if (gif89)
    {
        strncpy(GifVersionPrefix, GIF89_STAMP, GIF_STAMP_LEN);
    }
    else
    {
        strncpy(GifVersionPrefix, GIF87_STAMP, GIF_STAMP_LEN);
    }
    if (EGifPutScreenDesc(GifFileOut,
			  GifFileOut->SWidth,
			  GifFileOut->SHeight,
			  GifFileOut->SColorResolution,
			  GifFileOut->SBackGroundColor,
			  GifFileOut->SColorMap) == GIF_ERROR)
    {
	strncpy(GifVersionPrefix, SavedStamp, GIF_STAMP_LEN);
	return(GIF_ERROR);
    }
    strncpy(GifVersionPrefix, SavedStamp, GIF_STAMP_LEN);

    for (i = 0; i < GifFileOut->ImageCount; i++)
    {
	SavedImage	*sp = &GifFileOut->SavedImages[i];
	int		SavedHeight = sp->ImageDesc.Height;
	int		SavedWidth = sp->ImageDesc.Width;
	ExtensionBlock	*ep;

	/* this allows us to delete images by nuking their rasters */
	if (sp->RasterBits == NULL)
	    continue;
	
    if (sp->ExtensionBlocks)
	{
        for ( j = 0; j < sp->ExtensionBlockCount; j++) {
            ep = &sp->ExtensionBlocks[j];
            if (EGifPutExtension(GifFileOut,
               (ep->Function != 0) ? ep->Function : '\0',
               ep->ByteCount, ep->Bytes) == GIF_ERROR)
                return (GIF_ERROR);
        }
    }

	if (EGifPutImageDesc(GifFileOut,
			     sp->ImageDesc.Left,
			     sp->ImageDesc.Top,
			     SavedWidth,
			     SavedHeight,
			     sp->ImageDesc.Interlace,
			     sp->ImageDesc.ColorMap
			     ) == GIF_ERROR)
	    return(GIF_ERROR);

	for (j = 0; j < SavedHeight; j++)
	{
	    if (EGifPutLine(GifFileOut,
			    sp->RasterBits + j * SavedWidth,
			    SavedWidth) == GIF_ERROR)
		return(GIF_ERROR);
	}
    }

    if (EGifCloseFile(GifFileOut) == GIF_ERROR)
	return(GIF_ERROR);

    return(GIF_OK);
}
