/******************************************************************************
*   "Gif-Lib" - Yet another gif library.				      *
*									      *
* Written by:  Gershon Elber			IBM PC Ver 1.1,	Aug. 1990     *
*******************************************************************************
* The kernel of the GIF Decoding process can be found here.		      *
*******************************************************************************
* History:								      *
* 16 Jun 89 - Version 1.0 by Gershon Elber.				      *
*  3 Sep 90 - Version 1.1 by Gershon Elber (Support for Gif89, Unique names). *
******************************************************************************/


#if defined (__MSDOS__) && !defined(__DJGPP__) && !defined(__GNUC__)
#include <io.h>
#include <alloc.h>
#include <stdlib.h>
#include <sys\stat.h>
#else
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#endif /* __MSDOS__ */

#ifndef __MSDOS__
#include <stdlib.h>
#endif
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include "bm_gif_lib.h"
#include "bm_gif_lib_private.h"

#   include	<sioBlocked.h>
#   include	<sioLzw.h>
#   include	<appDebugon.h>

#define COMMENT_EXT_FUNC_CODE	0xfe /* Extension function code for comment. */

static int DGifGetWord(GifFileType *GifFile, int *Word);
static int DGifSetupDecompress(GifFileType *GifFile);

/************************************************************************/
/*									*/
/*  Read wrapper.							*/
/*									*/
/************************************************************************/

static int GifRead(	GifFileType *		gft,
			unsigned char *		buf,
			int			len )
    {
    GifFilePrivateType *	gfpt= (GifFilePrivateType *)gft->Private;

    return sioInReadBytes( gfpt->gfptSis, buf, len );
    }

/******************************************************************************
*   Update a new gif file, given its file handle.                             *
*   Returns GifFileType pointer dynamically allocated which serves as the gif *
*   info record. _GifError is cleared if succesfull.                          *
******************************************************************************/

GifFileType *DGifOpenFileHandle(	SimpleInputStream *	sis )
{
    unsigned char		Buf[GIF_STAMP_LEN+1];
    GifFileType *		GifFile;
    GifFilePrivateType *	Private;

    if ((GifFile = (GifFileType *) malloc(sizeof(GifFileType))) == NULL) {
        _GifError = D_GIF_ERR_NOT_ENOUGH_MEM;
        return NULL;
    }

    memset(GifFile, '\0', sizeof(GifFileType));

    if ((Private = (GifFilePrivateType *) malloc(sizeof(GifFilePrivateType)))
    == NULL) {
        _GifError = D_GIF_ERR_NOT_ENOUGH_MEM;
        free((char *) GifFile);
        return NULL;
    }

    GifFile->Private = (VoidPtr) Private;
    Private->gfptSis = sis;
    Private->gfptSisBlocked= (SimpleInputStream *)0;
    Private->gfptSisLzw= (SimpleInputStream *)0;
    Private->FileState = FILE_STATE_READ;

    /* Lets see if this is a GIF file: */
    if (GifRead(GifFile,Buf, GIF_STAMP_LEN) != GIF_STAMP_LEN) {
        _GifError = D_GIF_ERR_READ_FAILED;
        free((char *) Private);
        free((char *) GifFile);
        return NULL;
    }

    /* The GIF Version number is ignored at this time. Maybe we should do    */
    /* something more useful with it.                         */
    Buf[GIF_STAMP_LEN] = 0;
    if (strncmp(GIF_STAMP, (char *)Buf, GIF_VERSION_POS) != 0) {
        _GifError = D_GIF_ERR_NOT_GIF_FILE;
        free((char *) Private);
        free((char *) GifFile);
        return NULL;
    }

    if (DGifGetScreenDesc(GifFile) == GIF_ERROR) {
        free((char *) Private);
        free((char *) GifFile);
        return NULL;
    }

    _GifError = 0;

    return GifFile;
}

/******************************************************************************
*   This routine should be called before any other DGif calls. Note that      *
* this routine is called automatically from DGif file open routines.	      *
******************************************************************************/
int DGifGetScreenDesc(GifFileType *GifFile)
{
    int i, BitsPerPixel;
    GifByteType Buf[3];
    GifFilePrivateType *Private = (GifFilePrivateType *) GifFile->Private;

    if (!IS_READABLE(Private)) {
	/* This file was NOT open for reading: */
	_GifError = D_GIF_ERR_NOT_READABLE;
	return GIF_ERROR;
    }

    /* Put the screen descriptor into the file: */
    if (DGifGetWord(GifFile, &GifFile->SWidth) == GIF_ERROR ||
	DGifGetWord(GifFile, &GifFile->SHeight) == GIF_ERROR)
	return GIF_ERROR;

    if (GifRead( GifFile, Buf, 3) != 3) {
	_GifError = D_GIF_ERR_READ_FAILED;
	return GIF_ERROR;
    }
    GifFile->SColorResolution = (((Buf[0] & 0x70) + 1) >> 4) + 1;
    BitsPerPixel = (Buf[0] & 0x07) + 1;
    GifFile->SBackGroundColor = Buf[1];
    if (Buf[0] & 0x80) {		     /* Do we have global color map? */

	GifFile->SColorMap = MakeMapObject(1 << BitsPerPixel, NULL);

	/* Get the global color map: */
	for (i = 0; i < GifFile->SColorMap->ColorCount; i++) {
	    if (GifRead(GifFile, Buf, 3) != 3) {
		_GifError = D_GIF_ERR_READ_FAILED;
		return GIF_ERROR;
	    }
	    GifFile->SColorMap->Colors[i].Red = Buf[0];
	    GifFile->SColorMap->Colors[i].Green = Buf[1];
	    GifFile->SColorMap->Colors[i].Blue = Buf[2];
	}
    }

    return GIF_OK;
}

/******************************************************************************
*   This routine should be called before any attemp to read an image.         *
******************************************************************************/
int DGifGetRecordType(GifFileType *GifFile, GifRecordType *Type)
{
    GifByteType Buf;
    GifFilePrivateType *Private = (GifFilePrivateType *) GifFile->Private;

    if (!IS_READABLE(Private)) {
	/* This file was NOT open for reading: */
	_GifError = D_GIF_ERR_NOT_READABLE;
	return GIF_ERROR;
    }

    if (GifRead( GifFile, &Buf, 1) != 1) {
	_GifError = D_GIF_ERR_READ_FAILED;
	return GIF_ERROR;
    }

    switch (Buf) {
	case ',':
	    *Type = IMAGE_DESC_RECORD_TYPE;
	    break;
	case '!':
	    *Type = EXTENSION_RECORD_TYPE;
	    break;
	case ';':
	    *Type = TERMINATE_RECORD_TYPE;
	    break;
	default:
	    *Type = UNDEFINED_RECORD_TYPE;
	    _GifError = D_GIF_ERR_WRONG_RECORD;
	    return GIF_ERROR;
    }

    return GIF_OK;
}

/******************************************************************************
*   This routine should be called before any attemp to read an image.         *
*   Note it is assumed the Image desc. header (',') has been read.	      *
******************************************************************************/
int DGifGetImageDesc(GifFileType *GifFile)
{
    int i, BitsPerPixel;
    GifByteType Buf[3];
    GifFilePrivateType *Private = (GifFilePrivateType *) GifFile->Private;
	SavedImage	*sp;

    if (!IS_READABLE(Private)) {
	/* This file was NOT open for reading: */
	_GifError = D_GIF_ERR_NOT_READABLE;
	return GIF_ERROR;
    }

    if (DGifGetWord(GifFile, &GifFile->Image.Left) == GIF_ERROR ||
	DGifGetWord(GifFile, &GifFile->Image.Top) == GIF_ERROR ||
	DGifGetWord(GifFile, &GifFile->Image.Width) == GIF_ERROR ||
	DGifGetWord(GifFile, &GifFile->Image.Height) == GIF_ERROR)
	return GIF_ERROR;
    if (GifRead(GifFile,Buf, 1) != 1) {
	_GifError = D_GIF_ERR_READ_FAILED;
	return GIF_ERROR;
    }
    BitsPerPixel = (Buf[0] & 0x07) + 1;
    GifFile->Image.Interlace = (Buf[0] & 0x40);
    if (Buf[0] & 0x80) {	    /* Does this image have local color map? */

	if (GifFile->Image.ColorMap && GifFile->SavedImages == NULL)
	    FreeMapObject(GifFile->Image.ColorMap);

	GifFile->Image.ColorMap = MakeMapObject(1 << BitsPerPixel, NULL);
    
	/* Get the image local color map: */
	for (i = 0; i < GifFile->Image.ColorMap->ColorCount; i++) {
	    if (GifRead(GifFile,Buf, 3) != 3) {
		_GifError = D_GIF_ERR_READ_FAILED;
		return GIF_ERROR;
	    }
	    GifFile->Image.ColorMap->Colors[i].Red = Buf[0];
	    GifFile->Image.ColorMap->Colors[i].Green = Buf[1];
	    GifFile->Image.ColorMap->Colors[i].Blue = Buf[2];
	}
    }

    if (GifFile->SavedImages) {
	    if ((GifFile->SavedImages = (SavedImage *)realloc(GifFile->SavedImages,
		     sizeof(SavedImage) * (GifFile->ImageCount + 1))) == NULL) {
	        _GifError = D_GIF_ERR_NOT_ENOUGH_MEM;
	        return GIF_ERROR;
	    }
    } else {
        if ((GifFile->SavedImages =
             (SavedImage *)malloc(sizeof(SavedImage))) == NULL) {
            _GifError = D_GIF_ERR_NOT_ENOUGH_MEM;
            return GIF_ERROR;
        }
    }

	sp = &GifFile->SavedImages[GifFile->ImageCount];
	memcpy(&sp->ImageDesc, &GifFile->Image, sizeof(GifImageDesc));

    if (GifFile->Image.ColorMap != NULL) {
	    sp->ImageDesc.ColorMap= MakeMapObject(
				    GifFile->Image.ColorMap->ColorCount,
				    GifFile->Image.ColorMap->Colors );
    }

    sp->RasterBits = (GifPixelType *)NULL;
    sp->ExtensionBlockCount = 0;
    sp->ExtensionBlocks = (ExtensionBlock *)NULL;

    GifFile->ImageCount++;

    Private->PixelCount = (long) GifFile->Image.Width *
			    (long) GifFile->Image.Height;

    DGifSetupDecompress(GifFile);  /* Reset decompress algorithm parameters. */

    return GIF_OK;
}

/************************************************************************/
/*									*/
/*  Get one full scanned line (Line) of length LineLen from GIF file.	*/
/*									*/
/*  3)  The image is finished.. Close (and drain) to blocked input	*/
/*	stream.								*/
/*									*/
/************************************************************************/

int DGifGetLine(GifFileType *GifFile, GifPixelType *Line, int LineLen)
{
    GifFilePrivateType *	gfpt= (GifFilePrivateType *)GifFile->Private;

    if (!IS_READABLE(gfpt)) {
	/* This file was NOT open for reading: */
	_GifError = D_GIF_ERR_NOT_READABLE;
	return GIF_ERROR;
    }

    if (!LineLen) LineLen = GifFile->Image.Width;

    if  ( LineLen > gfpt->PixelCount )
	{
	LLDEB(LineLen,gfpt->PixelCount);
	_GifError = D_GIF_ERR_DATA_TOO_BIG;
	return GIF_ERROR;
	}

    gfpt->PixelCount -= LineLen;

    if  ( sioInReadBytes( gfpt->gfptSisLzw, Line, LineLen ) == LineLen )
	{
	if  ( gfpt->PixelCount == 0 )
	    {
	    if  ( ! gfpt->gfptSisLzw )
		{ XDEB(gfpt->gfptSisLzw); return GIF_ERROR;	}

	    if  ( sioInClose( gfpt->gfptSisLzw ) )
		{ XDEB(gfpt->gfptSisLzw); return GIF_ERROR;	}

	    gfpt->gfptSisLzw= (SimpleInputStream *)0;

	    if  ( ! gfpt->gfptSisBlocked )
		{ XDEB(gfpt->gfptSisBlocked); return GIF_ERROR;	}

	    if  ( sioInClose( gfpt->gfptSisBlocked ) )
		{ XDEB(gfpt->gfptSisBlocked); return GIF_ERROR;	}

	    gfpt->gfptSisBlocked= (SimpleInputStream *)0;
	    }

	return GIF_OK;
	}
    else{ LDEB(LineLen); return GIF_ERROR;	}
}

/************************************************************************/
/*									*/
/*  Get an extension block (see GIF manual) from gif file. This		*/
/*  routine only returns the first data block, and			*/
/*  DGifGetExtensionNext shouldbe called after this one until NULL	*/
/*  extension is returned.						*/
/*  The Extension should NOT be freed by the user (not dynamically	*/
/*  allocated).								*/
/*  Note it is assumed the Extension desc. header ('!') has been read.	*/
/*									*/
/************************************************************************/

int DGifGetExtension(	GifFileType *		GifFile,
			int *			ExtCode,
			GifByteType		Extension[256] )
{
    GifByteType			Buf;
    GifFilePrivateType *	Private= (GifFilePrivateType *)GifFile->Private;
    int				got;

    if (!IS_READABLE(Private)) {
	/* This file was NOT open for reading: */
	_GifError = D_GIF_ERR_NOT_READABLE;
	return GIF_ERROR;
    }

    if (GifRead(GifFile,&Buf, 1) != 1) {
	_GifError = D_GIF_ERR_READ_FAILED;
	return GIF_ERROR;
    }

    *ExtCode = Buf;

    if  ( DGifGetExtensionNext( GifFile, &got, Extension) != GIF_OK )
	{ LDEB(1); return GIF_ERROR;	}
    if  ( ! got )
	{ LDEB(got); return GIF_ERROR;	}

    return GIF_OK;
}

/************************************************************************/
/*									*/
/*  Get a following extension block (see GIF manual) from gif file.	*/
/*  This routine should be called until NULL Extension is returned.	*/
/*  The Extension should NOT be freed by the user (not dynamically	*/
/*  allocated).								*/
/*									*/
/************************************************************************/

int DGifGetExtensionNext(	GifFileType *	GifFile,
				int *		pGot,
				GifByteType	Extension[256] )
{
    int				byte;
    GifFilePrivateType *	Private= (GifFilePrivateType *)GifFile->Private;

    byte= sioInGetCharacter( Private->gfptSis );
    if  ( byte == EOF )
	{
	XDEB(byte);
	_GifError = D_GIF_ERR_READ_FAILED;
	return GIF_ERROR;
	}

    if  ( byte == 0 )
	{ *pGot= 0; return GIF_OK; }

    if  ( GifRead( GifFile, Extension+ 1, byte ) != byte )
	{
	LDEB(byte);
	_GifError = D_GIF_ERR_READ_FAILED; return GIF_ERROR;
	}

    *pGot= 1;
    Extension[0]= byte;
    return GIF_OK;
}

/******************************************************************************
*   This routine should be called last, to close the GIF file.		      *
******************************************************************************/
int DGifCloseFile(GifFileType *GifFile)
{
    GifFilePrivateType *Private;

    if (GifFile == NULL) return GIF_ERROR;

    Private = (GifFilePrivateType *) GifFile->Private;

    if (!IS_READABLE(Private)) {
	/* This file was NOT open for reading: */
	_GifError = D_GIF_ERR_NOT_READABLE;
	return GIF_ERROR;
    }

    if (GifFile->Image.ColorMap)
    {
	FreeMapObject(GifFile->Image.ColorMap);
        GifFile->Image.ColorMap = NULL;
    }

    if (GifFile->SColorMap)
    {
	FreeMapObject(GifFile->SColorMap);
	GifFile->SColorMap = NULL;
    }

    if (Private)
    {
	free((char *) Private);
	Private = NULL;
    }

    if (GifFile->SavedImages)
    {
	FreeSavedImages(GifFile);
	GifFile = NULL;
    }

    free(GifFile);

    return GIF_OK;
}

/******************************************************************************
*   Get 2 bytes (word) from the given file:				      *
******************************************************************************/
static int DGifGetWord(GifFileType *GifFile, int *Word)
{
    unsigned char c[2];

    if (GifRead(GifFile,c, 2) != 2) {
	_GifError = D_GIF_ERR_READ_FAILED;
	return GIF_ERROR;
    }

    *Word = (((unsigned int) c[1]) << 8) + c[0];
    return GIF_OK;
}

/************************************************************************/
/*									*/
/*  Setup the LZ decompression for this image				*/
/*									*/
/*  3)  Open the blocked input stream. (Push it on the unstructured	*/
/*	one.)								*/
/*  4)  Open the LZW stream (Push it on the blocked one.)		*/
/*									*/
/************************************************************************/

static int DGifSetupDecompress(	GifFileType *	GifFile )
{
    int				codeSize;
    GifFilePrivateType *	gfpt= (GifFilePrivateType *)GifFile->Private;

    codeSize= sioInGetCharacter( gfpt->gfptSis );
    if  ( codeSize == EOF )
	{ LDEB(codeSize); return GIF_ERROR;	}

    /*  3  */
    if  ( gfpt->gfptSisBlocked )
	{ XDEB(gfpt->gfptSisBlocked);	}

    gfpt->gfptSisBlocked= sioInBlockedOpen( gfpt->gfptSis );
    if  ( ! gfpt->gfptSisBlocked )
	{ XDEB(gfpt->gfptSisBlocked); return GIF_ERROR;	}

    /*  4  */
    if  ( gfpt->gfptSisLzw )
	{ XDEB(gfpt->gfptSisLzw);	}

    gfpt->gfptSisLzw= sioInLzwOpen( gfpt->gfptSisBlocked, codeSize );
    if  ( ! gfpt->gfptSisLzw )
	{ XDEB(gfpt->gfptSisLzw); return GIF_ERROR;	}

    return GIF_OK;
}

/******************************************************************************
* This routine reads an entire GIF into core, hanging all its state info off  *
* the GifFileType pointer.  Call DGifOpenFileName() or DGifOpenFileHandle()   *
* first to initialize I/O.  Its inverse is EGifSpew().			      *
* 
 ******************************************************************************/

int DGifSlurp(GifFileType *GifFile)
{
    int			ImageSize;
    GifRecordType	RecordType;
    SavedImage *	sp;
    GifByteType		ExtData[256];
    SavedImage		temp_save;

    temp_save.ExtensionBlocks=NULL;
    temp_save.ExtensionBlockCount=0;
  
    do {
	if (DGifGetRecordType(GifFile, &RecordType) == GIF_ERROR)
	    return(GIF_ERROR);

	switch (RecordType) {
	    case IMAGE_DESC_RECORD_TYPE:
		if (DGifGetImageDesc(GifFile) == GIF_ERROR)
		    return(GIF_ERROR);

		sp = &GifFile->SavedImages[GifFile->ImageCount-1];
		ImageSize = sp->ImageDesc.Width * sp->ImageDesc.Height;

		sp->RasterBits
		    = (GifPixelType*) malloc(ImageSize * sizeof(GifPixelType));

		if (DGifGetLine(GifFile, sp->RasterBits, ImageSize)
		    == GIF_ERROR)
		    return(GIF_ERROR);

        if (temp_save.ExtensionBlocks) {
            sp->ExtensionBlocks = temp_save.ExtensionBlocks;
            sp->ExtensionBlockCount = temp_save.ExtensionBlockCount;

            temp_save.ExtensionBlocks = NULL;
            temp_save.ExtensionBlockCount=0;

        /* FIXME: The following is wrong.  It is left in only for backwards
         * compatibility.  Someday it should go away.  Use the
         * sp->ExtensionBlocks->Function variable instead.
         */
            sp->Function = sp->ExtensionBlocks[0].Function;

        }

		break;

	    case EXTENSION_RECORD_TYPE:
		if  ( DGifGetExtension( GifFile, &temp_save.Function,
						    ExtData ) == GIF_ERROR )
		    { return(GIF_ERROR);	}
		{
		int	got= 1;

		while( got )
		    {
		    /* Create an extension block with our data */
		    if  ( AddExtensionBlock ( &temp_save, ExtData[0],
					(char *)&ExtData[1] ) == GIF_ERROR )
			{ return (GIF_ERROR); 	}
            
		    if  ( DGifGetExtensionNext( GifFile, &got, ExtData )
								== GIF_ERROR )
			{ return(GIF_ERROR);	}

		    temp_save.Function = 0;
		    }
		}
		break;

	    case TERMINATE_RECORD_TYPE:
		break;

	    default:	/* Should be trapped by DGifGetRecordType */
		break;
	}
    } while (RecordType != TERMINATE_RECORD_TYPE);

    /* Just in case the Gif has an extension block without an associated
     * image... (Should we save this into a savefile structure with no image
     * instead?  Have to check if the present writing code can handle that as
     * well....
     */
    if (temp_save.ExtensionBlocks)
        FreeExtension(&temp_save);
    
    return(GIF_OK);
}

