/************************************************************************/
/*									*/
/*  Print Postscript.							*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<time.h>

#   include	<appPs.h>

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Print a bounding box.						*/
/*									*/
/************************************************************************/

void appPsBoundingBoxComment(  const PrintingState *	ps,
			    const DocumentGeometry *	dgPage,
			    const char *		comment,
			    const char *		orientationComment )
    {
    double			yshift= 0.0;
    const DocumentGeometry *	dgSheet= &(ps->psPrinterGeometry);

    if  ( dgPage->dgPageHighTwips >= dgPage->dgPageWideTwips )
	{
	if  ( dgPage->dgPageHighTwips < dgSheet->dgPageHighTwips )
	    {
	    yshift= ( dgSheet->dgPageHighTwips- dgPage->dgPageHighTwips )/20;
	    }

	fprintf( ps->psFile, "%%%%%s: %g %g %g %g\n",
		comment,
		dgPage->dgLeftMarginTwips/20.0,
		dgPage->dgBottomMarginTwips/20.0+ yshift,
		(dgPage->dgPageWideTwips-
			    dgPage->dgRightMarginTwips)/20.0,
		(dgPage->dgPageHighTwips-
			    dgPage->dgTopMarginTwips)/20.0+ yshift );
	}
    else{
	if  ( dgPage->dgPageWideTwips < dgSheet->dgPageHighTwips )
	    {
	    yshift= ( dgSheet->dgPageHighTwips- dgPage->dgPageWideTwips )/20;
	    }

	fprintf( ps->psFile, "%%%%%s: %g %g %g %g\n",
		comment,
		dgPage->dgTopMarginTwips/20.0,
		dgPage->dgLeftMarginTwips/20.0+ yshift,
		(dgPage->dgPageHighTwips-
			    dgPage->dgBottomMarginTwips)/20.0,
		(dgPage->dgPageWideTwips-
			    dgPage->dgRightMarginTwips)/20.0+ yshift );
	}

    fprintf( ps->psFile, "%%%%%s: %s\n",
				    orientationComment, ps->psOrientation );
    }

/************************************************************************/
/*									*/
/*  Initialisation for 'nup' printing.					*/
/*									*/
/*  1)  Hack: When the page is higher than the sheet, shift by the page	*/
/*	size. This gives better results in case of miscongiguration.	*/
/*									*/
/************************************************************************/

int appPsGetNupFactor(	double *			pFac,
			int *				pRotate,
			int *				pYShift,
			const DocumentGeometry *	dgPage,
			const DocumentGeometry *	dgSheet,
			int				nup,
			int				horizontal )
    {
    double			xf;
    double			yf;

    switch( nup )
	{
	case 1:
	    if  ( dgPage->dgPageHighTwips >= dgPage->dgPageWideTwips	||
		  dgPage->dgPageWideTwips <= dgSheet->dgPageWideTwips	)
		{
		/*  1  */
		if  ( dgPage->dgPageHighTwips > dgSheet->dgPageHighTwips )
		    { *pYShift= dgPage->dgPageHighTwips;	}
		else{ *pYShift= dgSheet->dgPageHighTwips;	}

		*pFac= 1.0; *pRotate= 0;
		return 0;
		}
	    else{
		if  ( dgPage->dgPageWideTwips < dgSheet->dgPageHighTwips )
		    {
		    *pYShift= dgSheet->dgPageHighTwips-
						dgPage->dgPageWideTwips;
		    }
		else{ *pYShift= 0.0;	}

		*pFac= 1.0; *pRotate= 1;
		return 0;
		}

	case 2:
	    if  ( dgPage->dgPageHighTwips >= dgPage->dgPageWideTwips )
		{
		xf= ( 1.0* dgSheet->dgPageHighTwips )/
					    ( 2.0* dgPage->dgPageWideTwips );
		yf= ( 1.0* dgSheet->dgPageWideTwips )/
					    ( 1.0* dgPage->dgPageHighTwips );

		if  ( xf > 1.0 )	{ xf=  1.0;	}
		if  ( yf > 1.0 )	{ yf=  1.0;	}

		if  ( xf > yf )
		    { xf=  yf;	}

		*pFac= xf; *pRotate= 1; return 0;
		}
	    else{
		xf= ( 1.0* dgSheet->dgPageWideTwips )/
					    ( 1.0* dgPage->dgPageWideTwips );
		yf= ( 1.0* dgSheet->dgPageHighTwips )/
					    ( 2.0* dgPage->dgPageHighTwips );

		if  ( xf > 1.0 )	{ xf=  1.0;	}
		if  ( yf > 1.0 )	{ yf=  1.0;	}

		if  ( xf > yf )
		    { xf=  yf;	}

		*pFac= xf; *pRotate= 0; return 0;
		}

	case 4:
	    if  ( dgPage->dgPageHighTwips >= dgPage->dgPageWideTwips )
		{
		xf= ( 1.0* dgSheet->dgPageWideTwips )/
					( 2.0* dgPage->dgPageWideTwips );
		yf= ( 1.0* dgSheet->dgPageHighTwips )/
					( 2.0* dgPage->dgPageHighTwips );

		if  ( xf > 1.0 )	{ xf=  1.0;	}
		if  ( yf > 1.0 )	{ yf=  1.0;	}

		if  ( xf > yf )
		    { xf=  yf;	}

		*pYShift= 0.0; *pFac= xf; *pRotate= 0;
		return 0;
		}
	    else{
		xf= ( 1.0* dgSheet->dgPageHighTwips )/
					( 2.0* dgPage->dgPageWideTwips );
		yf= ( 1.0* dgSheet->dgPageWideTwips )/
					( 2.0* dgPage->dgPageHighTwips );

		if  ( xf > 1.0 )	{ xf=  1.0;	}
		if  ( yf > 1.0 )	{ yf=  1.0;	}

		if  ( xf > yf )
		    { xf=  yf;	}

		if  ( dgPage->dgPageWideTwips < dgSheet->dgPageHighTwips )
		    {
		    *pYShift= dgSheet->dgPageHighTwips-
						dgPage->dgPageWideTwips;
		    }
		else{ *pYShift= 0.0;	}

		*pFac= xf; *pRotate= 1;
		return 0;
		}

	default:
	    LDEB(nup); return -1;
	}
    }

int appPsSetNup(	PrintingState *			ps,
			const DocumentGeometry *	dgPage,
			int				nup,
			int				horizontal )
    {
    int				yshift= 0;
    const DocumentGeometry *	dgSheet= &(ps->psPrinterGeometry);

    NupTransform *		nt;

    double			fac;
    int				rotate;

    if  ( appPsGetNupFactor( &fac, &rotate, &yshift,
					dgPage, dgSheet, nup, horizontal ) )
	{ LDEB(nup); return -1;	}

    switch( nup )
	{
	case 1:
	    if  ( rotate )
		{
		ps->psTransform.ntAxx= 0.0;
		ps->psTransform.ntAxy= fac/ 20.0;
		ps->psTransform.ntAyx= fac/ 20.0;
		ps->psTransform.ntAyy= 0.0;
		ps->psTransform.ntTx=  0.0;
		ps->psTransform.ntTy=  yshift/20.0;

		ps->psOrientation= "Landscape";
		}
	    else{
		ps->psTransform.ntAxx= fac/ 20.0;
		ps->psTransform.ntAxy= 0.0;
		ps->psTransform.ntAyx= 0.0;
		ps->psTransform.ntAyy= -fac/ 20.0;
		ps->psTransform.ntTx=  0.0;
		ps->psTransform.ntTy=  yshift/20.0;

		ps->psOrientation= "Portrait";
		}

	    ps->psNup= nup;
	    return 0;

	case 2:
	    nt= realloc( ps->psNupTransforms, nup* sizeof( NupTransform ) );
	    if  ( ! nt )
		{ XDEB(nt); return -1;	}
	    ps->psNupTransforms= nt;
	    ps->psNup= nup;

	    if  ( rotate )
		{
		nt[0].ntAxx= 0.0;
		nt[0].ntAxy= fac/ 20.0;
		nt[0].ntAyx= fac/ 20.0;
		nt[0].ntAyy= 0.0;
		nt[0].ntTx=  0.0;
		nt[0].ntTy=  0.0;

		nt[1]= nt[0];
		nt[1].ntTx=  0.0;
		nt[1].ntTy=  ( dgSheet->dgPageHighTwips/ 2 )/ 20.0;

		ps->psOrientation= "Landscape";
		}
	    else{
		nt[0].ntAxx=  fac/ 20.0;
		nt[0].ntAxy=  0.0;
		nt[0].ntAyx=  0.0;
		nt[0].ntAyy= -fac/ 20.0;
		nt[0].ntTx=  0.0;
		nt[0].ntTy=  ( 1.0* dgSheet->dgPageHighTwips )/ 20.0;

		nt[1]= nt[0];
		nt[1].ntTx=  0.0;
		nt[1].ntTy=  ( 0.5* dgSheet->dgPageHighTwips )/ 20.0;

		ps->psOrientation= "Portrait";
		}

	    ps->psTransform= nt[0];
	    return 0;

	case 4:
	    nt= realloc( ps->psNupTransforms, nup* sizeof( NupTransform ) );
	    if  ( ! nt )
		{ XDEB(nt); return -1;	}
	    ps->psNupTransforms= nt;
	    ps->psNup= nup;

	    if  ( rotate )
		{
		nt[0].ntAxx= 0.0;
		nt[0].ntAxy= fac/ 20.0;
		nt[0].ntAyx= fac/ 20.0;
		nt[0].ntAyy= 0.0;
		nt[0].ntTx=  0.0;
		nt[0].ntTy=  0.0;

		ps->psOrientation= "Landscape";
		}
	    else{
		nt[0].ntAxx= fac/ 20.0;
		nt[0].ntAxy= 0.0;
		nt[0].ntAyx= 0.0;
		nt[0].ntAyy= -fac/ 20.0;
		nt[0].ntTx=  0.0;
		nt[0].ntTy=  ( 1.0* dgSheet->dgPageHighTwips )/ 20.0;

		ps->psOrientation= "Portrait";
		}

	    if  ( horizontal )
		{
		nt[1]= nt[0];
		nt[2]= nt[0];

		if  ( rotate )
		    {
		    nt[1].ntTx=  0.0;
		    nt[1].ntTy=  ( 0.5* dgSheet->dgPageHighTwips )/ 20.0;

		    nt[2].ntTx=  ( 0.5* dgSheet->dgPageWideTwips )/ 20.0;
		    nt[2].ntTy=  0.0;
		    }
		else{
		    nt[1].ntTx=  ( 0.5* dgSheet->dgPageWideTwips )/ 20.0;
		    nt[1].ntTy=  ( 1.0* dgSheet->dgPageHighTwips )/ 20.0;

		    nt[2].ntTx=  0.0;
		    nt[2].ntTy=  ( 0.5* dgSheet->dgPageHighTwips )/ 20.0;
		    }
		}
	    else{
		nt[1]= nt[0];
		nt[2]= nt[0];

		if  ( rotate )
		    {
		    nt[1].ntTx=  ( 0.5* dgSheet->dgPageWideTwips )/ 20.0;
		    nt[1].ntTy=  0.0;

		    nt[2].ntTx=  0.0;
		    nt[2].ntTy=  ( 0.5* dgSheet->dgPageHighTwips )/ 20.0;
		    }
		else{
		    nt[1].ntTx=  0.0;
		    nt[1].ntTy=  ( 0.5* dgSheet->dgPageHighTwips )/ 20.0;

		    nt[2].ntTx=  ( 0.5* dgSheet->dgPageWideTwips )/ 20.0;
		    nt[2].ntTy=  ( 1.0* dgSheet->dgPageHighTwips )/ 20.0;
		    }
		}

	    nt[3]= nt[0];
	    nt[3].ntTx=  ( 0.5* dgSheet->dgPageWideTwips )/ 20.0;
	    nt[3].ntTy=  ( 0.5* dgSheet->dgPageHighTwips )/ 20.0;

	    ps->psTransform= nt[0];
	    return 0;

	default:
	    LDEB(nup); return -1;
	}
    }

/************************************************************************/
/*									*/
/*  Issue a character string.						*/
/*									*/
/************************************************************************/

void appPsPrintString(	FILE *			f,
			const unsigned char *	s,
			int			len )
    {
    int		i;

    for ( i= 0; i < len; s++, i++ )
	{
	if  ( *s == '(' || *s == ')' || *s == '\\' )
	    { putc( '\\', f ); }
	putc( *s, f );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Set the font, this depends on the assumption that fonts have been	*/
/*  defined for the different fonts in the document.			*/
/*									*/
/************************************************************************/

void appPsSetFont(	FILE *		f,
			const char *	prefix,
			TextAttribute	ta )
    {
    int		fontSizeTwips= 10* ta.taFontSizeHalfPoints;

    if  ( ta.taSuperSub == DOCfontSUPERSCRIPT	||
	  ta.taSuperSub == DOCfontSUBSCRIPT	)
	{ fontSizeTwips= ( 6* fontSizeTwips )/ 10; }

    if  ( fontSizeTwips == 0 )
	{ LDEB(fontSizeTwips); fontSizeTwips= 1;	}

    fprintf( f, "[ %d 0 0 %d 0 0 ] %s%d",
		fontSizeTwips, -fontSizeTwips, prefix, ta.taFontNumber );

    if  ( ta.taFontIsBold )
	{ putc( 'b', f );	}
    if  ( ta.taFontIsSlanted )
	{ putc( 'i', f );	}
    putc( '\n', f );

    return;
    }

/************************************************************************/
/*									*/
/*  Collect the fonts in a document					*/
/*  This has two functions:						*/
/*	1)  They are needed for the DSC comments.			*/
/*	2)  This makes it possible to define shorthands in the prologue	*/
/*									*/
/************************************************************************/

int appPsRememberAfi(	AfmFontInfo *		afi,
			TextAttribute		ta,
			const char *		prefix,
			int			appearsInText,
			PostScriptFont **	pFontList,
			int *			pCount )
    {
    int			i;
    PostScriptFont *	psf= *pFontList;

    for ( i= 0; i < *pCount; psf++, i++ )
	{
	if  ( psf->psfAttributes.taFontNumber == ta.taFontNumber	&&
	      psf->psfAttributes.taFontIsBold == ta.taFontIsBold	&&
	      psf->psfAttributes.taFontIsSlanted == ta.taFontIsSlanted	&&
	      ! strcmp( psf->psfFontPrefix, prefix )			)
	    { break;	}
	}

    if  ( i >= *pCount )
	{
	char *			s;

	psf= (PostScriptFont *)
		    realloc( *pFontList, (i+ 1)* sizeof(PostScriptFont) );
	if  ( ! psf )
	    { LXDEB(i,psf); return -1;	}
	*pFontList= psf;
	*pCount= i+ 1;

	(*pFontList)[i].psfAfi= afi;
	(*pFontList)[i].psfAttributes= ta;
	(*pFontList)[i].psfAppearsInText= appearsInText;

	strcpy( (*pFontList)[i].psfFontPrefix, prefix );

	s= (*pFontList)[i].psfFontId;
	sprintf( s, "%s%d", prefix, ta.taFontNumber );
	if  ( ta.taFontIsBold )
	    { s += strlen( s ); s[0]= 'b'; s[1]= '\0'; s++; }
	if  ( ta.taFontIsSlanted )
	    { s += strlen( s ); s[0]= 'i'; s[1]= '\0'; s++; }
	}
    else{
	if  ( appearsInText )
	    { psf->psfAppearsInText= 1;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Initialise a printing session.					*/
/*									*/
/************************************************************************/

void appPsInitPrintingState(	PrintingState *	ps )
    {
    ps->psFile= (FILE *)0;
    ps->psCurrentPhysicalFont= -1;
    ps->psPagesPrinted= 0;
    ps->psSheetsPrinted= 0;

    ps->psTransform.ntAxx= 1.0;
    ps->psTransform.ntAxy= 0.0;
    ps->psTransform.ntAyx= 0.0;
    ps->psTransform.ntAyy= 1.0;
    ps->psTransform.ntTx=  0.0;
    ps->psTransform.ntTy=  0.0;

    ps->psOrientation= (char *)0;
    appInitDocumentGeometry( &(ps->psPrinterGeometry) );

    ps->psNup= 1;
    ps->psNupTransforms= (NupTransform *)0;

    ps->psInLink= 0;
    ps->psCurrentColor= PScolorNONE;

    ps->psLinkParticulesDone= 0;
    ps->psLinkRectLeft= -1;
    ps->psLinkFile= (const char *)0;
    ps->psLinkFileSize= 0;
    ps->psLinkMark= (const char *)0;
    ps->psLinkMarkSize= 0;

    return;
    }

void appPsCleanPrintingState(	PrintingState *	ps )
    {
    if  ( ps->psNupTransforms )
	{ free( ps->psNupTransforms );	}

    return;
    }

/************************************************************************/
/*									*/
/*  Switch to the 'Link Color'						*/
/*									*/
/************************************************************************/

void appPsSetLinkColor(	PrintingState *	ps )
    {
    if  ( ps->psCurrentColor != PScolorLINK )
	{
	fprintf( ps->psFile, "0.0 0.0 0.8 setrgbcolor\n" );
	ps->psCurrentColor= PScolorLINK;
	}
    }

void appPsSetTextColor(	PrintingState *	ps )
    {
    if  ( ps->psCurrentColor != PScolorTEXT )
	{
	fprintf( ps->psFile, "0 setgray\n" );
	ps->psCurrentColor= PScolorTEXT;
	}
    }

/************************************************************************/
/*									*/
/*  Go to the next page.						*/
/*									*/
/************************************************************************/

static void appPsPageOperator(		const char *		operator,
					const PrintingState *	ps,
					int			documentPage )
    {
    fprintf( ps->psFile, "%s %% Page %d # %d Sheet %d\n", operator,
				    documentPage+ 1,
				    ps->psPagesPrinted+ 1,
				    ps->psSheetsPrinted+ 1 );
    }

void appPsStartPage(	PrintingState *			ps,
			int				documentPage,
			const DocumentGeometry *	dgPage )
    {
    if  ( ps->psNup == 1			||
	  ps->psPagesPrinted % ps->psNup == 0	)
	{
	if  ( ps->psNup == 1 )
	    {
	    fprintf( ps->psFile, "%%%%Page: %d %d\n",
				    documentPage+ 1, ps->psSheetsPrinted+ 1 );
	    }
	else{
	    fprintf( ps->psFile, "%%%%Page: (%d ..) %d\n",
				    documentPage+ 1, ps->psSheetsPrinted+ 1 );
	    }

	appPsBoundingBoxComment( ps, dgPage,
				    "PageBoundingBox", "PageOrientation" );

	fprintf( ps->psFile, "%%%%BeginPageSetup\n" );

	ps->psCurrentTransform= ps->psTransform;
	}
    else{
	ps->psCurrentTransform=
			ps->psNupTransforms[ps->psPagesPrinted % ps->psNup];
	}

    appPsPageOperator( "gsave", ps, documentPage );

    fprintf( ps->psFile, "[ %g %g %g %g %g %g ] concat\n",
					ps->psCurrentTransform.ntAxx,
					ps->psCurrentTransform.ntAxy,
					ps->psCurrentTransform.ntAyx,
					ps->psCurrentTransform.ntAyy,
					ps->psCurrentTransform.ntTx,
					ps->psCurrentTransform.ntTy );

    ps->psCurrentColor= PScolorNONE;
    if  ( ps->psInLink )
	{ appPsSetLinkColor( ps );	}
    else{ appPsSetTextColor( ps );	}

    if  ( ps->psNup == 1			||
	  ps->psPagesPrinted % ps->psNup == 0	)
	{ fprintf( ps->psFile, "%%%%EndPageSetup\n" );	}

    return;
    }

void appPsFinishPage(	PrintingState *		ps,
			int			documentPage,
			int			asLast )
    {
    if  ( asLast					||
	  ps->psNup == 1				||
	  ( ps->psPagesPrinted+ 1 ) % ps->psNup == 0	)
	{
	appPsPageOperator( "showpage grestore", ps, documentPage );

	fprintf( ps->psFile, "%%%%PageTrailer\n" );

	ps->psPagesPrinted++;
	ps->psSheetsPrinted++;
	}
    else{
	appPsPageOperator( "grestore", ps, documentPage );

	ps->psPagesPrinted++;
	}

    if  ( asLast )
	{
	fprintf( ps->psFile, "%%%%Trailer\n" );
	fprintf( ps->psFile, "%%%%Pages: %d\n", ps->psSheetsPrinted );
	fprintf( ps->psFile, "%%%%EOF\n" );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Make function names for setting the fonts in a list.		*/
/*									*/
/************************************************************************/

void appPsFontNames(	FILE *				f,
			PostScriptFont *		fontList,
			int				fontCount,
			int				allFonts )
    {
    static const char *	ISO_SUFFIX= "-ISO1";

    PostScriptFont *	psf;

    int			i;
    int			started= 0;

    psf= fontList;
    for ( i= 0; i < fontCount; psf++, i++ )
	{
	AfmFontInfo *	afi= psf->psfAfi;

	if  ( psf->psfAppearsInText && ! allFonts )
	    { continue;	}

	if  ( ! started )
	    { fprintf( f, "\n" ); started= 1;	}

	if  ( afi->afiMapToIso )
	    {
	    int			done;
	    PostScriptFont *	psd;

	    psd= fontList;
	    for ( done= 0; done < i; psd++, done++ )
		{
		if  ( psd->psfAfi->afiMapToIso				    &&
		      ! strcmp( afi->afiFontName, psd->psfAfi->afiFontName ) )
		    { break;	}
		}

	    if  ( done >= i )
		{
		fprintf( f, "/%s findfont dup length dict begin\n",
							afi->afiFontName );
		fprintf( f, "  {\n" );
		fprintf( f, "    1 index /FID ne\n" );
		fprintf( f, "      { def } { pop pop } ifelse\n" );
		fprintf( f, "  } forall\n");
		fprintf( f, "  /Encoding ISOLatin1Encoding def currentdict\n" );
		fprintf( f, "end " );

		fprintf( f, "/%s%s exch definefont pop\n\n",
						afi->afiFontName, ISO_SUFFIX );
		}
	    }
	}

    psf= fontList;
    for ( i= 0; i < fontCount; psf++, i++ )
	{
	fprintf( f, "/%s\t{ /%s%s exch selectfont } def\n",
				psf->psfFontId,
				psf->psfAfi->afiFontName,
				psf->psfAfi->afiMapToIso?ISO_SUFFIX:"" );
	}

    fprintf( f, "\n" );

    return;
    }

/************************************************************************/
/*									*/
/*  Write the header for an EPS file.					*/
/*									*/
/************************************************************************/

void appPsWriteEpsHeader(	FILE *			f,
				const char *		creator,
				const char *		title,
				int			pointsWide,
				int			pointsHigh )
    {
    long			now;

    now= time( (long *)0 );

    fprintf( f, "%%!PS-Adobe-3.0 EPSF-3.0\n" );

    if  ( creator )
	{ fprintf( f, "%%%%Creator: %s\n", creator ); }

    if  ( title )
	{ fprintf( f, "%%%%Title: %s\n", title );	}

    fprintf( f, "%%%%CreationDate: %s", ctime(&now) );
    fprintf( f, "%%%%BoundingBox: 0 0 %d %d\n", pointsWide, pointsHigh );
    fprintf( f, "%%%%EndComments\n");

    return;
    }

/************************************************************************/
/*									*/
/*  Define a procedure from an array of lines of PostScript code.	*/
/*  Empty lines and lines completely consisting of a comment are	*/
/*  skipped.								*/
/*									*/
/************************************************************************/

void appPsDefineProcedure(	FILE *			f,
				const char **		lines,
				int			count )
    {
    int		i;

    for ( i= 0; i < count; lines++, i++ )
	{
	const char *	s= *lines;

	while( isspace( *s ) )
	    { s++;	}

	if  ( ! *s || *s == '%' )
	    { continue;	}

	fprintf( f, "%s\n", *lines );
	}

    fprintf( f, "\n" );
    }

/************************************************************************/
/*									*/
/*  Write the header for an EPS file.					*/
/*									*/
/************************************************************************/

static const char * appPsRectfillEmulation[]=
    {
    "%%rectfill emulation for one rectangle only",
    "/rectfill where",
    "    { pop }",
    "    { /rectfill",
    "        { 4 2 roll moveto 2 copy",
    "            0 exch rlineto 0 rlineto ",
    "            neg 0 exch rlineto pop closepath fill",
    "        } bind def",
    "    }",
    "ifelse",
    };

void appPsSetRectfillEmulation(	FILE *	f )
    {
    appPsDefineProcedure( f, appPsRectfillEmulation,
		    sizeof(appPsRectfillEmulation)/sizeof(const char *) );

    return;
    }

static const char * appPsSelectfontEmulation[]=
    {
    "%%selectfont emulation",
    "/selectfont where",
    "    { pop }",
    "    { /selectfont",
    "        { exch findfont exch dup type /arraytype eq",
    "          { makefont } { scalefont } ifelse",
    "          setfont",
    "        } bind def",
    "    }",
    "ifelse",
    };

void appPsSetSelectfontEmulation(	FILE *	f )
    {
    appPsDefineProcedure( f, appPsSelectfontEmulation,
		    sizeof(appPsSelectfontEmulation)/sizeof(const char *) );

    return;
    }

/************************************************************************/
/*									*/
/*  Define procedures to Begin/End the inclusion of an EPS graphic.	*/
/*									*/
/*  *)  Copied directly from the EPS spec.				*/
/*  1)  I spent some time wondering why the 'begin' operator was not	*/
/*	matched by an 'end' operator. The matching end is in the loop	*/
/*	as 'op_count' is remembered before the copy of 'userdict' is	*/
/*	pushed.								*/
/*									*/
/************************************************************************/

static const char * appBeginEPSF[]=
    {
    "/BeginEPSF",
    "    {",
    "    /b4_Inc_state save def          % Save state for cleanup",
    "    /dict_count countdictstack def  % Count objects on dict stack",
    "    /op_count count 1 sub def       % Count objects on operand stack",
	 /*  1  */
    "    userdict begin                  % Push userdict on dict stack",
    "    /showpage { } def               % Redefine showpage, { } = null proc",
    "    0 setgray 0 setlinecap          % Prepare graphics state",
    "    1 setlinewidth 0 setlinejoin",
    "    10 setmiterlimit [ ] 0 setdash newpath",
    "    /languagelevel where            % If level not equal to 1 then",
    "    { pop languagelevel             % set strokeadjust and",
    "        1 ne                        % overprint to their defaults.",
    "            { false setstrokeadjust false setoverprint",
    "            } if",
    "        } if",
    "    } bind def",
    };

static const char * appEndEPSF[]=
    {
    "/EndEPSF",
    "    {",
    "    count op_count sub {pop} repeat % Clean up stacks",
	 /*  1  */
    "    countdictstack dict_count sub {end} repeat",
    "    b4_Inc_state restore",
    "    } bind def",
    };

void appPsDefineEpsProcs(	FILE *		f )
    {
    appPsDefineProcedure( f, appBeginEPSF,
				sizeof(appBeginEPSF)/sizeof(const char *) );

    appPsDefineProcedure( f, appEndEPSF,
				sizeof(appEndEPSF)/sizeof(const char *) );

    return;
    }

void appPsBeginEpsObject(	PrintingState *		ps,
				int			x0Twips,
				int			y0Twips,
				int			llxTwips,
				int			llyTwips,
				int			urxTwips,
				int			uryTwips,
				const unsigned char *	file )
    {
    if  ( ! file )
	{ file= (const unsigned char *)"??";	}

    fprintf( ps->psFile, "BeginEPSF\n" );

    fprintf( ps->psFile, "[ %d %d %d %d %d %d ] concat\n",
				    20, 0, 0, -20, x0Twips, y0Twips );

    fprintf( ps->psFile, "newpath %d %d moveto ",
					    llxTwips/20, llyTwips/20 );
    fprintf( ps->psFile, "%d %d lineto ",
					    urxTwips/20, llyTwips/20 );
    fprintf( ps->psFile, "%d %d lineto ",
					    urxTwips/20, uryTwips/20 );
    fprintf( ps->psFile, "%d %d lineto ",
					    llxTwips/20, uryTwips/20 );
    fprintf( ps->psFile, "closepath clip\n" );

    fprintf( ps->psFile, "%%%%BeginDocument: (" );
    appPsPrintString( ps->psFile, file, strlen( (char *)file ) );
    fprintf( ps->psFile, ")\n" );

    return;
    }


void appPsEndEpsObject(	PrintingState *		ps )
    {
    fprintf( ps->psFile, "%%%%EndDocument\n" );
    fprintf( ps->psFile, "EndEPSF\n" );

    return;
    }
