/************************************************************************/
/*									*/
/*  Geometry calculations.						*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<limits.h>
#   define	y0	math_y0
#   define	y1	math_y1
#   include	<math.h>
#   undef	y0
#   undef	y1

#   include	"tedApp.h"
#   include	"docPs.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Calculate border width.						*/
/*									*/
/************************************************************************/

int tedBorderThick(		int *				pWide,
				const BorderProperties *	bp,
				const AppDrawingData *		add )
    {
    int			wide= bp->bpPenWideTwips+ bp->bpSpacingTwips;
    int			thick= bp->bpPenWideTwips;

    if  ( bp->bpStyle == DOCbsNONE )
	{ thick= 0;	}

    if  ( thick > 0 )
	{
	thick *= add->addMagnifiedPixelsPerTwip;
	wide  *= add->addMagnifiedPixelsPerTwip;

	if  ( thick == 0 )
	    { thick= 1;	}

	*pWide= wide;
	}
    else{ *pWide= 0;	}

    return thick;
    }

/************************************************************************/
/*									*/
/*  Calculate the width of a string.					*/
/*									*/
/************************************************************************/

static int tedTextWidth(	const AppPhysicalFont *		apf,
				const AppDrawingData *		add,
				const unsigned char *		s,
				int				len )
    {
    int		wide;
    int		fontAscent;
    int		fontDescent;

    appDrawTextExtents( &wide, &fontAscent, &fontDescent,
			    add, apf->apfFontStruct, (const char *)s, len );

    return wide;
    }

static int tedVisibleTextWidth(	const AppPhysicalFont *		apf,
				const AppDrawingData *		add,
				const unsigned char *		s,
				int				len )
    {
    int		wide;
    int		fontAscent;
    int		fontDescent;

    while( len > 0 && s[len- 1] == ' ' )
	{ len--;	}

    if  ( len == 0 )
	{ return 0;	}

    appDrawTextExtents( &wide, &fontAscent, &fontDescent,
			    add, apf->apfFontStruct, (const char *)s, len );

    return wide;
    }

/************************************************************************/
/*									*/
/*  Place as many particules on a line as possible.			*/
/*									*/
/*  1)  Accept at least one particule.					*/
/*  9)  Predictable undefied behaviour.					*/
/*									*/
/************************************************************************/

static int tedLayoutParticules(	const BufferItem *		bi,
				AppDrawingData *		add,
				const FormattingFrame *		ffPixels,
				TextParticule *			tp,
				ParticuleData *			pd,
				int				particuleCount,
				int				x0 )
    {
    int				accepted;
    int				width;

    int				len= 0;
    int				xFrom= x0;
    const unsigned char *	from= bi->biParaString+ tp->tpStroff;

    int				physicalFont= tp->tpPhysicalFont;
    AppPhysicalFontList *	apfl= &(add->addPhysicalFontList);
    AppPhysicalFont *		apf= apfl->apflFonts+ physicalFont;

    accepted= 0;
    while( accepted < particuleCount )
	{
	switch( tp->tpKind )
	    {
	    int		x1;

	    case DOCkindTEXT:
		while( accepted < particuleCount		&&
		       tp->tpPhysicalFont == physicalFont	&&
		       tp->tpKind == DOCkindTEXT		)
		    {
		    len += tp->tpStrlen;
		    width= tedTextWidth( apf, add, from, len );

		    pd->pdVisiblePixels= tedVisibleTextWidth(
						apf, add,
						bi->biParaString+ tp->tpStroff,
						tp->tpStrlen );
		    pd->pdWhiteUnits= xFrom+ width- x0- pd->pdVisiblePixels;

		    tp->tpX0= x0;
		    tp->tpPixelsWide= xFrom+ width- x0;
		    x0= xFrom+ width;

		    accepted++; tp++;
		    pd++; break;
		    }

		break;
	    case DOCkindTAB:
		x1= TWIPStoPIXELS( add->addMagnifiedPixelsPerTwip,
							pd->pdTabPosition );

		pd->pdVisiblePixels= 0;
		pd->pdWhiteUnits= x1- x0;
		tp->tpObjectNumber= pd->pdTabNumber;

		tp->tpX0= x0;
		tp->tpPixelsWide= x1- x0;
		x0= x1;

		accepted++; tp++;
		pd++;
		break;

	    case DOCkindOBJECT:
		width= bi->biParaObjects[tp->tpObjectNumber].ioPixelsWide;

		pd->pdVisiblePixels= width;
		pd->pdWhiteUnits= 0;

		tp->tpX0= x0;
		tp->tpPixelsWide= width;
		x0 += width;

		accepted++; tp++;
		pd++;
		break;

	    case DOCkindFIELDSTART:
	    case DOCkindFIELDEND:
	    case DOCkindXE:
	    case DOCkindTC:
	    case DOCkindLINEBREAK:
	    case DOCkindPAGEBREAK:
		if  ( pd )
		    { pd->pdVisiblePixels= 0; pd->pdWhiteUnits= 0;	}

		tp->tpX0= x0;
		tp->tpPixelsWide= 0;
		accepted++; tp++;
		if  ( pd )
		    { pd++;	}
		break;

	    default:
		LDEB(tp->tpKind); return -1;
	    }

	if  ( accepted >= particuleCount )
	    { return accepted;	}

	physicalFont= tp->tpPhysicalFont;
	apf= apfl->apflFonts+ physicalFont;
	from= bi->biParaString+ tp->tpStroff;
	xFrom= x0;
	len= 0;
	}

    return accepted;
    }

/************************************************************************/
/*									*/
/*  Determine the height of a line.					*/
/*									*/
/************************************************************************/

static void tedLineHeight(	const BufferItem *	bi,
				const TextParticule *	tp,
				const AppDrawingData *	add,
				int			particuleCount,
				int *			pAscent,
				int *			pDescent )
    {
    const AppPhysicalFontList *	apfl= &(add->addPhysicalFontList);
    int				physicalFont= -1;
    AppPhysicalFont *		apf;
    int				ascent= 0;
    int				descent= 0;
    int				objectHeight= 0;

    InsertedObject *		io;

    if  ( particuleCount <= 0 )
	{ LDEB(particuleCount); return;	}

    while( particuleCount > 0 )
	{
	switch( tp->tpKind )
	    {
	    case DOCkindTEXT:
	    case DOCkindTAB:
		physicalFont= tp->tpPhysicalFont;
		apf= apfl->apflFonts+ physicalFont;

		if  ( ascent < apf->apfAscentPixels )
		    { ascent=  apf->apfAscentPixels;	}
		if  ( descent < apf->apfDescentPixels )
		    { descent=  apf->apfDescentPixels; }
		break;

	    case DOCkindOBJECT:
		physicalFont= tp->tpPhysicalFont;
		apf= apfl->apflFonts+ physicalFont;

		if  ( ascent < apf->apfAscentPixels )
		    { ascent=  apf->apfAscentPixels;	}
		if  ( descent < apf->apfDescentPixels )
		    { descent=  apf->apfDescentPixels; }

		io= bi->biParaObjects+ tp->tpObjectNumber;

		if  ( objectHeight < io->ioPixelsHigh )
		    { objectHeight=  io->ioPixelsHigh;	}
		break;

	    case DOCkindFIELDSTART:
	    case DOCkindFIELDEND:
	    case DOCkindXE:
	    case DOCkindTC:
	    case DOCkindLINEBREAK:
	    case DOCkindPAGEBREAK:
		break;

	    default:
		LDEB(tp->tpKind); break;
	    }

	tp++; particuleCount--;
	}

    if  ( ascent < objectHeight )
	{ ascent=  objectHeight; }

    *pAscent= ascent;
    *pDescent= descent;
    }

/************************************************************************/
/*									*/
/*  Justify the particules in a line of text.				*/
/*									*/
/*  1)  Start justification after the last tab of the line. Justifying	*/
/*	after anything else than a left tab is ridiculous. Simply	*/
/*	refuse to.							*/
/*  2)  Ignore organisational particules such as the delimitation of	*/
/*	links and bookmarks at the end of the line.			*/
/*									*/
/************************************************************************/

static void tedJustifyLine(	const BufferItem *	bi,
				TextParticule *		tp,
				ParticuleData *		pd,
				int			accepted,
				int			x1TextLines )
    {
    TextParticule *	tpp;
    ParticuleData *	pdd;

    int			i;
    int			extra;
    int			totalWeight;
    int			step;
    int			left;

    /*  1  */
    left= 0;
    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	if  ( tpp->tpKind == DOCkindTAB )
	    { left= i+ 1;	}
	}

    if  ( left > 0 )
	{
	if  ( pd[left-1].pdTabKind != DOCtkLEFT )
	    { LDEB(pd[left-1].pdTabKind); return;	}

	tp += left; pd += left; accepted -= left;
	}

    /*  2  */
    while( accepted > 0 && tp[accepted- 1].tpStrlen == 0 )
	{ accepted--;	}

    if  ( accepted < 2 )
	{ /*LDEB(accepted);*/ return;	}

    extra= x1TextLines- tp[accepted- 1].tpX0- pd[accepted- 1].pdVisiblePixels;

    totalWeight= 0;
    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	pdd->pdWhiteUnits= 0;
	pdd->pdCorrectBy= 0;

	if  ( bi->biParaString[tpp->tpStroff+tpp->tpStrlen-1] == ' ' )
	    {
	    pdd->pdWhiteUnits=
		sqrt( (double)pdd[0].pdWidth+ (double)pdd[1].pdVisibleWidth );

	    totalWeight += pdd->pdWhiteUnits;
	    }
	}

    left= extra;
    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	if  ( pdd->pdWhiteUnits > 0 )
	    {
	    step= ( extra* pdd->pdWhiteUnits )/ totalWeight;
	    if  ( step > left )
		{ step= left;	}

	    pdd->pdCorrectBy += step;

	    left -= step;
	    }
	}

    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	if  ( pdd->pdWhiteUnits > 0 )
	    {
	    step= 1;
	    if  ( step > left )
		{ step= left;	}

	    pdd->pdCorrectBy += step;

	    left -= step;
	    }
	}

    step= 0;
    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	tpp->tpX0 += step;
	if  ( pdd->pdWhiteUnits > 0 )
	    {
	    tpp->tpPixelsWide += pdd->pdCorrectBy;
	    step += pdd->pdCorrectBy;
	    }
	}

    tpp->tpX0 += step;

    return;
    }

/************************************************************************/
/*									*/
/*  Recalculate the geometry from the start of the selection to the	*/
/*  end of the line.							*/
/*									*/
/*  'part'	is known to be the first particule of the line.		*/
/*									*/
/************************************************************************/

static void tedSqueezeParticules(	TextParticule *		tp,
					ParticuleData *		pd,
					int			accepted,
					int			x0,
					int			x1 )
    {
    TextParticule *	tpp= tp+ accepted- 1;
    ParticuleData *	pdd= pd+ accepted- 1;
    int			i;
    int			whitePixels= 0;
    int			shortage= tpp->tpX0+ pdd->pdVisiblePixels- x1;
    int			xx0;
    int			done;

    for ( i= 0; i < accepted- 1; i++ )
	{
	if  ( pd[i].pdWhiteUnits > 0 )
	    { whitePixels += pd[i].pdWhiteUnits; }

	pd[i].pdCorrectBy= 0;
	}

    if  ( whitePixels < shortage )
	{
	/*
	LLDEB(whitePixels,shortage);
	LLLDEB(x0,x1,tpp->tpX0+ pdd->pdVisiblePixels);
	LLDEB(whitePixels,tpp->tpX0+ pdd->pdVisiblePixels-x1);

	for ( i= 0; i < accepted; i++ )
	    { docListParticule( 0, "SH", part+ i, bi, tp+ i ); }
	*/
	shortage= whitePixels;
	}

    done= 0;

    for ( i= 0; i < accepted- 1; i++ )
	{
	int		step;

	if  ( pd[i].pdWhiteUnits > 0 )
	    {
	    step= ( pd[i].pdWhiteUnits* shortage )/ whitePixels;

	    if  ( step > 0 && done < shortage )
		{
		if  ( step > shortage- done )
		    { step= shortage- done;	}

		pd[i].pdCorrectBy += step;
		done += step;
		}
	    }
	}

    for ( i= 0; i < accepted- 1 && done < shortage; i++ )
	{
	if  ( pd[i].pdWhiteUnits > pd[i].pdCorrectBy )
	    { pd[i].pdCorrectBy++; done++; }
	}

    if  ( done != shortage )
	{ LLDEB(done,shortage);	}

    xx0= x0; whitePixels= 0;
    for ( i= 0; i < accepted- 1; i++ )
	{
	tp[i].tpX0 -= whitePixels;

	if  ( pd[i].pdCorrectBy > 0 )
	    {
	    tp[i].tpPixelsWide -= pd[i].pdCorrectBy;
	    whitePixels += pd[i].pdCorrectBy;
	    }
	}

    tp[i].tpX0 -= whitePixels;
    tp[i].tpPixelsWide= x1- tp[i].tpX0;

    return;
    }

/************************************************************************/
/*									*/
/*  Layout one line on screen, reconciling PostScript and X11 geometry.	*/
/*									*/
/************************************************************************/

static int tedLayoutLine(	const BufferItem *	bi,
				AppDrawingData *	add,
				const FormattingFrame *	ff,
				TextParticule *		tp,
				ParticuleData *		pd,
				int			particuleCount,
				int			x0 )
    {
    int			past= 0;
    int			part= 0;
    int			x1;

    int			accepted;

    while( past < particuleCount )
	{
	int		squeeze= 0;
	int		includeTab= 0;
	int		j;

	TextParticule *	tpp;
	ParticuleData *	pdd;

	j= 0;
	while( past < particuleCount && tp[j].tpKind != DOCkindTAB )
	    { j++; past++;	}

	if  ( past < particuleCount )
	    { includeTab= 1; past++;	}

	accepted= tedLayoutParticules( bi, add, ff, tp, pd, past- part, x0 );

	if  ( accepted != past- part )
	    { LLDEB(accepted,past- part); return -1;	}

	tpp= tp+ accepted- 1;
	pdd= pd+ accepted- 1;

	if  ( includeTab )
	    {
	    x1= TWIPStoPIXELS( add->addMagnifiedPixelsPerTwip,
						    pdd->pdX0+ pdd->pdWidth );

	    if  ( tpp->tpX0+ pdd->pdVisiblePixels > x1	)
		{
		/*
		LLDEB(tpp->tpX0+ tpp->tpPixelsWide,x1);
		LLDEB(tpp->tpX0+ pdd->pdVisiblePixels,x1);
		docListParticule( 0, "WIDE", part, bi, tpp );
		*/

		squeeze= 1;
		}
	    else{
		if  ( tpp->tpX0+ tpp->tpPixelsWide != x1 )
		    {
		    if  ( tpp->tpX0 > x1 )
			{
			LLDEB(tpp->tpX0+ tpp->tpPixelsWide,x1);
			tpp->tpX0= x1;
			tpp->tpPixelsWide= 0;
			}
		    else{
			tpp->tpPixelsWide= x1- tpp->tpX0;
			}
		    }
		}
	    }
	else{
	    x1= ff->ffX1TextLinesPixels;

	    if  ( tpp->tpX0+ pdd->pdVisiblePixels > x1	)
		{
		/*
		LLDEB(tpp->tpX0+ tpp->tpPixelsWide,x1);
		LLDEB(tpp->tpX0+ pdd->pdVisiblePixels,x1);
		docListParticule( 0, "WIDE", part, bi, tpp );
		*/

		squeeze= 1;
		}
	    }

	if  ( squeeze )
	    { tedSqueezeParticules( tp, pd, accepted, x0, x1 ); }

	part += accepted; tp += accepted; pd += accepted;
	x0= x1; past= part;
	}

    return 0;
    }

static int tedLayoutScreenLine(	TextLine *			tl,
				const BufferItem *		bi,
				int				part,
				int				accepted,
				AppDrawingData *		add,
				const FormattingFrame *		ff,
				TextParticule *			tp,
				ParticuleData *			pd )
    {
    int		xShift;
    int		x1;

    int		x0Pixels;
    int		ascentPixels;
    int		descentPixels;

    if  ( part == 0 )
	{ x0Pixels= ff->ffX0FirstLinePixels;	}
    else{ x0Pixels= ff->ffX0TextLinesPixels;	}

    if  ( tedLayoutLine( bi, add, ff, tp, pd, accepted, x0Pixels ) )
	{ LDEB(1); return -1;	}

    switch( bi->biParaAlignment )
	{
	case DOCiaLEFT:
	    xShift= 0; break;
	case DOCiaRIGHT:
	    x1= tp[accepted-1].tpX0+ pd[accepted-1].pdVisiblePixels;
	    xShift= ff->ffX1TextLinesPixels- x1;
	    break;
	case DOCiaCENTERED:
	    x1= tp[accepted-1].tpX0+ pd[accepted-1].pdVisiblePixels;
	    xShift= ( ff->ffX1TextLinesPixels- x1 )/ 2;
	    break;
	case DOCiaJUSTIFIED:
	    xShift= 0;
	    if  ( part+ accepted < bi->biParaParticuleCount )
		{
		tedJustifyLine( bi, tp, pd, accepted,
						ff->ffX1TextLinesPixels );
		}
	    break;
	default:
	    LDEB(bi->biParaAlignment); xShift= 0; break;
	}

    if  ( xShift > 0 )
	{
	int	i;

	for ( i= 0; i < accepted; i++ )
	    { tp[i].tpX0 += xShift; }
	}

    tedLineHeight( bi, bi->biParaParticules+ part, add, accepted,
					    &ascentPixels, &descentPixels );

    return tl->tlParticuleCount;
    }

/************************************************************************/
/*									*/
/*  Derive the frame for a paragraph from the page rectangle and the	*/
/*  paragraph properties.						*/
/*									*/
/*  For paragraphs inside a table cell, geometry is derived from the	*/
/*  table column.							*/
/*									*/
/************************************************************************/

void tedParagraphFramePixels( 	FormattingFrame *	ff,
				const AppDrawingData *	add,
				const BufferItem *	bi )
    {
    double		xfac= add->addMagnifiedPixelsPerTwip;

    ff->ffX0TextLinesPixels= TWIPStoPIXELS( xfac, ff->ffX0TextLinesTwips );
    ff->ffX0FirstLinePixels= TWIPStoPIXELS( xfac, ff->ffX0FirstLineTwips );
    ff->ffX1TextLinesPixels= TWIPStoPIXELS( xfac, ff->ffX1TextLinesTwips );

    return;
    }

/************************************************************************/
/*									*/
/*  Prepare for the layout of a paragraph for displaying.		*/
/*									*/
/************************************************************************/

void tedParaScreenGeometry(	const BufferItem *		bi,
				const DocumentProperties *	dp,
				const AppDrawingData *		add )
    {
    double		xfac= add->addMagnifiedPixelsPerTwip;

    int			tab;
    TabStop *		ts;

    FormattingFrame	ff;

    const int		bottom= -1;
    const int		stripHigh= -1;
    const int		page= 0;
    const int		column= 0;

    docParagraphFrameTwips( &ff, bottom, stripHigh, bi, dp, page, column );

    tedParagraphFramePixels( &ff, add, bi );

    ts= bi->biParaTabStops;
    for ( tab= 0; tab < bi->biParaTabCount; ts++, tab++ )
	{
	ts->tsPixels= TWIPStoPIXELS( xfac, ff.ffX0GeometryTwips+ ts->tsTwips );
	}

    return;
    }

static int tedLayoutStartScreenPara(	BufferItem *			bi,
					AppDrawingData *		add,
					const DocumentProperties *	dp )
    {
    int			part;
    TextParticule *	tp;

    double		xfac= add->addMagnifiedPixelsPerTwip;

    tp= bi->biParaParticules;
    for ( part= 0; part < bi->biParaParticuleCount; tp++, part++ )
	{
	if  ( tp->tpKind != DOCkindOBJECT )
	    { continue;	}

	if  ( 1 || tp->tpPhysicalFont < 0 )
	    {
	    InsertedObject *	io= bi->biParaObjects+ tp->tpObjectNumber;

	    if  ( tp->tpObjectNumber < 0			||
		  tp->tpObjectNumber >= bi->biParaObjectCount	)
		{
		LLDEB(tp->tpObjectNumber,bi->biParaObjectCount);
		return -1;
		}

	    if  ( io->ioKind == DOCokMACPICT )
		{
		if  ( io->ioTwipsWide == 0 )
		    { io->ioTwipsWide= 20* io->ioXExtent;	}
		if  ( io->ioTwipsHigh == 0 )
		    { io->ioTwipsHigh= 20* io->ioYExtent;	}
		}

	    io->ioPixelsWide= TWIPStoPIXELS( xfac,
				( io->ioScaleX* io->ioTwipsWide )/ 100 );
	    io->ioPixelsHigh= TWIPStoPIXELS( xfac,
				(  io->ioScaleY* io->ioTwipsHigh )/ 100 );
	    }
	}

    tedParaScreenGeometry( bi, dp, add );

    return 0;
    }

static int tedPlaceStartScreenPara( BufferItem *			bi,
				    AppDrawingData *		add,
				    const DocumentProperties *	dp )
    { return 0;	}

static int tedLayoutStartScreenRow(	BufferItem *			rowBi,
					AppDrawingData *		add,
					const DocumentProperties *	dp )
    {
    int			i;
    CellProperties *	cp;

    double		xfac= add->addMagnifiedPixelsPerTwip;

    rowBi->biRowHalfGapWidthPixels= TWIPStoPIXELS( xfac,
						rowBi->biRowHalfGapWidthTwips );
    rowBi->biRowLeftIndentPixels= TWIPStoPIXELS( xfac,
						rowBi->biRowLeftIndentTwips );

    cp= rowBi->biRowCells;
    for ( i= 0; i < rowBi->biRowCellCount; cp++, i++ )
	{
	cp->cpRightBoundaryPixels=
			    TWIPStoPIXELS( xfac, cp->cpRightBoundaryTwips );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Various layout routines.						*/
/*									*/
/************************************************************************/

static void tedFillLayoutJob(	LayoutJob *	lj )
    {
    lj->ljLayoutScreen.slStartRow= tedLayoutStartScreenRow;
    lj->ljLayoutScreen.slStartParagraph= tedLayoutStartScreenPara;
    lj->ljLayoutScreen.slScreenFrame= tedParagraphFramePixels;
    lj->ljLayoutScreen.slLayoutLine= tedLayoutScreenLine;

    lj->ljPlaceScreen.slStartRow= (START_ROW)0;
    lj->ljPlaceScreen.slStartParagraph= tedPlaceStartScreenPara;
    lj->ljPlaceScreen.slScreenFrame= tedParagraphFramePixels;
    lj->ljPlaceScreen.slLayoutLine= (LAYOUT_SCREEN_LINE)0;

    return;
    }

int tedLayoutItem(	BufferItem *		bi,
			const BufferDocument *	bd,
			int			inHeaderFooter,
			AppDrawingData *	add,
			DocumentRectangle *	drChanged )
    {
    const int		andParents= 1;
    LayoutJob		lj;

    tedFillLayoutJob( &lj );

    if  ( bi->biNumberInParent == 0 )
	{
	if  ( bi->biParent )
	    { lj.ljPosition= bi->biParent->biTopPosition;		}
	else{ docInitLayoutPosition( &(lj.ljPosition) );		}
	}
    else{
	BufferItem *	prevBi;

	prevBi= bi->biParent->biChildren[bi->biNumberInParent- 1];

	lj.ljPosition= prevBi->biBelowPosition;
	}

    lj.ljChangedRectanglePixels= drChanged;

    if  ( docPsLayoutItemAndParents( bi, add, bd, &lj, andParents ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Determine the box around a header or a footer.			*/
/*									*/
/************************************************************************/

int tedGetHeaderFooterBox(	DocumentRectangle *		dr,
				BufferItem *			sectBi,
				const DocumentProperties *	dp,
				AppDrawingData *		add,
				int				page,
				int				inHdFt )
    {
    HeaderFooter *		hf;
    const SectionProperties *	sp= &(sectBi->biSectProperties);
    const DocumentGeometry *	dgSect= &(sp->spDocumentGeometry);
    double			xfac= add->addMagnifiedPixelsPerTwip;

    DocumentRectangle		drBox;

    int				pageTopPixels= page* add->addPageStepPixels;

    drBox.drX0= TWIPStoPIXELS( xfac, dgSect->dgLeftMarginTwips );
    drBox.drX1= TWIPStoPIXELS( xfac, dgSect->dgPageWideTwips-
						dgSect->dgRightMarginTwips );

    if  ( docWhatPageHeader( &hf, sectBi, page, dp ) == inHdFt )
	{
	drBox.drY0= pageTopPixels+
				TWIPStoPIXELS( xfac, hf->hfY0ReservedTwips );
	drBox.drY1= pageTopPixels+
				TWIPStoPIXELS( xfac, hf->hfY1ReservedTwips );
	}
    else{
	if  ( docWhatPageFooter( &hf, sectBi, page, dp ) == inHdFt )
	    {
	    drBox.drY0= pageTopPixels+
				TWIPStoPIXELS( xfac, hf->hfY0ReservedTwips );
	    drBox.drY1= pageTopPixels+
				TWIPStoPIXELS( xfac, hf->hfY1ReservedTwips );
	    }
	else{ LDEB(inHdFt); return -1;	}
	}

    *dr= drBox; return 0;
    }

/************************************************************************/
/*									*/
/*  Calculate the layout of a page header.				*/
/*									*/
/************************************************************************/

int tedLayoutExternal(		HeaderFooter *			hf,
				int				page,
				const SectionProperties *	sp,
				const BufferDocument *		bd,
				AppDrawingData *		add,
				DocumentRectangle *		drChanged )
    {
    const int			andParents= 0;
    LayoutJob			lj;

    tedFillLayoutJob( &lj );

    lj.ljPosition.lpPage= page;
    lj.ljPosition.lpPageYTwips= hf->hfY0UsedTwips;
    lj.ljPosition.lpAtTopOfColumn= 1; /* not really */

    lj.ljChangedRectanglePixels= drChanged;

    if  ( docPsLayoutItemAndParents( hf->hfItem, add, bd, &lj, andParents ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

int tedRedoHeaderFooterLayout(	int *				pDone,
				HeaderFooter *			hf,
				int				page,
				const BufferItem *		sectBi,
				BufferDocument *		bd,
				AppDrawingData *		add,
				DocumentRectangle *		drChanged )
    {
    int		needed= hf->hfPageUsedFor != page;

    if  ( ! hf->hfItem )
	{ XDEB(hf->hfItem); return -1;	}
    if  ( ! needed )
	{ *pDone= needed; return 0;	}

    switch( hf->hfItem->biInHeaderFooter )
	{
	case DOCinSECT_HEADER:
	case DOCinFIRST_HEADER:
	case DOCinLEFT_HEADER:
	case DOCinRIGHT_HEADER:

	case DOCinSECT_FOOTER:
	case DOCinFIRST_FOOTER:
	case DOCinLEFT_FOOTER:
	case DOCinRIGHT_FOOTER:

	    if  ( docLayoutHeaderFooter( hf, page, bd, sectBi, add,
			    tedLayoutExternal, drChanged, tedCloseObject ) )
		{ LDEB(page); return -1; }

	    *pDone= needed;
	    return 0;

	default:
	    LDEB(hf->hfItem->biInHeaderFooter); return -1;
	}
    }

/************************************************************************/
/*									*/
/*  Redo the layout of a paragraph because its contents have changed	*/
/*  during editing.							*/
/*									*/
/************************************************************************/

int tedAdjustParagraphLayout(	BufferItem *			bi,
				int				line,
				int				stroffShift,
				int				upto,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				DocumentRectangle *		drChanged )
    {
    LayoutJob		lj;

    docFieldRefreshFlags( bi, bd );

    tedFillLayoutJob( &lj );

    /*  bogus  */
    lj.ljPosition.lpPage= 0;
    lj.ljPosition.lpPageYTwips= 0;

    lj.ljChangedRectanglePixels= drChanged;

    if  ( docPsAdjustParaLayout( bi, line, stroffShift, upto, add, bd, &lj ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Find the object in the document for an x,y position.		*/
/*									*/
/************************************************************************/

static BufferItem * tedFindItem(	BufferItem *		bi,
					const AppDrawingData *	add,
					int			x,
					int			y )
    {
    int				i;
    BufferItem *		child;
    BufferItem *		found;
    const CellProperties *	cp;
    int				page= y/ add->addPageStepPixels;

    switch( bi->biLevel )
	{
	case DOClevDOC:
	case DOClevSECT:
	case DOClevCELL:
	rowAsGroup:
	    for ( i= 0; i < bi->biChildCount; i++ )
		{
		BufferItem *	next= (BufferItem *)0;

		child= bi->biChildren[i];

		if  ( i+ 1 < bi->biChildCount )
		    { next= bi->biChildren[i+ 1];	}

		if  ( y < BI_BELOW_PIXELS( add, child )			||
		      ( next					&&
		        next->biTopPosition.lpPage > page	)	)
		    {
		    found= tedFindItem( child, add, x, y );
		    return found;
		    }
		}
	    if  ( bi->biChildCount > 0 )
		{
		child= bi->biChildren[bi->biChildCount- 1];

		found= tedFindItem( child, add, x, y );
		return found;
		}
	    LDEB(bi->biChildCount);
	    break;

	case DOClevROW:
	    if  ( ! bi->biRowHasTableParagraphs )
		{ goto rowAsGroup;	}

	    cp= bi->biRowProperties.rpCells;
	    for ( i= 0; i < bi->biChildCount; cp++, i++ )
		{
		BufferItem *	paraBi;
		int		j;
		int		right= cp->cpRightBoundaryPixels;

		if  ( cp->cpMergedWithLeft )
		    { continue; }

		if  ( cp->cpLeftInMergedRow )
		    {
		    const CellProperties *	cpp= cp;

		    for ( j= i; j < bi->biChildCount- 1; cpp++, j++ )
			{
			if  ( ! cpp[1].cpMergedWithLeft )
			    { break;	}

			right= cpp[1].cpRightBoundaryPixels;
			}
		    }

		if  ( x <= add->addDocRect.drX0+ right		||
		      i == bi->biChildCount- 1			)
		    {
		    child= bi->biChildren[i];
		    paraBi= tedFindItem( child, add, x, y );

		    if  ( paraBi )
			{ return paraBi;	}

		    if  ( child->biChildCount > 0 )
			{
			found= child->biChildren[child->biChildCount-1];
			return found;
			}
		    }
		}

	    LLDEB(x,y);
	    return (BufferItem *)0;

	case DOClevPARA:
	    return bi;

	default:
	    LDEB(bi->biLevel); return (BufferItem *)0;
	}

    LSDEB(y,docLevelStr(bi->biLevel)); return (BufferItem *)0;
    }

static int tedFindLine(	const BufferItem *	bi,
			const AppDrawingData *	add,
			int			x,
			int			y	)
    {
    int			page= y/ add->addPageStepPixels;

    int			i;
    TextLine *		tl;
    int			y1;

    if  ( y < BI_TOP_PIXELS( add, bi ) )
	{ return 0;	}

    if  ( bi->biLevel != DOClevPARA )
	{ LLDEB(bi->biLevel,DOClevPARA); return -1;	}

    tl= bi->biParaLines;
    for ( i= 0; i < bi->biParaLineCount; tl++, i++ )
	{
	y1= TL_BELOW_PIXELS( add, tl );

	if  ( y < y1						||
	      ( i+ 1 < bi->biParaLineCount		&&
	        tl[1].tlTopPosition.lpPage > page	)	)
	    { return i;	}
	}

    if  ( bi->biParaLineCount > 0 )
	{ return bi->biParaLineCount- 1;	}

    LLDEB(y,bi->biParaLineCount); return -1;
    }

int tedFindParticule(	TextLine *	tl,
			TextParticule *	tp,
			int		x,
			int		y	)
    {
    int				i;

    tp += tl->tlFirstParticule;

    if  ( x < tp->tpX0 )
	{ return tl->tlFirstParticule;	}

    for ( i= 0; i < tl->tlParticuleCount; tp++, i++ )
	{
	if  ( tp->tpX0 <= x &&  x < tp->tpX0+ tp->tpPixelsWide )
	    { return tl->tlFirstParticule+ i;	}
	}

    return tl->tlFirstParticule+ tl->tlParticuleCount- 1;
    }

/************************************************************************/
/*									*/
/*  Return the X position for an I bar.					*/
/*									*/
/************************************************************************/

int tedCalculateX(	const BufferItem *	bi,
			const TextParticule *	tp,
			const AppDrawingData *	add,
			int			stroff	)
    {
    const AppPhysicalFontList *	apfl= &(add->addPhysicalFontList);
    AppPhysicalFont *		apf;
    const unsigned char *	s;
    int				len;
    int				x;

    switch( tp->tpKind )
	{
	case DOCkindOBJECT:
	case DOCkindTAB:
	    len= stroff- tp->tpStroff;

	    if  ( len == 0 )
		{ x= tp->tpX0;	}
	    else{
		if  ( len == 1 )
		    { x= tp->tpX0+ tp->tpPixelsWide;	}
		else{
		    LLDEB(stroff,tp->tpStroff);
		    docListParticule( 0, "???", tp-bi->biParaParticules,
								    bi, tp );
		    docListItem( 0, bi );
		    x= tp->tpX0;
		    }
		}
	    return x;
	case DOCkindTEXT:
	    if  ( tp->tpPhysicalFont < 0 )
		{ LDEB(tp->tpPhysicalFont); return tp->tpX0;	}

	    s= bi->biParaString+ tp->tpStroff;
	    apf= apfl->apflFonts+ tp->tpPhysicalFont;
	    len= stroff- tp->tpStroff;

	    if  ( len == tp->tpStrlen )
		{ x= tp->tpX0+ tp->tpPixelsWide;			}
	    else{ x= tp->tpX0+ tedTextWidth( apf, add, s, len );	}
	    return x;
	case DOCkindFIELDSTART:
	case DOCkindFIELDEND:
	case DOCkindXE:
	case DOCkindTC:
	case DOCkindLINEBREAK:
	case DOCkindPAGEBREAK:
	    x= tp->tpX0;
	    return x;
	default:
	    LDEB(tp->tpKind);
	    x= tp->tpX0;
	    return x;
	}
    }

/************************************************************************/
/*									*/
/*  Return the offset in the paragraph string of a mouse event		*/
/*									*/
/*  1)  For Empty particules report the position of the beginning of	*/
/*	the particule.							*/
/*	For clicks in the left margin, report the position of the	*/
/*	particule.							*/
/*  2)  Note the <=, also consider the position after the last		*/
/*	character.							*/
/*									*/
/************************************************************************/

int tedFindStringOffset(	const TextParticule *	tp,
				const unsigned char *	paraString,
				const AppDrawingData *	add,
				int *			pBarX,
				int			x,
				int			y	)
    {
    const AppPhysicalFontList *	apfl= &(add->addPhysicalFontList);
    int				len;
    int				stroff= tp->tpStroff+ 1;
    const unsigned char *	s= paraString+ tp->tpStroff;
    AppPhysicalFont *		apf;
    int				before= tp->tpX0;
    int				past= tp->tpX0;

    /*  1  */
    if  ( x < before || tp->tpStrlen == 0 )
	{ *pBarX= tp->tpX0; return tp->tpStroff; }

    if  ( tp->tpPhysicalFont < 0 )
	{ LDEB(tp->tpPhysicalFont); return tp->tpStroff;	}

    apf= apfl->apflFonts+ tp->tpPhysicalFont;

    /*  2  */
    for ( len= 1; len <= tp->tpStrlen; stroff++, len++ )
	{
	int	between;

	if  ( len == tp->tpStrlen )
	    { past= tp->tpX0+ tp->tpPixelsWide;			}
	else{ past= tp->tpX0+ tedTextWidth( apf, add, s, len );	}

	between= ( before+ past )/ 2;

	if  ( before <= x && x < between )
	    { *pBarX= before; return stroff- 1;	}
	if  ( between <= x && x < past )
	    { *pBarX= past; return stroff;		}

	before= past;
	}

    *pBarX= tp->tpX0+ tp->tpPixelsWide; return tp->tpStroff+ tp->tpStrlen;
    }

/************************************************************************/
/*									*/
/*  Translate a Window coordinate to a position in a text buffer.	*/
/*									*/
/************************************************************************/

int tedFindPosition(	BufferItem *		rootBi,
			const AppDrawingData *	add,
			int			x,
			int			y,
			BufferPosition *	bp )
    {
    const BufferItem *	bi;

    int			part;
    int			stroff;
    int			barX;
    int			line;

    TextLine *		tl;

    bi= tedFindItem( rootBi, add, x, y );
    if  ( ! bi )
	{
	if  ( y < BI_TOP_PIXELS( add, rootBi ) )
	    {
	    if  ( docFirstPosition( rootBi, bp ) )
		{ LDEB(y); return -1; }

	    bi= bp->bpBi;
	    return 0;
	    }

	if  ( y >= BI_BELOW_PIXELS( add, rootBi ) )
	    {
	    if  ( docLastPosition( rootBi, bp ) )
		{ LDEB(y); return -1; }

	    bi= bp->bpBi;
	    /*
	    *pTl= bi->biParaLines+ bp->bpLine;
	    *pTp= bi->biParaParticules+ bp->bpParticule;
	    */

	    return 0;
	    }

	LDEB(y);
	return -1;
	}

    if  ( bi->biLevel != DOClevPARA )
	{ LLDEB(bi->biLevel,DOClevPARA); return -1;	}

    if  ( bi->biParaLineCount > 0				&&
	  y < TL_TOP_PIXELS( add, &(bi->biParaLines[0]) )	)
	{
	line= 0;
	part= 0;
	stroff= 0;
	barX= bi->biParaParticules[0].tpX0;
	}
    else{
	line= tedFindLine( bi, add, x, y );
	if  ( line < 0 )
	    {
	    if  ( bi->biParaLineCount < 1 )
		{ LDEB(bi->biParaLineCount); return -1;	}

	    if  ( y > BI_BELOW_PIXELS( add, bi )	&&
		  bi->biParaLineCount > 0		&&
		  bi->biParaParticuleCount > 0		)
		{
		line= bi->biParaLineCount- 1;
		part= bi->biParaParticuleCount- 1;
		stroff= bi->biParaStrlen;
		barX= bi->biParaParticules[part].tpX0+
				    bi->biParaParticules[part].tpPixelsWide;
		}
	    else{
		line= 0; part= 0; stroff= 0;
		if  ( bi->biParaParticuleCount == 0 )
		    { LLDEB(line,bi->biParaParticuleCount); return -1;	}
		barX= bi->biParaParticules[part].tpX0;
		}
	    }
	else{
	    part= tedFindParticule( bi->biParaLines+ line,
						bi->biParaParticules, x, y );
	    if  ( part < 0 )
		{ LLDEB(x,part); return -1;			}

	    stroff= tedFindStringOffset( bi->biParaParticules+ part,
					bi->biParaString, add, &barX, x, y );
	    if  ( stroff < 0 )
		{ LDEB(stroff); return -1;	}
	    }
	}

#   if 0
    {
    TextParticule *	tp= bi->biParaParticules+ part;

    appDebug( "(%3d,%3d): %.*s|%.*s\n", x, y,
			stroff- tp->tpStroff, bi->biParaString+ tp->tpStroff,
			tp->tpStroff+ tp->tpStrlen- stroff,
			bi->biParaString+ stroff );
    }
#   endif

    tl= bi->biParaLines+ line;

    bp->bpBi= (BufferItem *)bi;	/*  Discards const */
    bp->bpLine= line;
    bp->bpParticule= part;
    bp->bpStroff= stroff;

    bp->bpXPixels= barX;
    bp->bpTopPosition= tl->tlTopPosition;
    bp->bpY1Pixels= TL_BELOW_PIXELS( add, tl );
    bp->bpBaselinePixels= TL_BASE_PIXELS( add, tl );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Calculate the smallest rectangle that contains the selection.	*/
/*									*/
/*  1)  Same paragraph?							*/
/*  1b) For a single position, widen to contain the whole I-Bar.	*/
/*  2)  Same table row?							*/
/*  3)  Expose the whole cell.						*/
/*									*/
/************************************************************************/

#   define	IW		5

#   define	minX0(ff)	\
		((ff).ffX0TextLinesPixels < (ff).ffX0FirstLinePixels ? \
		 (ff).ffX0TextLinesPixels : (ff).ffX0FirstLinePixels )

void tedSelectionRectangle(	DocumentRectangle *		drClip,
				const AppDrawingData *		add,
				const DocumentProperties *	dp,
				const BufferSelection *		bs )
    {
    DocumentRectangle		dr;

    FormattingFrame		ff;
    const int			bottom= -1;
    const int			stripHigh= -1;

    drClip->drY0= BP_TOP_PIXELS( add, &(bs->bsBegin) );
    drClip->drY1= bs->bsEnd.bpY1Pixels;

    /*  1  */
    if  ( bs->bsEnd.bpBi == bs->bsBegin.bpBi )
	{
	if  ( bs->bsEnd.bpLine == bs->bsBegin.bpLine )
	    {
	    /*  1b  */
	    if  ( bs->bsEnd.bpStroff == bs->bsBegin.bpStroff )
		{
		drClip->drX0= bs->bsBegin.bpXPixels- IW;
		drClip->drX1= bs->bsEnd.bpXPixels+ IW;
		}
	    else{
		drClip->drX0= bs->bsBegin.bpXPixels;
		drClip->drX1= bs->bsEnd.bpXPixels;
		}

	    return;
	    }

	docParagraphFrameTwips( &ff, bottom, stripHigh,
				    bs->bsBegin.bpBi, dp,
				    bs->bsBegin.bpBi->biTopPosition.lpPage,
				    bs->bsBegin.bpBi->biTopPosition.lpColumn );
	tedParagraphFramePixels( &ff, add, bs->bsBegin.bpBi );

	drClip->drX0= minX0( ff );
	drClip->drX1= ff.ffX1TextLinesPixels;

	return;
	}

    /*  2  */
    if  ( ! bs->bsBegin.bpBi->biParaInTable		||
	  ! bs->bsEnd.bpBi->biParaInTable		||
	  ! docSelectionInsideRow( bs )			)
	{
	drClip->drX0= add->addBackRect.drX0;
	drClip->drX1= add->addBackRect.drX1;
	}

    /*  3  */
    if  ( bs->bsBegin.bpBi->biParaInTable )
	{
	docParagraphFrameTwips( &ff, bottom, stripHigh,
				    bs->bsBegin.bpBi, dp,
				    bs->bsBegin.bpBi->biTopPosition.lpPage,
				    bs->bsBegin.bpBi->biTopPosition.lpColumn );
	tedParagraphFramePixels( &ff, add, bs->bsBegin.bpBi );

	dr.drX0= minX0( ff );
	dr.drX1= ff.ffX1TextLinesPixels;

	if  ( docSelectionInsideCell( bs ) )
	    {
	    dr.drY0= BI_TOP_PIXELS( add, bs->bsBegin.bpBi );
	    dr.drY1= BI_BELOW_PIXELS( add, bs->bsBegin.bpBi );
	    }
	else{
	    BufferItem *	rowBi= bs->bsBegin.bpBi->biParent->biParent;

	    dr.drY0= BI_TOP_PIXELS( add, rowBi );
	    dr.drY1= BI_BELOW_PIXELS( add, rowBi );
	    }

	docUnionRectangle( drClip, drClip, &dr );
	}

    /*  3  */
    if  ( bs->bsEnd.bpBi->biParaInTable )
	{
	docParagraphFrameTwips( &ff, bottom, stripHigh,
				    bs->bsEnd.bpBi, dp,
				    bs->bsEnd.bpBi->biTopPosition.lpPage,
				    bs->bsEnd.bpBi->biTopPosition.lpColumn );
	tedParagraphFramePixels( &ff, add, bs->bsEnd.bpBi );

	dr.drX0= minX0( ff );
	dr.drX1= ff.ffX1TextLinesPixels;

	if  ( docSelectionInsideCell( bs ) )
	    {
	    dr.drY0= BI_TOP_PIXELS( add, bs->bsEnd.bpBi );
	    dr.drY1= BI_BELOW_PIXELS( add, bs->bsEnd.bpBi );
	    }
	else{
	    BufferItem *	rowBi= bs->bsEnd.bpBi->biParent->biParent;

	    dr.drY0= BI_TOP_PIXELS( add, rowBi );
	    dr.drY1= BI_BELOW_PIXELS( add, rowBi );
	    }

	docUnionRectangle( drClip, drClip, &dr );
	}

    return;
    }

void tedPositionCoordinates(	BufferPosition *	bp,
				const AppDrawingData *	add )
    {
    BufferItem *	bi;
    TextLine *		tl;
    TextParticule *	tp;

    bi= bp->bpBi;
    tl= bi->biParaLines+ bp->bpLine;
    tp= bi->biParaParticules+ bp->bpParticule;

    bp->bpXPixels= tedCalculateX( bi, tp, add, bp->bpStroff );
    bp->bpTopPosition= tl->tlTopPosition;
    bp->bpY1Pixels= TL_BELOW_PIXELS( add, tl );
    bp->bpBaselinePixels= TL_BASE_PIXELS( add, tl );

#   if 0
    Gives silly high I Bar.
    if  ( bp->bpLine == bi->biParaLineCount- 1 )
	{ bp->bpY1 += bi->biParaSpaceAfterPixels; }
#   endif

    return;
    }

void tedSelectionCoordinates(	BufferSelection *	bs,
				const AppDrawingData *	add )
    {
    tedPositionCoordinates( &(bs->bsBegin), add );
    tedPositionCoordinates( &(bs->bsEnd), add );
    }
