/************************************************************************/
/*									*/
/*  Save a BufferDocument into an RTF file.				*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<bitmap.h>

#   include	<appDebugon.h>

#   include	<charnames.h>
#   include	<psFont.h>
#   include	"docRtf.h"

/************************************************************************/
/*									*/
/*  Forward declaration..						*/
/*									*/
/************************************************************************/

static int docRtfSaveItem(	SimpleOutputStream *	sos,
				int *			pCol,
				const BufferItem *	bi,
				const BufferDocument *	bd,
				const BufferSelection *	bs,
				RtfWritingContext *	rwc );

/************************************************************************/
/*									*/
/*  Save a paragraph in RTF format.					*/
/*									*/
/*  Column counting is approximate as it is only for cosmetic reasons.	*/
/*									*/
/************************************************************************/

static void docRtfParaSaveProperties( SimpleOutputStream *	sos,
				int				fromDefault,
				int				saveIntbl,
				int *				pPropertyChange,
				int *				pCol,
				const ParagraphProperties *	newPP,
				const ParagraphProperties *	prevPP )
    {
    ParagraphProperties		ppp;
    unsigned int		updMask;

    PROPmaskCLEAR( &updMask );

    if  ( ! fromDefault )
	{
	PROPmaskFILL( &updMask, PPprop_COUNT );

	updMask= docParaPropertyDifference( newPP, prevPP, updMask );

	if  ( ! updMask )
	    { return;	}
	}

    docInitParagraphProperties( &ppp );

    /*  1  */
    docRtfWriteTag( "\\pard", pCol, sos );
    *pPropertyChange= 1;

    if  ( saveIntbl && newPP->ppInTable )
	{ docRtfWriteTag( "\\intbl", pCol, sos );	}

    PROPmaskCLEAR( &updMask );
    PROPmaskFILL( &updMask, PPprop_COUNT );

    updMask= docParaPropertyDifference( newPP, &ppp, updMask );

    docCleanParagraphProperties( &ppp );

    docRtfSaveParagraphProperties( sos, pCol, updMask, newPP );

    return;
    }

int docRtfSaveRuler(	SimpleOutputStream *		sos,
			const ParagraphProperties *	pp )
    {
    int				col= 0;
    int				propChange= 1;
    int				saveIntbl= 0;
    int				noRestart= 0;

    ParagraphProperties		refPP;

    docInitParagraphProperties( &refPP );

    docRtfWriteDestinationBegin( "\\ruler", &col, sos );
    docRtfWriteTag( "\\pard", &col, sos );

    docRtfParaSaveProperties( sos, noRestart, saveIntbl,
					&propChange, &col, pp, &refPP );

    docRtfWriteDestinationEnd( &col, sos );

    docCleanParagraphProperties( &refPP );

    return 0;
    }

static int docRtfSaveObject(	SimpleOutputStream *		sos,
				int *				pCol,
				const BufferItem *		bi,
				const TextParticule *		tp )
    {
    InsertedObject *	io= bi->biParaObjects+ tp->tpObjectNumber;

    docRtfWriteNextLine( pCol, sos );

    switch( io->ioKind )
	{
	case DOCokPICTWMETAFILE:
	    docRtfWriteDestinationBegin( "\\pict", pCol, sos );
	    docRtfWriteTag( "\\wmetafile8", pCol, sos );

	    docRtfSavePictureTags( io, pCol, sos );

	    docRtfWriteMemoryBuffer( &(io->ioObjectData), pCol, sos );
	    return 0;

	case DOCokMACPICT:
	    docRtfWriteDestinationBegin( "\\pict", pCol, sos );
	    docRtfWriteTag( "\\macpict", pCol, sos );

	    docRtfSavePictureTags( io, pCol, sos );

	    docRtfWriteMemoryBuffer( &(io->ioObjectData), pCol, sos );

	    return 0;

	case DOCokPICTPNGBLIP:
	    docRtfWriteDestinationBegin( "\\pict", pCol, sos );
	    docRtfWriteTag( "\\pngblip", pCol, sos );

	    docRtfSavePictureTags( io, pCol, sos );

	    docRtfWriteMemoryBuffer( &(io->ioObjectData), pCol, sos );
	    return 0;

	case DOCokPICTJPEGBLIP:
	    docRtfWriteDestinationBegin( "\\pict", pCol, sos );
	    docRtfWriteTag( "\\jpegblip", pCol, sos );

	    docRtfSavePictureTags( io, pCol, sos );

	    docRtfWriteMemoryBuffer( &(io->ioObjectData), pCol, sos );
	    return 0;

	case DOCokPICTEMFBLIP:
	    docRtfWriteDestinationBegin( "\\pict", pCol, sos );
	    docRtfWriteTag( "\\emfblip", pCol, sos );

	    docRtfSavePictureTags( io, pCol, sos );

	    docRtfWriteMemoryBuffer( &(io->ioObjectData), pCol, sos );
	    return 0;

	case DOCokOLEOBJECT:
	    docRtfWriteDestinationBegin( "\\object", pCol, sos );
	    docRtfWriteTag( "\\objemb", pCol, sos );

	    if  ( io->ioObjectClass )
		{
		docRtfWriteDestinationBegin( "\\*\\objclass ", pCol, sos );
		sioOutPutString( (char *)io->ioObjectClass, sos );
		docRtfWriteDestinationEnd( pCol, sos );
		}

	    if  ( io->ioObjectName )
		{
		docRtfWriteDestinationBegin( "\\*\\objname ", pCol, sos );
		sioOutPutString( (char *)io->ioObjectName, sos );
		docRtfWriteDestinationEnd( pCol, sos );
		}

	    docRtfWriteArgTag( "\\objw", pCol, io->ioTwipsWide, sos );
	    docRtfWriteArgTag( "\\objh", pCol, io->ioTwipsHigh, sos );
	    if  ( io->ioScaleX != 100 )
		{ docRtfWriteArgTag( "\\objscalex", pCol, io->ioScaleX, sos ); }
	    if  ( io->ioScaleY != 100 )
		{ docRtfWriteArgTag( "\\objscaley", pCol, io->ioScaleY, sos ); }

	    docRtfWriteDestinationBegin( "\\*\\objdata ", pCol, sos );
	    docRtfWriteMemoryBuffer( &io->ioObjectData, pCol, sos );

	    if  ( io->ioResultKind == DOCokPICTWMETAFILE )
		{
		docRtfWriteDestinationBegin(
			    "\\result {\\pict\\wmetafile8", pCol, sos );

		docRtfSavePictureTags( io, pCol, sos );

		docRtfWriteMemoryBuffer( &io->ioResultData, pCol, sos );

		docRtfWriteDestinationEnd( pCol, sos );
		}

	    docRtfWriteDestinationEnd( pCol, sos );
	    return 0;

	case DOCokINCLUDEPICTURE:
	    return 0;

	default:
	    LDEB(io->ioKind); return -1;
	}

    /*  Not reached ..
    return 0;
    */
    }

/************************************************************************/
/*									*/
/*  Reserve a number of columns in the output file.			*/
/*									*/
/************************************************************************/

static void docRtfReserveColumns(	SimpleOutputStream *	sos,
					int			cols,
					int *			pCol,
					RtfWritingContext *	rwc )
    {
    int				col= *pCol;

    if  ( *pCol > 0 && *pCol+ cols > 72 )
	{
	docRtfWriteNextLine( pCol, sos );
	rwc->rwcHasPrecedingTags= 0;
	}

    *pCol= col; return;
    }

/************************************************************************/
/*									*/
/*  Finish/Begin writing a hyperlink.					*/
/*									*/
/************************************************************************/

static void docRtfFinishFldrslt(	SimpleOutputStream *	sos,
					int *			pCol,
					RtfWritingContext *	rwc )
    {
    PushedAttribute *	pa;

    docRtfWriteDestinationEnd( pCol, sos );
    docRtfWriteDestinationEnd( pCol, sos );
    docRtfWriteDestinationEnd( pCol, sos );

    rwc->rwcHasPrecedingTags= 0;
    rwc->rwcInFldrslt--;

    pa= rwc->rwcOutsideFldrsltAttribute;
    if  ( ! pa )
	{ XDEB(pa); return;	}

    rwc->rwcTextAttribute= pa->paTextAttribute;
    rwc->rwcOutsideFldrsltAttribute= pa->paNext;

    free( pa );

    return;
    }

static void docRtfWriteStartField(	SimpleOutputStream *	sos,
					const DocumentField *	df,
					int *			pCol,
					RtfWritingContext *	rwc )
    {
    PushedAttribute *		pa;
    const unsigned char *	bytes= df->dfInstructions.mbBytes;
    int				byteCount= df->dfInstructions.mbSize;

    docRtfWriteDestinationBegin( "\\field", pCol, sos );
    docRtfWriteDestinationBegin( "\\*\\fldinst", pCol, sos );
    docRtfWriteDestinationBegin( "", pCol, sos );

    while( byteCount > 1 && bytes[byteCount- 1] == ' ' )
	{ byteCount--;	}

    docRtfReserveColumns( sos, byteCount, pCol, rwc );

    docRtfEscapeString( bytes, (const unsigned char *)0, pCol, byteCount, sos );

    sioOutPutCharacter( ' ', sos ); *pCol += 1;
    docRtfWriteDestinationEnd( pCol, sos );
    docRtfWriteDestinationEnd( pCol, sos );

    docRtfWriteDestinationBegin( "\\fldrslt", pCol, sos );
    docRtfWriteDestinationBegin( "", pCol, sos );

    rwc->rwcHasPrecedingTags= 0;
    rwc->rwcInFldrslt++;

    pa= malloc( sizeof( PushedAttribute ) );
    if  ( ! pa )
	{ XDEB(pa); return;	}
    pa->paNext= rwc->rwcOutsideFldrsltAttribute;
    rwc->rwcOutsideFldrsltAttribute= pa;
    pa->paTextAttribute= rwc->rwcTextAttribute;

    return;
    }

static void docRtfStartBookmark(	SimpleOutputStream *	sos,
					const DocumentField *	df,
					int *			pCol,
					RtfWritingContext *	rwc )
    {
    const char *		markName= (char *)0;
    int				markSize;

    if  ( ! docFieldGetBookmark( df, &markName, &markSize ) )
	{
	docRtfWriteDestinationBegin( "\\*\\bkmkstart ", pCol, sos );

	docRtfEscapeString( (const unsigned char *)markName, 
						    (const unsigned char *)0,
						    pCol, markSize, sos );

	docRtfWriteDestinationEnd( pCol, sos );
	rwc->rwcHasPrecedingTags= 0;
	}

    return;
    }

static void docRtfFinishBookmark(	SimpleOutputStream *	sos,
					const DocumentField *	df,
					int *			pCol,
					RtfWritingContext *	rwc )
    {
    const char *		markName= (char *)0;
    int				markSize;

    if  ( ! docFieldGetBookmark( df, &markName, &markSize ) )
	{
	docRtfWriteDestinationBegin( "\\*\\bkmkend ", pCol, sos );

	docRtfEscapeString( (const unsigned char *)markName, 
						    (const unsigned char *)0,
						    pCol, markSize, sos );

	docRtfWriteDestinationEnd( pCol, sos );
	rwc->rwcHasPrecedingTags= 0;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Save Cell properties.						*/
/*									*/
/************************************************************************/

static void docRtfSaveCellProperties(	const CellProperties *	cp,
					int *			pCol,
					int			shiftLeft,
					SimpleOutputStream *	sos )
    {
    int		didAnything= 0;

    if  ( cp->cpLeftInMergedRow )
	{ docRtfWriteTag( "\\clmgf", pCol, sos ); didAnything= 1;	}
    if  ( cp->cpMergedWithLeft )
	{ docRtfWriteTag( "\\clmrg", pCol, sos ); didAnything= 1;	}
    if  ( cp->cpTopInMergedColumn )
	{ docRtfWriteTag( "\\clvmgf", pCol, sos ); didAnything= 1;	}
    if  ( cp->cpMergedWithAbove )
	{ docRtfWriteTag( "\\clvmrg", pCol, sos ); didAnything= 1;	}

    docRtfWriteNextLine( pCol, sos );

    if  ( cp->cpTopBorder.bpStyle != DOCbsNONE )
	{ docRtfSaveBorder( "\\clbrdrt", pCol, &(cp->cpTopBorder), sos ); }
    if  ( cp->cpLeftBorder.bpStyle != DOCbsNONE )
	{ docRtfSaveBorder( "\\clbrdrl", pCol, &(cp->cpLeftBorder), sos ); }
    if  ( cp->cpRightBorder.bpStyle != DOCbsNONE )
	{ docRtfSaveBorder( "\\clbrdrr", pCol, &(cp->cpRightBorder), sos ); }
    if  ( cp->cpBottomBorder.bpStyle != DOCbsNONE )
	{ docRtfSaveBorder( "\\clbrdrb", pCol, &(cp->cpBottomBorder), sos ); }

    switch( cp->cpShadingPattern )
	{
	case DOCspSOLID:
	    break;

	case DOCspBGBDIAG:
	    docRtfWriteTag( "\\clbgbdiag", pCol, sos );
	    break;
	case DOCspBGCROSS:
	    docRtfWriteTag( "\\clbgcross", pCol, sos );
	    break;
	case DOCspBGDCROSS:
	    docRtfWriteTag( "\\clbgdcross", pCol, sos );
	    break;
	case DOCspBGDKBDIAG:
	    docRtfWriteTag( "\\clbgdkbdiag", pCol, sos );
	    break;
	case DOCspBGDKCROSS:
	    docRtfWriteTag( "\\clbgdkcross", pCol, sos );
	    break;
	case DOCspBGDKDCROSS:
	    docRtfWriteTag( "\\clbgdkdcross", pCol, sos );
	    break;
	case DOCspBGDKFDIAG:
	    docRtfWriteTag( "\\clbgdkfdiag", pCol, sos );
	    break;
	case DOCspBGDKHORIZ:
	    docRtfWriteTag( "\\clbgdkhor", pCol, sos );
	    break;
	case DOCspBGDKVERT:
	    docRtfWriteTag( "\\clbgdkvert", pCol, sos );
	    break;
	case DOCspBGFDIAG:
	    docRtfWriteTag( "\\clbgfdiag", pCol, sos );
	    break;
	case DOCspBGHORIZ:
	    docRtfWriteTag( "\\clbghoriz", pCol, sos );
	    break;
	case DOCspBGVERT:
	    docRtfWriteTag( "\\clbgvert", pCol, sos );
	    break;

	default:
	    LDEB(cp->cpShadingPattern);
	    break;
	}

    docRtfWriteArgTag( "\\cellx", pCol,
				cp->cpRightBoundaryTwips- shiftLeft, sos );

    docRtfWriteNextLine( pCol, sos );
    }

static void docRtfSaveRowProperties(	const RowProperties *	rp,
					int *			pCol,
					int			col0,
					int			col1,
					SimpleOutputStream *	sos )
    {
    const CellProperties *	cp= rp->rpCells;

    int				col;
    int				shiftLeft= 0;

    docRtfWriteTag( "\\trowd", pCol, sos );

    if  ( rp->rpHalfGapWidthTwips != 0 )
	{ docRtfWriteArgTag( "\\trgaph", pCol, rp->rpHalfGapWidthTwips, sos ); }

    if  ( rp->rpLeftIndentTwips != 0 )
	{ docRtfWriteArgTag( "\\trleft", pCol, rp->rpLeftIndentTwips, sos ); }

    if  ( rp->rpHeightTwips != 0 )
	{ docRtfWriteArgTag( "\\trrh", pCol, rp->rpHeightTwips, sos ); }

    if  ( rp->rpIsTableHeader )
	{ docRtfWriteTag( "\\trhdr", pCol, sos ); }
    if  ( rp->rpKeepOnPage )
	{ docRtfWriteTag( "\\trkeep", pCol, sos ); }

    docRtfWriteNextLine( pCol, sos );

    if  ( rp->rpTopBorder.bpStyle != DOCbsNONE )
	{ docRtfSaveBorder( "\\trbrdrt", pCol, &(rp->rpTopBorder), sos ); }
    if  ( rp->rpLeftBorder.bpStyle != DOCbsNONE )
	{ docRtfSaveBorder( "\\trbrdrl", pCol, &(rp->rpLeftBorder), sos ); }
    if  ( rp->rpRightBorder.bpStyle != DOCbsNONE )
	{ docRtfSaveBorder( "\\trbrdrr", pCol, &(rp->rpRightBorder), sos ); }
    if  ( rp->rpBottomBorder.bpStyle != DOCbsNONE )
	{ docRtfSaveBorder( "\\trbrdrb", pCol, &(rp->rpBottomBorder), sos ); }

    if  ( rp->rpHasVerticalBorders )
	{
	docRtfSaveBorder( "\\trbrdrv", pCol, &(rp->rpVerticalBorder), sos );
	}
    if  ( rp->rpHasHorizontalBorders )
	{
	docRtfSaveBorder( "\\trbrdrh", pCol, &(rp->rpHorizontalBorder), sos );
	}

    if  ( col0 > 0 )
	{ shiftLeft= cp[col0-1].cpRightBoundaryTwips; }

    for ( col= 0; col < rp->rpCellCount; cp++, col++ )
	{
	if  ( ( col0 < 0 || col >= col0 )	&&
	      ( col1 < 0 || col <= col1 )	)
	    { docRtfSaveCellProperties( cp, pCol, shiftLeft, sos );	}
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Enter/Leave a table.						*/
/*									*/
/************************************************************************/

static int docRtfPushTable(	RtfWritingContext *	rwc,
				int *			pCol,
				SimpleOutputStream *	sos,
				const BufferItem *	rowBi,
				const BufferSelection *	bs )
    {
    int			col0= -1;
    int			col1= -1;

    docRtfWriteNextLine( pCol, sos );
    docRtfWriteDestinationBegin( "", pCol, sos );

    rwc->rwcInTable= 1;

    rwc->rwcOutsideTableTextAttribute= rwc->rwcTextAttribute;

    if  ( docCopyParagraphProperties(
			    &(rwc->rwcOutsideTableParagraphProperties),
			    &(rwc->rwcParagraphProperties) ) )
	{ LDEB(1); return -1;	}

    if  ( bs )
	{ col0= bs->bsCol0; col1= bs->bsCol1; 	}

    docRtfSaveRowProperties( &(rowBi->biRowProperties), pCol, col0, col1, sos );

    if  ( docCopyRowProperties( &(rwc->rwcRowProperties),
					&(rowBi->biRowProperties) ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

static int docRtfPopTable(	RtfWritingContext *	rwc,
				int *			pCol,
				SimpleOutputStream *	sos )
    {
    docRtfWriteDestinationEnd( pCol, sos );
    docRtfWriteNextLine( pCol, sos );

    rwc->rwcInTable= 0;

    docCleanRowProperties( &(rwc->rwcRowProperties) );
    docInitRowProperties( &(rwc->rwcRowProperties) );

    rwc->rwcTextAttribute= rwc->rwcOutsideTableTextAttribute;

    if  ( docCopyParagraphProperties(
			    &(rwc->rwcParagraphProperties),
			    &(rwc->rwcOutsideTableParagraphProperties) ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save an inidividual header, footer, or footnote.			*/
/*									*/
/************************************************************************/

static int docRtfSaveExternalItem(	SimpleOutputStream *	sos,
					int *			pCol,
					const BufferItem *	bi,
					const BufferDocument *	bd,
					const char *		tag,
					RtfWritingContext *	rwc )
    {
    int			i;

    int			savedInTable= rwc->rwcInTable;

    if  ( ! bi )
	{ return 0;	}

    if  ( bi->biLevel != DOClevSECT )
	{ SDEB(docLevelStr(bi->biLevel)); return -1;	}

    docRtfWriteDestinationBegin( tag, pCol, sos );

    docCleanParagraphProperties( &(rwc->rwcParagraphProperties) );
    docInitParagraphProperties( &(rwc->rwcParagraphProperties) );
    docInitTextAttribute( &(rwc->rwcTextAttribute) );

    rwc->rwcInTable= 0;

    docRtfWriteTag( "\\pard", pCol, sos );
    docRtfWriteTag( "\\plain", pCol, sos );
    docRtfWriteNextLine( pCol, sos );

    for ( i= 0; i < bi->biChildCount; i++ )
	{
	if  ( docRtfSaveItem( sos, pCol, bi->biChildren[i], bd,
						(BufferSelection *)0, rwc ) )
	    { LDEB(i); return -1;	}
	}

    docCleanParagraphProperties( &(rwc->rwcParagraphProperties) );
    docInitParagraphProperties( &(rwc->rwcParagraphProperties) );
    docInitTextAttribute( &(rwc->rwcTextAttribute) );

    while( rwc->rwcInFldrslt > 0 )
	{ docRtfFinishFldrslt( sos, pCol, rwc );	}

    if  ( rwc->rwcInTable && docRtfPopTable( rwc, pCol, sos ) )
	{ LDEB(1);	}

    docRtfWriteDestinationEnd( pCol, sos );
    docRtfWriteNextLine( pCol, sos );

    rwc->rwcInTable= savedInTable;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save a paragraph.							*/
/*									*/
/*  1)  When we are saving a selection, and the selection is inside a	*/
/*	table cell, do not set the \intbl flag.				*/
/*  2)  To make WP 8 happy, always save 'intbl' for the first paragraph	*/
/*	in a table row.							*/
/*									*/
/************************************************************************/

static void docRtfSwitchTextAttributes(	SimpleOutputStream *	sos,
					int *			pCol,
					const TextParticule *	tp,
					RtfWritingContext *	rwc )
    {
    if  ( tp->tpTextAttribute.taFontNumber >= 0 )
	{
	unsigned int	chgMask= TAupdALL;
	unsigned int	updMask= 0;

	PROPmaskUNSET( &chgMask, TApropSHOWASLINK );

	docUpdateTextAttribute( &updMask, chgMask,
					    &(rwc->rwcTextAttribute),
					    tp->tpTextAttribute );

	if  ( updMask )
	    {
	    docRtfSaveTextAttribute( sos, pCol,
				    updMask, &(tp->tpTextAttribute) );
	    rwc->rwcHasPrecedingTags= 1;
	    }
	}

    return;
    }

static int docRtfSaveParaItem(	SimpleOutputStream *	sos,
				int *			pCol,
				const BufferDocument *	bd,
				const BufferItem *	bi,
				const BufferSelection *	bs,
				RtfWritingContext *	rwc )
    {
    TextParticule *			tp;
    unsigned char *			s;

    int					stroffUpto;
    int					part= 0;
    int					stroff= 0;

    int					saveIntbl;
    int					restartFromDefault= 0;
    int					propChange= 0;

    const DocumentField *		df;
    const FieldKindInformation *	fki;

    saveIntbl= ! bs || ! docSelectionInsideCell( bs );

    if  ( saveIntbl )
	{
	/*  2  */
	if  ( bi->biParaInTable				&&
	      bi->biNumberInParent == 0			&&
	      bi->biParent				&&
	      bi->biParent->biNumberInParent == 0	)
	    { restartFromDefault= 1; }
	}

    docRtfParaSaveProperties( sos, restartFromDefault, saveIntbl,
					&propChange, pCol,
					&(bi->biParaProperties),
					&(rwc->rwcParagraphProperties) );

    if  ( propChange )
	{ rwc->rwcHasPrecedingTags= 1;	}

    if  ( restartFromDefault || propChange )
	{
	/*  1  */
	docInitTextAttribute( &(rwc->rwcTextAttribute) );
	rwc->rwcTextAttribute.taFontSizeHalfPoints= 24;

	/*  2  */
	docRtfWriteTag( "\\plain", pCol, sos );
	rwc->rwcHasPrecedingTags= 1;
	}

    if  ( bi->biParaParticuleCount == 0 )
	{ LDEB(bi->biParaParticuleCount);	}

    if  ( bs && bs->bsBegin.bpBi == bi )
	{
	part= bs->bsBegin.bpParticule;
	stroff= bs->bsBegin.bpStroff;
	}

    tp= bi->biParaParticules+ part;
    s= bi->biParaString+ stroff;

    if  ( bs && bs->bsEnd.bpBi == bi )
	{ stroffUpto= bs->bsEnd.bpStroff;	}
    else{ stroffUpto= -1;			}

    for ( ; part < bi->biParaParticuleCount; part++, tp++ )
	{
	docRtfSwitchTextAttributes( sos, pCol, tp, rwc );

	if  ( rwc->rwcSaveAsLink )
	    {
	    docRtfWriteStartField( sos, &(rwc->rwcSaveAsHyperlinkField),
								pCol, rwc );

	    rwc->rwcSaveAsLink= 0;

	    if  ( rwc->rwcSaveAsLinkAsRef )
		{
		docRtfWriteStartField( sos, &(rwc->rwcSaveAsRefField),
								pCol, rwc );
		}
	    else{
		if  ( rwc->rwcSaveAsLinkAsPageref )
		    {
		    docRtfWriteStartField( sos,
				    &(rwc->rwcSaveAsPagerefField), pCol, rwc );
		    }
		}
	    }

	switch( tp->tpKind )
	    {
	    int		n;

	    case DOCkindTAB:
		if  ( stroffUpto >= 0 && stroff >= stroffUpto )
		    { break;	}

		docRtfWriteTag( "\\tab", pCol, sos ); s++; stroff++;
		rwc->rwcHasPrecedingTags= 1;
		continue;

	    case DOCkindLINEBREAK:
		if  ( stroffUpto >= 0 && stroff >= stroffUpto )
		    { break;	}

		docRtfWriteTag( "\\line", pCol, sos );
		docRtfWriteNextLine( pCol, sos );
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 1 */
		rwc->rwcHasPrecedingTags= 0;
		continue;

	    case DOCkindPAGEBREAK:
		if  ( stroffUpto >= 0 && stroff >= stroffUpto )
		    { break;	}

		docRtfWriteTag( "\\page", pCol, sos );
		docRtfWriteNextLine( pCol, sos );
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 1 */
		rwc->rwcHasPrecedingTags= 0;
		continue;

	    case DOCkindTEXT:
		if  ( stroffUpto >= 0 && stroff >= stroffUpto )
		    { break;	}

		if  ( stroffUpto >= 0				&&
		      tp->tpStroff+ tp->tpStrlen > stroffUpto	)
		    { n= stroffUpto- stroff;			}
		else{ n= tp->tpStroff+ tp->tpStrlen- stroff;	}

		if  ( *pCol > 0					&&
		      *pCol+ n > 72				&&
		      n < 72					)
		    { docRtfWriteNextLine( pCol, sos ); }

		if  ( rwc->rwcHasPrecedingTags )
		    {
		    if  ( *pCol > 0 )
			{ sioOutPutString( " ", sos ); *pCol += 1;	}
		    rwc->rwcHasPrecedingTags= 0;
		    }

		docRtfEscapeString( s, rwc->rwcOutputMapping, pCol, n, sos );
		stroff += n; s += n;

		continue;
	    case DOCkindOBJECT:
		if  ( stroffUpto >= 0 && stroff >= stroffUpto )
		    { break;	}

		if  ( docRtfSaveObject( sos, pCol, bi, tp ) )
		    { LDEB(tp->tpKind); }
		s += tp->tpStrlen; stroff += tp->tpStrlen;
		rwc->rwcHasPrecedingTags= 0;
		continue;

	    case DOCkindFIELDSTART:
		if  ( stroffUpto >= 0 && stroff >= stroffUpto )
		    { break;	}

		df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;
		fki= DOC_FieldKinds+ df->dfKind;

		if  ( fki->fkiIsFieldInRtf		&&
		      fki->fkiLevel == DOClevTEXT	)
		    {
		    docRtfWriteStartField( sos, df, pCol, rwc );
		    rwc->rwcHasPrecedingTags= 0;
		    }

		if  ( df->dfKind == DOCfkBOOKMARK	&&
		      rwc->rwcSaveBookmarks		)
		    {
		    docRtfStartBookmark( sos, df, pCol, rwc );
		    rwc->rwcHasPrecedingTags= 0;
		    }

		if  ( df->dfKind == DOCfkFOOTNOTE )
		    {
		    if  ( docRtfSaveExternalItem( sos, pCol,
					df->dfReferencedItem, bd,
					"\\footnote", rwc )	)
			{ LDEB(1); return -1;	}
		    }

		if  ( df->dfKind == DOCfkCHFTN )
		    {
		    docRtfWriteTag( "\\chftn", pCol, sos );
		    rwc->rwcHasPrecedingTags= 1;

		    /*HACK*/
		    s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */

		    while( part < bi->biParaParticuleCount- 1	&&
			   tp[1].tpKind != DOCkindFIELDEND	)
			{
			s += tp[1].tpStrlen; stroff += tp[1].tpStrlen;
			part++; tp++;
			}

		    continue;
		    }

		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		continue;

	    case DOCkindFIELDEND:
		df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;
		fki= DOC_FieldKinds+ df->dfKind;

		if  ( df->dfKind == DOCfkBOOKMARK	&&
		      rwc->rwcSaveBookmarks		)
		    {
		    docRtfFinishBookmark( sos, df, pCol, rwc );
		    rwc->rwcHasPrecedingTags= 0;
		    }

		if  ( fki->fkiIsFieldInRtf		&&
		      fki->fkiLevel == DOClevTEXT	)
		    {
		    if  ( rwc->rwcInFldrslt )
			{ docRtfFinishFldrslt( sos, pCol, rwc );	}
		    }

		if  ( df->dfKind == DOCfkFOOTNOTE	||
		      df->dfKind == DOCfkCHFTN		)
		    { /* nothing */	}

		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		rwc->rwcHasPrecedingTags= 0;
		continue;

	    case DOCkindXE:
		if  ( stroffUpto >= 0 && stroff >= stroffUpto )
		    { break;	}

		if  (  rwc->rwcSaveBookmarks )
		    {
		    df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;

		    sioOutPutString( "{\\xe {", sos ); *pCol += 5;
		    docRtfEscapeString( df->dfInstructions.mbBytes,
				    (const unsigned char *)0,
				    pCol, df->dfInstructions.mbSize, sos );
		    sioOutPutString( "}}", sos ); *pCol += 2;

		    rwc->rwcHasPrecedingTags= 0;
		    }
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		continue;

	    case DOCkindTC:
		if  ( stroffUpto >= 0 && stroff >= stroffUpto )
		    { break;	}

		if  (  rwc->rwcSaveBookmarks )
		    {
		    df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;

		    sioOutPutString( "{\\tc {", sos ); *pCol += 5;
		    docRtfEscapeString( df->dfInstructions.mbBytes,
				    (const unsigned char *)0,
				    pCol, df->dfInstructions.mbSize, sos );
		    sioOutPutString( "}}", sos ); *pCol += 2;

		    rwc->rwcHasPrecedingTags= 0;
		    }
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		continue;

	    default:
		LDEB(tp->tpKind);
		s += tp->tpStrlen; stroff += tp->tpStrlen;
		continue;
	    }

	break;
	}

    if  ( stroff == bi->biParaStrlen )
	{
	if  ( rwc->rwcSaveAsLinkAsRef		&&
	      rwc->rwcSaveAsLinkAsPageref	)
	    {
	    if  ( rwc->rwcInFldrslt > 0 )
		{ docRtfFinishFldrslt( sos, pCol, rwc );	}

	    docRtfWriteTag( "\\tab", pCol, sos );

	    docRtfWriteStartField( sos, &(rwc->rwcSaveAsPagerefField),
								pCol, rwc );

	    sioOutPutString( "?", sos ); *pCol += 1;

	    rwc->rwcSaveAsLinkAsRef= 0;
	    rwc->rwcSaveAsLinkAsPageref= 0;
	    }

	while( rwc->rwcInFldrslt > 0 )
	    { docRtfFinishFldrslt( sos, pCol, rwc ); }

	if  ( ! bi->biParaInTable					||
	      bi->biNumberInParent < bi->biParent->biChildCount- 1	)
	    {
	    docRtfWriteTag( "\\par", pCol, sos );
	    docRtfWriteNextLine( pCol, sos );
	    rwc->rwcHasPrecedingTags= 0;
	    }
	}

    if  ( docCopyParagraphProperties( &(rwc->rwcParagraphProperties),
						&(bi->biParaProperties) ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save a Row level item.						*/
/*									*/
/************************************************************************/

static int docRtfSaveRowItem(	SimpleOutputStream *	sos,
				int *			pCol,
				const BufferDocument *	bd,
				const BufferItem *	rowBi,
				const BufferSelection *	bs,
				RtfWritingContext *	rwc )
    {
    int				col;

    if  ( ! bs								||
	  bs->bsBegin.bpBi->biParent != bs->bsEnd.bpBi->biParent	)
	{
	if  ( ! docEqualRows( &(rowBi->biRowProperties),
						&(rwc->rwcRowProperties) ) )
	    {
	    if  ( rwc->rwcInTable && docRtfPopTable( rwc, pCol, sos ) )
		{ LDEB(1);	}
	    }

	if  ( ! rwc->rwcInTable					&&
	      docRtfPushTable( rwc, pCol, sos, rowBi, bs )	)
	    { LDEB(1);	}
	}

    for ( col= 0; col < rowBi->biChildCount; col++ )
	{
	int		par;
	BufferItem *	cellBi= rowBi->biChildren[col];

	if  ( bs && docCompareItemPositions( cellBi, bs->bsBegin.bpBi ) < 0 )
	    { continue;	}

	if  ( bs && docCompareItemPositions( cellBi, bs->bsEnd.bpBi ) > 0 )
	    { continue;	}

	if  ( ! bs						||
	      ( ( bs->bsCol0 < 0 || col >= bs->bsCol0 )	&&
		( bs->bsCol1 < 0 || col <= bs->bsCol1 )	)	)
	    {
	    for ( par= 0; par < cellBi->biChildCount; par++ )
		{
		BufferItem *	paraBi= cellBi->biChildren[par];

		if  ( bs && docCompareItemPositions( paraBi,
						    bs->bsBegin.bpBi ) < 0 )
		    { continue;	}

		if  ( bs && docCompareItemPositions( paraBi,
						    bs->bsEnd.bpBi ) > 0 )
		    { continue;	}

		if  ( docRtfSaveParaItem( sos, pCol, bd, paraBi, bs, rwc ) )
		    { LLDEB(col,par); return -1;	}
		}

	    docRtfWriteTag( "\\cell", pCol, sos );
	    if  ( col != rowBi->biChildCount- 1 )
		{ docRtfWriteNextLine( pCol, sos );	}
	    }
	}

    docRtfWriteTag( "\\row", pCol, sos );
    docRtfWriteNextLine( pCol, sos );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save the headers and footers of a Section.				*/
/*									*/
/************************************************************************/

static int docRtfSaveHeadersFooters(	SimpleOutputStream *	sos,
					int *			pCol,
					const BufferItem *	bi,
					const BufferDocument *	bd,
					RtfWritingContext *	rwc )
    {
    if  ( docRtfSaveExternalItem( sos, pCol, bi->biSectHeader.hfItem,
						bd, "\\header", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( docRtfSaveExternalItem( sos, pCol, bi->biSectFirstPageHeader.hfItem,
						bd, "\\headerf", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( docRtfSaveExternalItem( sos, pCol, bi->biSectLeftPageHeader.hfItem,
						bd, "\\headerl", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( docRtfSaveExternalItem( sos, pCol, bi->biSectRightPageHeader.hfItem,
						bd, "\\headerr", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( docRtfSaveExternalItem( sos, pCol, bi->biSectFooter.hfItem,
						bd, "\\footer", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( docRtfSaveExternalItem( sos, pCol, bi->biSectFirstPageFooter.hfItem,
						bd, "\\footerf", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( docRtfSaveExternalItem( sos, pCol, bi->biSectLeftPageFooter.hfItem,
						bd, "\\footerl", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( docRtfSaveExternalItem( sos, pCol, bi->biSectRightPageFooter.hfItem,
						bd, "\\footerr", rwc )	)
	{ LDEB(1); return -1;	}

    return 0;
    }

static int docSaveSectionProperties(	SimpleOutputStream *		sos,
					int *				pCol,
					const DocumentProperties *	dp,
					const BufferSelection *		bs,
					const SectionProperties *	sp,
					RtfWritingContext *		rwc )
    {
    SectionProperties		spDef;
    unsigned int		updMask;

    updMask= DGupdALL;
    docInitSectionProperties( &spDef );
    PROPmaskUNSET( &updMask, DGpropHEADER_POSITION );
    PROPmaskUNSET( &updMask, DGpropFOOTER_POSITION );

    appSetDocumentGeometry( &(spDef.spDocumentGeometry),
				    &(dp->dpGeometry), &updMask, updMask );

    PROPmaskCLEAR( &updMask );
    PROPmaskFILL( &updMask, SPprop_COUNT );

    updMask= docSectPropertyDifference( &spDef, sp, updMask );

    docCleanSectionProperties( &spDef );
    if  ( bs )
	{ PROPmaskUNSET( &updMask, SPpropTITLEPG );	}

    docRtfWriteNextLine( pCol, sos );
    docRtfWriteTag( "\\sectd", pCol, sos );

    docRtfSaveSectionProperties( sos,
				rwc->rwcOutputMapping, pCol, updMask, sp );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save a buffer item, hierarchically descending to the leaves of the	*/
/*  document tree.							*/
/*									*/
/************************************************************************/

static int docRtfSaveItem(	SimpleOutputStream *	sos,
				int *			pCol,
				const BufferItem *	bi,
				const BufferDocument *	bd,
				const BufferSelection *	bs,
				RtfWritingContext *	rwc )
    {
    const DocumentProperties *	dp= &(bd->bdProperties);

    int				i;

    if  ( bs )
	{
	if  ( docCompareItemPositions( bi, bs->bsBegin.bpBi ) < 0	)
	    { return 0;	}

	if  ( docCompareItemPositions( bi, bs->bsEnd.bpBi ) > 0		)
	    { return 0;	}
	}

    switch( bi->biLevel )
	{
	case DOClevDOC:
	case DOClevCELL:
	rowAsGroup:
	    for ( i= 0; i < bi->biChildCount; i++ )
		{
		if  ( docRtfSaveItem( sos, pCol,
					    bi->biChildren[i], bd, bs, rwc ) )
		    { LDEB(i); return -1;	}
		}
	    break;
	case DOClevSECT:

	    if  ( docSaveSectionProperties( sos, pCol, dp, bs,
					    &(bi->biSectProperties), rwc ) )
		{ LDEB(1); return -1;	}

	    if  ( ! bs							&&
		  docRtfSaveHeadersFooters( sos, pCol, bi, bd, rwc )	)
		{ LDEB(1); return -1;	}

	    for ( i= 0; i < bi->biChildCount; i++ )
		{
		if  ( docRtfSaveItem( sos, pCol,
					bi->biChildren[i], bd, bs, rwc ) )
		    { LDEB(i); return -1;	}
		}

	    if  ( rwc->rwcInTable && docRtfPopTable( rwc, pCol, sos ) )
		{ LDEB(1);	}

	    while( rwc->rwcInFldrslt > 0 )
		{ docRtfFinishFldrslt( sos, pCol, rwc );	}

	    docRtfWriteTag( "\\sect", pCol, sos );
	    docRtfWriteNextLine( pCol, sos );
	    break;

	case DOClevROW:
	    if  ( ! bi->biRowHasTableParagraphs )
		{
		if  ( rwc->rwcInTable && docRtfPopTable( rwc, pCol, sos ) )
		    { LDEB(1);	}
		goto rowAsGroup;
		}

	    if  ( docRtfSaveRowItem( sos, pCol, bd, bi, bs, rwc ) )
		{ LDEB(1); return -1;	}

	    break;

	case DOClevPARA:
	    if  ( docRtfSaveParaItem( sos, pCol, bd, bi, bs, rwc ) )
		{ LDEB(1); return -1;	}
	    break;

	default:
	    LDEB(bi->biLevel); return -1;
	}

    return 0;
    }

static void docRtfInitWritingContext(	RtfWritingContext *	rwc )
    {
    int		i;

    docInitTextAttribute( &(rwc->rwcTextAttribute) );
    docInitParagraphProperties( &(rwc->rwcParagraphProperties) );
    docInitRowProperties( &(rwc->rwcRowProperties) );

    docInitTextAttribute( &(rwc->rwcOutsideTableTextAttribute) );
    docInitParagraphProperties( &(rwc->rwcOutsideTableParagraphProperties) );

    rwc->rwcOutsideFldrsltAttribute= (PushedAttribute *)0;

    rwc->rwcInFldrslt= 0;
    rwc->rwcInTable= 0;

    rwc->rwcSaveBookmarks= 1;

    rwc->rwcSaveAsLink= 0;
    rwc->rwcSaveAsLinkAsRef= 0;
    rwc->rwcSaveAsLinkAsPageref= 0;
    docInitField( &(rwc->rwcSaveAsHyperlinkField) );
    docInitField( &(rwc->rwcSaveAsRefField) );
    docInitField( &(rwc->rwcSaveAsPagerefField) );

    for ( i= 0; i < 256; i++ )
	{ rwc->rwcOutputMapping[i]= i;	}
    }

static void docRtfCleanWritingContext(	RtfWritingContext *	rwc )
    {
    docCleanParagraphProperties( &rwc->rwcParagraphProperties );
    docCleanRowProperties( &rwc->rwcRowProperties );

    docCleanParagraphProperties( &rwc->rwcOutsideTableParagraphProperties );

    docCleanField( (BufferDocument *)0, &(rwc->rwcSaveAsHyperlinkField) );
    docCleanField( (BufferDocument *)0, &(rwc->rwcSaveAsRefField) );
    docCleanField( (BufferDocument *)0, &(rwc->rwcSaveAsPagerefField) );
    }

int docRtfSaveDocument(		SimpleOutputStream *	sos,
				BufferDocument *	bd,
				const SelectionScope *	ss,
				const BufferSelection *	bs,
				int			saveBookmarks )
    {
    const BufferItem *		bi= &bd->bdItem;
    int				col= 0;

    RtfWritingContext		rwc;

    if  ( bs )
	{
	const int	nearest= 1;

	bi= docGetSelectionRoot( bd, nearest, ss, bs );
	if  ( ! bi )
	    { XDEB(bi); return -1;	}
	}

    docRtfInitWritingContext( &rwc );
    rwc.rwcSaveBookmarks= saveBookmarks;

    docRtfWriteDestinationBegin( "\\rtf0\\ansi", &col, sos );
    docRtfWriteNextLine( &col, sos );

    if  ( docRtfSaveDocumentProperties( sos, &col, &rwc, bd ) )
	{ LDEB(1); return -1;	}

    if  ( docRtfSaveItem( sos, &col, bi, bd, bs, &rwc ) )
	{ LDEB(bi->biLevel); return -1; }

    while( rwc.rwcInFldrslt > 0 )
	{ docRtfFinishFldrslt( sos, &col, &rwc ); }

    docRtfWriteDestinationEnd( &col, sos );
    docRtfWriteNextLine( &col, sos );

    docRtfCleanWritingContext( &rwc );

    return 0;
    }

int docRtfSaveSelectionAsLink(	SimpleOutputStream *	sos,
				const BufferDocument *	bd,
				const SelectionScope *	ss,
				const BufferSelection *	bs,
				int			asRef,
				int			asPageref,
				const unsigned char *	fileName,
				int			fileSize,
				const unsigned char *	markName,
				int			markSize )
    {
    const BufferItem *		bi= &bd->bdItem;
    int				col= 0;

    RtfWritingContext		rwc;

    docRtfInitWritingContext( &rwc );

    rwc.rwcSaveBookmarks= 0;
    rwc.rwcSaveAsLink= 1;
    rwc.rwcSaveAsLinkAsRef= asRef;
    rwc.rwcSaveAsLinkAsPageref= asPageref;

    if  ( docFieldSetHyperlink( &(rwc.rwcSaveAsHyperlinkField),
						    fileName, fileSize,
						    markName, markSize ) )
	{ LDEB(1); docRtfCleanWritingContext( &rwc ); return -1;	}

    if  ( asRef && docFieldSetRef( &(rwc.rwcSaveAsRefField),
						    markName, markSize ) )
	{ LDEB(asRef); docRtfCleanWritingContext( &rwc ); return -1;	}

    if  ( asPageref && docFieldSetPageref( &(rwc.rwcSaveAsPagerefField),
						    markName, markSize ) )
	{ LDEB(asPageref); docRtfCleanWritingContext( &rwc ); return -1; }

    docRtfWriteDestinationBegin( "\\rtf0\\ansi", &col, sos );
    docRtfWriteNextLine( &col, sos );

    if  ( docRtfSaveDocumentProperties( sos, &col, &rwc, bd ) )
	{ LDEB(1); docRtfCleanWritingContext( &rwc ); return -1;	}

    docRtfWriteNextLine( &col, sos );

    if  ( docRtfSaveItem( sos, &col, bi, bd, bs, &rwc ) )
	{ LDEB(bi->biLevel); docRtfCleanWritingContext( &rwc ); return -1; }

    while( rwc.rwcInFldrslt > 0 )
	{ docRtfFinishFldrslt( sos, &col, &rwc ); }

    docRtfWriteDestinationEnd( &col, sos );
    docRtfWriteNextLine( &col, sos );

    docRtfCleanWritingContext( &rwc );

    return 0;
    }
