/************************************************************************/
/*									*/
/*  Layout of a document. Layout of a series of paragraphs in a common	*/
/*  parent.								*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>

#   define	y0	math_y0
#   define	y1	math_y1
#   include	<math.h>
#   undef	y0
#   undef	y1

#   include	<appImage.h>
#   include	"docPs.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Initialize the formatting of a paragraph by determining its frame	*/
/*									*/
/************************************************************************/

static int docPsLayoutStartParagraph(
				int *				pToNextPage,
				const ParagraphLayoutContext *	plc,
				BufferItem *			paraBi,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				int				bottomTwips,
				int				stripHigh,
				ParagraphLayoutPosition *	plp )
    {
    const DocumentProperties *	dp= &(bd->bdProperties);
    const DocumentGeometry *	dg;
    const BufferItem *		sectBi= paraBi;

    const ScreenLayout *	sl= &(plc->plcScreenLayout);

    while( sectBi && sectBi->biLevel != DOClevSECT )
	{ sectBi= sectBi->biParent;	}

    if  ( ! sectBi )
	{ XDEB(sectBi); return -1;	}

    dg= &(sectBi->biSectDocumentGeometry);

    /*  1  */
    if  ( paraBi->biInHeaderFooter == DOCinBODY			&&
	  ! paraBi->biParaInTable				&&
	  paraBi->biParaStartsOnNewPage				&&
	  ! plp->plpPos.lpAtTopOfColumn				)
	{ *pToNextPage= 1; return 0; }

    docParagraphFrameTwips( &(plp->plpFormattingFrame),
					bottomTwips, stripHigh, paraBi, dp,
					plp->plpPos.lpPage,
					plp->plpPos.lpColumn );

    if  ( sl && sl->slScreenFrame )
	{ (*sl->slScreenFrame)( &(plp->plpFormattingFrame), add, paraBi ); }

    if  ( (*plc->plcStartParagraph)( paraBi, add, dp ) )
	{ LDEB(1); return -1;	}

    if  ( sl && sl->slStartParagraph			&&
	  (*sl->slStartParagraph)( paraBi, add, dp )	)
	{ LDEB(1); return -1;	}

    *pToNextPage= 0;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Layout as much of a series of paragraph as fits on the current	*/
/*  page.								*/
/*									*/
/*  1)  When the paragraph starts on a new page, refuse to proceed on	*/
/*	the current one.						*/
/*  2)  When Widow/Orphan control is active and we could only place	*/
/*	one line on the page without finishing the paragraph. Force	*/
/*	the formatting of this paragraph to restart on the next page.	*/
/*  3)  When the paragraph is to be kept on one page, and it is not	*/
/*	possible to finish it on this page, Force the formatting of	*/
/*	paragraph to restart on the next page.				*/
/*  4)  When Widow/Orphan control is active and we finish the paragraph	*/
/*	by placing its last line on this page, skip back to the		*/
/*	penulimate line.						*/
/*									*/
/************************************************************************/

int docPsLayoutParagraphsInStrip(
				const ParagraphLayoutContext *	plc,
				BufferItem *			cellBi,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				int				bottomTwips,
				int				stripHigh,
				ParagraphLayoutJob *		plj )
    {
    int				changed= 0;
    ParagraphLayoutPosition *	plp= &(plj->pljPos);
    CellLayoutPosition *	clp= &(plp->plpProgress);

    if  ( cellBi->biLevel != DOClevCELL )
	{ SDEB(docLevelStr(cellBi->biLevel)); return -1;	}

    while( clp->clpPara < plj->pljParaUpto )
	{
	BufferItem *		paraBi= cellBi->biChildren[clp->clpPara];
	int			accepted;
	int			prevLine;
	LayoutPosition		lpBefore;

	lpBefore= plp->plpPos;

	if  ( clp->clpPart == 0 )
	    {
	    int		toNextPage= 0;

	    if  ( docPsLayoutStartParagraph( &toNextPage, plc, paraBi, add, bd,
					    bottomTwips, stripHigh, plp ) )
		{ LDEB(1); return -1;	}

	    if  ( toNextPage )
		{ return 0;	}

	    paraBi->biTopPosition= plp->plpPos;
	    }

	prevLine= clp->clpLine;

	accepted= (*plc->plcLayoutLines)( plc,
			    paraBi, clp->clpPart, add, bd,
			    &(plp->plpFormattingFrame),
			    &(clp->clpLine), &(plp->plpPos), &(plp->plpPos) );

	if  ( accepted < 0 )
	    { LDEB(accepted); return -1;	}

	/*  2  */
	if  ( paraBi->biParaWidowControl				&&
	      ! lpBefore.lpAtTopOfColumn				&&
	      clp->clpLine == 1						&&
	      clp->clpPart+ accepted < paraBi->biParaParticuleCount	&&
	      ! paraBi->biParaLines[0].tlHasPageBreak			)
	    {
	    clp->clpLine= 0;
	    clp->clpPart= 0;
	    break;
	    }

	/*  3  */
	if  ( ( paraBi->biParaKeepOnPage	||
	        paraBi->biParaKeepWithNext	)			&&
	      ! lpBefore.lpAtTopOfColumn				&&
	      clp->clpPart+ accepted < paraBi->biParaParticuleCount	&&
	      paraBi->biParaLineCount >= clp->clpLine			&&
	      clp->clpLine > 0						&&
	      ! paraBi->biParaLines[clp->clpLine- 1].tlHasPageBreak	)
	    {
	    clp->clpLine= 0;
	    clp->clpPart= 0;
	    break;
	    }

	if  ( accepted > 0 )
	    { plj->pljAdvanced= 1;	}

	clp->clpPart += accepted;

	if  ( clp->clpPart < paraBi->biParaParticuleCount		&&
	      ( accepted > 0 || ! lpBefore.lpAtTopOfColumn )		)
	    { break;	}

	/*  4  */
	if  ( paraBi->biParaWidowControl				&&
	      clp->clpLine > 1						&&
	      clp->clpLine- prevLine == 1				)
	    {
	    TextLine *		tl= paraBi->biParaLines+ prevLine- 1;

	    if  ( ! tl[0].tlHasPageBreak				&&
		  ! tl[0].tlTopPosition.lpAtTopOfColumn			&&
		  tl[0].tlTopPosition.lpPage < tl[1].tlTopPosition.lpPage )
		{
		clp->clpLine= prevLine- 1;
		clp->clpPart= tl->tlFirstParticule;
		plp->plpPos= tl[1].tlTopPosition;
		continue;
		}
	    }

	(*plc->plcAdjustBottom)( &changed, paraBi, &(plp->plpPos), add,
					    plc->plcChangedRectanglePixels );

	clp->clpLine= 0;
	clp->clpPart= 0;
	clp->clpPara++;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Find a start point for the formatter to layout lines. A certain	*/
/*  work ahead is necessary to decide whether work on line layout can	*/
/*  be used or not. docPsAdvanceParagraphLayout() is used to decide	*/
/*  what work is final and what possibly is to be redone.		*/
/*									*/
/*  docPsAdvanceParagraphLayout() is called when a column is full. In	*/
/*  this case, everything that is not final yet needs to be moved to	*/
/*  the next page.							*/
/*									*/
/*  -   Insure that the constraints comming from Widow/Orphan control	*/
/*	are been obeyed.						*/
/*  -   Insure that paragraphs that should not be divided over pages	*/
/*	are not split.							*/
/*  -   Insure that paragraphs that should be with the next one as a	*/
/*	header really are with it. (Experimenting with Word shows that	*/
/*	paragraphs that should be with the next one should be		*/
/*	completely with it.)						*/
/*									*/
/*  When any of the above is violated, find a solution that respects	*/
/*  the requirements by pushing offending paragraphs or lines down.	*/
/*									*/
/*  docPsAdvanceParagraphLayout() shifts 'plp0' to the last position	*/
/*  that is not below 'plp1' that is not influenced by the layout of	*/
/*  position on 'plp1' or below. Everything before 'plp1' is formatted	*/
/*  and plp1->plpPos is the position where formatting would continue	*/
/*  if docPsAdvanceParagraphLayout() moved 'plp0' onto 'plp1'. (As it	*/
/*  often does).							*/
/*									*/
/*  0)  Avoid superfluous work and the handling of this exception	*/
/*	below: Commit an explit page break.				*/
/*									*/
/*  1)  Find the place to begin. Anything on a previous page is final:	*/
/*	It will not be shifted to the next page. (Paragraph)		*/
/*  2)	Idem: Line. The line found here might very well be irrelevant	*/
/*	because a different paragraph to commit is found below.		*/
/*  3)	Commit all paragraphs before the current one that do not have	*/
/*	the 'Keep with Next' flag.					*/
/*	flag.								*/
/*  4)	If the paragraph to be committed is before the current one	*/
/*	commit the position we have found. This can be:			*/
/*	-   The first line on the current page (from 2)			*/
/*	    [ This case is however excluded by the exception code (0)].	*/
/*	-   The head of a series of KeepWithNext paragraphs. (from 3)	*/
/*  5)	When this paragraph is to be kept together, restart from its	*/
/*	beginning.							*/
/*  6)	When Widow/Orphan control is active.. Do not commit the first	*/
/*	line before the whole paragraph is committed.			*/
/*  7)	When Widow/Orphan control is active.. Do not commit the last	*/
/*	line that has been formatted unless the paragraph is complete.	*/
/*									*/
/*  8)	And finally when no exception applies, start from where we are	*/
/*	now.								*/
/*									*/
/************************************************************************/

void docPsAdvanceParagraphLayout(
				CellLayoutPosition *		clp0,
				const CellLayoutPosition *	clp1,
				int				page,
				const BufferItem *		cellBi )
    {
    const BufferItem *		paraBi;
    const TextLine *		tl;

    int				para;

    int				line1;

    /*  0  */
    if  ( clp1->clpPara < cellBi->biChildCount )
	{
	paraBi= cellBi->biChildren[clp1->clpPara];
	line1= clp1->clpLine;

	if  ( line1- 1 >= 0				&&
	      line1- 1 < paraBi->biParaLineCount	)
	    {
	    const TextLine *	tl= paraBi->biParaLines+ line1- 1;

	    if  ( tl->tlHasPageBreak )
		{
		*clp0= *clp1;
		return;
		}
	    }
	}

    /*  1  */
    paraBi= cellBi->biChildren[clp0->clpPara];
    while( clp0->clpPara < clp1->clpPara )
	{
	paraBi= cellBi->biChildren[clp0->clpPara];

	if  ( paraBi->biBelowPosition.lpPage >= page )
	    { break;	}

	clp0->clpPara++;
	clp0->clpLine= 0;
	clp0->clpPart= 0;
	}

    /*  2  */
    if  ( clp0->clpPara < clp1->clpPara )
	{ line1= paraBi->biParaLineCount;	}
    else{ line1= clp1->clpLine;			}

    tl= paraBi->biParaLines+ clp0->clpLine;
    while( clp0->clpLine < line1 )
	{
	if  ( tl->tlTopPosition.lpPage >= page )
	    { break;	}

	clp0->clpLine++; tl++;
	}

    /*  3  */
    for ( para= clp0->clpPara; para < clp1->clpPara; para++ )
	{
	paraBi= cellBi->biChildren[para];

	if  ( ! paraBi->biParaKeepWithNext )
	    {
	    clp0->clpPara= para+ 1;
	    clp0->clpLine= 0;
	    clp0->clpPart= 0;
	    }
	}

    /*  4  */
    if  ( clp0->clpPara < clp1->clpPara )
	{ return;	}

    if  ( clp0->clpPara != clp1->clpPara )
	{ LLDEB(clp0->clpPara,clp1->clpPara );	}

    if  ( clp0->clpPara >= cellBi->biChildCount )
	{ return;	}

    /*  5  */
    paraBi= cellBi->biChildren[clp0->clpPara];
    if  ( paraBi->biParaKeepOnPage	||
	  paraBi->biParaKeepWithNext	)
	{
	clp0->clpLine= 0;
	clp0->clpPart= 0;

	return;
	}

    /*  6  */
    if  ( paraBi->biParaWidowControl	&&
	  clp1->clpLine == 1		&&
	  paraBi->biParaLineCount >= 1	)
	{
	tl= paraBi->biParaLines+ 0;

	if  ( tl->tlFirstParticule+ tl->tlParticuleCount <
					paraBi->biParaParticuleCount	)
	    {
	    clp0->clpLine= 0;
	    clp0->clpPart= 0;

	    return;
	    }
	}

    /*  7  */
    if  ( paraBi->biParaWidowControl	&&
	  clp1->clpLine > 0		)
	{
	tl= paraBi->biParaLines+ clp1->clpLine- 1;

	if  ( tl->tlFirstParticule+ tl->tlParticuleCount <
					paraBi->biParaParticuleCount	)
	    {
	    clp0->clpLine= clp1->clpLine- 1;
	    clp0->clpPart= tl->tlFirstParticule;

	    return;
	    }
	}

    /*  8  */
    *clp0= *clp1;

    return;
    }

/************************************************************************/
/*									*/
/*  Format the lines in a series of paragraphs. On the way keep an	*/
/*  administration on where to rebegin formatting at a page break.	*/
/*									*/
/************************************************************************/

int docPsLayoutParagraphs(	const ParagraphLayoutContext *	plc,
				BufferItem *			cellBi,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				ParagraphLayoutJob *		plj )
    {
    const DocumentProperties *	dp= &(bd->bdProperties);
    int				toNextPage= 0;

    const int			bottomTwips= -1;
    const int			stripHigh= -1;

    LayoutPosition		lpBefore;

    const ScreenLayout *	sl= &(plc->plcScreenLayout);

    lpBefore= plj->pljPos.plpPos;

    if  ( docPsLayoutParagraphsInStrip( plc, cellBi,
				    add, bd, bottomTwips, stripHigh, plj ) )
	{ LDEB(1); return -1;	}

    toNextPage= plj->pljPos.plpProgress.clpPara < plj->pljParaUpto;

    while( toNextPage )
	{
	CellLayoutPosition		clp0;

	clp0= plj->pljPos0.plpProgress;

	docPsAdvanceParagraphLayout( &clp0, &(plj->pljPos.plpProgress),
						    lpBefore.lpPage, cellBi );

	docPsToNextPage( cellBi, dp, add, &(plj->pljPos.plpPos) );

	if  ( clp0.clpPara >    plj->pljPos0.plpProgress.clpPara	||
	      ( clp0.clpPara == plj->pljPos0.plpProgress.clpPara &&
	        clp0.clpPart >  plj->pljPos0.plpProgress.clpPart )	)
	    {
	    plj->pljPos0.plpProgress= clp0;
	    plj->pljPos.plpProgress= clp0;
	    }
	else{
	    plj->pljPos0.plpProgress= plj->pljPos.plpProgress;
	    }

	docPsParagraphColumnFrame( sl, cellBi, add, bd,
						bottomTwips, stripHigh, plj );

	lpBefore= plj->pljPos.plpPos;

	if  ( docPsLayoutParagraphsInStrip( plc, cellBi,
				    add, bd, bottomTwips, stripHigh, plj ) )
	    { LDEB(1); return -1;	}

	toNextPage= plj->pljPos.plpProgress.clpPara < plj->pljParaUpto;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Calculate the formatting frame for a paragraph layout.		*/
/*									*/
/************************************************************************/

void docPsParagraphColumnFrame(	const ScreenLayout *		sl,
				const BufferItem *		cellBi,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				int				bottomTwips,
				int				stripHigh,
				ParagraphLayoutJob *		plj )
					
    {
    const DocumentProperties *	dp= &(bd->bdProperties);

    const BufferItem *		paraBi;

    paraBi= cellBi->biChildren[plj->pljPos.plpProgress.clpPara];

    docParagraphFrameTwips( &(plj->pljPos.plpFormattingFrame),
					bottomTwips, stripHigh, paraBi, dp,
					plj->pljPos.plpPos.lpPage,
					plj->pljPos.plpPos.lpColumn );

    if  ( sl && sl->slScreenFrame )
	{
	(*sl->slScreenFrame)( &(plj->pljPos.plpFormattingFrame),
							    add, paraBi );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Initialize paragraph layout progress.				*/
/*									*/
/************************************************************************/

static void docPsInitParagraphLayoutPosition( ParagraphLayoutPosition *	plp )
    {
    plp->plpProgress.clpPara= 0;
    plp->plpProgress.clpLine= 0;
    plp->plpProgress.clpPart= 0;

    docInitLayoutPosition( &(plp->plpPos) );

    return;
    }

void docPsInitParagraphLayoutJob(	ParagraphLayoutJob *	plj )
    {
    plj->pljAdvanced= 0;
    plj->pljParaUpto= 0;

    docPsInitParagraphLayoutPosition( &(plj->pljPos) );
    docPsInitParagraphLayoutPosition( &(plj->pljPos0) );

    return;
    }

void docPsBeginParagraphLayoutProgress(	ParagraphLayoutJob *	plj,
					int			para,
					int			line,
					int			part,
					int			paraUpto,
					const LayoutPosition *	lp )
    {
    docPsInitParagraphLayoutJob( plj );

    plj->pljPos.plpProgress.clpPara= para;
    plj->pljPos.plpProgress.clpLine= line;
    plj->pljPos.plpProgress.clpPart= part;

    plj->pljParaUpto= paraUpto;

    plj->pljPos.plpPos= *lp;

    return;
    }

