/************************************************************************/
/*									*/
/*  Layout of a document. Do the layout inside lies of text.		*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   define	y0	math_y0
#   define	y1	math_y1
#   include	<math.h>
#   undef	y0
#   undef	y1

#   include	<appImage.h>
#   include	"docPs.h"
#   include	<appWinMeta.h>
#   include	<appMacPict.h>

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Find the position of the next tab.					*/
/*									*/
/************************************************************************/

static int docPsTabStop(	const BufferItem *		bi,
				int				x0Geometry,
				int				x0TextLines,
				int				tabInterval,
				int *				pTab,
				int *				pX,
				int				xPosition )
    {
    int				tab= -1;
    const ParagraphProperties *	pp= &(bi->biParaProperties);

    if  ( pp->ppTabCount == 0 )
	{
	if  ( xPosition < x0TextLines )
	    { *pX= x0TextLines; *pTab= tab; return 0; }

	if  ( tabInterval == 0 )
	    {
	    LLDEB(pp->ppTabCount,tabInterval);
	    *pX= xPosition; *pTab= tab; return 0;
	    }
	else{
	    *pTab= tab;

	    xPosition -= x0Geometry;
	    *pX= ( xPosition+ 1 )/ tabInterval;
	    *pX *= tabInterval;
	    *pX += tabInterval;
	    *pX += x0Geometry;
	    return 0;
	    }
	}
    else{
	TabStop *	ts= pp->ppTabStops;

	for ( tab= 0; tab < pp->ppTabCount; ts++, tab++ )
	    {
	    if  ( ts->tsTwips > xPosition- x0Geometry )
		{ break;	}
	    }

	if  ( tab >= pp->ppTabCount )
	    {
	    if  ( tabInterval > 0 )
		{
		xPosition -= x0Geometry;
		*pX= ( xPosition+ 1 )/ tabInterval;
		*pX *= tabInterval;
		*pX += tabInterval;
		*pX += x0Geometry;
		return 0;
		}
	    else{ LDEB(tabInterval); return 1;	}
	    }

	*pTab= tab;
	*pX= ts->tsTwips+ x0Geometry;
	return 0;
	}
    }

/************************************************************************/
/*									*/
/*  Calculate the layout of the next stretch of text upto the point	*/
/*  where it can be folded.						*/
/*									*/
/************************************************************************/

static int docPsLayoutWord(	const BufferItem *		bi,
				int *				pX1,
				int *				pVisibleX1,
				int *				pWordAscent,
				int *				pWordDescent,
				const DocumentFontList *	dfl,
				AppPhysicalFontList *		apfl,
				const TextParticule *		tp,
				ParticuleData *			pd,
				int				particuleCount,
				int				x0 )
    {
    int				particuleAscent;
    int				particuleDescent;
    int				fontAscent;
    int				fontDescent;
    int				wordAscent= 0;
    int				wordDescent= 0;

    int				accepted;
    int				width;
    int				decWidth;

    int				len= 0;
    int				xFrom= x0;
    const unsigned char *	from= bi->biParaString+ tp->tpStroff;

    int				x1= x0;
    int				visibleX1= x0;

    int				physicalFont= tp->tpPhysicalFont;

    accepted= 0;
    while( accepted < particuleCount && tp->tpKind == DOCkindTEXT )
	{
	AfmFontInfo *	afi;
	TextAttribute	ta;

	ta= tp->tpTextAttribute;
	afi= docPsPrintGetAfi( apfl, tp->tpPhysicalFont );
	if  ( ! afi )
	    { XDEB(afi); return -1;	}

	while( accepted < particuleCount		&&
	       tp->tpPhysicalFont == physicalFont	&&
	       tp->tpKind == DOCkindTEXT		)
	    {
	    AfmBBox		abb;
	    const int		withKerning= 0;
	    int			size;

	    int			dec;
	    int			d;

	    size= 10* tp->tpTextAttribute.taFontSizeHalfPoints;

	    if  ( tp->tpTextAttribute.taSuperSub == DOCfontSUPERSCRIPT ||
		  tp->tpTextAttribute.taSuperSub == DOCfontSUBSCRIPT   )
		{ size= ( 6* size )/ 10; }

	    len += tp->tpStrlen;

	    width= psCalculateStringExtents( &abb, from, len,
						    size, withKerning, afi );

	    dec= -1;
	    for ( d= 0; d < len; d++ )
		{
		if  ( from[d] == '.' || from[d] == ','	)
		    { dec= d;	}
		}

	    if  ( dec >= 0 )
		{
		AfmBBox		abbDec;

		decWidth= psCalculateStringExtents( &abbDec, from, dec,
						    size, withKerning, afi );
		}
	    else{ decWidth= width;	}

	    x1= xFrom+ width;
	    visibleX1= xFrom+ ( abb.abbRight- abb.abbLeft );

	    pd->pdAfi= afi;
	    pd->pdX0= x0;
	    pd->pdWidth= xFrom+ width- x0;
	    pd->pdDecWidth= xFrom+ decWidth- x0;
	    pd->pdVisibleWidth= xFrom+ ( abb.abbRight- abb.abbLeft )- x0;
	    pd->pdTabNumber= -1;
	    pd->pdTabKind= -1;
	    pd->pdTabPosition= 0;
	    pd->pdVisiblePixels= 0;
	    pd->pdWhiteUnits= 0;
	    pd->pdCorrectBy= 0;
	    x0= x1;

	    particuleAscent= abb.abbTop;
	    particuleDescent= abb.abbBottom;
	    fontAscent= ( size* afi->afiFontBBox.abbTop+ 500 ) / 1000;
	    fontDescent= fontAscent- size;

	    if  ( wordAscent < particuleAscent	&& 
		  wordAscent < fontAscent	)
		{ wordAscent=  fontAscent;	}

	    if  ( wordDescent > particuleDescent	&&
		  wordDescent > fontDescent		)
		{ wordDescent=  fontDescent; }

	    accepted++; tp++; pd++;

	    if  ( ( len > 0 && from[len- 1] == ' ' )	||
		  accepted >= particuleCount		)
		{ break;	}

	    if  ( ! tp->tpTextAttribute.taFontIsSlanted	&&
		  ta.taFontIsSlanted			)
		{
		int		extra;

		if  ( ta.taFontSizeHalfPoints <
				    tp->tpTextAttribute.taFontSizeHalfPoints )
		    { extra= ta.taFontSizeHalfPoints;			}
		else{ extra= tp->tpTextAttribute.taFontSizeHalfPoints;	}

		extra = (int)( -afi->afiTanItalicAngle* 10* extra );

		x0 += extra;
		}
	    }

	if  ( ( len > 0 && from[len- 1] == ' ' )	||
	      accepted >= particuleCount		)
	    { break;	}

	ta= tp->tpTextAttribute;
	physicalFont= tp->tpPhysicalFont;
	from= bi->biParaString+ tp->tpStroff;
	xFrom= x0;
	len= 0;
	}

    if  ( accepted < particuleCount			&&
	  ! tp->tpTextAttribute.taFontIsSlanted		&&
	  tp[-1].tpTextAttribute.taFontIsSlanted	)
	{
	int		extra;
	AfmFontInfo *	afi;

	if  ( tp[-1].tpTextAttribute.taFontSizeHalfPoints	<
	      tp->tpTextAttribute.taFontSizeHalfPoints	)
	    { extra= tp[-1].tpTextAttribute.taFontSizeHalfPoints; }
	else{ extra= tp->tpTextAttribute.taFontSizeHalfPoints; }

	afi= docPsPrintGetAfi( apfl, tp[-1].tpPhysicalFont );
	if  ( ! afi )
	    { XDEB(afi); return -1;	}

	extra= (int) ( -afi->afiTanItalicAngle* 10* extra );

	x1 += extra;
	}

    *pWordAscent= wordAscent;
    *pWordDescent= wordDescent;
    *pX1= x1;
    *pVisibleX1= visibleX1;

    return accepted;
    }

/************************************************************************/
/*									*/
/*  Place as many particules on a line as possible, but stop on a tab.	*/
/*									*/
/************************************************************************/

#   define	PSfoundNOTHING		0
#   define	PSfoundTAB		1
#   define	PSfoundLINEFULL		2
#   define	PSfoundLINEBREAK	3
#   define	PSfoundPAGEBREAK	4

static int docPsLayoutText(	const BufferItem *		bi,
				int *				pFound,
				int *				pWordCount,
				int *				pX1,
				int *				pTextAscent,
				int *				pTextDescent,
				int				x1TextLines,
				const DocumentFontList *	dfl,
				AppPhysicalFontList *		apfl,
				const TextParticule *		tp,
				ParticuleData *			pd,
				int				particuleCount,
				int				acceptAtLeast,
				int				x0 )
    {
    int				textAscent= 0;
    int				textDescent= 0;

    int				accepted;
    int				found= PSfoundNOTHING;
    int				wordCount= 0;

    accepted= 0;
    while( accepted < particuleCount && found == PSfoundNOTHING )
	{
	int		done;

	switch( tp->tpKind )
	    {
	    int			x1;
	    int			visibleX1;

	    int			wordAscent;
	    int			wordDescent;

	    case DOCkindTEXT:
	    textCase:
		done= docPsLayoutWord( bi, &x1, &visibleX1,
						&wordAscent, &wordDescent,
						dfl, apfl, tp, pd,
						particuleCount- accepted, x0 );
		if  ( done < 1 )
		    { LDEB(done); return -1;	}
			    
		if  ( visibleX1 >= x1TextLines		&&
		      accepted >= acceptAtLeast		)
		    {
		    *pFound= PSfoundLINEFULL;
		    *pWordCount= wordCount;
		    *pX1= x0;
		    *pTextAscent= textAscent;
		    *pTextDescent= textDescent;
		    return accepted;
		    }

		if  ( textAscent < wordAscent )
		    { textAscent=  wordAscent;		}
		if  ( textDescent > wordDescent )
		    { textDescent=  wordDescent;	}

		wordCount++; accepted += done; tp += done; pd += done;
		x0= x1;
		break;

	    case DOCkindFIELDSTART:
		done= 0;
		while( accepted < particuleCount	&&
		       tp->tpKind == DOCkindFIELDSTART	)
		    {
		    pd->pdAfi= (AfmFontInfo *)0;
		    pd->pdX0= x0;
		    pd->pdWidth= 0;
		    pd->pdDecWidth= 0;
		    pd->pdVisibleWidth= 0;
		    pd->pdTabNumber= -1;
		    pd->pdTabKind= -1;
		    pd->pdTabPosition= 0;
		    pd->pdVisiblePixels= 0;
		    pd->pdWhiteUnits= 0;
		    pd->pdCorrectBy= 0;

		    if  ( accepted < acceptAtLeast )
			{ acceptAtLeast++;	}

		    accepted++; tp++; pd++;
		    }

		if  ( accepted < particuleCount		&&
		      tp->tpKind == DOCkindTEXT		)
		    { goto textCase;	}

		break;

	    case DOCkindTAB:
		found= PSfoundTAB;
		break;

	    case DOCkindFIELDEND:
	    case DOCkindXE:
	    case DOCkindTC:
		pd->pdAfi= (AfmFontInfo *)0;
		pd->pdX0= x0;
		pd->pdWidth= 0;
		pd->pdDecWidth= 0;
		pd->pdVisibleWidth= 0;
		pd->pdTabNumber= -1;
		pd->pdTabKind= -1;
		pd->pdTabPosition= 0;
		pd->pdVisiblePixels= 0;
		pd->pdWhiteUnits= 0;
		pd->pdCorrectBy= 0;

		accepted++; tp++; pd++; break;

	    case DOCkindOBJECT:
		{
		InsertedObject *	io;
		int			width;

		io= bi->biParaObjects+ tp->tpObjectNumber;

		width= ( io->ioScaleX* io->ioTwipsWide )/ 100;

		if  ( x0+ width >= x1TextLines		&&
		      accepted >= acceptAtLeast		)
		    {
		    *pFound= PSfoundLINEFULL;
		    *pWordCount= wordCount;
		    *pX1= x0;
		    *pTextAscent= textAscent;
		    *pTextDescent= textDescent;
		    return accepted;
		    }

		wordAscent= ( io->ioScaleY* io->ioTwipsHigh )/ 100;
		if  ( textAscent < wordAscent )
		    { textAscent=  wordAscent;	}

		pd->pdAfi= (AfmFontInfo *)0;
		pd->pdX0= x0;
		pd->pdWidth= width;
		pd->pdDecWidth= width;
		pd->pdVisibleWidth= width;
		pd->pdTabNumber= -1;
		pd->pdTabKind= -1;
		pd->pdTabPosition= 0;
		pd->pdVisiblePixels= 0;
		pd->pdWhiteUnits= 0;
		pd->pdCorrectBy= 0;
		x0 += width;

		accepted++; tp++; pd++; break;
		}

	    case DOCkindLINEBREAK:
	    case DOCkindPAGEBREAK:
		pd->pdAfi= (AfmFontInfo *)0;
		pd->pdX0= x0;
		pd->pdWidth= 0;
		pd->pdDecWidth= 0;
		pd->pdVisibleWidth= 0;
		pd->pdTabNumber= -1;
		pd->pdTabKind= -1;
		pd->pdTabPosition= 0;
		pd->pdVisiblePixels= 0;
		pd->pdWhiteUnits= 0;
		pd->pdCorrectBy= 0;

		if  ( tp->tpKind == DOCkindPAGEBREAK	&&
		      ! bi->biParaInTable		)
		    { found= PSfoundPAGEBREAK;	}
		else{ found= PSfoundLINEBREAK;	}

		accepted++; tp++; pd++; break;

	    default:
		LDEB(tp->tpKind); return -1;
	    }
	}

    *pFound= found;
    *pWordCount= wordCount;
    *pX1= x0;
    *pTextAscent= textAscent;
    *pTextDescent= textDescent;

    return accepted;
    }

/************************************************************************/
/*									*/
/*  Place as many particules on a line as possible. The caller can	*/
/*  require that a certain number of particules are accepted even if	*/
/*  these do not fit on the line. This is to prevent loops. A formatter	*/
/*  typically wants to be sure that at least one particule is accepted,	*/
/*  such that it can know that it actually advances.			*/
/*									*/
/************************************************************************/

static int docPsLayoutParticules( const BufferItem *		bi,
				int *				pFound,
				int *				pWordCount,
				int *				pLineAscent,
				int *				pLineDescent,
				int				tabInterval,
				const FormattingFrame *		ff,
				const DocumentFontList *	dfl,
				AppPhysicalFontList *		apfl,
				const TextParticule *		tp,
				ParticuleData *			pd,
				int				particuleCount,
				int				acceptAtLeast,
				int				x0 )
    {
    int				lineAscent= 0;
    int				lineDescent= 0;

    int				accepted;
    int				wordCount= 0;

    int				tabKind= DOCtkLEFT;
    int				tabX0= 0;
    int				tabX1= 0;
    int				tabParticule= -1;

    const TextParticule *	tpp= tp;
    ParticuleData *		pdd= pd;

    accepted= 0;
    while( accepted < particuleCount )
	{
	int		done;
	int		found;

	int		textAscent;
	int		textDescent;
	int		textWordCount;

	int		x1;
	AfmFontInfo *	afi;
	int		tab;

	int		visibleSinceTab;
	int		shift= 0;

	if  ( accepted >= acceptAtLeast )
	    { acceptAtLeast= 0;	}

	done= docPsLayoutText( bi, &found, &textWordCount, &x1,
				    &textAscent, &textDescent,
				    ff->ffX1TextLinesTwips, dfl, apfl, tpp, pdd,
				    particuleCount- accepted,
				    acceptAtLeast, x0 );
	if  ( done < 0 )
	    { LDEB(done); return -1;	}

	if  ( lineAscent < textAscent )
	    { lineAscent=  textAscent;	}
	if  ( lineDescent > textDescent )
	    { lineDescent=  textDescent;	}

	switch( tabKind )
	    {
	    case DOCtkLEFT:
		shift= 0; break;

	    case DOCtkDECIMAL:
		if  ( done > 0 && tabParticule >= 0 )
		    {
		    int		d;

		    visibleSinceTab=
			pdd[done-1].pdX0+ pdd[done-1].pdVisibleWidth- tabX0;

		    for ( d= done+ accepted- 1; d >= accepted; d-- )
			{
			if  ( pd[d].pdDecWidth <  pd[d].pdWidth )
			    {
			    visibleSinceTab=
					pd[d].pdX0+ pd[d].pdDecWidth- tabX0;
			    break;
			    }
			}

		    shift= tabX1- visibleSinceTab- tabX0;
		    if  ( shift < 0 )
			{ LDEB(shift); shift= 0;	}
		    else{ x1= tabX1;			}
		    pd[tabParticule].pdWidth= shift;
		    pd[tabParticule].pdVisibleWidth= 0;
		    }
		else{
		    if  ( tabParticule < 0 )
			{ LLDEB(done,tabParticule);	}
		    else{
			x1= tabX1;
			pd[tabParticule].pdWidth= tabX1- tabX0;
			pd[tabParticule].pdVisibleWidth= 0;
			}
		    }
		break;

	    case DOCtkRIGHT:
		if  ( done > 0 && tabParticule >= 0 )
		    {
		    visibleSinceTab=
			pdd[done-1].pdX0+ pdd[done-1].pdVisibleWidth- tabX0;

		    shift= tabX1- visibleSinceTab- tabX0;
		    if  ( shift < 0 )
			{ LDEB(shift); shift= 0;	}
		    else{ x1= tabX1;			}
		    pd[tabParticule].pdWidth= shift;
		    pd[tabParticule].pdVisibleWidth= 0;
		    }
		else{
		    if  ( tabParticule < 0 )
			{ LLDEB(done,tabParticule);	}
		    else{
			x1= tabX1;
			pd[tabParticule].pdWidth= tabX1- tabX0;
			pd[tabParticule].pdVisibleWidth= 0;
			}
		    }
		break;

	    case DOCtkCENTRE:
		if  ( done > 0 && tabParticule >= 0 )
		    {
		    visibleSinceTab=
			pdd[done-1].pdX0+ pdd[done-1].pdVisibleWidth- tabX0;

		    shift= tabX1- visibleSinceTab/ 2- tabX0;
		    if  ( shift < 0 )
			{ LDEB(shift); shift= 0;	}
		    else{ x1 += shift;			}
		    pd[tabParticule].pdWidth= shift;
		    pd[tabParticule].pdVisibleWidth= 0;
		    }
		else{
		    if  ( tabParticule >= 0 )
			{
			x1= tabX1;
			pd[tabParticule].pdWidth= tabX1- tabX0;
			pd[tabParticule].pdVisibleWidth= 0;
			}
		    }
		break;

	    default:
		LDEB(tabKind);
		shift= 0; break;
	    }

	if  ( shift > 0 )
	    {
	    int		i;

	    for ( i= 0; i < done; i++ )
		{ pdd[i].pdX0 += shift; }
	    }

	wordCount += textWordCount;
	accepted += done; tpp += done; pdd += done;
	x0= x1;

	switch( found )
	    {
	    case PSfoundTAB:
		break;

	    case PSfoundNOTHING:
	    case PSfoundLINEFULL:
		*pFound= found;
		*pWordCount= wordCount;
		*pLineAscent= lineAscent; *pLineDescent= lineDescent;
		return accepted;

	    case PSfoundLINEBREAK:
		*pFound= found;
		*pWordCount= wordCount;
		*pLineAscent= lineAscent; *pLineDescent= lineDescent;
		return accepted;

	    case PSfoundPAGEBREAK:
		*pFound= found;
		*pWordCount= wordCount;
		*pLineAscent= lineAscent; *pLineDescent= lineDescent;
		return accepted;

	    default:
		LDEB(found); return -1;
	    }

	afi= docPsPrintGetAfi( apfl, tpp->tpPhysicalFont );
	if  ( ! afi )
	    { XDEB(afi); return -1;	}

	tab= -1;
	if  ( docPsTabStop( bi, ff->ffX0GeometryTwips, ff->ffX0TextLinesTwips,
						tabInterval, &tab, &x1, x0 ) )
	    {
	    if  ( accepted < 1 )
		{ LLDEB(particuleCount,accepted);	}

	    *pFound= PSfoundLINEFULL;
	    *pWordCount= wordCount;
	    *pLineAscent= lineAscent; *pLineDescent= lineDescent;
	    return accepted;
	    }

	if  ( tab >= 0 )
	    { tabKind= bi->biParaTabStops[tab].tsKind;	}
	else{ tabKind= DOCtkLEFT;			}
	tabX0= x0;
	tabX1= x1;
	tabParticule= accepted;

	pdd->pdAfi= afi;
	pdd->pdX0= x0;
	pdd->pdWidth= x1- x0;
	pdd->pdDecWidth= x1- x0;
	pdd->pdVisibleWidth= 0;
	pdd->pdTabNumber= tab;
	pdd->pdTabKind= tabKind;
	pdd->pdTabPosition= x1;
	pdd->pdVisiblePixels= 0;
	pdd->pdWhiteUnits= 0;
	pdd->pdCorrectBy= 0;

	if  ( tabKind == DOCtkLEFT )
	    { x0= x1;	}

	accepted++; tpp++; pdd++;

	}

    if  ( accepted < 1 )
	{ LLDEB(particuleCount,accepted);	}

    *pFound= PSfoundNOTHING;
    *pWordCount= wordCount;
    *pLineAscent= lineAscent; *pLineDescent= lineDescent;

    return accepted;
    }

/************************************************************************/
/*									*/
/*  Justify the particules in a line of text.				*/
/*									*/
/*  1)  Start justification after the last tab of the line. Justifying	*/
/*	after anything else than a left tab is ridiculous. Simply	*/
/*	refuse to.							*/
/*  2)  Ignore organisational particules such as the delimitation of	*/
/*	links and bookmarks at the end of the line.			*/
/*									*/
/************************************************************************/

static void docPsJustifyLine(	const BufferItem *	bi,
				const TextParticule *	tp,
				ParticuleData *		pd,
				int			accepted,
				int			x1TestLines )
    {
    int				i;
    int				extra;
    int				totalWeight;
    int				step;
    int				left;

    ParticuleData *		pdd;
    const TextParticule *	tpp;

    /*  1  */
    left= 0;
    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	if  ( tpp->tpKind == DOCkindTAB )
	    { left= i+ 1;	}
	}

    if  ( left > 0 )
	{
	if  ( pd[left-1].pdTabKind != DOCtkLEFT )
	    { LDEB(pd[left-1].pdTabKind); return;	}

	tp += left; pd += left; accepted -= left;
	}

    /*  2  */
    while( accepted > 0 && tp[accepted- 1].tpStrlen == 0 )
	{ accepted--;	}

    if  ( accepted < 2 )
	{ /* LDEB(accepted); */ return;	}

    extra= x1TestLines- pd[accepted- 1].pdX0- pd[accepted- 1].pdVisibleWidth;

    if  ( extra < 0 )
	{
	LLDEB(extra,x1TestLines);
	tpp= tp; pdd= pd;
	for ( i= 0; i < accepted; tpp++, pdd++, i++ )
	    {
	    appDebug( "%-4s: %5d..%5d \"%*.*s\"\n",
		    docKindStr( tpp->tpKind ),
		    pdd->pdX0, pdd->pdX0+ pdd->pdVisibleWidth,
		    tpp->tpStrlen, tpp->tpStrlen,
		    bi->biParaString+ tpp->tpStroff );
	    }

	extra= 0;
	}

    totalWeight= 0;
    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	pdd->pdWhiteUnits= 0;
	pdd->pdCorrectBy= 0;

	if  ( bi->biParaString[tpp->tpStroff+tpp->tpStrlen-1] == ' ' )
	    {
	    pdd->pdWhiteUnits=
		sqrt( (double)pdd[0].pdWidth+ (double)pdd[1].pdVisibleWidth );

	    totalWeight += pdd->pdWhiteUnits;
	    }
	}

    left= extra;
    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	if  ( pdd->pdWhiteUnits > 0 )
	    {
	    step= ( extra* pdd->pdWhiteUnits )/ totalWeight;
	    if  ( step > left )
		{ step= left;	}

	    pdd->pdCorrectBy += step;

	    left -= step;
	    }
	}

    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	if  ( pdd->pdWhiteUnits > 0 )
	    {
	    step= 1;
	    if  ( step > left )
		{ step= left;	}

	    pdd->pdCorrectBy += step;

	    left -= step;
	    }
	}

    step= 0;
    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	pdd->pdX0 += step;
	if  ( pdd->pdWhiteUnits > 0 )
	    {
	    pdd->pdWidth += pdd->pdCorrectBy;
	    step += pdd->pdCorrectBy;
	    }
	}

    pdd->pdX0 += step;

    return;
    }

/************************************************************************/
/*									*/
/*  Place as many particules on a line as possible.			*/
/*									*/
/*  1)  Accept at least one particule.					*/
/*  A)  It looks nicer when fields do not start at the end of the line.	*/
/*	For the formatting it self this is irrelevant, but when a field	*/
/*	opens on the line before its contents, the shading of the	*/
/*	selection looks silly.						*/
/*  9)  Make \page mean \line in headers and footers and inside tables.	*/
/*									*/
/************************************************************************/

int docPsLineBox(	TextLine *			tl,
			const BufferItem *		bi,
			int				part,
			int				tabInterval,
			const DocumentFontList *	dfl,
			AppPhysicalFontList *		apfl,
			const TextParticule *		tp,
			ParticuleData *			pd,
			const FormattingFrame *		ff )
    {
    int			lineAscent;
    int			lineDescent;
    int			ascent;
    int			descent;
    int			xShift;

    int			accepted;
    int			found;
    int			wordCount;

    int			x0;
    int			visibleX1Twips;

    unsigned char *	from;

    if  ( part == 0 )
	{ x0= ff->ffX0FirstLineTwips;	}
    else{ x0= ff->ffX0TextLinesTwips;	}

    from= bi->biParaString+ tp->tpStroff;

    /*  1  */
    accepted= docPsLayoutParticules( bi, &found, &wordCount,
				    &lineAscent, &lineDescent,
				    tabInterval, ff, dfl, apfl, tp, pd,
				    bi->biParaParticuleCount- part, 1, x0 );

    if  ( accepted <= 0 )
	{ LDEB(accepted); return -1;	}

    /*  A  */
    while( accepted > 1							&&
	   bi->biParaParticules[part+accepted-1].tpKind
					    == DOCkindFIELDSTART	)
	{ accepted--;	}

    tl->tlWordCount= wordCount;

    ascent= bi->biParaAscentTwips;
    descent= -bi->biParaDescentTwips;

    if  ( lineAscent > bi->biParaAscentTwips )
	{ ascent= lineAscent;	}
    if  ( lineDescent < bi->biParaDescentTwips )
	{ descent= -lineDescent; }

    tl->tlLineAscentTwips= ascent;
    tl->tlLineHeightTwips= ascent+ descent;
    tl->tlLineSpacingTwips= ascent+ descent+ bi->biParaLeadingTwips;

    if  ( bi->biParaLineSpacingTwips != 0			&&
	  ( part+ accepted < bi->biParaParticuleCount	||
	    bi->biParaLineSpacingIsMultiple		)	)
	{
	if  ( bi->biParaLineSpacingTwips > 0 )
	    {
	    if  ( bi->biParaLineSpacingTwips > tl->tlLineSpacingTwips )
		{ tl->tlLineSpacingTwips= bi->biParaLineSpacingTwips; }
	    }
	else{ tl->tlLineSpacingTwips= -bi->biParaLineSpacingTwips; }
	}

    if  ( accepted == 0 )
	{ visibleX1Twips= x0;	}
    else{
	ParticuleData *	pdd= pd+ accepted- 1;

	visibleX1Twips= pdd->pdX0+ pdd->pdVisibleWidth;
	}

    switch( bi->biParaAlignment )
	{
	case DOCiaLEFT:
	    xShift= 0; break;
	case DOCiaRIGHT:
	    xShift= ff->ffX1TextLinesTwips- visibleX1Twips;
	    break;
	case DOCiaCENTERED:
	    xShift= ( ff->ffX1TextLinesTwips- visibleX1Twips )/ 2;
	    break;
	case DOCiaJUSTIFIED:
	    xShift= 0;
	    if  ( found == PSfoundLINEFULL )
		{
		docPsJustifyLine( bi, tp, pd, accepted,
						    ff->ffX1TextLinesTwips );
		}
	    break;
	default:
	    LDEB(bi->biParaAlignment); xShift= 0;
	    break;
	}

    if  ( xShift != 0 )
	{
	int		i;
	ParticuleData *	pdd= pd;

	for ( i= 0; i < accepted; pdd++, i++ )
	    { pdd->pdX0 += xShift;	}
	}

    /*  9 */
    if  ( found == PSfoundPAGEBREAK		&&
	  bi->biInHeaderFooter == DOCinBODY	&&
	  ! bi->biParaInTable			)
	{ tl->tlHasPageBreak= 1;	}
    else{ tl->tlHasPageBreak= 0;	}

    return accepted;
    }

