/************************************************************************/
/*									*/
/*  Buffer administration routines.					*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<limits.h>

#   include	<appDebugon.h>

#   include	<appUnit.h>
#   include	"docBuf.h"

/************************************************************************/
/*									*/
/*  Manage fields inside the text.					*/
/*									*/
/************************************************************************/

/************************************************************************/
/*									*/
/*  Editing and field balancing code.					*/
/*									*/
/*  Extend a selection such that it does not partially cover a text	*/
/*  level field. Besides find out whether the whole selection is inside	*/
/*  a field with a read-only result.					*/
/*									*/
/************************************************************************/

static int docTextFieldLevels(		int			textFieldLevel,
					int *			pReadonly,
					int *			pRonlyLevel,
					int *			pHigest,
					int *			pLowest,
					const BufferDocument *	bd,
					const BufferItem *	bi,
					int			part,
					int			partUpto )
    {
    const TextParticule *	tp;
    int				readonly= *pReadonly;
    int				readOnlyLevel= *pRonlyLevel;
    int				highest= *pHigest;
    int				lowest= *pLowest;

    if  ( partUpto > bi->biParaParticuleCount )
	{
	LLDEB(partUpto,bi->biParaParticuleCount);
	partUpto= bi->biParaParticuleCount;
	}

    tp= bi->biParaParticules+ part;
    while( part < partUpto )
	{
	const DocumentField *		df;
	const FieldKindInformation *	fki;

	if  ( lowest > textFieldLevel )
	    { lowest=  textFieldLevel;	}
	if  ( highest < textFieldLevel )
	    { highest=  textFieldLevel;	}

	if  ( tp->tpKind == DOCkindFIELDSTART )
	    {
	    df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;
	    fki= DOC_FieldKinds+ df->dfKind;

	    if  ( fki->fkiLevel == DOClevTEXT )
		{
		textFieldLevel++;
		if  ( ! fki->fkiResultEditable )
		    {
		    if  ( ! readonly )
			{ readOnlyLevel= textFieldLevel- 1;	}

		    readonly++;
		    }
		}
	    }

	if  ( tp->tpKind == DOCkindFIELDEND )
	    {
	    df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;
	    fki= DOC_FieldKinds+ df->dfKind;

	    if  ( fki->fkiLevel == DOClevTEXT )
		{
		textFieldLevel--;
		if  ( textFieldLevel < 0 )
		    { LDEB(textFieldLevel); textFieldLevel= 0;	}
		if  ( ! fki->fkiResultEditable )
		    {
		    readonly--;
		    if  ( readonly < 0 )
			{ LDEB(readonly); readonly= 0;	}
		    }
		}
	    }

	part++; tp++;
	}

    *pReadonly= readonly;
    *pRonlyLevel= readOnlyLevel;
    *pHigest= highest;
    *pLowest= lowest;

    return textFieldLevel;
    }

static int docFieldBalanceBegin(	int *			pMoved,
					BufferPosition *	bp,
					const BufferDocument *	bd,
					int			beginLevel,
					int			lowest )
    {
    BufferItem *		bi= bp->bpBi;
    int				textFieldLevel= beginLevel;
    int				part;
    const TextParticule *	tp;

    part= bp->bpParticule;
    tp= bi->biParaParticules+ part;

    while( part > 0 )
	{
	const DocumentField *		df;
	const FieldKindInformation *	fki;

	if  ( tp->tpKind == DOCkindFIELDSTART )
	    {
	    df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;
	    fki= DOC_FieldKinds+ df->dfKind;

	    if  ( fki->fkiLevel == DOClevTEXT )
		{
		textFieldLevel--;
		if  ( textFieldLevel < 0 )
		    { LDEB(textFieldLevel); textFieldLevel= 0;	}
		}
	    }

	if  ( tp->tpKind == DOCkindFIELDEND )
	    {
	    df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;
	    fki= DOC_FieldKinds+ df->dfKind;

	    if  ( fki->fkiLevel == DOClevTEXT )
		{ textFieldLevel++; }
	    }

	if  ( textFieldLevel <= lowest )
	    { break;	}

	part--; tp--;
	}

    if  ( bp->bpParticule != part )
	{ *pMoved= 1;	}
    bp->bpParticule= part;
    bp->bpStroff= tp->tpStroff;

    if  ( docFindLineOfParticule( bi, part, &(bp->bpLine) ) )
	{ LDEB(part); return -1;	}

    return 0;
    }

static int docFieldBalanceEnd(		int *			pMoved,
					BufferPosition *	bp,
					const BufferDocument *	bd,
					int			endLevel,
					int			lowest )
    {
    BufferItem *		bi= bp->bpBi;
    int				textFieldLevel= endLevel;
    int				part;
    const TextParticule *	tp;

    part= bp->bpParticule;
    tp= bi->biParaParticules+ part;

    while( part < bi->biParaParticuleCount )
	{
	const DocumentField *		df;
	const FieldKindInformation *	fki;

	if  ( tp->tpKind == DOCkindFIELDSTART )
	    {
	    df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;
	    fki= DOC_FieldKinds+ df->dfKind;

	    if  ( fki->fkiLevel == DOClevTEXT )
		{ textFieldLevel++;	}
	    }

	if  ( tp->tpKind == DOCkindFIELDEND )
	    {
	    df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;
	    fki= DOC_FieldKinds+ df->dfKind;

	    if  ( fki->fkiLevel == DOClevTEXT )
		{
		textFieldLevel--;
		if  ( textFieldLevel < 0 )
		    { LDEB(textFieldLevel); textFieldLevel= 0;	}
		}
	    }

	if  ( textFieldLevel <= lowest )
	    { break;	}

	part++; tp++;
	}

    if  ( part >= bi->biParaParticuleCount )
	{ part=   bi->biParaParticuleCount- 1;	}

    if  ( bp->bpParticule != part )
	{ *pMoved= 1;	}
    bp->bpParticule= part;
    bp->bpStroff= tp->tpStroff+ tp->tpStrlen;

    if  ( docFindLineOfParticule( bi, part, &(bp->bpLine) ) )
	{ LDEB(part); return -1;	}

    return 0;
    }

int docBalanceFieldSelection(	int *			pBeginMoved,
				int *			pEndMoved,
				const BufferDocument *	bd,
				BufferPosition *	bpBegin,
				BufferPosition *	bpEnd )
    {
    int				beginLevel;
    int				endLevel;
    int				readonly;
    int				readOnlyLevel;

    int				highest= 0;
    int				lowest;

    int				beginMoved= 0;
    int				endMoved= 0;

    lowest= INT_MAX;
    beginLevel= 0;
    readonly= 0;
    readOnlyLevel= -1;
    beginLevel= docTextFieldLevels( beginLevel, &readonly, &readOnlyLevel,
						&highest, &lowest,
						bd, bpBegin->bpBi,
						0,
						bpBegin->bpParticule+ 1 );

    if  ( bpBegin->bpBi == bpEnd->bpBi )
	{
	lowest= beginLevel;
	endLevel= docTextFieldLevels( beginLevel, &readonly, &readOnlyLevel,
						&highest, &lowest,
						bd, bpEnd->bpBi,
						bpBegin->bpParticule+ 1,
						bpEnd->bpParticule+ 1 );
	}
    else{
	lowest= INT_MAX;
	beginLevel= 0;
	readonly= 0;
	readOnlyLevel= -1;
	endLevel= docTextFieldLevels( beginLevel, &readonly, &readOnlyLevel,
						&highest, &lowest,
						bd, bpEnd->bpBi,
						0,
						bpEnd->bpParticule+ 1 );
	lowest= 0;
	}

    if  ( beginLevel > lowest )
	{
	if  ( docFieldBalanceBegin( &beginMoved, bpBegin, bd,
							beginLevel, lowest ) )
	    { LDEB(1); return -1;	}
	}

    if  ( endLevel > lowest )
	{
	if  ( docFieldBalanceEnd( &endMoved, bpEnd, bd, endLevel, lowest ) )
	    { LDEB(1); return -1;	}
	}

    if  ( bpBegin->bpBi == bpEnd->bpBi	&&
	  readonly			)
	{
	lowest= INT_MAX;
	beginLevel= 0;
	readonly= 0;
	readOnlyLevel= -1;
	beginLevel= docTextFieldLevels( beginLevel, &readonly, &readOnlyLevel,
						&highest, &lowest,
						bd, bpBegin->bpBi,
						0,
						bpBegin->bpParticule );

	if  ( readonly )
	    {
	    if  ( docFieldBalanceBegin( &beginMoved, bpBegin, bd,
						beginLevel, readOnlyLevel ) )
		{ LDEB(1); return -1;	}

	    if  ( docFieldBalanceEnd( &endMoved, bpEnd, bd,
						endLevel, readOnlyLevel ) )
		{ LDEB(1); return -1;	}
	    }
	}

    *pBeginMoved= beginMoved;
    *pEndMoved= endMoved;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Refresh the field and hyperlink related flags in a BufferItem.	*/
/*									*/
/************************************************************************/

void docFieldRefreshFlags(	BufferItem *		bi,
				const BufferDocument *	bd )
    {
    int			i;
    int			part;
    TextParticule *	tp;
    int			inLink;

    switch( bi->biLevel )
	{
	case DOClevDOC:
	case DOClevCELL:
	case DOClevSECT:
	case DOClevROW:
	    for ( i= 0; i < bi->biChildCount; i++ )
		{ docFieldRefreshFlags( bi->biChildren[i], bd ); }
	    return;

	case DOClevPARA:
	    break;

	default:
	    LDEB(bi->biLevel); return;
	}

    inLink= 0;
    tp= bi->biParaParticules;
    for ( part= 0; part < bi->biParaParticuleCount; tp++, part++ )
	{
	DocumentField *		df;

	if  ( tp->tpKind == DOCkindFIELDEND )
	    {
	    df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;

	    if  ( df->dfKind == DOCfkHYPERLINK )
		{
		inLink--;

		if  ( inLink < 0 )
		    { LDEB(inLink); inLink= 0;	}
		}
	    }

	tp->tpTextAttribute.taShowAsLink= inLink != 0;

	if  ( tp->tpKind == DOCkindFIELDSTART )
	    {
	    df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;

	    if  ( df->dfKind == DOCfkHYPERLINK )
		{ inLink++;	}
	    }
	}

    return;
    }

static int docFieldCalculateBalance(	const BufferItem *	bi,
					int			part,
					int			partUpto )
    {
    int				fieldLevel= 0;
    const TextParticule *	tp;

    tp= bi->biParaParticules+ part;
    while( part < bi->biParaParticuleCount )
	{
	if  ( tp->tpKind == DOCkindFIELDSTART )
	    { fieldLevel++;	}
	if  ( tp->tpKind == DOCkindFIELDEND )
	    { fieldLevel--;	}

	part++; tp++;
	}

    return fieldLevel;
    }

/************************************************************************/
/*									*/
/*  Insert the particule that terminates a field.			*/
/*									*/
/*  1)  Find the particule.						*/
/*  2)  Should it be split?						*/
/*  3)  Insert end particule.						*/
/*									*/
/************************************************************************/

static int docFieldInsertEndParticule(	const BufferPosition *	bpEnd,
					int *			pEndPart,
					int			fieldNumber )
    {
    TextParticule *	tp;
    TextAttribute	ta;

    int			endPart= bpEnd->bpParticule;
    BufferItem *	bi= bpEnd->bpBi;

    /*  1  */
    tp= bi->biParaParticules+ endPart;
    if  ( tp->tpStrlen > 0				&&
	  bpEnd->bpStroff == tp->tpStroff+ tp->tpStrlen	)
	{ endPart++; tp++;	}

    if  ( endPart < bi->biParaParticuleCount )
	{ ta= tp[ 0].tpTextAttribute; }
    else{ ta= tp[-1].tpTextAttribute; }

    if  ( endPart < bi->biParaParticuleCount	&&
	  tp->tpStroff != bpEnd->bpStroff	)
	{
	int	len= tp->tpStroff+ tp->tpStrlen- bpEnd->bpStroff;

	tp->tpStrlen= bpEnd->bpStroff- tp->tpStroff;

	tp= docInsertTextParticule( bi, endPart+ 1,
					bpEnd->bpStroff, len, tp->tpKind, ta );
	if  ( ! tp )
	    { XDEB(tp); return -1;	}

	endPart++;
	}

    tp= docInsertTextParticule( bi, endPart,
				    bpEnd->bpStroff, 0, DOCkindFIELDEND, ta );
    if  ( ! tp )
	{ XDEB(tp); return -1;	}

    tp->tpObjectNumber= fieldNumber;

    *pEndPart= endPart; return 0;
    }

/************************************************************************/
/*									*/
/*  Insert the particule that starts a field.				*/
/*									*/
/*  1)  Find the particule.						*/
/*  2)  Should it be split?						*/
/*  3)  Insert start particule.						*/
/*									*/
/************************************************************************/

static int docFieldInsertStartParticule( const BufferPosition *	bpStart,
					int *			pStartPart,
					int *			pEndPart,
					int			fieldNumber )
    {
    TextParticule *	tp;
    TextAttribute	ta;
    int			endPart= *pEndPart;

    int			startPart;
    BufferItem *	bi= bpStart->bpBi;

    startPart= bpStart->bpParticule;
    tp= bi->biParaParticules+ startPart;
    if  ( tp->tpStrlen > 0					&&
	  bpStart->bpStroff == tp->tpStroff+ tp->tpStrlen	)
	{ startPart++; tp++;	}

    if  ( startPart < bi->biParaParticuleCount )
	{ ta= tp[ 0].tpTextAttribute; }
    else{ ta= tp[-1].tpTextAttribute; }

    /*  2  */
    if  ( startPart < bi->biParaParticuleCount	&&
	  tp->tpStroff != bpStart->bpStroff	)
	{
	int		stroff= tp->tpStroff;
	int		len= bpStart->bpStroff- tp->tpStroff;

	tp->tpStrlen= tp->tpStroff+ tp->tpStrlen- bpStart->bpStroff;
	tp->tpStroff= bpStart->bpStroff;

	tp= docInsertTextParticule( bi, startPart,
					    stroff, len, tp->tpKind, ta );
	if  ( ! tp )
	    { XDEB(tp); return -1;	}

	if  ( endPart >= startPart )
	    { endPart++;	}

	startPart++;
	}

    /*  3  */
    tp= docInsertTextParticule( bi, startPart,
				bpStart->bpStroff, 0, DOCkindFIELDSTART, ta );
    if  ( ! tp )
	{ XDEB(tp); return -1;	}
    tp->tpObjectNumber= fieldNumber;

    if  ( endPart >= startPart )
	{ endPart++;	}

    *pStartPart= startPart; *pEndPart= endPart; return 0;
    }

/************************************************************************/
/*									*/
/*  Surround a selected range of text by a field level field.		*/
/*									*/
/************************************************************************/

int docSurroundTextSelectionByField(	BufferSelection *	bsField,
					const BufferSelection *	bs,
					int			fieldNumber )
    {
    BufferSelection	bsNew;
    int			fieldBalance;

    if  ( bs->bsBegin.bpBi != bs->bsEnd.bpBi )
	{ XXDEB(bs->bsBegin.bpBi,bs->bsEnd.bpBi); return -1;	}

    bsNew= *bs;

    fieldBalance= docFieldCalculateBalance( bsNew.bsBegin.bpBi,
					    bsNew.bsBegin.bpParticule,
					    bsNew.bsEnd.bpParticule );
    if  ( fieldBalance != 0 )
	{
	LLDEB(bsNew.bsBegin.bpParticule,bsNew.bsEnd.bpParticule);
	LDEB(fieldBalance); return -1;
	}

    /*  find end, split?, insert end particule */
    if  ( docFieldInsertEndParticule( &(bsNew.bsEnd),
					    &(bsNew.bsEnd.bpParticule),
					    fieldNumber ) )
	{ LDEB(1); return -1;	}

    /*  find begin, split?, insert start particule */
    if  ( docFieldInsertStartParticule( &(bsNew.bsBegin),
					    &(bsNew.bsBegin.bpParticule),
					    &(bsNew.bsEnd.bpParticule),
					    fieldNumber ) )
	{ LDEB(1); return -1;	}

    *bsField= bsNew;

    return 0;
    }

