/************************************************************************/
/*									*/
/*  Text Editor Buffer structure.					*/
/*									*/
/************************************************************************/

#   ifndef	DOC_BUF_H
#   define	DOC_BUF_H

#   include	<time.h>

#   include	<bitmap.h>
#   include	<docFont.h>
#   include	"docShape.h"
#   include	<appGeo.h>
#   include	<sioGeneral.h>
#   include	<utilMemoryBuffer.h>

#   include	"docParaProperties.h"
#   include	"docSectProperties.h"
#   include	"docStyleSheet.h"
#   include	"docDocumentProperties.h"
#   include	"docField.h"
#   include	"docObject.h"

/************************************************************************/
/*									*/
/*  Forward declarations of structs.					*/
/*									*/
/************************************************************************/

struct BufferDocument;
struct BufferItem;
struct DocumentField;

/************************************************************************/
/*									*/
/*  An homogenous piece of text. Without hyphenation, it will not be	*/
/*  divided over lines.							*/
/*  It consists of some characters and then some spaces.		*/
/*									*/
/************************************************************************/

#   define	DOCkindUNKNOWN		0
#   define	DOCkindTEXT		1
#   define	DOCkindTAB		2
#   define	DOCkindOBJECT		3
#   define	DOCkindFIELDSTART	4
#   define	DOCkindFIELDEND		5
#   define	DOCkindTC		6
#   define	DOCkindXE		7
#   define	DOCkindLINEBREAK	8
#   define	DOCkindPAGEBREAK	9

typedef struct TextParticule
    {
    unsigned int	tpStroff:16;
    unsigned int	tpStrlen:12;
    unsigned int	tpX0:12;
    unsigned int	tpPixelsWide:12;
    unsigned int	tpKind:6;
    unsigned int	tpInField:1;
    short int		tpPhysicalFont;
    short int		tpObjectNumber;
    TextAttribute	tpTextAttribute;
    } TextParticule;

/************************************************************************/
/*									*/
/*  The geometry bounds inside which a layout job operates.		*/
/*									*/
/************************************************************************/

typedef struct FormattingFrame
    {
    int		ffX0GeometryTwips;
    int		ffX0TextLinesTwips;
    int		ffX0FirstLineTwips;

    int		ffX0TextLinesPixels;
    int		ffX0FirstLinePixels;

    int		ffX1GeometryTwips;
    int		ffX1TextLinesTwips;

    int		ffX1TextLinesPixels;

    int		ffPageFrameBottomTwips;

    int		ffY1;

    int		ffPageHigh;
    int		ffStripHigh;
    } FormattingFrame;

/************************************************************************/
/*									*/
/*  Used to layout lines of text.					*/
/*									*/
/*  A0 paper is 4* 30 cm high: 28.3465* 120* 20= 68031.6 Twips.		*/
/*  So an unsigned short limits us to A1 paper.				*/
/*									*/
/************************************************************************/

typedef struct LayoutPosition
    {
    unsigned short	lpPageYTwips;
    unsigned short	lpPage;
    unsigned char	lpColumn;
    unsigned char	lpAtTopOfColumn;
    } LayoutPosition;

/************************************************************************/
/*									*/
/*  Used to lay out the text on the page.				*/
/*  A text line consists of a series of particules.			*/
/*									*/
/*  1)  What part of the string in the paragraph.			*/
/*  2)  What particules.						*/
/*  3)  Where is it geo/typographically?				*/
/*  4)  Postscript geometry information.				*/
/*	lineSpacing includes the leading that is irrelevant for the	*/
/*	last line of a page.						*/
/*  5)  Screen geometry.						*/
/*									*/
/************************************************************************/

typedef struct TextLine
    {
								/*  1,2	*/
    int			tlStroff;
    int			tlFirstParticule;
    short int		tlStrlen;
    short int		tlParticuleCount;
    short int		tlWordCount;
								/*  3  */
    LayoutPosition	tlTopPosition;
								/*  4  */
    short int		tlLineAscentTwips;
    short int		tlLineHeightTwips;
    short int		tlLineSpacingTwips;
								/*  5  */
    short int		tlHasPageBreak;
    } TextLine;

/************************************************************************/
/*									*/
/*  Information on Lists.						*/
/*									*/
/************************************************************************/

typedef struct DocumentList
    {
    int			dlListId;
    int			dlListTemplateId;
    int			dlListIsSimple;
    int			dlRestartForEverySection;
    char *		dlListName;
    } DocumentList;

typedef struct CellProperties
    {
    int			cpRightBoundaryTwips;
    int			cpRightBoundaryPixels;

    int			cpForegroundColor;
    int			cpBackgroundColor;

    BorderProperties	cpTopBorder;
    BorderProperties	cpBottomBorder;
    BorderProperties	cpLeftBorder;
    BorderProperties	cpRightBorder;

    unsigned int	cpShadingLevel:10;
    unsigned int	cpShadingPattern:4;		/*  (enum)	*/

    unsigned int	cpLeftInMergedRow:1;
    unsigned int	cpMergedWithLeft:1;
    unsigned int	cpTopInMergedColumn:1;
    unsigned int	cpMergedWithAbove:1;

    unsigned int	cpVerticalTextAlignment:3;	/*  (enum)	*/
    } CellProperties;

#   define	docCopyCellProperties( cp1, cp2 ) \
					    ( ( *(cp1)= *(cp2) ), 0 )

#   define	docCleanCellProperties( cp ) ( ( *(cp)= *(cp) ), 0 )

typedef struct RowProperties
    {
    int			rpCellCount;
    CellProperties *	rpCells;

    int			rpHalfGapWidthTwips;
    int			rpHalfGapWidthPixels;

    int			rpHeightTwips;

    int			rpLeftIndentTwips;
    int			rpLeftIndentPixels;

			/************************************************/
			/*  Row borders are not stored in the word	*/
			/*  binary format, and seem to be irrelevant:	*/
			/************************************************/
    BorderProperties	rpTopBorder;
    BorderProperties	rpBottomBorder;
    BorderProperties	rpLeftBorder;
    BorderProperties	rpRightBorder;
    BorderProperties	rpHorizontalBorder;
    BorderProperties	rpVerticalBorder;

    ItemAlignment	rpAlignment:3;
    unsigned int	rpHasTableParagraphs:1;

    unsigned int	rpHasHorizontalBorders:1;
    unsigned int	rpHasVerticalBorders:1;

    unsigned int	rpIsTableHeader:1;
    unsigned int	rpKeepOnPage:1;
    } RowProperties;

/************************************************************************/
/*									*/
/*  Primitives for shapes.						*/
/*									*/
/************************************************************************/

typedef enum ShapeHorizontalAttachment
    {
    SHPshaPAGE= 0,
    SHPshaMARGIN,
    SHPshaCOLUMN
    } ShapeHorizontalAttachment;

typedef enum ShapeVerticalAttachment
    {
    SHPsvaPAGE= 0,
    SHPsvaMARGIN,
    SHPsvaPARAGRAPH
    } ShapeVerticalAttachment;

typedef enum ShapeWrapStyle
    {
    SHPswsTOPBOTTOM= 1,
    SHPswsAROUND,
    SHPswsNONE,
    SHPswsTIGHTLY,
    SHPswsTHROUGH
    } ShapeWrapStyle;

typedef enum ShapeWrapSide
    {
    SHPswsBOTH= 0,
    SHPswsLEFT,
    SHPswsRIGHT,
    SHPswsLARGEST
    } ShapeWrapSide;

typedef struct ShapeProperties
    {
    int			spTwipsLeftOfAnchor;
    int			spTwipsAboveAnchor;
    int			spTwipsBelowAnchor;
    int			spTwipsRightOfAnchor;

    unsigned int	spHorizontalAttachment:2;	/*  (enum)	*/
    unsigned int	spVerticalAttachment:2;		/*  (enum)	*/
    unsigned int	spWrapStyle:3;			/*  (enum)	*/
    unsigned int	spWrapSide:2;			/*  (enum)	*/
    unsigned int	spLockAnchor:1;
    } ShapeProperties;

typedef struct BufferPara
    {
    unsigned char *		btString;
    unsigned int		btStrlen;

    int				btParticuleCount;
    TextParticule *		btParticules;

    int				btObjectCount;
    InsertedObject *		btObjects;

    int				btLineCount;
    TextLine *			btLines;

    int				btShapeCount;
    DrawingShape *		btShapes;

    ParagraphProperties		btProperties;

    int				btAscentTwips;
    int				btDescentTwips;
    int				btLeadingTwips;

    int				btSpaceAboveLinesTwips;
    const BorderProperties *	btBorderAboveParagraph;

    int				btSpaceBelowLinesTwips;
    const BorderProperties *	btBorderBelowParagraph;
    } BufferPara;

typedef struct HeaderFooter
    {
    struct BufferItem *	hfItem;
    int			hfPageUsedFor;
    int			hfY0UsedTwips;
    int			hfY1UsedTwips;
    int			hfY0ReservedTwips;
    int			hfY1ReservedTwips;
    } HeaderFooter;

typedef struct BufferSect
    {
    SectionProperties		bsProperties;

    HeaderFooter		bsHeader;
    HeaderFooter		bsFirstPageHeader;
    HeaderFooter		bsLeftPageHeader;
    HeaderFooter		bsRightPageHeader;

    HeaderFooter		bsFooter;
    HeaderFooter		bsFirstPageFooter;
    HeaderFooter		bsLeftPageFooter;
    HeaderFooter		bsRightPageFooter;

				/****************************************/
				/*  Only used for Headers and Footers:	*/
				/*  A)  Normally not set. Is set by the	*/
				/*	formatter to pass information	*/
				/*	to calculate page numbers.	*/
				/****************************************/
    int				bsUseForPage;
    const struct BufferItem *	bsUseForSectBi;
#   define			biSectHeaderFooterUseForPage \
						BIU.biuSect.bsUseForPage
				/*  A  */
#   define			biSectHeaderFooterUseForSectBi \
						BIU.biuSect.bsUseForSectBi
    } BufferSect;

typedef struct BufferRow
    {
    RowProperties		bguRowProperties;
    } BufferRow;

/************************************************************************/
/*									*/
/*  Levels of nesting.							*/
/*									*/
/************************************************************************/

typedef enum ItemLevel
    {
    DOClevANY,
    DOClevOUT,
			/****************************************/
			/*  Ignore, Garbage values.		*/
			/****************************************/
    DOClevDOC,
			/****************************************/
			/*					*/
			/*  Document.				*/
			/*					*/
			/*  * biGroupChildren			*/
			/*  * biGroupChildCount			*/
			/****************************************/
    DOClevSECT,
			/****************************************/
			/*					*/
			/*  Section.				*/
			/*					*/
			/*  * biGroupChildren			*/
			/*  * biGroupChildCount			*/
			/*					*/
			/****************************************/

#	define	biSectHeader		BIU.biuSect.bsHeader
#	define	biSectFirstPageHeader	BIU.biuSect.bsFirstPageHeader
#	define	biSectLeftPageHeader	BIU.biuSect.bsLeftPageHeader
#	define	biSectRightPageHeader	BIU.biuSect.bsRightPageHeader

#	define	biSectFooter		BIU.biuSect.bsFooter
#	define	biSectFirstPageFooter	BIU.biuSect.bsFirstPageFooter
#	define	biSectLeftPageFooter	BIU.biuSect.bsLeftPageFooter
#	define	biSectRightPageFooter	BIU.biuSect.bsRightPageFooter

#	define	biSectProperties	BIU.biuSect.bsProperties

#	define	biSectDocumentGeometry	biSectProperties.spDocumentGeometry
#	define	biSectPageWideTwips	biSectProperties.spPageWideTwips
#	define	biSectPageHighTwips	biSectProperties.spPageHighTwips
#	define	biSectLeftMarginTwips	biSectProperties.spLeftMarginTwips
#	define	biSectTopMarginTwips	biSectProperties.spTopMarginTwips
#	define	biSectRightMarginTwips	biSectProperties.spRightMarginTwips
#	define	biSectBottomMarginTwips	biSectProperties.spBottomMarginTwips
#	define	biSectHeaderYTwips	biSectProperties.spHeaderYTwips
#	define	biSectFooterYTwips	biSectProperties.spFooterYTwips

#	define	biSectHasTitlePage	biSectProperties.spHasTitlePage
#	define	biSectBreakKind		biSectProperties.spBreakKind
#	define	biSectPageNumberStyle	biSectProperties.spPageNumberStyle
#	define	biSectRestartPageNumbers \
					biSectProperties.spRestartPageNumbers
#	define	biSectStartPageNumber	biSectProperties.spStartPageNumber

#	define	biSectParagraphNumbers	biSectProperties.spParagraphNumbers
#	define	biSectParagraphNumberCount \
					biSectProperties.spParagraphNumberCount

			/****************************************/

    DOClevROW,
			/****************************************/
			/*  Row.				*/
			/*  * biGroupChildren			*/
			/*  * biGroupChildCount			*/
			/****************************************/
#	define	biRowProperties		BIU.biuRow.bguRowProperties
#	define	biRowHasTableParagraphs	biRowProperties.rpHasTableParagraphs
#	define	biRowHalfGapWidthTwips	biRowProperties.rpHalfGapWidthTwips
#	define	biRowHalfGapWidthPixels \
					biRowProperties.rpHalfGapWidthPixels
#	define	biRowLeftIndentTwips	biRowProperties.rpLeftIndentTwips
#	define	biRowLeftIndentPixels	biRowProperties.rpLeftIndentPixels
#	define	biRowHeightTwips	biRowProperties.rpHeightTwips
#	define	biRowIsTableHeader	biRowProperties.rpIsTableHeader
#	define	biRowKeepOnPage		biRowProperties.rpKeepOnPage
#	define	biRowHasHorizontalBorders \
					biRowProperties.rpHasHorizontalBorders
#	define	biRowHasVerticalBorders \
					biRowProperties.rpHasVerticalBorders
#	define	biRowCells		biRowProperties.rpCells
#	define	biRowCellCount		biRowProperties.rpCellCount
#	define	biRowAlignment		biRowProperties.rpAlignment

#	define	biRowTopBorder		biRowProperties.rpTopBorder
#	define	biRowBottomBorder	biRowProperties.rpBottomBorder
#	define	biRowLeftBorder		biRowProperties.rpLeftBorder
#	define	biRowRightBorder	biRowProperties.rpRightBorder
#	define	biRowVerticalBorder	biRowProperties.rpVerticalBorder
#	define	biRowHorizontalBorder	biRowProperties.rpHorizontalBorder

    DOClevCELL,
			/****************************************/
			/*  Cell.				*/
			/*  * biGroupChildren			*/
			/*  * biGroupChildCount			*/
			/****************************************/
	/* No.. Information is stored with the row.
	define	biCellProperties	BIU.biuGroup.BGU.bguCellProperties
	*/

    DOClevPARA,
			/********************************/
			/*  Paragraph.			*/
			/********************************/
#	define	biParaString		BIU.biuPara.btString
#	define	biParaStrlen		BIU.biuPara.btStrlen
#	define	biParaParticuleCount	BIU.biuPara.btParticuleCount
#	define	biParaParticules	BIU.biuPara.btParticules
#	define	biParaObjectCount	BIU.biuPara.btObjectCount
#	define	biParaObjects		BIU.biuPara.btObjects
#	define	biParaLineCount		BIU.biuPara.btLineCount
#	define	biParaLines		BIU.biuPara.btLines
#	define	biParaShapeCount	BIU.biuPara.btShapeCount
#	define	biParaShapes		BIU.biuPara.btShapes
#	define	biParaProperties	BIU.biuPara.btProperties
#	define	biParaInTable		biParaProperties.ppInTable
#	define	biParaLineSpacingIsMultiple \
					biParaProperties.ppLineSpacingIsMultiple
#	define	biParaStartsOnNewPage	biParaProperties.ppStartsOnNewPage
#	define	biParaFirstIndentTwips	biParaProperties.ppFirstIndentTwips
#	define	biParaLeftIndentTwips	biParaProperties.ppLeftIndentTwips
#	define	biParaRightIndentTwips	biParaProperties.ppRightIndentTwips
#	define	biParaSpaceBeforeTwips	biParaProperties.ppSpaceBeforeTwips
#	define	biParaSpaceAfterTwips	biParaProperties.ppSpaceAfterTwips
#	define	biParaLineSpacingTwips	biParaProperties.ppLineSpacingTwips
#	define	biParaAlignment		biParaProperties.ppAlignment
#	define	biParaTabStops		biParaProperties.ppTabStops
#	define	biParaTabCount		biParaProperties.ppTabCount
#	define	biParaKeepOnPage	biParaProperties.ppKeepOnPage
#	define	biParaKeepWithNext	biParaProperties.ppKeepWithNext
#	define	biParaShadingLevel	biParaProperties.ppShadingLevel
#	define	biParaShadingPattern	biParaProperties.ppShadingPattern
#	define	biParaOutlineLevel	biParaProperties.ppOutlineLevel
#	define	biParaTopBorder		biParaProperties.ppTopBorder
#	define	biParaBottomBorder	biParaProperties.ppBottomBorder
#	define	biParaLeftBorder	biParaProperties.ppLeftBorder
#	define	biParaRightBorder	biParaProperties.ppRightBorder
#	define	biParaBoxBorder		biParaProperties.ppBoxBorder
#	define	biParaBetweenBorder	biParaProperties.ppBetweenBorder
#	define	biParaBar		biParaProperties.ppBar
#	define	biParaStyle		biParaProperties.ppStyle
#	define	biParaWidowControl	biParaProperties.ppWidowControl

#	define	biParaAscentTwips	BIU.biuPara.btAscentTwips
#	define	biParaDescentTwips	BIU.biuPara.btDescentTwips
#	define	biParaLeadingTwips	BIU.biuPara.btLeadingTwips

#	define	biParaSpaceAboveLinesTwips \
					BIU.biuPara.btSpaceAboveLinesTwips
#	define	biParaBorderAboveParagraph \
					BIU.biuPara.btBorderAboveParagraph
#	define	biParaSpaceBelowLinesTwips \
					BIU.biuPara.btSpaceBelowLinesTwips
#	define	biParaBorderBelowParagraph \
					BIU.biuPara.btBorderBelowParagraph

    DOClevTEXT
			/********************************/
			/*  Handeled inside TEXT.	*/
			/********************************/
    } ItemLevel;

typedef struct BufferItem
    {
    short int			biLevel;
    char			biInHeaderFooter;
    struct BufferItem *		biParent;
    int				biNumberInParent;
    struct BufferItem **	biChildren;
    int				biChildCount;


    LayoutPosition		biTopPosition;
    LayoutPosition		biBelowPosition;

    union	BIU
	{
	BufferPara	biuPara;
	BufferSect	biuSect;
	BufferRow	biuRow;
	} BIU;

    } BufferItem;

/************************************************************************/
/*									*/
/*  A position in a document.						*/
/*									*/
/*  1)  Geometry is derived from the position, NOT the other way round.	*/
/*									*/
/************************************************************************/

typedef struct BufferPosition
    {
    BufferItem *	bpBi;
    int			bpLine;
    int			bpParticule;
    int			bpStroff;

    /*  1  */
    int			bpXPixels;
    LayoutPosition	bpTopPosition;
    int			bpY1Pixels;
    int			bpBaselinePixels;
    } BufferPosition;

#			define	DOCinBODY		0
#			define	DOCinSECT_HEADER	1
#			define	DOCinFIRST_HEADER	2
#			define	DOCinLEFT_HEADER	3
#			define	DOCinRIGHT_HEADER	4
#			define	DOCinSECT_FOOTER	5
#			define	DOCinFIRST_FOOTER	6
#			define	DOCinLEFT_FOOTER	7
#			define	DOCinRIGHT_FOOTER	8
#			define	DOCinFOOTNOTE		9
			/* ?
#			define	DOCinENDNOTE		10
			*/

typedef struct SelectionScope
    {
    int			ssInHeaderFooter;
    int			ssHeaderFooterSection;
    int			ssHeaderFooterPage;
    } SelectionScope;

typedef struct BufferSelection
    {
    BufferPosition	bsBegin;
    BufferPosition	bsEnd;
    BufferPosition	bsAnchor;
    int			bsCol0;
    int			bsCol1;
    int			bsDirection;
    } BufferSelection;

typedef struct TableRectangle
    {
    int		trSectionNumber;

    int		trCol0;
    int		trCol1;
    int		trCol11;

    int		trRow00;
    int		trRow0;
    int		trRow1;
    int		trRow11;
    } TableRectangle;

#   define	docPositionsInsideCell(b,e) \
		((b)->bpBi->biParent == (e)->bpBi->biParent)
#   define	docPositionsInsideRow(b,e) \
		((b)->bpBi->biParent->biParent == (e)->bpBi->biParent->biParent)

#   define	docSelectionInsideCell(bs) \
		docPositionsInsideCell(&(bs)->bsBegin,&(bs)->bsEnd)
#   define	docSelectionInsideRow(bs) \
		docPositionsInsideRow(&(bs)->bsBegin,&(bs)->bsEnd)

/************************************************************************/
/*									*/
/*  The document as a whole.						*/
/*									*/
/************************************************************************/

typedef struct BufferDocument
    {
    BufferItem		bdItem;
    DocumentProperties	bdProperties;

    DocumentStyleSheet	bdStyleSheet;
    DocumentFieldList	bdFieldList;
    } BufferDocument;

/************************************************************************/
/*									*/
/*  Statistics about a document. Used in the 'Document Properties'	*/
/*  dialog.								*/
/*									*/
/************************************************************************/

typedef struct DocumentStatistics
    {
    int		dsPageCount;
    int		dsParagraphCount;
    int		dsLineCount;
    int		dsWordCount;
    int		dsCharacterCount;
    } DocumentStatistics;

/************************************************************************/
/*									*/
/*  Routine declarations.						*/
/*									*/
/************************************************************************/

extern BufferDocument *	docPlainReadFile( SimpleInputStream *	sis,
				    int *			pMxL,
				    const DocumentGeometry *	dg );

extern BufferDocument *	docRtfReadFile(	SimpleInputStream *	sis,
					int			defAnsigpg );

extern BufferDocument *	docLeafReadFile(	SimpleInputStream *	sis );
extern BufferDocument *	docWordReadFile(	SimpleInputStream *	sis );

extern BufferDocument * docNewFile(	const char *		fontName,
					int			fontSize,
					const DocumentGeometry * dg );

extern int docRtfSaveDocument(		SimpleOutputStream *	sos,
					BufferDocument *	bd,
					const SelectionScope *	ss,
					const BufferSelection *	bs,
					int			saveBookmarks );

extern int docRtfSaveSelectionAsLink(	SimpleOutputStream *	sos,
					const BufferDocument *	bd,
					const SelectionScope *	ss,
					const BufferSelection *	bs,
					int			asRef,
					int			asPageref,
					const unsigned char *	fileName,
					int			fileSize,
					const unsigned char *	markName,
					int			markSize );

extern int docHtmlSaveDocument(		SimpleOutputStream *	sos,
					const BufferDocument *	bd,
					int			asMimeAggr,
					const char *		mimeBoundary,
					const char *		filename );

extern int docPlainSaveDocument(	SimpleOutputStream *	sos,
					BufferDocument *	bd,
					const SelectionScope *	ss,
					const BufferSelection *	bs,
					int			fold,
					int			closed );

extern int docRtfSaveRuler(	SimpleOutputStream *		sos,
				const ParagraphProperties *	pp );

extern int docRtfReadRuler(	SimpleInputStream *	sis,
				ParagraphProperties *	pp );

extern int docInflateTextString(	BufferItem *    bi,
					int		by	);

extern void docInitTextLine(		TextLine *	tl );

extern TextLine * docInsertTextLine(	BufferItem *	bi,
					int		line );

extern void docCleanItem(	BufferDocument *	bd,
				BufferItem *		bi );

extern void docFreeItem(	BufferDocument *	bd,
				BufferItem *		bi );

extern void docFreeDocument( BufferDocument *	bd );

extern void docInitDocument(	BufferDocument *	bd	);

extern void docInitTabStop(	TabStop *	ts );

extern int docCopyRowProperties(	RowProperties *			to,
					const RowProperties *		from );

extern BufferItem * docInsertItem(	const BufferDocument *	bd,
					BufferItem *		parent,
					int			n,
					ItemLevel		level );

extern BufferItem * docCopyParaItem(	BufferDocument *	bdTo,
					const BufferDocument *	bdFrom,
					int *			fieldMap,
					unsigned int *		pFieldUpd,
					BufferItem *		biCellTo,
					int			n,
					BufferItem *		biParaFrom,
					const char *		filename );

extern BufferItem * docCopyRowItem(	BufferDocument *	bdTo,
					const BufferDocument *	bdFrom,
					int *			fieldMap,
					unsigned int *		pFieldUpd,
					BufferItem *		biSectTo,
					int			n,
					BufferItem *		biRowFrom,
					const char *		filename );

extern void docDeleteItems(	BufferDocument *	bd,
				BufferItem *		bi,
				int			first,
				int			count	);

extern TextParticule *	docInsertTextParticule(	BufferItem *	bi,
						int		n,
						int		off,
						int		len,
						int		kind,
						TextAttribute	ta	);

extern TextParticule * docCopyParticule(	BufferItem *		bi,
						int			n,
						int			off,
						int			len,
						int			kind,
						const TextParticule *	from );

void docDeleteParticules	(	BufferItem *	bi,
					int		first,
					int		count	);

extern void docDeleteLines	(	BufferItem *	bi,
					int		first,
					int		count	);

extern int docParaAddTab(	ParagraphProperties *	pp,
				const TabStop *		ts );

extern void docInitPosition(		BufferPosition *	bp );
extern void docInitSelection(		BufferSelection *	bs );
extern void docInitTableRectangle(	TableRectangle *	tr );

extern int docParaReplaceText(	BufferDocument *	bd,
				BufferItem *		bi,
				int			part,
				unsigned int		stroffBegin,
				int *			pPartShift,
				int *			pStroffShift,
				unsigned int		stroffEnd,
				const unsigned char *	addedText,
				unsigned int		addedLength,
				TextAttribute		addedAttribute,
				void *			voidadd,
				DOC_CLOSE_OBJECT	closeObject );

extern int docSplitParaItem(	BufferDocument *	bd,
				BufferItem **		pNewBi,
				BufferItem *		oldBi,
				int			stroff );

extern int docSplitGroupItem(	const BufferDocument *	bd,
				BufferItem **		pNewBi,
				BufferItem *		oldBi,
				int			n );

extern BufferItem * docNextParagraph(	BufferItem *	bi	);
extern BufferItem * docPrevParagraph(	BufferItem *	bi	);

extern BufferItem * docNextSection(	BufferItem *	bi	);
extern BufferItem * docPrevSection(	BufferItem *	bi	);

extern int docNextPosition(	BufferPosition *	bp	);

extern int docPrevPosition(	BufferPosition *	bp,
				int			lastOne );

extern int docNextWord(		BufferPosition *	bp,
				int			lastOne );
extern int docPrevWord(		BufferPosition *	bp,
				int			lastOne );

extern int docParaBegin(	BufferPosition *	bp,
				int			lastOne );
extern int docParaEnd(		BufferPosition *	bp,
				int			lastOne );

extern int docNextLine(		TextParticule **	pTp,
				TextLine **		pTl,
				BufferPosition *	bp	);
extern int docPrevLine(		TextParticule **	pTp,
				TextLine **		pTl,
				BufferPosition *	bp	);

extern int docFirstPosition(	BufferItem *		bi,
				BufferPosition *	bp	);

extern int docLastPosition(	BufferItem *		bi,
				BufferPosition *	bp	);

extern void docSetParaSelection(	BufferSelection *	bs,
				BufferItem *		bi,
				int			direction,
				int			start,
				int			length	);

extern int docComparePositions(	const BufferPosition *	bp1,
				const BufferPosition *	bp2,
				int			mindLine );

extern int docCompareItemPositions(	const BufferItem *	bi1,
					const BufferItem *	bi2 );

extern TextParticule * docParaSpecialParticule(	BufferItem *	bi,
					int			kind,
					int			part,
					int			stroff,
					int *			pPartShift,
					int *			pStroffShift );

extern int docFindLineAndParticule(	BufferItem *		bi,
					int			stroff,
					BufferPosition *	bp ,
					int			lastOne );

extern int docFindParticule(		const BufferItem *	bi,
					int			stroff,
					int *			pPart,
					int			lastOne );

extern int docFindLineOfParticule(	const BufferItem *	bi,
					int			part,
					int *			pLine );

extern void docDeleteItem(		BufferDocument *	bd,
				    BufferItem *		bi );

extern void docListItem(	int			indent,
				const BufferItem *	bi );

extern void docListParticule(	int			indent,
				const char *		label,
				int			n,
				const BufferItem *	bi,
				const TextParticule *	tp );

extern void docListTextLine(	int			indent,
				const char *		label,
				int			n,
				const BufferItem *	bi,
				const TextLine *	tl );

extern int docReplaceSelection(	BufferDocument *	bd,
				const BufferSelection *	bs,
				int *			pPartShift,
				int *			pStroffShift,
				const unsigned char *	addedText,
				int			addedLength,
				TextAttribute		addedAttribute,
				void *			voidadd,
				DOC_CLOSE_OBJECT	closeObject );

extern void docInitItem(	BufferItem *		bi,
				BufferItem *		parent,
				const BufferDocument *	bd,
				int			numberInParent,
				int			level,
				int			inHeaderFooter );

extern const char * docKindStr(		int			kind );
extern const char * docLevelStr(	int			level );
extern const char * docExternalKindStr(	int			level );
extern const char * docFieldKindStr(	int			kind );
extern const char * docAttributeStr(	TextAttribute		ta );

extern TextParticule *	docInsertObject(	BufferItem *		bi,
						InsertedObject **	pIo,
						int			n,
						int			off,
						TextAttribute		ta );

extern int docObjectSetData(	InsertedObject *	io,
				const unsigned char *	bytes,
				int			size );

extern int docSetResultData(	InsertedObject *	io,
				const unsigned char *	bytes,
				int			size );

extern int docSaveObjectTag(	InsertedObject *	io,
				const char *		tag,
				int			arg );

extern int docSaveResultTag(	InsertedObject *	io,
				const char *		tag,
				int			arg );

extern void docCloseItemObjects(	BufferDocument *	bd,
					BufferItem *		bi,
					void *			voidadd,
					DOC_CLOSE_OBJECT	closeObject );

extern void docCleanParticuleObject(	BufferItem *	bi,
					TextParticule *	tp );

extern InsertedObject *	docClaimObject(	int *			pNr,
					BufferItem *		bi );

extern InsertedObject * docClaimObjectCopy( BufferItem *	bi,
					int *			pNr,
					const InsertedObject *	ioFrom );

extern int docSetObjectName(	InsertedObject *	io,
				const unsigned char *	name,
				int			len );

extern int docSetObjectClass(	InsertedObject *	io,
				const unsigned char *	name,
				int			len );

extern DocumentField * docClaimField(	int *			pNr,
					DocumentFieldList *	dfl );

extern int docSetFieldInst(	DocumentField *		df,
				const unsigned char *	bytes,
				int			size );

extern int docAddToFieldInst(	DocumentField *		df,
				const unsigned char *	bytes,
				int			size );

extern void docInitRowProperties(	RowProperties *	rp );
extern void docCleanRowProperties(	RowProperties *	rp );
extern void docCleanInitRowProperties(	RowProperties *	rp );

extern void docInitCellProperties(	CellProperties *	cp );

extern int docInsertRowColumn(	RowProperties *			rp,
				int				n,
				const CellProperties *	cp );

extern int docAlignedColumns(	const RowProperties *	rp1,
				const RowProperties *	rp2 );

extern int docEqualRows(	const RowProperties *	rp1,
				const RowProperties *	rp2 );

extern int docDelimitTable(	BufferItem *		paraBi,
				BufferItem **		pSectBi,
				int *			pCol,
				int *			pRow0,
				int *			pRow,
				int *			pRow1 );

extern void docInitShapeProperties(	ShapeProperties *	sp );


extern int docGetBookmarkForPosition(	const BufferDocument *	bd,
					const BufferPosition *	bp,
					int *			pPart,
					int *			pEndPart,
					const char **		pMarkName,
					int *			pMarkSize );

extern int docGetHyperlinkForPosition(	const BufferDocument *	bd,
					const BufferPosition *	bp,
					int *			pStartPart,
					int *			pEndPart,
					const char **		pFileName,
					int *			pFileSize,
					const char **		pMarkName,
					int *			pMarkSize );

extern int docFieldSetHyperlink(	DocumentField *		df,
					const unsigned char *	file,
					int			fileSize,
					const unsigned char *	mark,
					int			markSize );

extern DocumentField *	docFindBookmarkField( const DocumentFieldList * dfl,
					const char *		markName,
					int			markSize );

extern int docFieldGetHyperlink(	const DocumentField *	df,
					const char **		pFileName,
					int *			pFileSize,
					const char **		pMarkName,
					int *			pMarkSize );

extern int docFieldGetBookmark(		const DocumentField *	df,
					const char **		pMarkName,
					int *			pMarkSize );

extern int docFieldGetPageref(		const DocumentField *	df,
					const char **		pMarkName,
					int *			pMarkSize );

extern int docFieldGetIncludePicture(	const DocumentField *	df,
					const char **		pFileName,
					int *			pFileSize );

extern int docFieldGetRef(		const DocumentField *	df,
					const char **		pMarkName,
					int *			pMarkSize );

extern int docFieldSetBookmark(		DocumentField *		df,
					const unsigned char *	mark,
					int			markSize );

extern int docFieldSetRef(		DocumentField *		df,
					const unsigned char *	mark,
					int			markSize );

extern int docFieldSetPageref(		DocumentField *		df,
					const unsigned char *	mark,
					int			markSize );

extern void docInitField(		DocumentField *	df );

extern void docCleanField(		BufferDocument *	bd,
					DocumentField *		df );

extern void docCleanFieldList(		BufferDocument *	bd,
					DocumentFieldList *	dfl );
extern void docInitFieldList(		DocumentFieldList *	dfl );

DocumentField *	docClaimFieldCopy(	DocumentFieldList *	dfl,
					int *			pNr,
					const DocumentField *	dfFrom );

extern int docCopyParticuleData(	BufferDocument *	bdTo,
					const BufferDocument *	bdFrom,
					int *			fieldMap,
					unsigned int *		pFieldUpd,
					BufferItem *		biTo,
					const BufferItem *	biFrom,
					TextParticule *		tpTo,
					const TextParticule *	tpFrom );

extern int docCopyParticules(	BufferDocument *	bdTo,
				const BufferDocument *	bdFrom,
				int *			fieldMap,
				unsigned int *		pFieldUpd,
				BufferItem *		biTo,
				const BufferItem *	biFrom,
				int			partTo,
				int			partFrom,
				int			countFrom,
				int *			pParticulesInserted,
				int *			pCharactersCopied,
				const char *		refFileName );

extern void docInitShape(	DrawingShape *	ds );
extern void docCleanShape(	DrawingShape *	ds );

extern DrawingShape * docClaimShape(	int *			pNr,
					BufferItem *		bi );

extern int docParticuleInField(		BufferItem *	bi,
					int		part );

extern int docSaveParticules(	BufferItem *		bi,
				TextAttribute		ta,
				const unsigned char	inputMapping[256],
				const unsigned char *	text,
				int			len );

extern int docSaveSpecialParticule(	BufferItem *		bi,
					TextAttribute		ta,
					int			kind );

extern void docLogRectangle(	const char *			label,
				const DocumentRectangle *	dr );

extern void docLogRectangles(	const char *			label1,
				const DocumentRectangle *	dr1,
				const char *			label2,
				const DocumentRectangle *	dr2 );

extern void docInitLayoutPosition(	LayoutPosition *	lp );

extern void docLinePositions(	BufferPosition *	bpLineBegin,
				BufferPosition *	bpLineEnd,
				const BufferItem *	bi,
				int			line );

extern void docInitDocumentStatistics(		DocumentStatistics *	ds );

extern void docCollectDocumentStatistics(	DocumentStatistics *	ds,
						const BufferDocument *	bd );

extern int docWhatPageHeader(	HeaderFooter **			pHf,
				BufferItem *			sectBi,
				int				page,
				const DocumentProperties *	dp );

extern int docWhatPageFooter(	HeaderFooter **			pHf,
				BufferItem *			sectBi,
				int				page,
				const DocumentProperties *	dp );

extern BufferItem * docGetSelectionRoot(	BufferDocument *	bd,
						int			nearest,
						const SelectionScope *	ss,
						const BufferSelection *	bs );

extern void docInitParagraphNumber(	ParagraphNumber *	pn );

extern TextParticule *	docInsertParticule(	BufferItem *	bi,
						int		n,
						int		off,
						int		len,
						int		kind );

extern int docCopyFieldRelative(	BufferDocument *	bdTo,
					const BufferDocument *	bdFrom,
					int *			fieldMap,
					unsigned int *		pFieldUpd,
					BufferItem *		biTo,
					const BufferItem *	biFrom,
					TextParticule *		tpTo,
					const TextParticule *	tpFrom,
					const char *		refFileName,
					int			refFileSize );

extern void docCleanObject(	InsertedObject *	io );

extern void docFieldRefreshFlags(	BufferItem *		bi,
					const BufferDocument *	bd );

extern void docAdaptBookmarkName(	int *		pLength,
					char *		mark );

extern void docInitDocumentList(	DocumentList *	dl );
extern void docCleanDocumentList(	DocumentList *	dl );


extern int docRecalculateTextLevelFields(
					int *			pChanged,
					BufferDocument *	bd,
					BufferItem *		bi,
					unsigned int		whenMask,
					void *			voidadd,
					DOC_CLOSE_OBJECT	closeObject );

extern int docRedivideStringInParticules(	BufferItem *	bi,
						int		strOff,
						int		strLen,
						int		part,
						int		partsFree,
						TextAttribute	ta );

extern int docBalanceFieldSelection(	int *			pBeginMoved,
					int *			pEndMoved,
					const BufferDocument *	bd,
					BufferPosition *	bpBegin,
					BufferPosition *	bpEnd );

extern void docInitSelectionScope(	SelectionScope *	ss );

extern int docSplitFieldInstructions(	const MemoryBuffer *	mb,
				FieldInstructionsComponent *	fic,
				int				count );

extern int docGetHeaderFooter(	HeaderFooter **			pHf,
				BufferItem **			pSectBi,
				const SelectionScope *		ss,
				const BufferSelection *		bs,
				const BufferDocument *		bd,
				int				which );

extern void docInitHeaderFooter(	HeaderFooter *		hf );
extern void docCleanHeaderFooter(	BufferDocument *	bd,
					HeaderFooter *		hf );

extern BufferItem * docMakeExternalItem(
				BufferDocument *		bd,
				const SectionProperties *	sp,
				int				inHdFt );

extern BufferItem * docInsertEmptyParagraph(	BufferDocument *	bd,
						BufferItem *		bi,
						TextAttribute		ta );

extern int docIntersectTableRectangle(	TableRectangle *		tr,
					const TableRectangle *		tr1,
					const TableRectangle *		tr2 );

extern void docExpandTableRectangleToWholeTable( TableRectangle *	tr );
extern void docExpandTableRectangleToWholeRows(	TableRectangle *	tr );
extern void docExpandTableRectangleToWholeColumns( TableRectangle *	tr );

extern int docShiftTableRectangleByRows( TableRectangle *	tr,
					int			rows );
extern int docShiftTableRectangleByColumns( TableRectangle *	tr,
					int			columns );

extern int docTableRectangleSelection(	BufferSelection *	bs,
					BufferItem **		pSectBi,
					BufferDocument *	bd,
					const SelectionScope *	ss,
					const TableRectangle *	tr );

extern int docMergeDocumentFontsIntoList( DocumentFontList *	dflTo,
					BufferDocument *	bdFrom );

extern int docFindBookmarkInDocument(	SelectionScope *	ss,
					BufferSelection *	bs,
					BufferDocument *	bd,
					const char *		markName,
					int			markSize );

extern int docSurroundTextSelectionByField( BufferSelection *	bsField,
					const BufferSelection *	bs,
					int			fieldNumber );

extern void docConstrainSelectionToOneParagraph( int *		pBeginMoved,
					int *			pEndMoved,
					BufferSelection *	bs );

extern void docUnionParaSelections(	BufferSelection *	bs,
					const BufferSelection *	bs1,
					const BufferSelection *	bs2 );

extern int *	docAllocateFieldMap(	const BufferDocument *	bdFrom );

extern int docCalculatePagerefFieldString(
					int *			pCalculated,
					int *			pNewSize,
					unsigned char *		target,
					int			targetSize,
					BufferDocument *	bd,
					const BufferItem *	paraBi,
					const DocumentField *	dfRef );

extern int docCalculatePageFieldString(	int *			pCalculated,
					int *			pNewSize,
					unsigned char *		target,
					int			targetSize,
					BufferDocument *	bd,
					const BufferItem *	paraBi,
					const DocumentField *	df );

extern int docCalculateNumpagesFieldString(
					int *			pCalculated,
					int *			pNewSize,
					unsigned char *		target,
					int			targetSize,
					BufferDocument *	bd,
					const BufferItem *	paraBi,
					const DocumentField *	df );

extern int docCalculateDocDateFieldString(
					int *			pCalculated,
					int *			pNewSize,
					unsigned char *		target,
					int			targetSize,
					BufferDocument *	bd,
					const BufferItem *	paraBi,
					const DocumentField *	df );

extern int docCalculateDocStringFieldString(
					int *			pCalculated,
					int *			pNewSize,
					unsigned char *		target,
					int			targetSize,
					BufferDocument *	bd,
					const BufferItem *	paraBi,
					const DocumentField *	df );

extern int docCalculateFilenameFieldString(
					int *			pCalculated,
					int *			pNewSize,
					unsigned char *		target,
					int			targetSize,
					BufferDocument *	bd,
					const BufferItem *	paraBi,
					const DocumentField *	df );

extern int docCalculateRefFieldString(	int *			pCalculated,
					int *			pNewSize,
					unsigned char *		target,
					int			targetSize,
					BufferDocument *	bd,
					const BufferItem *	paraBi,
					const DocumentField *	dfRef );

extern int docRecalculateIncludePictureField(
				int *				pCalculated,
				BufferDocument *		bd,
				int *				pPartShift,
				int *				pStroffShift,
				BufferItem *			bi,
				int				part,
				int				partCount,
				DocumentField *			df,
				void *				voidadd,
				DOC_CLOSE_OBJECT		closeObject );

extern int docParaStringReplace(	int *			pStroffShift,
					BufferItem *		bi,
					int			stroffBegin,
					int			stroffEnd,
					const unsigned char *	addedString,
					int			addedStrlen );

#   endif
