/*
 * solve.c - Addresses family independant resolution functions
 */

/**********************************************************************
 *  Copyright (C) 2002 Rmi Denis-Courmont.                           *
 *  This program is free software; you can redistribute and/or modify *
 *  it under the terms of the GNU General Public License as published *
 *  by the Free Software Foundation; version 2 of the license.        *
 *                                                                    *
 *  This program is distributed in the hope that it will be useful,   *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of    *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.              *
 *  See the GNU General Public License for more details.              *
 *                                                                    *
 *  You should have received a copy of the GNU General Pulic License  *
 *  along with this program; if not, you can get it from:             *
 *  http://www.gnu.org/copyleft/gpl.html                              *
 *                                                                    *
 *  You are highly encouraged to submit your modifications to         *
 *  remi@simphalempin.com for possible inclusion in the official      *
 *  distribution. By doing so, and unless otherwise stated, you give  *
 *  Rmi Denis-Courmont an unlimited, non-exclusive right to modify,  *
 *  reuse and/or relicense the code.                                  *
 **********************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <string.h> /* memset(), strcmp() */
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h> /* needed before sys/socket.h on FreeBSD */
#endif
#ifdef HAVE_SYS_SOCKET_H
# include <sys/socket.h> /* getsockname(), getpeername() */
#endif
#ifdef HAVE_NETDB_H
# include <netdb.h>
#endif

/*
 * Compares 2 socket addresses. Return 0 if they are identical,
 * a positive if they are different, a negative on error. It is assumed
 * that they are of the same size (otherwise, YOU know they are
 * different anyway, don't you?).
 *
 * Only nodes are compared, services are not.
 */
int sockaddrcmp(const struct sockaddr *a1, const struct sockaddr *a2,
	size_t alen)
{
	char n1[NI_MAXHOST], n2[NI_MAXHOST];
	/* Normally, we'd compare addr and res->ai_addr, but there is
	no address family independant way to do this (memcmp() won't
	work in many case (at least Linux/IPv4).

	Instead, we do compare numerical address strings. This requires
	yet another (but fortunately non-blocking) call of getnameinfo.

	It is moreover assumed that service names cannot be spoofed
	(Service Name Service has not been invented, right?).
	*/
	if (a1->sa_family != a2->sa_family)
		return 1;
	if (getnameinfo(a1, alen, n1, sizeof(n1), NULL, 0, NI_NUMERICHOST))
		/* FIXME: when EAI_FAMILY is returned, might use memcmp, or try
		some other family-dependant method (required for AF_UNIX). */
		return -1; /* unlikely - unless family not supported */
	if (getnameinfo(a2, alen, n2, sizeof(n2), NULL, 0, NI_NUMERICHOST))
		/* FIXME: see above. */
		return -1;

	return (strcmp(n1, n2) == 0) ? 0 : 1;
}

/*
 * Secure reverse DNS resolution.
 * NI_NOFQDN (flags option) will fail unless addr is on the same domain
 * as we are (this is absolutely normal). All other flags should work
 * correctly.
 *
 * In case of error, if *servbuf is true, the service name is ok.
 */
int secure_getnameinfo(const struct sockaddr *addr, size_t addrlen,
		char *namebuf, size_t namelen,
		char *servbuf, size_t servlen, int flags)
{
	int check;

	/* Gets service name once and for all */
	if ((check = getnameinfo(addr, addrlen, NULL, 0, servbuf, servlen,
			flags)) != 0) {
		*servbuf = 0; /* specific extension to getnameinfo:
		this lets the calling function know service name could
		not be found (should never happen, except if addr is
		invalid). */
		return check;
	}
	
	
	/* Reverse DNS request */
	if (((check = getnameinfo(addr, addrlen, namebuf, namelen, NULL, 0,
			flags)) != 0) || (flags & NI_NUMERICHOST))
		return check; /* If numeric host name is request, done. */
	else {
		struct addrinfo hints, *res, *info;
	
		/* Hostname DNS request (to prevent malicious users
		from DNS spoofing us). */
		memset(&hints, 0, sizeof(hints));
		hints.ai_family = addr->sa_family;
		if ((check = getaddrinfo(namebuf, NULL, &hints, &res)) != 0)
			return check;

		for (info = res; info != NULL; info = info->ai_next)
			if ((addrlen == info->ai_addrlen)
			 && !sockaddrcmp(addr, info->ai_addr, addrlen)) {
		 		freeaddrinfo(res);
			 	return 0;
			}

		/* DNS spoofing detected: use numeric address only */
		freeaddrinfo(res);
	}
	return getnameinfo(addr, addrlen, namebuf, namelen,
			NULL, 0, flags|NI_NUMERICHOST);
}


/*
 * Stores a full socket address in readable form.
 * Even in case of error, buf is filled with a valid nul-terminated string.
 */
int getsockaddr(int fd, int flags, int peer, char *buf, size_t len)
{
	struct sockaddr_storage addr;
	GETSOCKNAME_ARG3 addrlen = sizeof(addr);
	char nodename[NI_MAXHOST], service[NI_MAXSERV];
	int val;

	memset(&addr, 0, sizeof(addr));
	*service = 0;
	if ((peer) ? getpeername(fd, (GETSOCKNAME_ARG2 *)&addr, &addrlen)
		   : getsockname(fd, (GETSOCKNAME_ARG2 *)&addr, &addrlen))
		perror((peer) ? "getpeername" : "getsockname");
	else
	if ((val = secure_getnameinfo((const struct sockaddr *)&addr, addrlen,
			nodename, sizeof(nodename), service, sizeof(service),
			flags)) != 0)
		fprintf(stderr, _("Hostname lookup failure: %s\n"),
				gai_strerror(val));
	else {
		snprintf(buf, len, "[%s]:%s", nodename, service);
		buf[len - 1] = 0;
		return 0;
	}
	snprintf(buf, len, _("[unknown host]:%s"),
			(*service) ? service : _("unknown_service"));
	buf[len - 1] = 0;
	return -1;
}

/*
 * Extracts hostname/service number. Currently, 3 differents formats
 * are supported:

   [hostname]:port
   [hostname]
   port

 * where hostname is a resolvable host name or a valid host address,
 * and port is a known service name or port number.
 * If hostname is not specified, <nodename> is left untouched;
 * similarly, if <port> is not specified, <service> is not modified.
 *
 * Return 0 on success or true on error (string too long).
 * In this case, nodename and service are undefined.
 */
int parse_host(const char *host, char *nodename, char *service)
{
	if (*host == '[') { /* 1st or 2nd formats */
		const char *ptr;

		host++;
		if ((((ptr = strchr(host, ']')) == NULL))
		 || ((ptr - host) >= NI_MAXHOST))
			return -1;
		memcpy(nodename, host, (ptr - host) * sizeof(char));
		nodename[ptr - host] = 0;

		ptr++;
		if (*ptr) { /* 1st format */
			if (*ptr != ':')
				return -1;
			service[NI_MAXSERV - 1] = 0; /* there may be
				buggy strncpy() that do not do this */
			strncpy(service, ++ptr, NI_MAXSERV);
		}
		else /* 2nd format */
			return 0;

	} else { /* 3rd format */
		if (*host == ':') /* loudly prevent old format */
			return -1;
		service[NI_MAXSERV - 1] = 0; /* there might be
			wrong strncpy() implementations (?) */
		strncpy(service, host, NI_MAXSERV);
		*nodename = 0; /* empty */
	}
	return service[NI_MAXSERV - 1];	/* This is true if, and only if
		service was too small for strncpy(). */
}

