/*
 * socketinfo.c - protocol independant socket handling
 * (on top of getaddrinfo()).
 */

/**********************************************************************
 *  Copyright (C) 2002 Rmi Denis-Courmont.                           *
 *  This program is free software; you can redistribute and/or modify *
 *  it under the terms of the GNU General Public License as published *
 *  by the Free Software Foundation; version 2 of the license.        *
 *                                                                    *
 *  This program is distributed in the hope that it will be useful,   *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of    *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.              *
 *  See the GNU General Public License for more details.              *
 *                                                                    *
 *  You should have received a copy of the GNU General Pulic License  *
 *  along with this program; if not, you can get it from:             *
 *  http://www.gnu.org/copyleft/gpl.html                              *
 *                                                                    *
 *  You are highly encouraged to submit your modifications to         *
 *  remi@simphalempin.com for possible inclusion in the official      *
 *  distribution. By doing so, and unless otherwise stated, you give  *
 *  Rmi Denis-Courmont an unlimited, non-exclusive right to modify,  *
 *  reuse and/or relicense the code.                                  *
 **********************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h> /* fprintf() */
#include <stdlib.h> /* realloc(), free() */
#include <string.h> /* memset() */
#include <limits.h> /* INT_MAX */
#ifdef HAVE_UNISTD_H
# include <sys/time.h>
# include <sys/types.h>
# include <unistd.h> /* select(), fd_set, close() */
#endif
#ifdef HAVE_NETDB_H
# include <netdb.h>
#endif
#if defined(HAVE_SYS_SOCKET_H)
# include <sys/socket.h>
#else
# ifdef _Windows
#  include <winsock.h>
#  define perror( str ) stub_perror(str)
void stub_perror(const char *str);
# endif
#endif
#include "socketinfo.h"

/*
 * Creates a socket, and binds it if required.
 */
int socket_from_addrinfo(const struct addrinfo *info)
{
	int fd;

	if ((fd = socket(info->ai_family, info->ai_socktype,
			 info->ai_protocol)) != -1) {
		const int val = 1;
		setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (SETSOCKOPT_ARG4)&val, sizeof(val));
		
		if ((info->ai_flags & AI_PASSIVE)
		 && bind(fd, info->ai_addr, info->ai_addrlen)) {
			close(fd);
			fd = -1;
		}
	}
	return fd;
}

/*
 * Creates a socket connected to destname/destserv.
 * Return 0 on success, an EAI_* error code on getaddrinfo() failure,
 * -1 on other error (see errno).
 * hints, destname and destserv may be NULL.
 * AI_PASSIVE should not be set in hints->ai_flags (in such case, an
 * error will be returned).
 */
int socket_connect(int *fd, const struct addrinfo *hints,
		   const char *destname, const char *destserv)
{
	int val;
	struct addrinfo *info, *info2;

	if ((val = getaddrinfo(destname, destserv, hints, &info)) != 0)
		return val;

	info2 = info;
	do
		if ((val = socket_from_addrinfo(info2)) != -1)
			if (connect(val, info2->ai_addr, info2->ai_addrlen)) {
				close(val);
				val = -1;
			}
	while ((val == -1) && ((info2 = info2->ai_next) != NULL));
	
	freeaddrinfo(info);
	if (val != -1) {
		*fd = val;
		return 0;
	}
	return -1;
}

/*
 * Creates a table of listening sockets according to <hints>.
 * For correct operation, AI_PASSIVE has to be set in hints->ai_flags,
 * it will not work properly (it won't crash, but it will cause various
 * errors).
 * Returns 0 on success, and puts a pointer to a (-1)-terminated array
 * of socket descriptors. This array is to be freed by fd_freearray().
 * You have to close all descriptors yourself.
 * On I/O error, returns -1.
 * On getaddrinfo error, returns a EAI_* error code suitable for use
 * with gai_strerror() or socket_perror().
 */
static int *fd_arrayadd(int fd, fd_array *fdvptr, int *fdcptr);
int socket_listen(fd_array *arrayptr, const struct addrinfo *hints,
			const char *locname, const char *service)
{
	struct addrinfo *info, *info2;
	int val, fdc;
	fd_array fdv;

	if ((locname == NULL) && (service == NULL))
		service = "0"; /* trick for dynamic port allocation */

	if ((fdv = (int *)malloc(sizeof(int))) == NULL)
		return EAI_SYSTEM; /* not enough memory */
	fdc = 1;
	*fdv = -1;

	if ((val = getaddrinfo(locname, service, hints, &info)) != 0)
		return val;

	for (info2 = info; info2 != NULL; info2 = info2->ai_next)
		if ((val = socket_from_addrinfo(info2)) != -1) {
			if (listen(val, INT_MAX) == 0)
				fd_arrayadd(val, &fdv, &fdc);
			else {
				close(val);
				val = -1;
			}
		}

	freeaddrinfo(info);

	if (fdc != 1) {
		*arrayptr = fdv;
		return 0;
	}
	fd_freearray(fdv);
	return EAI_SYSTEM;
}

void fd_freearray(fd_array array)
{
	if (array != NULL)
		free(array);
}

int fd_closearray(fd_array array)
{
	int *fdp, fd, retval = 0;

	for (fdp = array; (fd = *fdp) != -1; fdp++)
		if (close(fd))
			retval = -1;
	fd_freearray(array);
	return retval;
}


/*
 * Converts an array of file descriptors into a select()-able fd_set.
 * Returns the biggest fd + 1.
 */
int fd_arraytoset(const int *array, fd_set *set)
{
	int fd, max = 0;
	
	FD_ZERO(set);
	while ((fd = (*array)) != -1) {
		FD_SET(fd, set);
		if (fd > max)
			max = fd;
		array++;
	}
	return max + 1;
}

/*
 * Adds a fd in an array of fd.
 * Returns NULL on error (*fdvptr is unchanged in this case).
 */
static int *fd_arrayadd(int fd, fd_array *fdvptr, int *fdcptr)
{
	int c;
	fd_array v;
	v = *fdvptr;
	c = *fdcptr;
	
	if ((v = (fd_array)realloc(v, (++c) * sizeof(int))) == NULL)
		return NULL;

	v[c-1] = -1;
	v[c-2] = fd;

	*fdvptr = v;
	*fdcptr = c;
	return v;
}

/*
 * Finds the first file descriptor in an array that is in a set, remove
 * it from the set and returns it.
 * Returns -1 if none were found.
 *
 * This function could be optimized if depending on how fd_set is
 * defined.
 */
int fd_findfromset(const int *array, fd_set *set)
{
	int fd;

	while ((fd = (*array)) != -1) {
		if (FD_ISSET(fd, set)) {
			FD_CLR(fd, set);
			return fd;
		}
		array++;
	}
	return -1;
}

/*
 * Accepts a connection from a table of socket assumed to all be
 * listening.
 * Returns the newly created socket, or -1 on error.
 */
int accept_array(const fd_array array)
{
	fd_set set;
	int maxfd, fd = -1;

	maxfd = fd_arraytoset(array, &set);

	if (select(maxfd, &set, NULL, NULL, NULL) > 0) {
		const int val = 1;
		fd = accept(fd_findfromset(array, &set), NULL, 0);
		if (fd != -1)
			setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (SETSOCKOPT_ARG4)&val, sizeof(val));
	}
		
	return fd;
}

/*
 * Displays a socket error message.
 */
void socket_perror(int num, const char *nodename, const char *service)
{
	if (num == EAI_SYSTEM) {
		fprintf(stderr, "[%s]:", (nodename != NULL) ? nodename : _("any"));
		perror((service != NULL) ? service : _("any"));
	} else
		fprintf(stderr, "[%s]:%s: %s\n",
			(nodename != NULL) ? nodename : _("any"),
			(service != NULL) ? service : _("any"),
			gai_strerror(num));
}

int sockhostinfo_listen(fd_array *array, const struct sockhostinfo *hostinfo)
{
	struct addrinfo hints;
	const char *nodename, *servicename;

	memset(&hints, 0, sizeof(hints));
	hints.ai_flags = AI_PASSIVE;
	hints.ai_family = hostinfo->family;
	hints.ai_socktype = hostinfo->socktype;
	hints.ai_protocol = hostinfo->protocol;

	if (!*(nodename = hostinfo->hostname))
		nodename = NULL;
	if (!*(servicename = hostinfo->service))
		servicename = NULL;

	return socket_listen(array, &hints, nodename, servicename);
}

int sockhostinfo_connect(int *fd, const struct sockhostinfo *hostinfo)
{
	struct addrinfo hints;
	const char *nodename, *servicename;

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = hostinfo->family;
	hints.ai_socktype = hostinfo->socktype;
	hints.ai_protocol = hostinfo->protocol;

	if (!*(nodename = hostinfo->hostname))
		nodename = NULL;
	if (!*(servicename = hostinfo->service))
		servicename = NULL;

	return socket_connect(fd, &hints, nodename, servicename);
}

void sockhostinfo_perror(int errnum, const struct sockhostinfo *info)
{
	const char *nodename, *servicename;

	if (!*(nodename = info->hostname))
		nodename = NULL;
	if (!*(servicename = info->service))
		servicename = NULL;
	return socket_perror(errnum, nodename, servicename);
}

