/*
 * getaddrinfo.c - replacement functions for getaddrinfo (& co.)
 * This is not thread-safe!!!
 */

/**********************************************************************
 *  Copyright (C) 2002 Rmi Denis-Courmont.                           *
 *  This program is free software; you can redistribute and/or modify *
 *  it under the terms of the GNU General Public License as published *
 *  by the Free Software Foundation; version 2 of the license.        *
 *                                                                    *
 *  This program is distributed in the hope that it will be useful,   *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of    *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.              *
 *  See the GNU General Public License for more details.              *
 *                                                                    *
 *  You should have received a copy of the GNU General Pulic License  *
 *  along with this program; if not, you can get it from:             *
 *  http://www.gnu.org/copyleft/gpl.html                              *
 *                                                                    *
 *  You are highly encouraged to submit your modifications to         *
 *  remi@simphalempin.com for possible inclusion in the official      *
 *  distribution. By doing so, and unless otherwise stated, you give  *
 *  Rmi Denis-Courmont an unlimited, non-exclusive right to modify,  *
 *  reuse and/or relicense the code.                                  *
 **********************************************************************/
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#if USE_GETADDRINFO_STUB
# include "getaddrinfo.h"

# include <string.h> /* strncpy() */
# include <stdio.h> /* snprintf() */
# ifdef HAVE_NETDB_H
#  include <netdb.h>
# endif
# ifdef HAVE_ARPA_INET_H
#  include <arpa/inet.h>
# endif
# ifdef HAVE_NETINET_IN_H
#  include <netinet/in.h>
# endif
# ifdef HAVE_ERRNO_H
#  include <errno.h>
# endif

# if !HAVE_GAI_STRERROR
static const char * __gai_errlist [] = {
	"Error 0",
	"Invalid flag used",
	"Host or service not found",
	"Temporary name service failure",
	"Non-recoverable name service failure",
	"No data for host name",
	"Unsupported address family",
	"Unsupported socket type",
	"Incompatible service for socket type",
	"Unavailable address family for host name",
	"Memory allocation failure",
	"System error"
};
static const char *__gai_unknownerr =
	"Unrecognized error number";
#  define _EAI_POSITIVE_MAX 11

const char *gai_strerror(int errnum)
{
	unsigned int errc;

	errc = (unsigned int)(-errnum);
	if (errc > _EAI_POSITIVE_MAX)
		return __gai_unknownerr;
	return __gai_errlist[errc];
}
#  undef _EAI_POSITIVE_MAX
# endif /* if !HAVE_GAI_STRERROR */

# if !(HAVE_GETNAMEINFO && HAVE_GETADDRINFO)

static int gai_error_from_herrno(void)
{
	switch(h_errno) {
		case HOST_NOT_FOUND:
			return EAI_NONAME;

		case NO_ADDRESS:
#  if (NO_ADDRESS != NO_DATA)
		case NO_DATA:
#  endif
			return EAI_NODATA;

		case NO_RECOVERY:
			return EAI_FAIL;

		case TRY_AGAIN:
			return EAI_AGAIN;

		default:
			return EAI_SYSTEM;
	}
}
# endif /* if !(HAVE_GETNAMEINFO && HAVE_GETADDRINFO) */

# if !HAVE_GETNAMEINFO
int getnameinfo (const struct sockaddr *sa, int salen,
		char *host, int hostlen,
		char *serv, int servlen, int flags)
{
	if (sa->sa_family != AF_INET)
		return EAI_FAMILY;
	else if (flags & ~_NI_MASK)
		return EAI_BADFLAGS;
	else {
		const struct sockaddr_in *addr;
		
		addr = (const struct sockaddr_in *)sa;

		if (host != NULL) {
			const char *hostname;
			int solved = 0;
			
			/* host name resolution */
			if (!(flags & NI_NUMERICHOST)) {
				struct hostent *hent;

				if ((hent = gethostbyaddr(
						(const void*)&addr->sin_addr,
						4, AF_INET)) != NULL) {
					hostname = hent->h_name;
					solved = 1;
				} else if (flags & NI_NAMEREQD)
					return gai_error_from_herrno();
			}
			if (!solved)
				hostname = inet_ntoa(addr->sin_addr);

			/* host name copy */
			strncpy(host, hostname, hostlen);
			host[hostlen - 1] = 0;
			
			if ((flags & NI_NOFQDN) && solved) {
				char *ptr;

				if ((ptr = strchr(host, '.')) != NULL)
					*ptr = 0;
			}
		}

		if (serv != NULL) {
			const char *servname = NULL;
	
			/* service name resolution */
			if (!(flags & NI_NUMERICSERV)) {
				struct servent *sent;

				if ((sent = getservbyport(addr->sin_port,
						(flags & NI_DGRAM) ? "udp"
						: "tcp")) != NULL)
					servname = sent->s_name;
			}
			/* service name copy */
			if (servname == NULL)
				snprintf(serv, servlen, "%u",
					(unsigned int)ntohs(addr->sin_port));
			else
				strncpy(serv, servname, servlen);
			serv[servlen - 1] = 0;
		}
	}
	return 0;
}
# endif /* if !HAVE_GETNAMEINFO */


# if !HAVE_GETADDRINFO
#  error Unimplemented
#  if 0
int getaddrinfo(const char *node, const char *service,
		const struct addrinfo *hints, struct addrinfo **res)
{
	int protocol = 0, socktype = 0;
	
	if (hints != NULL) {
		if (hints->ai_flags & ~_AI_MASK)
			return EAI_BADFLAGS;
		if (hints->ai_family != PF_INET)
			return EAI_FAMILY;
		switch (hints->ai_socktype) {
			case SOCK_STREAM:
				protocol = IPPROTO_TCP;
				break;

			case SOCK_DGRAM:
				protocol = IPPROTO_UDP;
				break;

			case SOCK_RAW:
			case 0:
				break;

			default:
				return EAI_SOCKTYPE;
		}
		if ((hints->ai_protocol) && (protocol)
		 && (protocol != hints->ai_protocol))
			return EAI_SERVICE;
	} else {
		struct hostent *host;
		struct servent *serv;
		struct sockaddr_in *addr;
		struct addrinfo *result;
		
		result = (struct addrinfo *)malloc(sizeof(struct addrinfo));
		if (result == NULL)
			return EAI_SYSTEM;
		addr = (struct sockaddr_in *)malloc(sizeof(struct sockaddr_in));
		if (addr == NULL) {
			int saved_errno;
			
			saved_errno = errno;
			free(result);
			errno = saved_errno;
			return EAI_SYSTEM
		}

		memset(result, 0, sizeof(struct addrinfo));
		memset(addr, 0, sizeof(struct sockaddr_in));
		result->ai_family = PF_INET;
		result->ai_socktype = /*FIXME*/;
	}
}
#  endif
# endif /* if !HAVE_GETADDRINFO */
#endif /* if USE_GETADDRINFO_STUB */

