/***************************************************************************
                          taxipilot.cpp  -  description
                             -------------------
    begin                : Don Mai 17 20:03:43 CEST 2001
    copyright            : (C) 2001, 2002 by Thomas Friedrichsmeier
    email                : Thomas.Friedrichsmeier@ruhr-uni-bochum.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include "defines.h"

#include <klocale.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include KAPP

#include <qlayout.h>
#include <qdom.h>
#include <qrect.h>
#include <qcolor.h>
#include <qfont.h>
//#include <qpalette.h>

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <time.h>

#include "taxipilot.h"

#include "taxi.h"
#include "level.h"
#include "passenger.h"
#include "cdp.h"
#include "taxipilotapp.h"
#include "movingobject.h"
#include "teleporter.h"
#include "centerofgravity.h"
#include "platform.h"
#include "message.h"
#include "timemanager.h"
#include "hotarea.h"

Taxipilot::Taxipilot(TaxipilotApp * parent, const char *name):QWidget(parent, name)
{
	app = parent;

	current_key_lr = current_key_ud = 0;
	toggle_flaps = false;
	game_running = false;
	start_screen_active = false;
	passengers_created = false;

	base_timer = passenger_timer = scroll_timer = credits_timer = canvas_timer = -1;

	srand((unsigned int) time((time_t *) NULL));	// randomize

//      QVBoxLayout *mainvbox = new QVBoxLayout ( this ); // create Layout
	QVBoxLayout *mainvbox = new QVBoxLayout;
	QHBoxLayout *centralhbox = new QHBoxLayout(this);
	QVBoxLayout *rightvbox = new QVBoxLayout;
	QHBoxLayout *bottomhbox = new QHBoxLayout;

	canvas = new QCanvas;

	view = new QCanvasView(canvas, this);
	view->setHScrollBarMode(QScrollView::AlwaysOff);
	view->setVScrollBarMode(QScrollView::AlwaysOff);

	fader = new QCanvasRectangle(canvas);
	fader->setPen(NoPen);
	fader->setBrush(QBrush::QBrush(Dense4Pattern));
	fader->setZ(FADER_Z);		// second topmost

//      centralhbox->addWidget ( view );
	mainvbox->addWidget(view);

	QFont labelFont(LABEL_FONT, LABEL_FONT_SIZE);

// One day I'll have to find out about colors:
//  QColorGroup grp( darkGreen, black, QColor( 128, 128, 128 ), QColor( 64, 64, 64 ), black, darkGreen, black );
//  QPalette pal( grp, grp, grp );

	QLabel *label;
	label = new QLabel(i18n("Ships"), this);
	label->setFont(labelFont);
	rightvbox->addWidget(label);

	shipsLCD = new QLCDNumber(2, this);
	rightvbox->addWidget(shipsLCD);

	label = new QLabel(i18n("Funds"), this);
	label->setFont(labelFont);
	rightvbox->addWidget(label);

	scoreLCD = new QLCDNumber(6, this);
	rightvbox->addWidget(scoreLCD);

	label = new QLabel(i18n("Fuel"), this);
	label->setFont(labelFont);
	rightvbox->addWidget(label);

	fuelDial = new EQDial(0, 100, 20, 1, this);
	fuelDial->setNotchTarget(10);
	fuelDial->setLimits(10, -1);
	fuelDial->setNotchesVisible(true);
	fuelDial->setTracking(false);
	fuelDial->setFocusPolicy(QWidget::NoFocus);
	rightvbox->addWidget(fuelDial);

	label = new QLabel(i18n("Speed"), this);
	label->setFont(labelFont);
	rightvbox->addWidget(label);

	speedDial = new EQDial(0, 100, 20, 30, this);
	speedDial->setNotchesVisible(true);
	speedDial->setTracking(false);
	speedDial->setFocusPolicy(QWidget::NoFocus);
	rightvbox->addWidget(speedDial);

	label = new QLabel(i18n("Fare"), this);
	label->setFont(labelFont);
	rightvbox->addWidget(label);

	fareLCD = new QLCDNumber(6, this);
	rightvbox->addWidget(fareLCD);

	label = new QLabel(i18n("Tip"), this);
	label->setFont(labelFont);
	rightvbox->addWidget(label);

	tipLCD = new QLCDNumber(6, this);
	rightvbox->addWidget(tipLCD);

//      centralhbox->addLayout ( rightvbox );
//      mainvbox->addLayout ( centralhbox );

	label = new QLabel(i18n("Status:"), this);
	label->setFont(labelFont);
	bottomhbox->addWidget(label);

	statusLabel = new QLabel(i18n("No Game running"), this);
	statusLabel->setFont(labelFont);
	statusLabel->setAlignment(AlignRight);
	bottomhbox->addWidget(statusLabel);

	bottomhbox->addSpacing(20);

	label = new QLabel(i18n("Destination:"), this);
	label->setFont(labelFont);
	bottomhbox->addWidget(label);

	destinationLabel = new QLabel(i18n("Nowhere"), this);
	destinationLabel->setFont(labelFont);
	destinationLabel->setAlignment(AlignRight);
	bottomhbox->addWidget(destinationLabel);

	mainvbox->addLayout(bottomhbox);

	rightvbox->addSpacing(destinationLabel->height());

	centralhbox->addLayout(mainvbox);
	centralhbox->addLayout(rightvbox);

	cdp = new Cdp(this);

	tm = new TimeManager (50, 10, cdp);
	init_timer = tm->startTimer (this, 10);		// this timer will get active after a.exec () in main and then fire initialization.
	pause_tm = new TimeManager (10, 10, cdp);
	pause_tm->pause ();
	message_handler = new Message (this, view); // needs to know about cdp

	passenger_exists = false;
	config_changed ();		// this updates what we store about the config
}

Taxipilot::~Taxipilot()
{
	if (passengers_created) {
		for (int i=0; i < num_passenger_files; i++) {
			delete all_passengers[i];
		}
		delete [] all_passengers;
	}

	delete[]level_files;
	delete[]passenger_files;
	delete fader;
	delete message_handler;

	if (start_screen_active) {
		delete_start_screen ();
	}

	if (game_running) {
		delete taxi;
		delete level;
	}

	delete tm;
	delete pause_tm;
	delete cdp;

	delete view;
	delete canvas;
}

/** This function really would be part of the constructor. It initializes Taxipilot by reading
the default mission, displaying the start-screen, and setting a few variables.
The thing is though, that it may also set the KAction gameNew to enabled or
disabled, which rests on this Action having been created in TaxiPIlotApp, which
in turn depends on having an instance of Taxipilot.
If you can think of something more elegant, please don't hesitate to tell me. */
void Taxipilot::init()
{
	num_passengers_on_start_screen = num_passenger_files = num_level_files = 0;

	setFocusPolicy(QWidget::StrongFocus);	// get Focus
	setFocus();

	cdp->init_debug (i18n ("No file opened yet"), false);	
	readMission(cdp->prefix_check(cdp->startupMission (), false));

	game_running = false;
	std_resumable = true;

	if (cdp->ok()) {
		display_start_screen();
		app->gameNew->setEnabled(true);
	} else {
		app->gameNew->setEnabled(false);
	}

}

bool Taxipilot::onActivePlatform()
{
	return (taxi->landed && (on_platform == active_platform));
}


/* This function is to handle virtually everything of significance */
void Taxipilot::timerTick(int id)
{						

	if (game_running) {	// do nothing if paused.
		if (id == scroll_timer) {
			do_scroll();
		} else if (id == base_timer) {	// base timer

			// whatever may happen, do the chores ( gravity only if not landed )
			do_chores();

			int might_be_landed = 0;		// number of potential platforms, taxi is in contact with
									// of course, > 1 means crashed

			// check for moving objects here
			for (int i = 0; i < level->num_objects; i++) {
				if (level->objects[i]->harmful()) {
					if (taxi->collidesWith(level->objects[i])) {
						if (level->objects[i]->platform) {
							might_be_landed++;	
						} else {
							crash(CRASH_BACKGR);
							return;
						}
					} else {
						if (taxi->flaps_avail && taxi->flaps_state) {
							if (taxi->flaps->collidesWith(level->objects[i])) {
								if (level->objects[i]->platform) {							
									might_be_landed++;
								} else {
									crash(CRASH_BACKGR);
									return;
								}
							}
						}
					}
				}			
			}

			// check for hot-areas here
			QRect dummy = taxi->boundingRect ();			
			for (int i=0; i < level->num_areas; i++) {
				level->areas[i]->check (dummy);
			}

			if (taxi->ready) {	// means: not teleporting already

				for (int i = 0; i < level->num_teleporters; i++) {
					if (level->teleporters[i]->check_teleportation (taxi, &score)) {
						scoreLCD->display(string.setNum(score, 'f', 2));
						i = level->num_teleporters; // skip further checks
					}
				}

			}

			if (!taxi->landed) {
				if (passenger_exists && (!passenger_onboard)) {	// checking when passenger undefined crashes the program!
					if (taxi->collidesWith(passenger)) {
						crash(CRASH_PASSENGER);
						return;	// skip everything else, it might crash if taxi got deleted
					}
					if (taxi->flaps_state && taxi->flaps_avail) {
						if (taxi->flaps->collidesWith(passenger)) {
							crash(CRASH_PASSENGER);
							return;
						}
					}
					if (current_key_lr) {	// only check for collision with flying anim, when accelerating
						if (taxi->flying_lr->collidesWith(passenger)) {
							crash(CRASH_PASSENGER);
							return;	// skip everything else, it might crash if taxi got deleted
						}
					}
					if (current_key_ud) {
						if (taxi->flying_ud->collidesWith(passenger)) {
							crash(CRASH_PASSENGER);
							return;	// skip everything else, it might crash if taxi got deleted
						}
					}
				}
			}

			if (!taxi->landed) {
				if (taxi->collidesWith(level)) {
					might_be_landed++;	
				} else if (taxi->flaps_avail) {
					if (taxi->flaps_state && taxi->flaps->collidesWith(level)) {
						might_be_landed++;
					}
				}
			}

			if (might_be_landed) {
				if (might_be_landed > 1) {		// in contact with more than one potential platform? -> crash!
					crash(CRASH_BACKGR);
					return;		// omit gravity
				} else {
					if (taxi->flaps_state == 2) {	// otherwise this is definitely a crash
						if ((on_platform = check_landed()) >= 0) {
							landing_func();		// do platform stuff
						} else {
							crash(CRASH_BACKGR);
							return;
						}
					} else {
						crash(CRASH_BACKGR);
						return;
        				}					
				}
			}

			if (!taxi->landed) {
				// nothing interesting happend (and we are flying)
				do_gravity();
			} else {			// hey, we're still landed. anything special to do?
				if (fueling) {
					int max_affordable;
					if (level->platform[on_platform]->fuel_price != 0) {
						max_affordable = (int) (score / level->platform[on_platform]->fuel_price);
					} else {
						max_affordable = level->platform[on_platform]->fuel_step;
					}
					if ((max_affordable <= 0) && (taxi->fuel_out)) {	// we're on the fuel platform, but can't afford any, but need it.
						display_message(LOW_FUEL);
					}
					if (level->platform[on_platform]->fuel_step < max_affordable) {
						score -=
							level->platform[on_platform]->fuel_price *
							taxi->fuel_up(level->platform[on_platform]->fuel_step);
					} else {
						score -= level->platform[on_platform]->fuel_price * taxi->fuel_up(max_affordable);
					}
					fuelDial->setValue(taxi->fuel_state());
					scoreLCD->display(string.setNum(score, 'f', 2));
				}
			}

		} else if (id == passenger_timer) {	// create new passenger

			int rand_num;
			do {
				rand_num = rand() % (level->num_platforms);
			} while ((!level->platform[rand_num]->accepts_passenger));
			active_platform = rand_num;

			passenger = all_passengers[(rand() % num_passenger_files)];

			passenger->initialize (level->platform[active_platform]->current_plat_base_x (), level->platform[active_platform]->current_plat_y (), level->platform[active_platform]);
			passenger_exists = true;

			tm->killTimer(passenger_timer);
			passenger_timer = -1;

			// everyting else is taken care of by passenger_ready ()

		} else if (id == canvas_timer) {	// advance taxi and update canvas
			taxi->advance (1);
			canvas->update ();
		}
	} else {					// stuff to do only when game is paused

		 if (id == credits_timer) {
			Credit_item *dummy;
			int store_current = credits.at();
			if (!credits.current()->item->visible()) {	// set off a new item
				dummy = credits.at(credits.at());
				dummy->item->move(view->width(), dummy->y);
				dummy->item->setAnimated(true);
				dummy->item->show();
				last_visible_credit++;
			}
			dummy = credits.at(first_visible_credit);
			if (dummy->item->x() < (-dummy->width)) {	// delete an item out of sight
				dummy->item->setAnimated(false);
				dummy->item->hide();
				first_visible_credit++;
				if (first_visible_credit >= ((int) credits.count())) {	// full cycle
					first_visible_credit = 0;
					last_visible_credit = -1;
					store_current = 0;
				}
			}
			if ((last_visible_credit < ((int) credits.count())) && (last_visible_credit >= 0)) {
				dummy = credits.at(last_visible_credit);
				if (dummy->item->x() < (view->width() - (dummy->width + dummy->space_to_next))) {	// trigger next item
					store_current++;
				}
			}
			if (store_current < ((int) credits.count())) {
				credits.at(store_current);
			} else {
				credits.at(credits.count() - 1);
			}

		} else if (id == canvas_timer) {	// advance credits and update canvas
			canvas->advance ();
		} else if (id == init_timer) {
			tm->killTimer (init_timer);
			init_timer = -1;
			init ();
		}

	}
}

void Taxipilot::do_chores()
{								// meter-updates... ( speed taken care of in do_gravity () )
	if (passenger_waiting || passenger_onboard) {	// run tip as soon as a passenger is waiting
		tipLCD->display(string.setNum((tip -= tip_step), 'f', 2));
		if (passenger_onboard) {	// run fare as soon as passenger has boarded
			fareLCD->display(string.setNum(fare, 'f', 2));
		}
	}
	taxi->stop_flying_animation();

/* Begin Keyboard-Input-Handling-Section
	This has to be after taxi->stop_flying_animation (), so the flying_animation remains visible. */
	if (current_key_lr || current_key_ud) {

		if (passenger_onboard) {
			fare += level->fare_step + level->fare_step;
		}

		switch (current_key_lr) {	// left or right
		case -1:
			taxi->left(-ACCELERATION * acceleration);
			break;
		case 1:
			taxi->right(+ACCELERATION * acceleration);
			break;
		default:{
				if (passenger_onboard) {
					fare -= level->fare_step;
				}
			}
		}

		switch (current_key_ud) {	// up or down
		case -1:
			taxi->up(-ACCELERATION * acceleration);
			break;
		case 1:
			taxi->down(+ACCELERATION * acceleration);
			break;
		default:{
				if (passenger_onboard) {
					fare -= level->fare_step;
				}
			}
		}

		fuelDial->setValue(taxi->fuel_state());
	}

	if (toggle_flaps) {
		taxi->toggle_flaps();
		toggle_flaps = false;
	}
/* End Keyboard-Input-Handling-Section */
}

/** Check, whether we landed on a platform and on which one */
int Taxipilot::check_landed()
{
	for (int i = 0; i < level->num_platforms; i++) {
		if (level->platform[i]->check_landed (taxi)) {
			return (i);
		}
	}
	return (-1);
}

/** Passenger has finished boarding */
void Taxipilot::picked_up()
{
	passenger_onboard = true;
	passenger_waiting = false;

	int rand_num;
	do {
		rand_num = rand() % (level->num_platforms);
	} while ((rand_num == active_platform) || (!level->platform[rand_num]->accepts_passenger));	// generate different platform
	active_platform = rand_num;

	destinationLabel->setText(level->platform[active_platform]->label);
	statusLabel->setText(i18n("Delivery"));

	fare = level->base_fare;
}

/** No descriptions */
void Taxipilot::passenger_abandoned()
{
	// complain!!!
}

/** Last passenger has now disappeared, a new one may be generated */
void Taxipilot::delivery_complete()
{

	fareLCD->display(string.setNum((fare = 0), 'f', 2));	// update display
	tipLCD->display(string.setNum((tip = 0), 'f', 2));

	passenger->discard ();
	passenger_exists = false;
	if (level->deliveries_to_go) {
		passenger_timer = tm->startTimer(this, PASSENGER_TP);	// to be randomized
	} else {					// this level is complete!

		int message_id = NEXT_LEVEL;

		if (level->extra_life) {
			message_id += EXTRA_LIFE;
			shipsLCD->display(++ships);
		}
		if (++current_level == num_level_files) {
			message_id += MISSION_COMPLETE;	// abort will be done by do_event_after_message ()
		}

		display_message(message_id);
	}
}

	/* passenger is now ready for pick-up */
void Taxipilot::passenger_ready()
{
	destinationLabel->setText(level->platform[active_platform]->label);
	statusLabel->setText(i18n("Pick up"));
	tip = level->min_base_tip + (rand() * (level->max_base_tip - level->min_base_tip) / (RAND_MAX + 1.0));
	tip_step = 0.2 / (double) cdp->cps () * passenger->impatience;

	passenger_waiting = true;
	if (onActivePlatform()) {	// in case we are already there
		passenger->pick_up (taxi->LEFTEDGE() + (int) (taxi->width() / 2));
	}
}

  /** Computes the gravity acting on the taxi and sets it. Also updates the speedDial */
void Taxipilot::do_gravity()
{

	if (taxi->ready) {
		double xv = taxi->xVelocity() + level->horiz_gravity;	// do the basic gravity right in place;
		double yv = taxi->yVelocity() + level->base_gravity;

		int taxi_pos_x = taxi->LEFTEDGE() + (taxi->width() / 2);
		int taxi_pos_y = taxi->TOPEDGE() + (taxi->height() / 2);

		for (int i = 0; i < level->num_grav_centers; i++) {	// do the advanced gravity
			level->grav_center[i]->calculate_gravity (taxi_pos_x, taxi_pos_y, &xv, &yv);
		}
/*
// make velocity sane (if taxi comes very close to centers of gravity, velocity may increase dramatically to speed in no
// sensible proportion to the frame rate)

		if (xv > 3) {
			xv = 3;
		} else if (xv < -3) {
			xv = -3;
		}
		if (yv > 3) {
			yv = 3;
		} else if (yv < -3) {
			yv = -3;
		} */
		
		taxi->reposition(-1, -1, xv, yv);	// this takes care of flaps as well.

		speedDial->setValue(16 * (taxi->speed = (sqrt(xv * xv + yv * yv))));
	}
}

void Taxipilot::landing_func()
{								// this function is responsible for everything involved when landed

	// first of all: stop
	speedDial->setValue(taxi->speed = 0);
	taxi->reposition(-1, level->platform[on_platform]->current_plat_y (), 0, 0, true);
	taxi->land (level->platform[on_platform]);

	// now let's see, if there's any fuel here
	if (level->platform[on_platform]->fuel_step > 0) {
		fueling = true;			// timerEvent will take care of the actual fueling
	} else {
		fueling = false;		// fueling is only looked at while landed, so it's good enough to set it when landing
		if (taxi->fuel_out) {
			display_message(LOW_FUEL);
		}
	}

	// finally let's see, whether we are on the active platform
	if (on_platform == active_platform) {	// we landed on a meaningful platform

		if (passenger_waiting) {

			passenger->pick_up (taxi->LEFTEDGE() + (int) (taxi->width() / 2));	// you may now come aboard!
		} else if (passenger_onboard) {

			passenger->got_delivered ((taxi->LEFTEDGE() + (int) (taxi->width() / 2)), level->platform[active_platform]); 	// there you go!
			passenger_onboard = false;
			passenger_waiting = false;

			if ((tip + fare) >= 0) {	// we won't be substracting money
				scoreLCD->display(string.setNum((score += (tip + fare)), 'f', 2));
				if (tip >= 0) {
					level->deliveries_to_go--;	// one successful delivery
				}
			}					// TODO: else complain about being too slow

			active_platform = -1;	// delete active platform

			destinationLabel->setText(i18n("Nowhere"));	// update display
			statusLabel->setText(i18n("Idle"));
		}
	}
}

void Taxipilot::crash(int how)
{								// handle the crash             

	int message_id = CRASH;

	if (--ships < 0) {
		message_id += GAME_OVER;
	}

	taxi->crash ();

	display_message(message_id + how);

}

/** No descriptions */
void Taxipilot::newGame()
{

	delete_start_screen();
	cdp->set_for_real (true);

	app->gameNew->setEnabled(false);
	app->gameAbort->setEnabled(true);

	if (!passengers_created) {                         // create all passengers
		all_passengers = new Passenger * [num_passenger_files];
		for (int i=0; i < num_passenger_files; i++) {
			all_passengers[i] = new Passenger(passenger_files[i], canvas, this);
		}
		passengers_created = true;
	}

	level = new Level(level_files[(current_level = 0)], canvas, this);

	view->setFixedSize(level->visible_width + 3, level->visible_height + 3);	// this has to happen after initialization of level, for this is where the canvas is initialized!
	// I do wonder, why I need this +3, though
	view->setContentsPos(current_offset_x = level->initial_offset_x, current_offset_y = level->initial_offset_y);

	fader->setSize(canvas->width(), canvas->height());

	taxi = new Taxi(taxi_file, canvas, level->initial_x, level->initial_y, this);

	fuelDial->setValue(taxi->fuel_state());
	scoreLCD->display(score = 0);
	shipsLCD->display(ships = initial_ships);

	//      app->setCaption (level->caption);

// initialize timers and variables

	base_timer = tm->startTimer(this, 1000 /  cdp->cps ());
	scroll_timer = tm->startTimer(this, 1000 / cdp->fps ());

	passenger_waiting = false;
	passenger_onboard = false;
	passenger_exists = false;
	fueling = false;

	passenger_timer = tm->startTimer(this, PASSENGER_TP);	// randomize ?

	game_running = true;

	lastStatus = (i18n("Idle"));
	resumeGame();				// this will set the AdvancePeriod.
}

/** Guess what! */
void Taxipilot::quitGame()
{
	pauseGame();
	if (3 == KMessageBox::questionYesNo(this, i18n("Really wanna quit?"), "", i18n("'course"), i18n("Never!"))) {
		kapp->quit ();
	}
	resumeGame();
}

/** Pauses the game. */
void Taxipilot::pauseGame()
{
	if (game_running) {
		app->gamePause->setEnabled(false);
                if (std_resumable) {
			display_message (PAUSE);
		}
		lastStatus = statusLabel->text();
		statusLabel->setText(i18n("Game paused"));
		fader->show();
		canvas->update();
	}
	tm->pause ();
	pause_tm->resume();
}

/** Resumes a paused game */
void Taxipilot::resumeGame()
{
	app->gamePause->setEnabled(true);
	canvas_timer = tm->setLastTimer (this, 1000 / cdp->fps ());
	statusLabel->setText(lastStatus);
	fader->hide();
	tm->resume ();
	pause_tm->pause();
	std_resumable = true;
}

/** Selects a new Mission (i.e. a list of level- and
passenger-description-files) */
void Taxipilot::selectMission()
{

	if (game_running) {
		ask_abortGame();
	}
	if (!game_running) {		// this is NOT an else! If game is still running now, user cancelled:
		delete_start_screen();

		QString file = KFileDialog::getOpenFileName(cdp->data_prefix, "*.mission", this, i18n("Select Mission"));

		if (!file.isNull()) {	// if no file selected, don't try to read mission!
			readMission(file);
		}

		if (cdp->ok()) {
			display_start_screen();
			pauseGame ();
			if (cdp->ok()) {
				resumeGame();
				cdp->setLastLoaded (file);
				app->gameNew->setEnabled(true);
			} else {
				app->gameNew->setEnabled(false);
				delete_start_screen();
			}
		} else {
			app->gameNew->setEnabled(false);
		}
	}
}

/** Reads in a mission-description file. */
void Taxipilot::readMission(const QString & file)
{

	QDomElement e;				// just some element
	QDomNode n;					// just some node
	QDomNodeList list;			// just a list of nodes

	cdp->init_debug(file, false);

	QDomElement xdocElement = cdp->open_config_file(file);
	mission_author = cdp->get_string_attribute("author", xdocElement, i18n("Anonymous"), 3);
	mission_title = cdp->get_string_attribute("title", xdocElement, i18n("Untitled"), 3);

	if (!cdp->isOk())
		return;					// Errors at this point ARE fatal!

	startscreen_file =
		cdp->prefix_check(cdp->get_string_attribute("startscreen", xdocElement, "default_start_screen.jpg", 1), true);

	e = cdp->get_element("initials", xdocElement, 1);	// find the initials-section
	shipsLCD->display((ships = initial_ships = cdp->get_int_attribute("ships", e, 1, 99, 3, 1)));

	e = cdp->get_element("levels", xdocElement, 0);	// find the levels-section
	list = cdp->get_node_list("data", e, 1, -1, 1, 0);	// get all level files

	n = list.item(0);
	
	if (num_level_files) {
		delete [] level_files;
	}
	level_files = new QString[list.length()];

	num_level_files = 0;
	while (!n.isNull()) {
		e = n.toElement();

		level_files[num_level_files] =
			cdp->prefix_check(cdp->get_string_attribute("file", e, "level1.level", 1), false);

		n = n.nextSibling();
		num_level_files++;
	}

	if (passengers_created) {						// in case previous passengers still exist
		for (int i=0; i < num_passenger_files; i++) {
			delete all_passengers[i];
		}
		delete [] all_passengers;
		passengers_created = false;
	}
	
	e = cdp->get_element("passengers", xdocElement, 0);	// find the passengers-section
	list = cdp->get_node_list("data", e, 1, -1, 1, 0);	// get all passenger files

	n = list.item(0);

	if (num_passenger_files) {
		delete [] passenger_files;
	}
	passenger_files = new QString[list.length()];
	if (num_passengers_on_start_screen) {
		delete [] passengers_on_start_screen;
	}
	passengers_on_start_screen = new Passengers_on_Startscreen[list.length()];	// this may allocate a few spare ones, but we really don't have to worry about that

	num_passenger_files = num_passengers_on_start_screen = 0;
	while (!n.isNull()) {
		e = n.toElement();

		passenger_files[num_passenger_files] =
			cdp->prefix_check(cdp->get_string_attribute("file", e, "passenger1.passenger", 1), false);

		if (cdp->get_bool_attribute("on_start_screen", e, false, 3)) {
			passengers_on_start_screen[num_passengers_on_start_screen].id = num_passenger_files;
			passengers_on_start_screen[num_passengers_on_start_screen].x =
				cdp->get_int_attribute("start_screen_x", e, -100, 3000, 200, 1);
			passengers_on_start_screen[num_passengers_on_start_screen].y =
				cdp->get_int_attribute("start_screen_y", e, -100, 3000, 200, 1);
			num_passengers_on_start_screen++;
		}

		n = n.nextSibling();
		num_passenger_files++;
	}

	e = cdp->get_element("taxi", xdocElement, 1);	// find the taxi-section
	taxi_file = cdp->prefix_check(cdp->get_string_attribute("file", e, "taxi1.taxi", 1), false);

	if (taxi_on_start_screen = cdp->get_bool_attribute("on_start_screen", e, false, 3)) {
		taxi_start_screen_x = cdp->get_int_attribute("start_screen_x", e, -100, 3000, 200, 1);
		taxi_start_screen_y = cdp->get_int_attribute("start_screen_y", e, -100, 3000, 200, 1);
	}

}

/** Asks whether game should really be aborted, then calls abortGame () */
void Taxipilot::ask_abortGame()
{

	pauseGame();
	if (3 != KMessageBox::questionYesNo(this, i18n("This will quit the current game!\nAre you sure?"))) {
//		resumeGame();
		return;
	}
	abortGame();
}

/** This function aborts the current game. Usually called by ask_abortGame, which
first asks, then calls this function */
void Taxipilot::abortGame()
{

	pauseGame();				// Usually we've already done this in ask_abortGame. It should'nt harm, though

	message_handler->kill ();
	game_running = false;
	tm->killTimers(this);

	if (passenger_exists) {
		passenger->discard();
	}

	delete level;
	delete taxi;

	app->gamePause->setEnabled(false);
	app->gameAbort->setEnabled(false);
	app->gameNew->setEnabled(true);

//	tm->killLastTimer ();
//	canvas->setAdvancePeriod(0);

// in case there is still a message 
	fader->hide();
	canvas->update();

	base_timer = -1;
	passenger_timer = -1;
	scroll_timer = -1;
	credits_timer = -1;
	canvas_timer = -1;

	statusLabel->setText(i18n("No Game running"));

	display_start_screen();
}

/** Generates and displays the current mission's startscreen */
void Taxipilot::display_start_screen()
{
	cdp->set_for_real (false);

	if (start_screen_active)
		return;
	start_screen_active = true;

	start_screen = new QCanvasPixmap(startscreen_file);

	canvas->setBackgroundPixmap(*start_screen);
	canvas->resize(start_screen->width(), start_screen->height());

	view->setFixedSize(canvas->width() + 3, canvas->height() + 3);

	for (int i = 0; i < num_passengers_on_start_screen; i++) {
		passengers_on_start_screen[i].pointer =	new Passenger(passenger_files[passengers_on_start_screen[i].id], canvas, this);
		passengers_on_start_screen[i].pointer->initialize (passengers_on_start_screen[i].x, passengers_on_start_screen[i].y, 0);
	}

	if (taxi_on_start_screen) {
		taxi = new Taxi(taxi_file, canvas, taxi_start_screen_x, taxi_start_screen_y, this);
	}

	resumeGame ();		// restart timers

	app->setCaption(mission_title);

	generate_credits();

}

/** Delete (the passengers on) the Startscreen */
void Taxipilot::delete_start_screen()
{
	if (!start_screen_active)
		return;
	start_screen_active = false;

	for (int i = 0; i < num_passengers_on_start_screen; i++) {
		delete passengers_on_start_screen[i].pointer;
	}

	if (taxi_on_start_screen) {
		delete taxi;
	}

	delete_credits();

	canvas->setBackgroundColor (QColor (0, 0, 0));
	delete start_screen;
}

/** Displays a message as QCanvasText on the Canvas (scrolling it into view). */
void Taxipilot::display_message(int message_id)
{

	std_resumable = false;
	QString message = "";

	if (message_id != PAUSE) pauseGame();
	else {
		message = i18n ("Game paused\nto continue:");
	}

	event_after_message = message_id;

	if (message_id & CRASH) {
		if (message_id & CRASH_PASSENGER) {
			message = i18n("You just killed an innocent passenger!\nAre you mad?!?");
		} else if (message_id & CRASH_BACKGR) {
			message = i18n("You just crashed your taxi.\nThat's gonna be expensive");
		}
		if (message_id & GAME_OVER) {
			message += i18n("\n\nYou crashed so many times now...\n...go find another job!\nGame Over!");
		} else if (passenger_onboard) {
			message += i18n("\n\nYour passenger decided to walk the rest of the way.");
		}
		if (taxi->fuel_out) {
			message += i18n("\nBetter watch your fuel next time.");
		}
	}

	if (message_id & NEXT_LEVEL) {
		message = i18n("Congratulations on finishing this level!\n");
		if (message_id & EXTRA_LIFE) {
			message += (i18n("Additionally you are awarded a new ship.\n"));
		}
		if (message_id & MISSION_COMPLETE) {
			message += (i18n("\nYou completed your mission.\nGo do something useful now."));
		}
	}
	// Don't display fuel-message if game over, or you completed a level.
	if (!(message_id & (NEXT_LEVEL | GAME_OVER)) && taxi->fuel_low) {
		message += i18n("\nSince you ran low on fuel, the people at '") +
			level->platform[level->exp_fuel_platform]->label +
			i18n("'\n generously offer you some fuel at only twice the regular price.");
		event_after_message |= LOW_FUEL;
	}

	message_handler->display_message (message, app->ack_key_name);

}

/** I really hate doing this, but I need to have a function that is called after a text
message was displayed, which will do, whatever needs to be done next.
Unfortunately, what needs to be done may vary. */
void Taxipilot::do_event_after_message()
{

	current_key_lr = current_key_ud = 0;	// in case we missed a key-release somewhere

	if (event_after_message & LOW_FUEL) {
		QString dummy;
		scoreLCD->display(dummy.setNum((score -= taxi->fuel_up(FUEL_UP_EMPTY_BY * cdp->cps () / 20) * level->platform[level->exp_fuel_platform]->fuel_price * 2), 'f', 2));	// don't check whether affordable
		fuelDial->setValue(taxi->fuel_state());
	}

	if (event_after_message & (GAME_OVER | MISSION_COMPLETE)) {	// no more lives/levels

		abortGame();
		return;

	} else if (event_after_message & CRASH) {	// continue after crash

		taxi->teleport_to(level->initial_x, level->initial_y, 0, 0, true);
		shipsLCD->display(ships);

		if (passenger_onboard || (event_after_message & CRASH_PASSENGER)) {
			passenger_onboard = false;
			passenger_waiting = false;
			destinationLabel->setText(i18n("Nowhere"));	// update display
			lastStatus = i18n("Idle");

			if (!level->deliveries_to_go) {	// If this was the last passenger to deliver, we would get stuck.
				level->deliveries_to_go = 1;
			}

			delivery_complete();	// of course it is not, but this will take care of the rest
		}

	} else if (event_after_message & NEXT_LEVEL) {	// next level

		taxi->landed = false;	// otherwise taxi would try to detach itself from (to be deleted) platform
		delete level;
		level = new Level(level_files[current_level], canvas, this);
		view->setFixedSize(level->visible_width + 3, level->visible_height + 3);	// this has to happend after initialization of level, for this is where the canvas is initialized!
		// I do wonder, why I need this +3, though
		view->setContentsPos(current_offset_x = level->initial_offset_x, current_offset_y = level->initial_offset_y);

		fader->setSize(canvas->width(), canvas->height());
		taxi->teleport_to(level->initial_x, level->initial_y, 0, 0, true);
		//              app->setCaption (level->caption);

		// initialize timers and variables

		passenger_waiting = false;
		passenger_onboard = false;
		passenger_exists = false;
		fueling = false;

		passenger_timer = tm->startTimer(this, PASSENGER_TP);	// next passenger may appear, too
	}							// else if MISSION_COMPLETE

	resumeGame();				// this will set the AdvancePeriod.
}

/** Takes care of any scrolling necessary (for oversized levels) */
void Taxipilot::do_scroll()
{
	bool changes = false;
	int dummy;
// scroll left or right

	if ((taxi->x() - current_offset_x) < level->margin) {
		if (current_offset_x > 0) {
			dummy =
				(int) ((level->margin - (taxi->x() - current_offset_x)) + taxi->xVelocity());
			if (current_offset_x > dummy) {
				current_offset_x -= dummy;
			} else {
				current_offset_x = 0;
			}
			changes = true;
		}
	} else if ((taxi->RIGHTEDGE() - current_offset_x) > (level->visible_width - level->margin)) {
		dummy =
			((taxi->RIGHTEDGE() - current_offset_x) - (level->visible_width - level->margin)) +
			(int) (taxi->xVelocity());
		if (current_offset_x < level->max_offset_x) {
			if ((level->max_offset_x - current_offset_x) > dummy) {
				current_offset_x += dummy;
			} else {
				current_offset_x = level->max_offset_x;
			}
			changes = true;
		}
	}
// scroll up or down

	if ((taxi->y() - current_offset_y) < level->margin) {
		if (current_offset_y > 0) {
			dummy =
				(int) ((level->margin - (taxi->y() - current_offset_y)) +
					   (taxi->yVelocity()));
			if (current_offset_y > dummy) {
				current_offset_y -= dummy;
			} else {
				current_offset_y = 0;
			}
			changes = true;
		}
	} else if ((taxi->BOTTOMEDGE() - current_offset_y) > (level->visible_height - level->margin)) {
		if (current_offset_y < level->max_offset_y) {
			dummy =
				((taxi->BOTTOMEDGE() - current_offset_y) - (level->visible_height - level->margin)) +
				(int) (taxi->yVelocity());
			if ((level->max_offset_y - current_offset_y) > dummy) {
				current_offset_y += dummy;
			} else {
				current_offset_y = level->max_offset_y;
			}
			changes = true;
		}
	}

	if (changes) {
		view->setContentsPos(current_offset_x, current_offset_y);
	}

}

/** Generates the credits text and graphics */
void Taxipilot::generate_credits()
{
	int i;
	Credit_item *dummy;
	QFont credits_font = QFont(CREDITS_FONT, CREDITS_FONT_SIZE);
	QFont title_font = QFont(CREDITS_FONT, CREDITS_FONT_SIZE);
	title_font.setItalic(true);
	QCanvasText *text_dummy;

	// General info:
	dummy = new Credit_item;
	text_dummy =
		new QCanvasText(i18n("Taxipilot Version ") + VERSION + i18n(" by Thomas Friedrichsmeier."), credits_font,
						canvas);
	text_dummy->setColor(CREDITS_FONT_COLOR);
	dummy->item = text_dummy;
	append_credit(dummy, 50);

	// Mission info:
	dummy = new Credit_item;
	text_dummy = new QCanvasText(i18n("This Mission: '"), credits_font, canvas);
	text_dummy->setColor(CREDITS_FONT_COLOR);
	dummy->item = text_dummy;
	append_credit(dummy, 0);

	dummy = new Credit_item;
	text_dummy = new QCanvasText(mission_title, title_font, canvas);
	text_dummy->setColor(CREDITS_FONT_COLOR);
	dummy->item = text_dummy;
	append_credit(dummy, 0);

	dummy = new Credit_item;
	text_dummy = new QCanvasText(i18n("' by ") + mission_author, credits_font, canvas);
	text_dummy->setColor(CREDITS_FONT_COLOR);
	dummy->item = text_dummy;
	append_credit(dummy, 50);

	// Taxi info:
	dummy = new Credit_item;
	text_dummy = new QCanvasText(i18n("The Taxi: '"), credits_font, canvas);
	text_dummy->setColor(CREDITS_FONT_COLOR);
	dummy->item = text_dummy;
	append_credit(dummy, 0);

	Taxi *credits_taxi = new Taxi(taxi_file, canvas, 0, 0, this, true);

	dummy = new Credit_item;
	text_dummy = new QCanvasText(credits_taxi->title, title_font, canvas);
	text_dummy->setColor(CREDITS_FONT_COLOR);
	dummy->item = text_dummy;
	append_credit(dummy, 0);

	dummy = new Credit_item;
	text_dummy = new QCanvasText("' (", credits_font, canvas);
	text_dummy->setColor(CREDITS_FONT_COLOR);
	dummy->item = text_dummy;
	append_credit(dummy, 0);

	dummy = new Credit_item;
	dummy->item = credits_taxi;
	append_credit(dummy, 0);

	dummy = new Credit_item;
	text_dummy = new QCanvasText(i18n(") by ") + credits_taxi->author, credits_font, canvas);
	text_dummy->setColor(CREDITS_FONT_COLOR);
	dummy->item = text_dummy;
	append_credit(dummy, 50);

	// Passenger_info:
	dummy = new Credit_item;
	text_dummy = new QCanvasText(i18n("The Passengers: "), credits_font, canvas);
	text_dummy->setColor(CREDITS_FONT_COLOR);
	dummy->item = text_dummy;
	append_credit(dummy, 10);

	for (i = 0; i < num_passenger_files; i++) {

		Passenger *credits_passenger = new Passenger(passenger_files[i], canvas, this);
		credits_passenger->initialize (0, 0, 0, true);

		dummy = new Credit_item;
		text_dummy = new QCanvasText("'", credits_font, canvas);
		text_dummy->setColor(CREDITS_FONT_COLOR);
		dummy->item = text_dummy;
		append_credit(dummy, 0);

		dummy = new Credit_item;
		text_dummy = new QCanvasText(credits_passenger->title, title_font, canvas);
		text_dummy->setColor(CREDITS_FONT_COLOR);
		dummy->item = text_dummy;
		append_credit(dummy, 0);

		dummy = new Credit_item;
		text_dummy = new QCanvasText("' (", credits_font, canvas);
		text_dummy->setColor(CREDITS_FONT_COLOR);
		dummy->item = text_dummy;
		append_credit(dummy, 0);

		dummy = new Credit_item;
		dummy->item = credits_passenger;
		append_credit(dummy, 0);

		dummy = new Credit_item;
		text_dummy = new QCanvasText(i18n(") by ") + credits_passenger->author, credits_font, canvas);
		text_dummy->setColor(CREDITS_FONT_COLOR);
		dummy->item = text_dummy;
		if (i == (num_passenger_files - 1)) {
			append_credit(dummy, 50);
		} else {
			append_credit(dummy, 20);
		}
	}

	// Level-info:
	dummy = new Credit_item;
	text_dummy = new QCanvasText(i18n("The Levels: "), credits_font, canvas);
	text_dummy->setColor(CREDITS_FONT_COLOR);
	dummy->item = text_dummy;
	append_credit(dummy, 10);

	for (i = 0; i < num_level_files; i++) {
		QString convert;

		Level *credits_level = new Level(level_files[i], canvas, this);

		dummy = new Credit_item;
		text_dummy = new QCanvasText(convert.setNum(i + 1) + ": '", credits_font, canvas);
		text_dummy->setColor(CREDITS_FONT_COLOR);
		dummy->item = text_dummy;
		append_credit(dummy, 0);

		dummy = new Credit_item;
		text_dummy = new QCanvasText(credits_level->caption, title_font, canvas);
		text_dummy->setColor(CREDITS_FONT_COLOR);
		dummy->item = text_dummy;
		append_credit(dummy, 0);

		dummy = new Credit_item;
		text_dummy = new QCanvasText(i18n("' by ") + credits_level->author, credits_font, canvas);
		text_dummy->setColor(CREDITS_FONT_COLOR);
		dummy->item = text_dummy;

		if (i == (num_level_files - 1)) {
			append_credit(dummy, 50);
		} else {
			append_credit(dummy, 20);
		}

		delete credits_level;	// not needed
	}

// set off animation    
	credits.first();			// go to first item
	first_visible_credit = 0;
	last_visible_credit = -1;
	credits_timer = tm->startTimer(this, CREDITS_TP);	// set off animation

}

/** Free all the memory occupied by the credits */
void Taxipilot::delete_credits()
{
	credits.setAutoDelete(true);

	while (!credits.isEmpty()) {
		if (credits.first()->item != 0) {	// just a security measure
			delete credits.first()->item;
		}
		credits.remove();		// since we called first (), this removes the first item.
	}

	tm->killTimer(credits_timer);
	credits_timer = -1;
}

/** This function is only ever needed by generate_credits */
void Taxipilot::append_credit(Credit_item * item, int space_after)
{

	item->item->setAnimated(false);
	item->item->setVelocity(-CREDITS_SPEED / cdp->fps (), 0);
	item->width = item->item->boundingRect().width();
	item->space_to_next = space_after;
	item->y = view->height() - item->item->boundingRect().height() - 5;
	item->item->move(view->width(), item->y);
	credits.append(item);

}

/** Pause the game when focus gets lost. */
void Taxipilot::focusOutEvent (QFocusEvent *e){
	if (game_running) app->gamePause->activate();
}

/** Receives a signal that the config changed. */
void Taxipilot::config_changed(){
	double x = (double) cdp->fps () * (double) cdp->cps ();
	acceleration = 2000 / x;
	if (passenger_exists) {
		tip_step = 0.2 / (double) cdp->cps () * passenger->impatience;
	}
	if (game_running) {
		level->config_changed ();	// the level is no QObject, so we just tell it by hand. (The level will tell the centers of gravity)
		taxi->config_changed ();
	}
	if (base_timer != -1) {
		tm->killTimer (base_timer);
	}
	base_timer = tm->startTimer (this, 1000 / cdp->cps ());		// AdvancePeriod is set in resumeGame ()
	if (scroll_timer != -1) {
		tm->killTimer (scroll_timer);
	}
	scroll_timer = tm->startTimer (this, 1000 / cdp->fps ());

	speedDial->setMaxValue (10000 / cdp->fps ());
	speedDial->setLimits (-1, 4000 / cdp->fps ());
	speedDial->setLineStep (1000 / cdp->fps ());
	speedDial->setPageStep (2000 / cdp->fps ());
	speedDial->setNotchTarget (5);
}
