/***************************************************************************
                          taxi.h  -  description
                             -------------------
    begin                : Fri May 18 2001
    copyright            : (C) 2001, 2002 by Thomas Friedrichsmeier
    email                : Thomas.Friedrichsmeier@ruhr-uni-bochum.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef TAXI_H
#define TAXI_H

#include <qcanvas.h>
#include <qstring.h>

#include "timedobject.h"

/**
  *@author Thomas Friedrichsmeier
  */

class Taxipilot;
class Cdp;
class QDomElement;
class Teleporter;
class Platform;

/** This class is basically the taxi-sprite. It is a lot like the Passenger-class
(see documentation there) and probably those two classes should eventually be derived from
a common base class (e.g. when finally adding movin object support */

class Taxi: public QCanvasSprite, public TimedObject {
  public:
	Taxi(const QString & file, QCanvas * canvas, int base_x, int base_y, Taxipilot * parent, bool for_credits_only =
		 false);
	~Taxi();
  /** Signals we are going down. See up () for details. */
	void down(double acc);
  /** Signals we are going up. The reason this is not simply done via setVelocity is
that taxi is eventually to do animation stuff. The double acceleration is passed as
a parameter instead of using some constant amount, so we may eventually
abstract from keyboard input, and allow (analog) joystick input */
	void up(double acc);
  /** See above */
	void right(double acc);
  /** See above */
	void left(double acc);
/** Stops the currently running flying animation, if any (called from Taxipilot::do_chores ()) */
	void stop_flying_animation();
  /** Takes care of teleportation (appearing, disappearing animations), setting new position
and speed. If appear_only is given, does not do disappearing animation. */
	void teleport_to(int x, int y, double speedfactor_x, double speedfactor_y, bool appear_only = false, Teleporter *dynamic = 0);
  /** Returns the current fuel level in percent of the taxi (Used by Taxipilot to set the
FuelDial) */
	int fuel_state();
	/** These two are set by fuel_state () and should be self-explanatory */
	bool fuel_low, fuel_out;
  /** Increases the taxi's fuel supply by step (max up to max_fuel). Returns actual
increase in fuel (so that's 0 if taxi is already full). */
	double fuel_up(int step);
  /** Toggles the taxis landing-flaps */
	void toggle_flaps();
/** Repositions the taxi. This also takes care of repositioning the landing flaps. */
	void reposition(int x, int y, double xv, double yv, bool align_bottom = false);;
/** Returns the (absolute) x-position the left landing-flaps is to be found at */
	int landing_flaps_x1();
/** Returns the (absolute) x-position the right landing-flap is to be found at */
	int landing_flaps_x2();
/** A wrapper for moveBy, taking care of the flaps if appropriate */
  void wrap_moveBy (double delta_x, double delta_y);
/** Adjusts the taxi to a fractional position, so it doesn't wiggle on a moving platform */
	void adjust (double fraction_x, double fraction_y);
/** Returns the (absolute) y-position the landing-flaps are to be found at */
  int landing_flaps_y();
/** Sprite for the flaps-animation */
	QCanvasSprite *flaps;

/** This additional sprite is needed for the flight-exhausts (harmful to passenger, no crashing
into levels */
	QCanvasSprite *flying_lr, *flying_ud;
/** Is taxi ready for flight (not teleporting) */
	bool ready;
/** x-offsets of the landing flaps, used to check, whether taxi is properly landed within platform boundaries */
	int landing_flaps_off_x1[2], landing_flaps_off_x2[2];

	friend class Taxipilot;		// we need to know absX2 and stuff

/** about data */
	QString title, author;
  /** Current status of the landing flaps: 0=in 1=half in/out 2=out. */
	int flaps_state;
/** The previous state of the flaps (i.e. before crash / teleport). Can only be in (flase) or out (true) */
	bool flaps_prev_state;
  /** Are flaps available for this taxi? */
	bool flaps_avail;
/** Is the taxi currently landed? */
	bool landed;
  private:						// Private methods
	double speed;				// store how fast we were going last.
/** Store the last speed when teleporting */
	double speed_x, speed_y;
	double fuel, max_fuel;
/** The pointer to the common debugging and prefixing class */
	Cdp *cdp;
  /** This function reads a taxi-description ( specs for the animations ) from a file.
read_taxi is only called by the constructor. */
	void read_taxi(const QString & file);
/** Tell the taxi it has landed on the specified paltform */
	void land (Platform *on);
/** Pointer to the platform the taxi is landed on (if it is landed) */
	Platform *platform_p;
/** stores the timer-id for triggering and animating the idle-animations, respectively.
"Idle" animation is a slightly misleading term, since the taxi will do these animations
regardless of its own activity (at least for now) */
	int idle_timer;
/** The number of different idle_animations this taxi has. Must be at least one. */
	int num_idle_animations;
/** Stores current frame for the taxi-sprite */
	int taxi_frame;
/** Stores current frame for the flying-exhausts-sprite */
	int flying_lr_frame, flying_ud_frame;

	bool dynamic_teleport;
	Teleporter *dynamic_teleporter;

/** All animations */
	struct Anim {
/** The actual animation as used by QT */
		QCanvasPixmapArray *anim[2];
/** This is the number of frames in the animation */
		int frames;
/** Simple things like waving may be repeated. */
		int repetitions;
/** Is the anim currently being played forward (+1) or reverse (-1)? */
		int direction;
/** Inverse of frame-rate */
		int frame_period;
/** Only the still-anim is continous */
		bool continuous;
	} still, teleporting, flying_right, flying_left, flying_up, flying_down, flaps_anim, *idle;

  /** This function is only used by read_taxi, and takes care of reading in
(direction-specific or non-specific) animations. */
	void read_animation(Anim * anim, QDomElement e);

/** When teleporting, these values have to be stored temporarily */
	int new_x, new_y;
	double new_speed_x, new_speed_y;
/** just so we don't have to calculate (width () / 2) and (height () / 2) all the time */
	int off_x, off_y;
	int double_off_y;
/** Whether left/right-acceleration is blocked when flaps are out */
	bool flaps_block;
/** Whether were currently turned left (0) or right (1) */
	int lr;

/** Sets of the animation anim for this taxi or it's flaps,  and triggers
a function-call when completed. Reverse means: play in reverse order; flaps means, the animation is for the flaps */
	void do_animation (Anim * anim, void (Taxi::*call_back) (), bool reverse=false, bool flaps=false);

	int flaps_anim_timer, anim_timer;
	int cflaps_frame, ctaxi_frame;
	Anim *current_anim, *cflaps_anim;
	int repetition;
	void (Taxi::*taxi_call_back) ();
	void (Taxi::*flaps_call_back) ();
  /** Taxi is to reappear (at global new_x, new_y) */
  void reappear ();
  /** Stuff to do when flaps have just finished going in */
  void flaps_in ();
  /** Stuff to do when the flaps just finished going out */
  void flaps_out();
  /** Starts idle_timer (not idle_animation) */
  void go_idle ();
  /** Does nothing */
  void do_nothing();

  /** Stuff to do when done appearing */
  void done_appearing ();
  /** Takes care of deleting the one or two QCanvasPixmapArrays associated with this
animation */
  void delete_animation (Anim &anim);

/** Tells the taxi it crashed. Does or will do stuff like setting of an appropriate animation,
sound and other things to do. */
	void crash();
/** The config has changed. Recalculate. */
	void config_changed();
/** Fuel-consumption (dependend on cdp->cps ()) */
	double fuel_con;
  protected:					// Protected methods
/** Mostly everyting will be done inside: timing for all animations */
	void timerTick(int id);
/** Reimplementation of the advance function to make sure that the flaps get moved
accordingly. */
	void advance (int phase);
};

#endif
