/***************************************************************************
                          platform.cpp  -  description
                             -------------------
    begin                : Tue Nov 6 2001
    copyright            : (C) 2001, 2002 by Thomas Friedrichsmeier
    email                : Thomas.Friedrichsmeier@ruhr-uni-bochum.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include "defines.h"

#include "platform.h"

#include <qcanvas.h>
#include <qstring.h>
#include <qdom.h>

#include <math.h>

#include "passenger.h"
#include "taxi.h"
#include "taxipilot.h"
#include "level.h"
#include "movingobject.h"
#include "cdp.h"
#include "event.h"

Platform::Platform(const QDomElement *description, Level *parent, bool draw_lines){

	draw_platform_lines = draw_lines;

	level = parent;
	tp = level->tp;
	cdp = level->cdp;

	visible = true;
	has_object = false;
	
	passenger_ontop = taxi_ontop = false;

	read_platform (description);

}

Platform::~Platform(){

	if (draw_platform_lines) {
		delete line;
	}

}

/** Read the platforms's description and creates it */
void Platform::read_platform(const QDomElement * description){

	QString convert, dummy;

	if (description->hasAttribute ("ident")) {
		object_ident = cdp->get_string_attribute ("ident", *description, "#bad value#", 1);
		object_promised = true;
	} else {
		last_x = last_y = 0;
		object_promised = false;
	}

	y = cdp->get_int_attribute("y", *description, 0, 3000, 100, 1);
	x1 = cdp->get_int_attribute("x1", *description, 0, 3000, 100, 1);
	x2 =	cdp->get_int_attribute("x2", *description, x1, 3000, x1 + 100, 1);

	if (cdp->get_bool_attribute("has_fuel", *description, false, 3)) {
		fuel_price = cdp->get_double_attribute("fuel_price", *description, 0, 100, 0.05, 2);
		fuel_step = cdp->get_int_attribute("fuel_step", *description, 1, 8000, 1, 3);
	} else {
		fuel_step = 0;
	}

	if ((accepts_passenger = cdp->get_bool_attribute("passengers", *description, false, 3))) {
		passenger_base_x = cdp->get_int_attribute("passenger_base_x", *description, 0, 3000, x1, 1);
	}

	label = cdp->get_string_attribute("label", *description, i18n("Platform ") + convert.setNum(level->num_platforms + 1), 3);

	if (draw_platform_lines) {
		line = new QCanvasLine(level->canvas());
		line->setPen(Qt::yellow);
		line->setPoints(x1, y, x2, y);
		line->setZ(LEVEL_Z + 1);
		line->show();
	}

// events
	if ((dummy = cdp->get_string_attribute ("land_event", *description, "NOTHING", 3)) == "NOTHING") {
		land_event = 0;
	} else {
		land_event = level->resolve_event (dummy);
	}
	if ((dummy = cdp->get_string_attribute ("take_off_event", *description, "NOTHING", 3)) == "NOTHING") {
		take_off_event = 0;
	} else {
		take_off_event = level->resolve_event (dummy);
	}
	if ((dummy = cdp->get_string_attribute ("arrive_event", *description, "NOTHING", 3)) == "NOTHING") {
		arrive_event = 0;
	} else {
		arrive_event = level->resolve_event (dummy);
	}
	if ((dummy = cdp->get_string_attribute ("leave_event", *description, "NOTHING", 3)) == "NOTHING") {
		leave_event = 0;
	} else {
		leave_event = level->resolve_event (dummy);
	}

}

/** Tells the platform, whether it is currently visible */
void Platform::set_visible(bool visibility){
	if (passenger_ontop) {
		passenger->setVisible (visibility);
	}
	if (taxi_ontop) {
		taxi->landed = false;
	}

	visible = visibility;
}

/** Fulfills the promise (assigns the promised object to this platform) */
void Platform::promise_fulfilled(MovingObject * obj){

	object = obj;
	has_object = true;
	
}

/** Drop a passenger on top of this paltform (moving platforms will use the pointer to
move the passenger with them) */
void Platform::attach_passenger (Passenger *pass){
	double dummy;

	passenger_ontop = true;
	passenger = pass;
	if (has_object) {
		// positions the passenger/taxi to the same fraction of the coordinates as the object,
		// to make sure that both move synchronous
		passenger->adjust (modf (object->x(), &dummy), modf (object->y(), &dummy));
	}

	// trigger arrive_event if any
	if (arrive_event) {
		arrive_event->trigger ();
	}

}

/** Detaches the passenger when it has boarded the taxi or disappeared */
void Platform::detach_passenger(){

	passenger_ontop = false;

	// trigger leave_event if any
	if (leave_event) {
		leave_event->trigger ();
	}

}

/** Tells the platform, it has moved (so it can then move the taxi / passenger) */
void Platform::movedBy (double dx, double dy) {
	if (taxi_ontop) {
		taxi->wrap_moveBy (dx, dy);
	}
	if (passenger_ontop) {
		passenger->move_passenger_by (dx, dy);
	}

	last_x += dx;
	last_y += dy;

	if (draw_platform_lines) {
		line->move (current_plat_x1 (), current_plat_y ());
	}
}

/** Same as above, but absolute positioning */
void Platform::movedTo (double x, double y) {
	if (taxi_ontop) {
		taxi->wrap_moveBy (x - last_x, y - last_y);
	}
	if (passenger_ontop) {
		passenger->move_passenger_by (x - last_x, y - last_y);
	}

	// set new reference position
	last_x = x;
	last_y = y;

	if (draw_platform_lines) {
		line->move (current_plat_x1 (), current_plat_y ());
	}
}

/** Checks whether the given taxi is landed on this platform and if so, takes appropriate
actions. */
bool Platform::check_landed (Taxi *taxi_p){

	double tolerance_down;
	
	tolerance_down = (taxi_p->yVelocity() * ( /*(int)*/ ((double) cdp->fps() / (double) cdp->cps ())/* + 1*/));

	if (has_object) {
		tolerance_down -= (object->current_vy () * (/* (int)*/ ((double) cdp->fps () / (double) cdp->cps ())/* + 1*/));
	}
	
	tolerance_down += 5;

	if ((taxi_p->landing_flaps_y () >= (current_plat_y () - taxi_p->yVelocity () - 2)) && ((taxi_p->landing_flaps_y () - current_plat_y ()) < tolerance_down)) {
		if ((taxi_p->landing_flaps_x1() >= current_plat_x1 ()) && (taxi_p->landing_flaps_x2() <= current_plat_x2 ())) {
			return (true);
		}
	}
	return (false);
}

/** Current y-position of the actual platform (i.e. taking into account any offset) */
int Platform::current_plat_y (){
	return ((int) last_y + y);
}

/** Current x1-position of the actual platform (i.e. taking into account any offset) */
int Platform::current_plat_x1(){
	return ((int) last_x + x1);
}

/** Current x2-position of the actual platform (i.e. taking into account any offset) */
int Platform::current_plat_x2(){
	return ((int) last_x + x2);
}

/** Current x-position of the actual platform's passenger_base_x (i.e. taking into account any offset) */
int Platform::current_plat_base_x(){
	return ((int) last_x + passenger_base_x);
}

/** Return current x-velocity of platform (if it is an object) */
double Platform::current_plat_vx () {
	if (has_object) {
		return object->current_vx ();
	} else {
		return 0;
	}
}

/** Return current y-velocity of platform (if it is an object) */
double Platform::current_plat_vy () {
	if (has_object) {
		return object->current_vy ();
	} else {
		return 0;
	}
}

/** A taxi has landed on top of this paltform (moving platforms will use the pointer to
move the taxi with them) */
void Platform::attach_taxi(Taxi * taxi_p){
	double dummy;

	if (!taxi_ontop) {
		taxi_ontop = true;
		taxi = taxi_p;
		if (has_object) {
			// positions the passenger/taxi to the same fraction of the coordinates as the object,
			// to make sure that both move synchronous
			taxi->adjust (modf (object->x(), &dummy), modf (object->y(), &dummy));
		}
		// trigger land_event if any
		if (land_event) {
			land_event->trigger ();
		}
        }

}

/** Detaches the taxi when it takes off */
void Platform::detach_taxi(){

	taxi_ontop = false;

	// trigger take_off_event if any
	if (take_off_event) {
		take_off_event->trigger ();
	}

}
