/***************************************************************************
                          cdp.cpp  -  description
                             -------------------
    begin                : Sun Aug 26 2001
    copyright            : (C) 2001, 2002 by Thomas Friedrichsmeier
    email                : Thomas.Friedrichsmeier@ruhr-uni-bochum.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include "defines.h"

#include <qfile.h>
#include <qfileinfo.h>
#include <qstrlist.h>
#include <qstring.h>
#include <qdom.h>
#include <qtextstream.h>
#include <qregexp.h>

#include <qwidget.h>

#include <klocale.h>
#include <kconfig.h>
#include <kglobal.h>
#include KSTDDIRS
#include <kcmdlineargs.h>
#include <kmessagebox.h>
#include KAPP

#include <stdlib.h>
#include <unistd.h>

#include "cdp.h"
#include "taxipilot.h"
#include "taxipilotapp.h"

Cdp::Cdp(Taxipilot * parent)
{

	// get the debug-level
	KCmdLineArgs *args = KCmdLineArgs::parsedArgs();
	QString opt = args->getOption("debug-level");
	debug_level = opt.toInt();
	draw_highlights = args->isSet ("highlight");
	opt_thorough_check = args->isSet ("thorough_check");

	// get the data-dirs
	alt_image_prefix = image_prefix = KGlobal::dirs()->findResourceDir("images", "images_tag");

	// store parent
	tp = parent;

	errors_occurred = false;

	// read global settings
	KConfig *config = kapp->config();

	config->setGroup( "Paths" );
	alt_data_prefix = data_prefix = config->readPathEntry ( "defaultMissionDir", KGlobal::dirs()->findResourceDir("game_data", "game_data_tag"));
	alt_image_prefix = image_prefix = data_prefix + "/images/";
	pref_useDefaultMission = config->readBoolEntry ( "useDefaultMission", true );
	pref_useLastMission = config->readBoolEntry ( "useLastMission", false );
	pref_useSelectedMission =config->readBoolEntry ( "useSelectedMission", false );
	pref_DefaultMission = config->readPathEntry ( "DefaultMission", data_prefix + "/mission1.mission" );
	pref_LastMission = config->readPathEntry ( "LastMission", data_prefix + "/mission1.mission" );
	pref_SelectedMission = config->readPathEntry ( "SelectedMission" );

	config->setGroup ( "Tuning" );
	pref_fps = config->readNumEntry( "fps", 30 );
	pref_cps = config->readNumEntry( "cps", 20 );

}

Cdp::~Cdp()
{
}

/** Add a directory prefix to a filename if appropriate (i.e. not an absolute path) and checks,
whether the file exists. image specifies, whether this is an image (use image-prefix instead of
data-prefix) */
QString Cdp::prefix_check(QString filename, bool image)
{
	QString pref, alt_pref;
	if (image) {
		pref = image_prefix;
		alt_pref = alt_image_prefix;
	} else {
		pref = data_prefix;
		alt_pref = alt_data_prefix;
	}

	if (QFileInfo (filename).isRelative()) {
		if (QFile::QFile(alt_pref + filename).exists()) {
			pref = alt_pref;
		}
		debug_msg(i18n("Adding prefix '") + pref + i18n("' to filename: '") + filename + "'", 3);
		filename = pref + filename;
	}
	QFile f(filename);
	if (!f.open(IO_ReadOnly))	// if we can't open the file for reading, this is a fatal error and qe quit.
		debug_msg(i18n("Could not open file for reading: ") + filename, 0);
	f.close();
	return filename;
}

/** Add a directory prefix to a filename_pattern if appropriate (i.e. not an absolute path) and checks,
whether all the files needed fro the animation exist */
QString Cdp::prefix_check_animation(QString filename_pattern, int frames)
{
	QString dummy1, dummy2;

	QString pref = image_prefix;

	if (QFileInfo (filename_pattern).isRelative ()) {
		dummy1 = filename_pattern;
		dummy1.replace(QRegExp("%1"), "0000");
		if (QFile::QFile(alt_image_prefix + dummy1).exists()) {
			pref = alt_image_prefix;
		}
		debug_msg(i18n("Adding prefix '") + pref + i18n("' to filename-pattern: '") + filename_pattern + "'", 3);
		filename_pattern = pref + filename_pattern;
	}
	if (frames == 0) {
		prefix_check(filename_pattern, "");	// Already prefixed
	} else {
		for (int i = 0; i < frames; i++) {
			dummy1 = filename_pattern;
			dummy2.setNum(i);
			while (dummy2.length() < 4) {
				dummy2.prepend('0');
			}
			prefix_check(dummy1.replace(QRegExp("%1"), dummy2), true);	// passed filename is already absolute, of course
		}
	}
	return filename_pattern;
}

/** Initializes the function that will take care of debugging messages and potential errors. in_file
is the filename which is currently to be checked. If exit_on_error == false, does not kill
the app on errors. This is useful, if even critical errors are not necessarly fatal, such as
when reading the mission. */
void Cdp::init_debug(QString in_file, bool exit_on_error)
{
	QString alt_path = QFileInfo (in_file).dirPath (true);

//    errors_occurred = false;
	die_on_error = exit_on_error;
	debug_in_file = in_file;
	alt_data_prefix = alt_path + "/";
	alt_image_prefix = alt_path + "/images/";
}

/** Returns inverse of errors_occurred, displaying a Dialog if errors_occured */
bool Cdp::ok()
{
	if (errors_occurred) {
		KMessageBox::sorry(tp->app,
						   i18n ("An error occured reading the configuration for this mission.\n(See output to stderr for details)\nPlease select a different mission via Settings->Select Mission."));
		errors_occurred = false;
		return (false);
	}
	return (true);
}

/** Appends an arbitrary string to the debug or error messages. If level == 0, this is a critical
error and the program will abort in do_debug (). If *in_node is specified a backtrace is shown */
void Cdp::debug_msg(QString message, int level, QDomNode const *in_node)
{
	QString color_code;

	if (level <= debug_level) {	// generate output only if level <= debug_level
		QString niveau;
		QString bt_string = "";
		switch (level) {
		case 0:
			color_code = "\033[01;31m";	// bright red
			niveau = i18n("Fatal error");
			break;
		case 1:
			color_code = "\033[01;35m";	// bright magenta
			niveau = i18n("Serious warning");
			break;
		case 2:
			color_code = "\033[01;32m";	// bright green
			niveau = i18n("Warning");
			break;
		case 3:
			color_code = "\033[0m";	// normal/black
			niveau =  i18n("Info");
			break;
		case 4:
			color_code = "\033[0m";	// normal/black
			niveau =  i18n("Translations");
			break;
		}
		if (in_node) {			// only create backtrace if in_node was supplied
			bt_string = i18n("In node '");
			if (!in_node->isNull()) {	// and node is valid
				bt_string += backtrace(*in_node) + "': ";
			} else {
				bt_string += i18n("#INVALID#") + "': ";
			}
		}
		if (isatty (STDERR_FILENO)) {
			fputs (color_code, stderr);		// ouput color_code only if this is a tty
		}
		fputs((i18n("In file '") + debug_in_file + "': " + bt_string + niveau + ": " + message + "\n"), stderr);
		if (isatty (STDERR_FILENO)) {
			fputs ("\033[0m", stderr);		// ouput color_code (back to normal) only if this is a tty
		}
	}

	if (level == 0) {
		if (die_on_error) {
			KMessageBox::error(tp, i18n("Error reading file '") + debug_in_file + "'\nExiting TaxiPilot.");
			exit(1);
		}
		errors_occurred = true;
	}

}

/** Checks for the existence of a certain integer XML-attribute in element. If given, returns its
value. If not given or not an int or not between min and max, return def and writes debugging message of level level */
int Cdp::get_int_attribute(QString attribute, const QDomElement & element, int min, int max, int def, int level)
{

	QString convert;

	if (!element.hasAttribute(attribute)) {
		debug_msg(attribute + i18n("-attribute not given. Assuming ") + convert.setNum(def), level, &element);
		return def;
	}

	convert = element.attribute(attribute);
	bool ok;

	int given = convert.toInt(&ok);

	if (!ok) {
		debug_msg(attribute + i18n("-attribute not an integer value. Assuming ") + convert.setNum(def), level,
				  &element);
		return def;
	}

	if ((given > max) || (given < min)) {
		debug_msg(attribute + i18n("-attribute outside valid range (") + convert.setNum(min) + " - " +
				  convert.setNum(max) + i18n(". Assuming ") + convert.setNum(def), level, &element);
		return def;
	}

	return given;
}

/** Checks for the existence of a certain double XML-attribute in element. If given, returns its
value. If not given or not a double number or not between min and max, returns def and writes debugging message of level level */
double Cdp::get_double_attribute(QString attribute, const QDomElement & element, double min, double max, double def,
								 int level)
{

	QString convert;

	if (!element.hasAttribute(attribute)) {
		debug_msg(attribute + i18n("-attribute not given. Assuming ") + convert.setNum(def), level, &element);
		return def;
	}

	convert = element.attribute(attribute);
	bool ok;

	double given = convert.toDouble(&ok);

	if (!ok) {
		debug_msg(attribute + i18n("-attribute not a number. Assuming ") + convert.setNum(def), level, &element);
		return def;
	}

	if ((given > max) || (given < min)) {
		debug_msg(attribute + i18n("-attribute outside valid range (") + convert.setNum(min) + " - " +
				  convert.setNum(max) + i18n(". Assuming ") + convert.setNum(def), level, &element);
		return def;
	}

	return given;
}

/** Checks for the existence of a certain string XML-attribute in element. If given, returns its
value. If not given or not existent, returns def and writes debugging message of level level */
QString Cdp::get_string_attribute(QString attribute, const QDomElement & element, QString def, int level, bool trans)
{

	if (!element.hasAttribute(attribute)) {
		debug_msg(attribute + i18n("-attribute not given. Assuming '") + def + "'", level, &element);
		return def;
	}

	QString dummy = element.attribute(attribute);
	if (!trans) {
		return dummy;
	}
	if (translations.find (dummy) == translations.end ()) {
		debug_msg(i18n("No translation found for string '") + dummy + "'", 4, &element);
	} else {
		debug_msg(i18n("Translating string '") + dummy + i18n ("' to '") + translations[dummy] + "'", 4, &element);
		return (translations[dummy]);
	}
	return (dummy);
}

/** Checks for the existence of a certain bool XML-attribute in element. If given, returns its
value. If not given or not existent, returns def and writes debugging message of level level */
bool Cdp::get_bool_attribute(QString attribute, const QDomElement & element, bool def, int level)
{

	QString defstring = "no";
	if (def)
		defstring = "yes";

	if (!element.hasAttribute(attribute)) {
		debug_msg(attribute + i18n("-attribute not given. Assuming '") + defstring + "'", level, &element);
		return def;
	}

	QString value = element.attribute(attribute);

	if ((value == "true") || (value == "yes")) {
		return true;
	}
	// (else)
	if ((value == "false") || (value == "no")) {
		return false;
	}
	// (else)
	debug_msg(attribute + i18n("-attribute not yes/true or no/false. Assuming '") + defstring + "'", level, &element);
	return def;
}

  /** Opens the config-file file, parses it into XML, and returns document element */
QDomElement Cdp::open_config_file(QString const &file)
{
	QDomDocument doc;
	QFile f(file);
	if (!f.open(IO_ReadOnly))
		debug_msg(i18n("Could not open file for reading"), 0);
	if (!doc.setContent(&f)) {
		f.close();
		debug_msg(i18n("Error parsing XML-file"), 0);
	}
	f.close();
		
	// get translations-file (if any)
	translations.clear ();
	QFile t(file + "." + i18n("en_US"));	// TRANSLATORS: set "en_US" to your language
	if (!t.open(IO_ReadOnly))
		debug_msg(i18n("Could not find/open translations-file for reading"), 4);
	else {
		QTextStream l (&t);
		while ( !l.eof() ) {        // until end of file...
			QString dummy = l.readLine();
			QString key = dummy.left (dummy.find (":::::"));
			QString trans = dummy.right (dummy.length () - dummy.find (":::::") - 5);
			translations.insert (key, trans);
		}
		t.close();
	}
	
	return doc.documentElement();
}

/** Traces back the nodes in the current file, so as to display them in the debugging-output */
QString Cdp::backtrace(QDomNode const &node)
{
	QStrList list;
	QString return_string;

	QDomNode node_copy = node;
	while (!((node_copy.isDocument()) || (node_copy.isNull()))) {
		list.append(node_copy.nodeName());
		node_copy = node_copy.parentNode();
	}

	do {
		return_string += list.current();
		if (list.prev()) {
			return_string += "->";
		}
	} while (list.current());

	return (return_string);
}

/** Gets all child-nodes named name of element parent, checking whether there are at least
min, at most max nodes and min_if_any nodes if any, otherwise stating an error at
level level. (if name = "", don't check for name, but return all child nodes, if max < 0, there is no maximum) */
QDomNodeList Cdp::get_node_list(QString const &name, QDomElement const &parent, int min, int max, int min_if_any,
								int level)
{
	QDomNodeList empty;

	if (parent.isNull()) {
		debug_msg(i18n("Unspecified parsing error! (Wrong filetype?)"), 0);
		return empty;
	} else {
		QString convert;
		QDomNodeList list;
		if (name != "") {
			list = parent.elementsByTagName(name);
		} else {
			list = parent.childNodes();
		}

		int num = list.length();
		if (num < min) {
			debug_msg(i18n("Only ") + convert.setNum(num) + " " + name + i18n("-elements given. But at least ") +
					  convert.setNum(min) + i18n(" required."), level, &parent);
		} else if ((max >= 0) && (num > max)) {
			debug_msg(convert.setNum(num) + " " + name + i18n("-elements given. Maximum is ") + convert.setNum(min),
					  level, &parent);
		} else if ((num != 0) && (num < min_if_any)) {
			debug_msg(i18n("Only ") + convert.setNum(num) + " " + name + i18n("-elements given. But at least ") +
					  convert.setNum(min) + i18n(" required if any. Assuming none"), level, &parent);
			return (empty);
		}

		return (list);
	}
}

  /** Get the child-element named name of element parent. If no such element exists
	or there are more than one element, outputs error at level level. */
QDomElement Cdp::get_element(QString const &name, QDomElement const &parent, int level)
{

	QDomNodeList list = get_node_list(name, parent, 0, 1, 1, level);

#ifndef NEW_KDE
	QDomNode node = list.item(0);
	QDomElement e = node.toElement();

	if (e.tagName() != name) {	// if no nodes by that name were found, you CANNOT reely on
		// QT (2) returning an empty list!
		debug_msg(name + i18n("-element not given."), level, &parent);
		QDomElement empty;
		return (empty);
	}
	return (e);
#else
	if (!list.count ()) {
		debug_msg(name + i18n("-element not given."), level, &parent);
		QDomElement empty;
		return (empty);
	}

	QDomNode node = list.item(0);
	return (node.toElement ());
#endif
}

/** Checks for the existence of a certain string XML-attribute in element. If given, checks
whether the attribute's value matches any of possible_values, which is a ';' seperated
list of allowed strings. If not given, not valid or not existent, returns def and writes
debugging message of level level. */
QString Cdp::get_multi_choice_attribute(QString attribute, const QDomElement & element, QString possible_values, QString def, int level){

	QString allowed_values = possible_values;

	QString value = get_string_attribute(attribute, element, "", level, false);	// do not translate

	while (possible_values != "") {
		int i = possible_values.findRev (';');
		if (i >= 0) {
			if (value == possible_values.mid (i+1, possible_values.length())) {
				return (value);
			}
			possible_values.truncate (i);
		} else {
			if (value == possible_values) {
				return (value);
			}
			possible_values = "";
		}
	}

	debug_msg ( i18n ("Attribute '") + attribute + i18n ("' does not allow value '") + value + i18n ("'. Valid values are: '")
				+ allowed_values + i18n ("'. Assuming: '") + def + "'.", level, &element );
	return def;
}

/** Returns true, if the commandline-option highlight was set, false otherwise */
bool Cdp::draw_highlight (){
	return (draw_highlights);
}

/** Returns true, if the commandline-option thorough_check was set, false otherwise */
bool Cdp::thorough_check(){
	return (opt_thorough_check);
}

/** Returns whether we're reading for real (not just for the credits/startscreen) */
bool Cdp::for_real (){
	return (read_for_real);
}

/** Sets whether reading is going to be for real */
void Cdp::set_for_real (bool real){
	read_for_real = real;
}

/** Return frames-per-second setting */
int Cdp::fps (){
	return pref_fps;
}

/** Return calculations-per-second setting */
int Cdp::cps (){
	return pref_cps;
}

/** Return the startup-Mission */
QString Cdp::startupMission (){
	if (pref_useLastMission) return pref_LastMission;
	if (pref_useDefaultMission) return pref_DefaultMission;
	return pref_SelectedMission;
}

/** Return the default mission path */
QString Cdp::defaultPath (){
	return data_prefix;
}

/** Signals, that the mission mission has just been opened (and therefore is the one
last chosen) */
void Cdp::setLastLoaded (QString &mission){
	pref_LastMission = mission;
	set_new_config ();
}

/** Saves the current config (and emits a config_changed ()) */
void Cdp::set_new_config (){

	tp->config_changed ();

// write settings
	KConfig *config = kapp->config();

	if( config ) {
		config->setGroup( "Paths" );
		config->writeEntry ( "defaultMissionDir", data_prefix );
		config->writeEntry ( "useDefaultMission", pref_useDefaultMission );
		config->writeEntry ( "useLastMission", pref_useLastMission );
		config->writeEntry ( "useSelectedMission", pref_useSelectedMission );
		config->writeEntry ( "DefaultMission", pref_DefaultMission );
		config->writeEntry ( "LastMission", pref_LastMission );
		config->writeEntry ( "SelectedMission", pref_SelectedMission );

		config->setGroup ( "Tuning" );
		config->writeEntry( "fps", pref_fps );
		config->writeEntry( "cps", pref_cps );

		config->sync();
        }

}

/** Like ok, but does not yet give feedback to the user, if errors occured. Used in
constructors to prevent crashes, while still making it possible to continue parsing. */
bool Cdp::isOk (){
	return (!errors_occurred);
}

/** Checks, whether the given element is really of type/tagName "name" and returns false otherwise
+ outputs error-message at level level */
bool Cdp::verify_tag (QString const &name, QDomElement const & element, int level) {
	if (element.tagName () == name) {
		return true;
	}
	debug_msg (i18n ("Tag of type '") + name + i18n ("' found, but '") + element.tagName () + i18n ("' expected."), level, &element);
	return false;
}