/***************************************************************************
                          level.cpp  -  description
                             -------------------
    begin                : Mon May 21 2001
    copyright            : (C) 2001 by Thomas Friedrichsmeier
    email                : Thomas.Friedrichsmeier@ruhr-uni-bochum.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <kglobal.h>
#include <klocale.h>
#include <kstddirs.h>

#include <qpoint.h>
#include <qdom.h>
#include <qpen.h>
#include <qlist.h>

#include "level.h"

#include "taxipilot.h"
#include "taxipilotapp.h"
#include "cdp.h"
#include "movingobject.h"
#include "teleporter.h"
#include "centerofgravity.h"
#include "platform.h"

#include "defines.h"

Level::Level(const QString & file, QCanvas * canvas, Taxipilot * parent):QCanvasSprite(0, canvas)
{

	setAnimated(false);			// level never moves

	tp = parent;
	cdp = tp->cdp;
	p_canvas = canvas;

	if (cdp->for_real()) {
		draw_platform_lines = cdp->draw_platform_lines ();
	} else {
		draw_platform_lines = false;
	}

	read_level(file);

	if (cdp->for_real()) {
		canvas->setBackgroundPixmap(*background);
		canvas->resize(width(), height());
		parent->app->setCaption(caption);

		config_changed ();
		setZ(LEVEL_Z);
		show();
	}
}

Level::~Level()
{
	if (num_platforms) {
		for (int i = 0; i < num_platforms; i++) {
			delete platform[i];
		}
		delete[]platform;
	}

	if (num_teleporters) {
		for (int i = 0; i < num_teleporters; i++) {		
			delete teleporters[i];			
		}
		delete[]teleporters;
	}

	if (num_grav_centers) {
		for (int i = 0; i < num_grav_centers; i++) {		
			delete grav_center[i];			
		}
		delete[]grav_center;
	}

	if (num_objects) {
		for (int i = 0; i < num_objects; i++) {
			delete objects[i];
		}
		delete[]objects;
	}

	if (cdp->for_real () || cdp->thorough_check ()) {
		hide ();
		delete image;
		canvas()->setBackgroundColor (QColor (0, 0, 0));
	}	

}

/** I consider this function to be mostly straight-forward. I do however know close to nothing about XML
(definitely no more than was neccessary to write up this code. Go ahead and change it to make more sense! */
void Level::read_level(const QString & file)
{

	QString convert;			// used to convert numbers
	QDomElement e;				// just some element
	QDomNode n;					// just some node
	QDomNodeList list;			// just a list of nodes

	QList <Promise> promises;
	QList <Promise> definitions;
	promises.setAutoDelete (true);
	definitions.setAutoDelete (true);

	cdp->init_debug(file, true);

	QDomElement xdocElement = cdp->open_config_file(file);
	caption = cdp->get_string_attribute("caption", xdocElement, i18n("No Title"), 2);
	author = cdp->get_string_attribute("author", xdocElement, i18n("Anonymous"), 3);

	if (cdp->for_real() || cdp->thorough_check()) {		// all the rest not needed for credits

		e = cdp->get_element("completion", xdocElement, 1);	// get completion tag
		deliveries_to_go = cdp->get_int_attribute("deliveries", e, 1, 9999, 3, 1);
		extra_life = cdp->get_bool_attribute("extra_life", e, false, 2);

		e = cdp->get_element("image", xdocElement, 0);	// get image-section
		background =
			new QCanvasPixmap(cdp->prefix_check(cdp->get_string_attribute("background", e, "testbackgr2.png", 1), true));
		image = new QCanvasPixmapArray(cdp->prefix_check(cdp->get_string_attribute("level", e, "testlevel2.png", 1), true));

		e = cdp->get_element("platforms", xdocElement, 0);	// get platforms section
		list = cdp->get_node_list("data", e, 2, -1, 2, 0);	// get all platform data ( at least two platforms! )

		exp_fuel_platform = -1;
		num_platforms = 0;
		platform = new Platform * [list.length()];
		n = list.item(0);

		while (!n.isNull()) {		// get all data for this platform
			e = n.toElement();

			platform[num_platforms] = new Platform (&e, this, draw_platform_lines);		

			if (platform[num_platforms]->fuel_step) {
				if (exp_fuel_platform < 0) {
					exp_fuel_platform = num_platforms;
				} else if (platform[exp_fuel_platform]->fuel_price < platform[num_platforms]->fuel_price) {
					exp_fuel_platform = num_platforms;
				}
			}

			if (platform[num_platforms]->object_promised) {
				promise_pointer = new Promise;
				promise_pointer->ident = platform[num_platforms]->object_ident;
				promise_pointer->plat_point = platform[num_platforms];
				promise_pointer->type = 1;
				promises.append (promise_pointer);
			}

			n = n.nextSibling();
			num_platforms++;
		}

		if (exp_fuel_platform < 0) {
			cdp->debug_msg(i18n("No fuel-platform specified. Assuming one on Platform 1"), 2);
			platform[0]->fuel_step = 1;
			platform[0]->fuel_price = 0.1;
			exp_fuel_platform = 0;
		}

		e = cdp->get_element("teleporters", xdocElement, 3);	// now get the Teleporters
		num_teleporters = 0;
		int i =0;

		if (!(e.isNull())) {		// did we really find a "teleporters"-element?
       	
			list = cdp->get_node_list("data", e, 2, -1, 2, 1);	// get all teleporter data ( at least two if any! )

			teleporters = new Teleporter* [list.length()];
			num_teleporters = list.length();
			n = list.item(0);

			while (!n.isNull()) {	// get all data for this teleporter
				e = n.toElement();

				teleporters[i] = new Teleporter (&e, this, canvas());

				if (teleporters[i]->object_promised) {
					promise_pointer = new Promise;
					promise_pointer->ident = teleporters[i]->object_ident;
					promise_pointer->tp_point = teleporters[i];
					promise_pointer->type = 0;
					promises.append (promise_pointer);
				}

				n = n.nextSibling();
				i++;
			}
		}


		e = cdp->get_element("objects", xdocElement, 3);	// now get the MovingObjects
		num_objects = 0;

		if (!(e.isNull())) {		// did we really find an objects-section?

			list = cdp->get_node_list("object", e, 1, -1, 1, 2);	// get each single object in the section

			objects = new MovingObject * [list.length()];
			n = list.item(0);

			while (!n.isNull()) {	// create this object
				e = n.toElement ();

				objects[num_objects] = new MovingObject ( &e, tp, p_canvas );

				if (objects[num_objects]->has_ident) {
					promise_pointer = new Promise;
					promise_pointer->ident = objects[num_objects]->ident;
					promise_pointer->obj_point = objects[num_objects];
					definitions.append (promise_pointer);
				}

				n = n.nextSibling();
				num_objects++;
			}
		}


		e = cdp->get_element("gravity", xdocElement, 1);	// now get the gravity-section
		n_base_gravity = cdp->get_double_attribute("base_gravity", e, -5, 5, 0.01, 2);
		n_horiz_gravity = cdp->get_double_attribute("horiz_gravity", e, -5, 5, 0, 2);

		list = cdp->get_node_list("data", e, 0, -1, 0, 3);	// now get the Centers of Gravity
		num_grav_centers = 0;

		grav_center = new CenterOfGravity * [list.length()];
		n = list.item(0);

		while (!n.isNull()) {		// get all data for this center
			e = n.toElement();

			grav_center[num_grav_centers] = new CenterOfGravity (&e, this);

			if (grav_center[num_grav_centers]->object_promised) {
				promise_pointer = new Promise;
				promise_pointer->ident = grav_center[num_grav_centers]->object_ident;
				promise_pointer->cog_point = grav_center[num_grav_centers];
				promise_pointer->type = 2;
				promises.append (promise_pointer);
			}

			n = n.nextSibling();
			num_grav_centers++;
		}

		e = cdp->get_element("tips", xdocElement, 2);	// find the tips-tag
		min_base_tip = cdp->get_double_attribute("min", e, 0, 2000, 10, 2);
		max_base_tip = cdp->get_double_attribute("max", e, 0, 2000, 30, 2);

		e = cdp->get_element("fare", xdocElement, 2);	// find the fare-tag
		base_fare = cdp->get_double_attribute("base", e, 0, 200, 4.5, 2);
		fare_step = cdp->get_double_attribute("step", e, 0, 1, 0.01, 2);

		e = cdp->get_element("initial_position", xdocElement, 1);	// find the initial_position-tag
		initial_x = cdp->get_int_attribute("x", e, 0, 3000, 100, 1);
		initial_y = cdp->get_int_attribute("y", e, 0, 3000, 100, 1);

		setSequence(image);			// need this (width, height) before we can take care of oversized levels

		e = cdp->get_element("oversize", xdocElement, 3);
		if (!(e.isNull())) {
			visible_width = cdp->get_int_attribute("visible_width", e, 200, width(), width(), 2);
			visible_height = cdp->get_int_attribute("visible_height", e, 200, height(), height(), 2);
			max_offset_x = width() - visible_width;
			max_offset_y = height() - visible_height;
			initial_offset_x = cdp->get_int_attribute("initial_offset_x", e, 0, max_offset_x, 0, 3);
			initial_offset_y = cdp->get_int_attribute("initial_offset_y", e, 0, max_offset_y, 0, 3);
			margin = cdp->get_int_attribute("margin", e, 0, visible_width / 2, 50, 3);
		} else {					// this else of course has the same effect as the if, when the tag is not supplied.
			visible_width = width();	// it's only separate to suppress error-messages if the (optional) tag is not supplied.
			visible_height = height();
			max_offset_x = max_offset_y = initial_offset_x = initial_offset_y = margin = 0;
		}

	// Finally, check whether all promises have been kept / assign promised objects
	
		promise_pointer = promises.first ();

		while (promise_pointer) {
			definitions.first();
			bool found = false;
			while (definitions.current() && !found) {
				if (definitions.current()->ident == promise_pointer->ident) {
					switch (promise_pointer->type) {
						case 0:	if (definitions.current ()->obj_point->teleporter) {
									found = true;
									promise_pointer->tp_point->promise_fulfilled (definitions.current()->obj_point);
									definitions.remove ();
							        }
								break;
						case 1:	if (definitions.current ()->obj_point->platform) {
									found = true;
									promise_pointer->plat_point->promise_fulfilled (definitions.current()->obj_point);
									definitions.remove ();
							        }
								break;
						case 2:	if (definitions.current ()->obj_point->center) {
									found = true;
									promise_pointer->cog_point->promise_fulfilled (definitions.current()->obj_point);
									definitions.remove ();
								}
								break;
						default:	break;
					}
				}			
				definitions.next();
			}
       	
			if (found) {
				promises.remove ();
				promise_pointer = promises.current();
			} else {
				QString type_str;
				switch (promise_pointer->type) {
					case 0:	type_str = "Teleporter";
							break;
					case 1:	type_str = "Platform";
							break;
					case 2:	type_str = "Center-of-Gravity";
							break;
					default:	type_str = "Bad type";
							break;
				}
				cdp->debug_msg ("Promised object of type '" + type_str + "', named '" + promise_pointer->ident +
								"' was not defined.", 0);
				promises.next ();
			}
		}	
	
		promise_pointer = definitions.first ();
		while (promise_pointer) {
			cdp->debug_msg ("Object named '" + promise_pointer->ident + "' was defined but never promised.", 2);
			promise_pointer = definitions.next ();
		}

		 cdp->debug_msg ("Finished loading level", 3);
	} else {
		num_teleporters = num_grav_centers = num_platforms = num_objects = 0;
	}
}

/** No descriptions */
void Level::config_changed(){
	base_gravity = n_base_gravity * 2000.0 / ((double) cdp->cps () * (double) cdp->fps ());
	horiz_gravity = n_horiz_gravity * 2000.0 / ((double) cdp->cps () * (double) cdp->fps ());

	for (int i = 0; i < num_grav_centers; i++) {
		grav_center[i]->config_changed ();
	}
}
