/***************************************************************************
                          centerofgravity.cpp  -  description
                             -------------------
    begin                : Sun Nov 4 2001
    copyright            : (C) 2001 by Thomas Friedrichsmeier
    email                : Thomas.Friedrichsmeier@ruhr-uni-bochum.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "centerofgravity.h"

#include <qstring.h>
#include <qdom.h>

#include <math.h>

#include <klocale.h>

#include "level.h"
#include "cdp.h"
#include "movingobject.h"

CenterOfGravity::CenterOfGravity(const QDomElement *description, Level *parent){

	cdp = parent->cdp;

	has_object = false;

	read_center (description);

	config_changed ();
}

CenterOfGravity::~CenterOfGravity(){
}

/** Takes the current position and speed of the taxi as arguments, changes the
velocity (vx, vy) - vars to the new value (which will be set by Taxipilot, when
all Centers have been calculated). */
void CenterOfGravity::calculate_gravity (int const taxi_pos_x, int const taxi_pos_y, double *vx, double *vy){

	int dx, dy;	

	if (!has_object) {
		dx = static_x - taxi_pos_x;
		dy = static_y - taxi_pos_y;
	} else {
		dx = (int) object->x() + center_off_x - taxi_pos_x;
		dy = (int) object->y() + center_off_y - taxi_pos_y;		
	}

	double dxv, dyv;	// changes in x/y-Velocity
	double force;

	if (dx || dy) { 	// do not divide by zero
		if (potential == 0) {
			force = gravity;
		} else if (potential == 4) {
			force = (gravity / ((dx * dx + dy * dy) * (dx * dx + dy * dy)));
		} else if (potential == 1) {
			force = (gravity / (sqrt(dx * dx + dy * dy)));
		} else {			// default (2)
			force = (gravity / (dx * dx + dy * dy));
		}
       }

// experimental: this is to lower the risk of erratic acceleration. force == 1 means instant speed-up to 100 pixels per second!
	if (force > force_limit) {
		force = force_limit;
	} else if (force < -force_limit) {
		force = -force_limit;
	}

	if (dx == 0) {		// this is to avoid division by zero
		dxv = 0;
		if (dy == 0) {
			// we're right on the center of gravity. do nothing
			dyv = 0;
		} else {
			dyv = fabs(force);	// we need to take care of signs below, anyway.
		}
	} else if (dy == 0) {
		dyv = 0;
		if (dx == 0) {
			dxv = 0;
		} else {
			dxv = fabs(force);
		}
	} else {			// not in danger of dividing by zero
		dxv = sqrt((force * force) / (1 + (((double) (dy * dy)) / ((double) (dx * dx)))));	// can this be done without sqrt ()?
		dyv = sqrt((force * force) - (dxv * dxv));	// having a block on maths, currently
	}
		if ((force < 0)) {	// force is negative, meaning pushing away
		if (dx > 0) {
			dxv = dxv * -1;
		}
		if (dy > 0) {
			dyv = dyv * -1;
		}
	} else {			// force is positive, meaning attraction
		if (dx < 0) {
			dxv = dxv * -1;
		}
		if (dy < 0) {
			dyv = dyv * -1;
		}
	}

	*vx += dxv;
	*vy += dyv;

}

/** Reads in the description for this center of gravity */
void CenterOfGravity::read_center (const QDomElement *description){

	if (description->hasAttribute ("ident")) {
		object_ident = cdp->get_string_attribute ("ident", *description, "#bad value#", 1);
		object_promised = true;
	} else {
		object_promised = false;		
		static_x = cdp->get_int_attribute("x", *description, 0, 3000, 100, 1);
		static_y = cdp->get_int_attribute("y", *description, 0, 3000, 100, 1);
	}

	n_gravity = cdp->get_double_attribute("gravity", *description, -10000, 10000, 10, 1);
	potential = cdp->get_int_attribute("potential", *description, 0, 4, 2, 1);

	if (potential == 3) {	// not allowed
		cdp->debug_msg(i18n("Center of Gravity potential=3 is not allowed. Changing to 2"), 1);
	}

}

/** Fulfills the promise (assigns the promised object to this center) */
void CenterOfGravity::promise_fulfilled(MovingObject * obj){
	object = obj;
	has_object = true;

	center_off_x = object->boundingRect ().width () / 2;
	center_off_y = object->boundingRect ().height () / 2;
}

/** No descriptions */
void CenterOfGravity::config_changed (){
	double x = (double) cdp->fps () * (double) cdp->cps ();
	gravity = n_gravity * (2000.0 / x);
	force_limit = 1000.0 / x;
}
