/*
 * Sylpheed -- a GTK+ based, lightweight, and fast e-mail client
 * Copyright (C) 1999-2002 Hiroyuki Yamamoto
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include "defs.h"

#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "intl.h"
#include "folder.h"
#include "prefs.h"
#include "prefs_folder_item.h"
#include "manage_window.h"
#include "folderview.h"
#include "inc.h"
#include "menu.h"

typedef struct _PrefsFolderItemDialog	PrefsFolderItemDialog;

struct _PrefsFolderItemDialog
{
	PrefsDialog *dialog;
	FolderItem *item;

	GtkWidget *name_entry;
	GtkWidget *type_optmenu;
};

static PrefsFolderItemDialog *prefs_folder_item_create
					(FolderItem		*item);
static void prefs_folder_item_general_create
					(PrefsFolderItemDialog	*dialog);
static void prefs_folder_item_set_dialog(PrefsFolderItemDialog	*dialog);

static void prefs_folder_item_ok_cb	(GtkWidget		*widget, 
					 PrefsFolderItemDialog	*dialog);
static void prefs_folder_item_apply_cb	(GtkWidget		*widget, 
					 PrefsFolderItemDialog	*dialog);
static void prefs_folder_item_cancel_cb	(GtkWidget		*widget, 
					 PrefsFolderItemDialog	*dialog);
static gint prefs_folder_item_delete_cb	(GtkWidget		*widget,
					 GdkEventAny		*event, 
					 PrefsFolderItemDialog	*dialog);
static void prefs_folder_item_key_press_cb
					(GtkWidget		*widget,
					 GdkEventKey		*event,
					 PrefsFolderItemDialog	*dialog);

void prefs_folder_item_open(FolderItem *item)
{
	PrefsFolderItemDialog *dialog;

	g_return_if_fail(item != NULL);

	inc_lock();

	dialog = prefs_folder_item_create(item);

	manage_window_set_transient(GTK_WINDOW(dialog->dialog->window));

	prefs_folder_item_set_dialog(dialog);

	gtk_widget_show_all(dialog->dialog->window);
}

PrefsFolderItemDialog *prefs_folder_item_create(FolderItem *item)
{
	PrefsFolderItemDialog *new_dialog;
	PrefsDialog *dialog;

	new_dialog = g_new0(PrefsFolderItemDialog, 1);

	dialog = g_new0(PrefsDialog, 1);
	prefs_dialog_create(dialog);

	gtk_window_set_title(GTK_WINDOW(dialog->window), _("Folder Property"));
	gtk_signal_connect(GTK_OBJECT(dialog->window), "delete_event",
			   GTK_SIGNAL_FUNC(prefs_folder_item_delete_cb),
			   new_dialog);
	gtk_signal_connect(GTK_OBJECT(dialog->window), "key_press_event",
			   GTK_SIGNAL_FUNC(prefs_folder_item_key_press_cb),
			   new_dialog);
	MANAGE_WINDOW_SIGNALS_CONNECT(dialog->window);

	gtk_signal_connect(GTK_OBJECT(dialog->ok_btn), "clicked",
			   GTK_SIGNAL_FUNC(prefs_folder_item_ok_cb),
			   new_dialog);
	gtk_signal_connect(GTK_OBJECT(dialog->apply_btn), "clicked",
			   GTK_SIGNAL_FUNC(prefs_folder_item_apply_cb),
			   new_dialog);
	gtk_signal_connect(GTK_OBJECT(dialog->cancel_btn), "clicked",
			   GTK_SIGNAL_FUNC(prefs_folder_item_cancel_cb),
			   new_dialog);

	new_dialog->dialog = dialog;
	new_dialog->item = item;

	prefs_folder_item_general_create(new_dialog);

	SET_NOTEBOOK_LABEL(dialog->notebook, _("General"), 0);

	return new_dialog;
}

static void prefs_folder_item_general_create(PrefsFolderItemDialog *dialog)
{
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *entry;
	GtkWidget *optmenu;
	GtkWidget *optmenu_menu;
	GtkWidget *menuitem;

	vbox = gtk_vbox_new(FALSE, VSPACING);
	gtk_container_add(GTK_CONTAINER(dialog->dialog->notebook), vbox);
	gtk_container_set_border_width(GTK_CONTAINER (vbox), VBOX_BORDER);

	hbox = gtk_hbox_new(FALSE, 8);
	gtk_box_pack_start(GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	label = gtk_label_new(_("Name"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

	entry = gtk_entry_new();
	gtk_entry_set_editable(GTK_ENTRY(entry), FALSE);
	gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);

	hbox = gtk_hbox_new(FALSE, 8);
	gtk_box_pack_start(GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	label = gtk_label_new(_("Type"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

	optmenu = gtk_option_menu_new();
	gtk_box_pack_start(GTK_BOX(hbox), optmenu, FALSE, FALSE, 0);

	optmenu_menu = gtk_menu_new();

	MENUITEM_ADD(optmenu_menu, menuitem, _("Normal"), F_NORMAL);
	MENUITEM_ADD(optmenu_menu, menuitem, _("Inbox") , F_INBOX);
	MENUITEM_ADD(optmenu_menu, menuitem, _("Sent")  , F_OUTBOX);
	MENUITEM_ADD(optmenu_menu, menuitem, _("Drafts"), F_DRAFT);
	MENUITEM_ADD(optmenu_menu, menuitem, _("Queue") , F_QUEUE);
	MENUITEM_ADD(optmenu_menu, menuitem, _("Trash") , F_TRASH);

	gtk_option_menu_set_menu(GTK_OPTION_MENU(optmenu), optmenu_menu);

	dialog->name_entry = entry;
	dialog->type_optmenu = optmenu;
}

static void prefs_folder_item_set_dialog(PrefsFolderItemDialog *dialog)
{
	GtkWidget *menu;
	GtkWidget *menuitem;
	GtkOptionMenu *optmenu = GTK_OPTION_MENU(dialog->type_optmenu);
	GList *cur;
	SpecialFolderItemType type;
	gint n;
	guint index = 0;

	gtk_entry_set_text(GTK_ENTRY(dialog->name_entry), dialog->item->name);

	menu = gtk_option_menu_get_menu(optmenu);
	for (cur = GTK_MENU_SHELL(menu)->children, n = 0;
	     cur != NULL; cur = cur->next, n++) {
		menuitem = GTK_WIDGET(cur->data);
		type = (SpecialFolderItemType)
			gtk_object_get_user_data(GTK_OBJECT(menuitem));
		if (type != F_NORMAL &&
		    FOLDER_TYPE(dialog->item->folder) == F_NEWS)
			gtk_widget_set_sensitive(menuitem, FALSE);
		if (dialog->item->stype == type)
			index = n;
	}

	gtk_option_menu_set_history(optmenu, index);
}

void prefs_folder_item_destroy(PrefsFolderItemDialog *dialog) 
{
	prefs_dialog_destroy(dialog->dialog);
	g_free(dialog->dialog);
	g_free(dialog);

	inc_unlock();
}

static void prefs_folder_item_ok_cb(GtkWidget *widget,
				    PrefsFolderItemDialog *dialog)
{
	prefs_folder_item_apply_cb(widget, dialog);
	prefs_folder_item_destroy(dialog);
}

static void prefs_folder_item_apply_cb(GtkWidget *widget,
				       PrefsFolderItemDialog *dialog)
{
	GtkWidget *menu;
	GtkWidget *menuitem;
	GtkOptionMenu *optmenu = GTK_OPTION_MENU(dialog->type_optmenu);
	SpecialFolderItemType type;
	FolderItem *item = dialog->item;
	Folder *folder = item->folder;
	FolderItem *prev_item = NULL;

	menu = gtk_option_menu_get_menu(optmenu);
	menuitem = gtk_menu_get_active(GTK_MENU(menu));
	type = (SpecialFolderItemType)
		gtk_object_get_user_data(GTK_OBJECT(menuitem));

	if (dialog->item->stype == type)
		return;

	switch (type) {
	case F_NORMAL:
		break;
	case F_INBOX:
		if (folder->inbox)
			folder->inbox->stype = F_NORMAL;
		prev_item = folder->inbox;
		folder->inbox = item;
		break;
	case F_OUTBOX:
		if (folder->outbox)
			folder->outbox->stype = F_NORMAL;
		prev_item = folder->outbox;
		folder->outbox = item;
		break;
	case F_DRAFT:
		if (folder->draft)
			folder->draft->stype = F_NORMAL;
		prev_item = folder->draft;
		folder->draft = item;
		break;
	case F_QUEUE:
		if (folder->queue)
			folder->queue->stype = F_NORMAL;
		prev_item = folder->queue;
		folder->queue = item;
		break;
	case F_TRASH:
		if (folder->trash)
			folder->trash->stype = F_NORMAL;
		prev_item = folder->trash;
		folder->trash = item;
		break;
	}

	item->stype = type;

	if (prev_item)
		folderview_update_item(prev_item, FALSE);
	folderview_update_item(item, FALSE);
}

static void prefs_folder_item_cancel_cb(GtkWidget *widget,
					PrefsFolderItemDialog *dialog)
{
	prefs_folder_item_destroy(dialog);
}

static gint prefs_folder_item_delete_cb(GtkWidget *widget, GdkEventAny *event,
					PrefsFolderItemDialog *dialog)
{
	prefs_folder_item_destroy(dialog);
	return TRUE;
}

static void prefs_folder_item_key_press_cb(GtkWidget *widget,
					   GdkEventKey *event,
					   PrefsFolderItemDialog *dialog)
{
	if (event && event->keyval == GDK_Escape)
		prefs_folder_item_cancel_cb(widget, dialog);
}
