/*	$OpenBSD: script.c,v 1.23 2005/04/11 19:59:07 deraadt Exp $	*/
/*	$NetBSD: script.c,v 1.3 1994/12/21 08:55:43 jtc Exp $	*/

/*
 * Copyright (c) 2001 Theo de Raadt
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Copyright (c) 1980, 1992, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/time.h>

#include <errno.h>
#include <fcntl.h>
#include <paths.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <tzfile.h>
#include <unistd.h>

#include <util.h>
#include <err.h>

int	master, slave;
volatile sig_atomic_t child;
pid_t	subchild;
char	*fname = NULL;

volatile sig_atomic_t dead;
volatile sig_atomic_t sigdeadstatus;
volatile sig_atomic_t flush;

struct	termios tt;

__dead void done(int);
void dooutput(const char *, int, int);
void doshell(char **);
void fail(void);
void finish(int);
void scriptflush(int);
void handlesigwinch(int);
void close_output(FILE *, int);
FILE *open_output(const char *, int);
int silent = 0;

#define OUTPUT_WRITE 0
#define OUTPUT_APPEND 1
#define OUTPUT_PIPE 2

int
main(int argc, char *argv[])
{
	extern char *__progname;
	struct sigaction sa;
	struct termios rtt;
	struct winsize win;
	char ibuf[BUFSIZ];
	ssize_t cc, off;
	int oflg, ch, cflg;

	oflg = OUTPUT_WRITE;
	cflg = 0;
	silent = 0;
	while ((ch = getopt(argc, argv, "acp:s")) != -1)
		switch(ch) {
		case 'a':
			oflg = OUTPUT_APPEND;
			break;
		case 'c':
			cflg = 1;
			break;
		case 'p':
			oflg = OUTPUT_PIPE;
			fname = optarg;
			break;
		case 's':
			silent = 1;
			break;
		default:
			fprintf(stderr, "usage: %s [-as] [-p command] [file|-c command...]\n", __progname);
			exit(1);
		}
	argc -= optind;
	argv += optind;

	if (!fname) {
		if (cflg == 0 && argc > 0)
			fname = argv[0];
		else {
			fname = "typescript";
		}
	}

	(void)tcgetattr(STDIN_FILENO, &tt);
	(void)ioctl(STDIN_FILENO, TIOCGWINSZ, &win);
	if (openpty(&master, &slave, NULL, &tt, &win) == -1)
		err(1, "openpty");

	if (!silent) {
		switch(oflg) {
		case OUTPUT_WRITE:
		case OUTPUT_APPEND:
			(void)printf("Script started, output file is %s\n", fname);
			break;
		case OUTPUT_PIPE:
			(void)printf("Script started, output piped to %s\n", fname);
			break;
		}
	}
	rtt = tt;
	cfmakeraw(&rtt);
	rtt.c_lflag &= ~ECHO;
	(void)tcsetattr(STDIN_FILENO, TCSAFLUSH, &rtt);

	bzero(&sa, sizeof sa);
	sigemptyset(&sa.sa_mask);
	sa.sa_handler = finish;
	(void)sigaction(SIGCHLD, &sa, NULL);

	sa.sa_handler = handlesigwinch;
	sa.sa_flags = SA_RESTART;
	(void)sigaction(SIGWINCH, &sa, NULL);

	child = fork();
	if (child < 0) {
		warn("fork");
		fail();
	}
	if (child == 0) {
		subchild = child = fork();
		if (child < 0) {
			warn("fork");
			fail();
		}
		if (child)
			dooutput(fname, oflg, silent);
		else
			doshell(cflg ? argv : NULL);
	}

	while (1) {
		if (dead)
			break;
		cc = read(STDIN_FILENO, ibuf, BUFSIZ);
		if (cc == -1 && errno == EINTR)
			continue;
		if (cc <= 0)
			break;
		for (off = 0; off < cc; ) {
			ssize_t n = write(master, ibuf + off, cc - off);
			if (n == 0)
				break;	/* skip writing */
			if (n > 0)
				off += n;
		}
	}
	done(sigdeadstatus);
}

/* ARGSUSED */
void
finish(int signo)
{
	int save_errno = errno;
	int status, e = 1;
	pid_t pid;

	while ((pid = wait3(&status, WNOHANG, 0)) > 0) {
		if (pid == (pid_t)child) {
			if (WIFEXITED(status))
				e = WEXITSTATUS(status);
		}
	}
	dead = 1;
	sigdeadstatus = e;
	errno = save_errno;
}

/* ARGSUSED */
void
handlesigwinch(int signo)
{
	int save_errno = errno;
	struct winsize win;
	pid_t pgrp;

	if (ioctl(STDIN_FILENO, TIOCGWINSZ, &win) != -1) {
		ioctl(slave, TIOCSWINSZ, &win);
		if (ioctl(slave, TIOCGPGRP, &pgrp) != -1)
			killpg(pgrp, SIGWINCH);
	}
	errno = save_errno;
}

FILE *
open_output(const char *name, int flg)
{
	switch (flg) {
	case OUTPUT_WRITE:
		return fopen(name, "w");
	case OUTPUT_APPEND:
		return fopen(name, "a");
	case OUTPUT_PIPE:
		return popen(name, "w");
	}
}

void
close_output(FILE *f, int flg)
{
	switch(flg) {
	case OUTPUT_WRITE:
	case OUTPUT_APPEND:
		(void)fclose(f);
		break;
	case OUTPUT_PIPE:
		(void)pclose(f);
		break;
	}
}

void
dooutput(const char *fname, int oflg, int silent)
{
	struct sigaction sa;
	struct itimerval value;
	sigset_t blkalrm;
	char obuf[BUFSIZ];
	time_t tvec;
	ssize_t outcc = 0, cc, off;
	FILE *fscript;

	if ((fscript = open_output(fname, oflg)) == NULL) {
		int save_errno = errno;
		(void)tcsetattr(STDIN_FILENO, TCSAFLUSH, &tt);
		errno = save_errno;
		err(1, "%s", fname);
	}

	(void)close(STDIN_FILENO);

	if (!silent) {
		tvec = time(NULL);
		(void)fprintf(fscript, "Script started on %s", ctime(&tvec));
	}

	sigemptyset(&blkalrm);
	sigaddset(&blkalrm, SIGALRM);
	bzero(&sa, sizeof sa);
	sigemptyset(&sa.sa_mask);
	sa.sa_handler = scriptflush;
	(void)sigaction(SIGALRM, &sa, NULL);

	value.it_interval.tv_sec = SECSPERMIN / 2;
	value.it_interval.tv_usec = 0;
	value.it_value = value.it_interval;
	(void)setitimer(ITIMER_REAL, &value, NULL);
	for (;;) {
		if (flush) {
			if (outcc) {
				(void)fflush(fscript);
				outcc = 0;
			}
			flush = 0;
		}
		cc = read(master, obuf, sizeof (obuf));
		if (cc == -1 && errno == EINTR)
			continue;
		if (cc <= 0)
			break;
		sigprocmask(SIG_BLOCK, &blkalrm, NULL);
		for (off = 0; off < cc; ) {
			ssize_t n = write(STDOUT_FILENO, obuf + off, cc - off);
			if (n == 0)
				break;	/* skip writing */
			if (n > 0)
				off += n;
		}
		(void)fwrite(obuf, 1, cc, fscript);
		outcc += cc;
		sigprocmask(SIG_UNBLOCK, &blkalrm, NULL);
	}
	if (!silent) {
		tvec = time(NULL);
		(void)fprintf(fscript,"\nScript done on %s", ctime(&tvec));
	}
	(void)close_output(fscript, oflg);
	done(0);
}

/* ARGSUSED */
void
scriptflush(int signo)
{
	flush = 1;
}

void
doshell(char **argv)
{
	(void)close(master);
	login_tty(slave);
	if (argv) {
		setsid();
		execvp(argv[0], argv);
		warn("%s", argv[0]);
	} else {
		char *shell;

		shell = getenv("SHELL");
		if (shell == NULL)
			shell = _PATH_BSHELL;

		execl(shell, shell, "-i", (char *)NULL);
		warn("%s", shell);
	}
	fail();
}

void
fail(void)
{

	(void)kill(0, SIGTERM);
	done(1);
}

void
done(int eval)
{
	if (subchild) {
		(void)close(master);
	} else {
		(void)tcsetattr(STDIN_FILENO, TCSAFLUSH, &tt);
		if (!silent)
			(void)printf("Script done, output file is %s\n", fname);
	}
	exit(eval);
}
