/*	$Id: sudoku.c,v 1.11 2007/01/24 13:46:55 mbalmer Exp $ */

/*
 * Copyright (c) 2006, 2007 Marc Balmer <mbalmer@openbsd.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <ctype.h>
#include <err.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

int verbose;

int complete(char f[10][9][9]);
void show(char f[10][9][9], int num);

int
set(char f[10][9][9], int x, int y, char num)
{
	int x2, y2, x3, y3, num2;
	if (f[(int)num][x][y])
		return -1;

	f[0][x][y] = num;
	for (x2 = 0; x2 < 9; x2++)
		f[(int)num][x2][y] = 1;
	for (y2 = 0; y2 < 9; y2++)
		f[(int)num][x][y2] = 1;
	for (num2 = 1; num2 < 10; num2++)
		f[num2][x][y] = 1;

	if (x > 5)
		x2 = 6;
	else if (x > 2)
		x2 = 3;
	else
		x2 = 0;

	if (y > 5)
		y2 = 6;
	else if (y > 2)
		y2 = 3;
	else
		y2 = 0;

	for (x3 = x2; x3 < x2 + 3; x3++)
		for (y3 = y2; y3 < y2 + 3; y3++)
			f[(int)num][x3][y3] = 1;

	return 0;
}

/* find the number with lowest number of possibilities */
int
findnum(char f[10][9][9], int *nx, int *ny)
{
	int num, count, n, nn;
	int x, y, lx = 0, ly = 0;

	count = 9 * 9 + 1;
	nn = 0;
	for (num = 1; num < 10; num++) {
		n = 0;
		for (x = 0; x < 9; x++) {
			for (y = 0; y < 9; y++) {
				if (!f[num][x][y]) {
					++n;
					lx = x;
					ly = y;
				}
			}
		}
		if (n && n < count) {
			count = n;
			nn = num;
			*nx = lx;
			*ny = ly;
		}
	}
	return nn;
}

int
find(char f[10][9][9])
{
	int x, y, fx, fy, lx, ly;
	int count, tot, last;
	char num;

	tot = 0;

	if (verbose > 1)
		printf("checking for fields that allow only one number\n");

	for (x = 0; x < 9; x++) {
		for (y = 0; y < 9; y++) { 
			for (num = 1, last = 1, count = 0; num < 10; num++) {
				if (!f[(int)num][x][y]) {
					last = num;
					++count;
				}
			}
			if (count == 1) {
				if (verbose > 1)
					printf("%d at %d %d\n", last, x + 1,
					    y + 1);
				set(f, x, y, last);
				++tot;
			}
		}
	}

	if (verbose > 1)
		printf("checking for fields with one place left\n");

	for (num = 1; num < 10; num++) {
		for (fx = 0; fx < 3; fx++) {
			for (fy = 0; fy < 3; fy++) {
				count = lx = ly = 0;
				for (x = 0; x < 3; x++) {
					for (y = 0; y < 3; y++) {
						if (!f[(int)num][fx*3+x][fy*3+y]) {
							lx = fx*3+x;
							ly = fy*3+y;
							++count;
						}
					}
				}
				if (count == 1) {
					if (verbose > 1)
						printf("%d at %d %d\n", num,
						    lx + 1, ly + 1);
					set(f, lx, ly, num);
					++tot;
				}
			}
		}
	}

	if (verbose > 1)
		printf("checking for columns with one place left\n");

	for (num = 1; num < 10; num++) {
		for (x = 0; x < 9; x++) {
			count = ly = 0;
			for (y = 0; y < 9; y++) {
				if (!f[(int)num][x][y]) {
					ly = y;
					++count;
				}
			}
			if (count == 1) {
				if (verbose > 1)
					printf("%d at %d %d\n", num, x + 1,
					    ly + 1);
				set(f, x, ly, num);
				++tot;
			}
		}
	}

	if (verbose > 1)
		printf("checking for rows with one place left\n");

	for (num = 1; num < 10; num++) {
		for (y = 0; y < 9; y++) {
			count = lx = 0;
			for (x = 0; x < 9; x++) {
				if (!f[(int)num][x][y]) {
					lx = x;
					++count;
				}
			}
			if (count == 1) {
				if (verbose > 1)
					printf("%d at %d %d\n", num, lx + 1,
					    y + 1);
				set(f, lx, y, num);
				++tot;
			}
		}
	}
	return tot;
}

int
permute(int n)
{
	int m;

	m = n--;
	while (n > 1)
		m*= n--;
	return m;
}

void
recurse(int **dst, int *p, int *cp, int len, int lvl)
{
	int n, q;

	for (n = 0; n < len; n++) {
		*(cp + lvl) = *(p + n);
		/*recurse with array that does not containt this element */
		if (len > 1) {
			q = *p;
			*p = *(p + n);
			*(p + n) = q;
			recurse(dst, p + 1, cp, len - 1, lvl + 1);
			q = *(p + n);
			*(p + n) = *p;
			*p = q;
		} else {
			for (q = 0; q <= lvl; q++) {
				*(*dst) = *(cp + q);
				(*dst)++;
			}
		}
	}
}

int
guess_in_field(char f[10][9][9], int fx, int fy)
{
	char p[10][9][9];
	int nums[9];
	int tnums[9];	/* numbers to try */
	int unums[9];	/* used numbers */
	int fp[9][2];	/* free places */
	int num;	/* number of free places */
	int n, m, x, y;
	int np;		/* num of permuations */
	int cp;
	int *cnums;
	int *perms, *dst;

	if (verbose > 1)
		printf("finding which numbers to try on field %d %d\n", fx + 1,
		    fy + 1);

	bzero(nums, sizeof(nums));
	for (n = 0, x = 0; x < 3; x++)
		for (y = 0; y < 3; y++)
			nums[f[0][fx*3+x][fy*3+y]-1] = 1;

	for (num = n = m = 0; n < 9; n++)
		if (!nums[n]) {
			tnums[m++] = n + 1;
			++num;
		}

	np = permute(num);

	if (verbose > 1) {
		printf("will try");
		for (n = 0; n < num; n++)
			printf(" %d", tnums[n]);
		printf(" in %d permutations\n", np);
	}

	if (verbose > 1)
		printf("%d free places, %d permutations\n", num, np);

	if (verbose > 1)
		printf("finding free places in field %d %d\n", fx + 1, fy + 1);

	for (n = 0, x = 0; x < 3; x++) {
		for (y = 0; y < 3; y++) {
			if (!f[0][fx*3+x][fy*3+y]) {
				fp[n][0] = x;
				fp[n][1] = y;
				++n;
			}
		}
	}

	/*
	 * try all combinations to put the numbers on this field, until
	 * we can completely fill this field
	 */
	perms = calloc(np * num, sizeof(int));
	dst = perms;
	
	if (perms == NULL)
		errx(1, "out of memory");
	recurse(&dst, tnums, unums, num, 0);
 
	for (cp = 0; cp < np; cp++) {
		cnums = perms + (cp * num);
		bcopy(f, p, sizeof(p));
		for (n = 0; n < num; n++) {
			if (set(p, fx*3+fp[n][0], fy*3+fp[n][1], *(cnums + n)))
				break;
		}
		if (n < num)
			continue;

		if (verbose) {
			printf("\ntrying this board:\n\n");
			show(p, 0);
		}

		while (find(p))
			;
		if (complete(p)) {
			bcopy(p, f, sizeof(p));
			return 0;
		}
	}
	return -1;
}

int
guess(char f[10][9][9])
{
	int x, y, fx, fy, lx, ly;
	int num, numf, n;
	int fn[3][3];
	int fields[9][2];	/* fields with free places */

	if (verbose > 1)
		printf("counting free places per field\n");

	bzero(fn, sizeof(fn));
	lx = ly = -1;
	for (numf = fx = 0; fx < 3; fx++)
		for (fy = 0; fy < 3; fy++)
			for (x = 0; x < 3; x++)
				for (y = 0; y < 3; y++)
					if (!f[0][fx*3+x][fy*3+y]) {
						++fn[fx][fy];
						if (fx != lx || fy != ly) {
							fields[numf][0] = fx;
							fields[numf][1] = fy;
							lx = fx;
							ly = fy;
							++numf;
						}
					}

	if (verbose > 1)
		printf("%d fields have free places\n", numf);

	if (verbose > 1)
		printf("finding the field with the least free places\n");

	for (num = 9, fx = 0; fx < 3; fx++) {
		for (fy = 0; fy < 3; fy++) {
			if (verbose > 1)
				printf("field %d %d has %d free places\n",
				    fx + 1, fy + 1, fn[fx][fy]);
			if (fn[fx][fy] > 0 && fn[fx][fy] < num) {
				lx = fx;
				ly = fy;
				num = fn[fx][fy];
			}
		}
	}

	if (verbose > 1)
		printf("field %d %d has the least (%d) free places\n", lx + 1,
		    ly + 1, num);

	for (n = 0; n < numf; n++)
		guess_in_field(f, fields[n][0], fields[n][1]);

	return -1;
}

int
complete(char f[10][9][9])
{
	int x, y;

	for (x = 0; x < 9; x++)
		for (y = 0; y < 9; y++)
			if (!f[0][x][y])
				return 0;
	return 1;
}

void
show(char f[10][9][9], int num)
{
	int x, y;

	for (y = 0; y < 9; y++) {
		if (y && y % 3 == 0) {
			if (num)
				printf("---+---+---  ");
			printf("---+---+---\n");
		}
		for (x = 0; x < 9; x++) {
			if (x && x % 3 == 0)
				printf("|");
			printf("%c", f[0][x][y] ? '0' + f[0][x][y] : ' ');
		}
		if (num) {
			printf("  ");
			for (x = 0; x < 9; x++) {
				if (x && x % 3 == 0)
					printf("|");
				printf("%c", f[num][x][y] ? 'X' : ' ');
			}
		}
		printf("\n");
	}
}

int
main(int argc, char *argv[])
{
	char f[10][9][9];
	FILE *fp;
	char line[128];
	char *s;
	char num = 0;
	int x, y, n, ch;

	bzero(f, sizeof(f));

	while ((ch = getopt(argc, argv, "v")) != -1) {
		switch (ch) {
		case 'v':
			++verbose;
			break;
		}
	}
	argc -= optind;
	argv += optind;

	if (argc == 1 && ((fp = fopen(argv[0], "r")) !=NULL)) {
		x = y = 0;
		while (fgets(line, sizeof(line), fp) != NULL) {
			if (line[0] == '#')
				continue;
			for (x = 0; x < 9 && line[x] != '\0' &&
			    line[x] != '\n'; x++) {
				if (!isdigit(line[x]) && line[x] != ' ')
					errx(1, "illegal symbol in %s: %c",
					    argv[0], line[x]);
				if (line[x] == ' ')
					continue;
				if (set(f, x, y, line[x] - '0'))
					errx(1, "error in %s: %c can not be "
					    "set at %d %d", argv[0], line[x],
					    x + 1, y + 1);
			}
			++y;
		}
		fclose(fp);
	}
	show(f, 0);
	printf("\n");

	while (find(f))
		++num;
	if (num)
		show(f, 0);

	if (!complete(f)) {
		guess(f);
		printf("\n");
		show(f, 0);
	}

	if (complete(f)) {
		if (verbose)
			printf("complete\n");
		exit(0);
	}

	if (verbose)
		printf("could not solve puzzle, not even by guessing\n");

	do {
		printf("enter x y num\n");

		if ((s = fgets(line, sizeof(line), stdin)) != NULL) {
			if (line[0] == 'g') {
				if ((num = findnum(f, &x, &y)) != 0) {
					printf("set guessed %d at %d %d\n",
					    num, x + 1, y + 1);
					if (set(f, x, y, num))
						printf("illegal position\n");
					else {
						while (find(f))
							;
						show(f, 0);
					}
				} else {
					if (!complete(f))
						printf("unsolved\n");
				}
			} else { 
				x = y = num = 0;
				sscanf(line, "%d %d %d", &x, &y, &n);
				num = n;
				if (x > 0 && x < 10 && y == 0)
					show(f, x);
				else if (x > 0 && x < 10 && y > 0 && y < 10
				    && num > 0 && num < 10) {
					--x;
					--y;
					if (set(f, x, y, num))
						printf("illegal position for "
						    "%d\n", num);
					else {
						show(f, num);
						while (find(f))
							;
						show(f, 0);
					}
				} else
					printf("illegal values: %d %d %d\n", x,
					    y, num);
			}
		}
	} while (s != NULL && !complete(f));
	return 0;
}
