/*
 * This file is part of sudognu.
 *
 * Copyright (C) 2007 Jens Baaran, Germany.
 ******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "sudoku.h"

/*
 ******************************************************************************/
void sudokus_to_svg(char *filename, t_sudoku sudokus[4]) {

	// page width and height 10% smaller than a4 to allow printing programs
	// to add margins
	float pageWidth = 210.0 * 0.90, pageHeight = 297.0 * 0.90;
	float thickLineWidth = 0.5, thinLineWidth = 0.2;
	int titleFontSz = 40, sudokuFontSz = 12, numberFontSz = 24;
	char sudTitle[512];
	char fn[516] = "sudoku";
	
	// at least one square width between sudokus
	int size = sudokus[0].size;
	float squareSizeX = (pageWidth - 6) / (2*size+1);
	
	// at least 3 squares for page title, one square for sudoku titles
	float squareSizeY = (pageHeight - 6) / (2*size+5);
	
	float squareSize   = squareSizeX < squareSizeY ? squareSizeX : squareSizeY;
	float sudokuWidth  = size * squareSize;
	float sudokuHeight = size * squareSize;
	float pagePadW   = (pageWidth - 6)  - squareSize * (2*size+1);
	float pagePadH   = (pageHeight - 6) - squareSize * (2*size+5);
	
	float x0, y0;
	
	// svg file name
	if (strlen(filename) != 0) strncpy(fn,filename,511);
	strcat(fn,".svg");
	FILE *f=fopen(fn,"w");
	
	// print xml / svg head
	fprintf(f,"<?xml version=\"1.0\" standalone=\"no\"?>\n\
		<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 1.1//EN\"\n\
		\"http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd\">\n\
		<svg width=\"%.1fmm\" height=\"%.1fmm\" version=\"1.1\"\n\
		xmlns=\"http://www.w3.org/2000/svg\">\n\n\
		<desc>Sudoku</desc>\n\n",pageWidth,pageHeight);

	// print page title
	fprintf(f,"<g id=\"pagetitle\" font-size=\"%d\" font-weight=\"bold\" text-anchor=\"middle\">\n\
		<text x=\"%.1fmm\" y=\"%.1fmm\">SUDOKU</text>\n\
		</g>\n\n", titleFontSz, pageWidth / 2, squareSize * 1.5 + pagePadH/3);
	
	// print sudokus: bottom right corner of bottom right sudoku is fixed at
	// (pageWidth, pageHeight)
	// x0, y0 are coordinates of top left corner of sudoku
	x0 = pageWidth  - 2 * sudokuWidth  -     squareSize - 3 - pagePadW/2;
	y0 = pageHeight - 2 * sudokuHeight - 2 * squareSize - 3 - pagePadH/3;
	if (sudokus[0].rating > 0) {
		get_sudoku_title(0,(int)sudokus[0].rating,sudTitle);
		grid_to_svg(f, sudokus[0].grid, size, x0, y0, squareSize,
			thickLineWidth, thinLineWidth, sudokuFontSz, numberFontSz, sudTitle);
	}

	x0 = pageWidth  -     sudokuWidth - 3 - pagePadW/2;
	if (sudokus[1].rating > 0) {
		get_sudoku_title(1,(int)sudokus[1].rating,sudTitle);
		grid_to_svg(f, sudokus[1].grid, size, x0, y0, squareSize,
			thickLineWidth, thinLineWidth, sudokuFontSz, numberFontSz, sudTitle);
	}

	x0 = pageWidth  - 2 * sudokuWidth  -     squareSize - 3 - pagePadW/2;
	y0 = pageHeight -     sudokuHeight - 3 - pagePadH/3;
	if (sudokus[2].rating > 0) {
		get_sudoku_title(2,(int)sudokus[2].rating,sudTitle);
		grid_to_svg(f, sudokus[2].grid, size, x0, y0, squareSize,
			thickLineWidth, thinLineWidth, sudokuFontSz, numberFontSz, sudTitle);
	}

	x0 = pageWidth  -     sudokuWidth - 3 - pagePadW/2;
	if (sudokus[3].rating > 0) {
		get_sudoku_title(3,(int)sudokus[3].rating,sudTitle);
		grid_to_svg(f, sudokus[3].grid, size, x0, y0, squareSize,
			thickLineWidth, thinLineWidth, sudokuFontSz, numberFontSz, sudTitle);
	}

	// print svg foot
	fprintf(f,"</svg>\n");
	
	//
	fclose(f);
}

/*
 ******************************************************************************/
void grid_to_svg(FILE *f, t_field grid[SIZE2][SIZE2], int size,  float x0, float y0, float squareSize,
	float thickLineWidth, float thinLineWidth, int sudokuFontSz, int numberFontSz, char *sudTitle) {
	
	float sudSize = size * squareSize;
	float sudSIZE = SIZE * squareSize;

	int i, j, n;
	
	// print sudoku title
	fprintf(f,"<g id=\"sudokutitle\" font-size=\"%d\">\n\
		<text x=\"%.1fmm\" y=\"%.1fmm\">%s</text>\n\
		</g>\n\n", sudokuFontSz, x0+2.0, y0-2.0, sudTitle);
	
	// print thick lines
	fprintf(f,"<g id=\"thicklines\" stroke-width=\"%.1fmm\" stroke=\"black\">\n\
		<rect  x=\"%.1fmm\" y=\"%.1fmm\" width=\"%.1fmm\" height=\"%.1fmm\" fill=\"none\" />\n",
		thickLineWidth, x0, y0, sudSize, sudSize);
	for (i=1; i<SIZE; i++) {
		fprintf(f,"<line x1=\"%.1fmm\" y1=\"%.1fmm\" x2=\"%.1fmm\" y2=\"%.1fmm\" />\n",
			x0 + i * sudSIZE, y0, x0 + i * sudSIZE, y0 + sudSize);
		fprintf(f,"<line x1=\"%.1fmm\" y1=\"%.1fmm\" x2=\"%.1fmm\" y2=\"%.1fmm\" />\n",
			x0, y0 + i * sudSIZE, x0 + sudSize, y0 + i * sudSIZE);
	}
	fprintf(f,"</g>\n\n");
	
	// print thin lines
	fprintf(f,"<g id=\"thinlines\" stroke-width=\"%.1fmm\" stroke=\"black\">\n",
		thinLineWidth);
	for (i=0; i<SIZE; i++) {
		for (j=1; j<SIZE; j++) {
			n = i * SIZE + j;
			fprintf(f,"<line x1=\"%.1fmm\" y1=\"%.1fmm\" x2=\"%.1fmm\" y2=\"%.1fmm\" />\n",
				x0 + n * squareSize, y0, x0 + n * squareSize, y0 + sudSize);
			fprintf(f,"<line x1=\"%.1fmm\" y1=\"%.1fmm\" x2=\"%.1fmm\" y2=\"%.1fmm\" />\n",
				x0, y0 + n * squareSize, x0 + sudSize, y0 + n * squareSize);
		}
	}
	fprintf(f,"</g>\n\n");

	// print numbers
	fprintf(f,"<g id=\"sudokunumbers\" font-size=\"%d\" text-anchor=\"middle\">\n",
		numberFontSz);
	for (i=0; i<size; i++) {
		for (j=0; j<size; j++) {
			if (grid[i][j].value > 0) {
				fprintf(f,"<text x=\"%.1fmm\" y=\"%.1fmm\" dy=\"%d\">%d</text>\n",
					x0 + (j + 0.5) * squareSize, y0 + (i + 0.5) * squareSize - 1, numberFontSz/2,
					grid[i][j].value);
			}
		}		
	}
	fprintf(f,"</g>\n\n\n");
}

/*
 ******************************************************************************/
void get_sudoku_title(int isud, int rating, char *title) {
	char dstr[20];
	
	if      (rating > 199) strcpy(dstr,"fiendish");
	else if (rating >  99) strcpy(dstr,"evil");
	else if (rating >   9) strcpy(dstr,"hard");
	else if (rating >   1) strcpy(dstr,"moderate");
	else if (rating ==  1) strcpy(dstr,"easy");
	else                   strcpy(dstr,"");
	snprintf(title, 200, "Sudoku %d, difficulty %d - %s",isud+1,rating,dstr);
}
