/*
 * This file is part of sudognu.
 *
 * Copyright (C) 2007 Jens Baaran, Germany.
 ******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "sudoku.h"

/*
 ******************************************************************************/
int hidden_single(t_sudoku *sudoku, int doElim, int verbosity) {
	
	int size = sudoku->size;
	int r, c, cc, nc, br, bc, nhs=0, row, col, cnd;
	char ent[7], estr[512];

	for (cc = 0; cc < size; cc++) {
		for (c = 0; c < size; c++) {
			// check columns
			r = nc = 0;
			while((nc < 2) && (r < size)) {
				if (sudoku->grid[r][c].cand[cc] != 0) {
					nc++;
					row = r;
				}
				r++;
			}
			if (nc == 1) {
				nhs++;
				if (doElim == 1) {
					col = c;
					cnd = cc;
					strcpy(ent,"column");
					goto eliminate;
				}
			}
			// check rows
			r = nc = 0;
			while((nc < 2) && (r < size)) {
				if (sudoku->grid[c][r].cand[cc] != 0) {
					nc++;
					col = r;
				}
				r++;
			}
			if (nc == 1) {
				nhs++;
				if (doElim == 1) {
					row = c;
					cnd = cc;
					strcpy(ent,"row");
					goto eliminate;
				}
			}
			// check blocks
			r = nc = 0;
			while((nc < 2) && (r < size)) {
				br = (c / SIZE) * SIZE + (r / SIZE);
				bc = (c % SIZE) * SIZE + (r % SIZE);
				if (sudoku->grid[br][bc].cand[cc] != 0) {
					nc++;
					row = br; col = bc; 
				}
				r++;
			}
			if (nc == 1) {
				nhs++;
				if (doElim == 1) {
					cnd = cc;
					strcpy(ent,"block");
					goto eliminate;
				}
			}
		}
	}
	
	if (doElim == 0)
		return(nhs);
	else
		return(-1);

	eliminate:
	
	// 1. explanation string
	snprintf(estr,511,".  r%dc%d, (%d) %s hidden single",
		row+1,col+1,sudoku->grid[row][col].cand[cnd],ent);

	// 2. put value in and update candidates
	sudoku->grid[row][col].value = sudoku->grid[row][col].cand[cnd];
	if (sudoku->grid[row][col].stat == stat_free) sudoku->grid[row][col].stat = stat_hsingle;
	remove_candidates(sudoku,row,col);

	// 3. print explanation
	fprint_explain(estr,*sudoku,verbosity);
	
	return(1);
}

/* eliminate hidden singles for candidate cc (needed for two-color technique)
 ******************************************************************************/
int hsinglecc(t_sudoku *s, int cc) {

	int ret, i, j, kr, kc, kb, row, col, nelim, snelim = 0;
	int nccrow, ncccol, nccblk;

	do {
		nelim = 0;
		for (i=0; i<SIZE2; i++) {
			nccrow = ncccol = nccblk = 0;
			for (j=0; j<SIZE2; j++) {
				if (s->grid[i][j].cand[cc] != 0) {
					nccrow++;
					kr = j;
				}
				if (s->grid[j][i].cand[cc] != 0) {
					ncccol++;
					kc = j;
				}
				col = (i / SIZE) * SIZE + (j / SIZE);
				row = (i % SIZE) * SIZE + (j % SIZE);
				if (s->grid[row][col].cand[cc] != 0) {
					nccblk++;
					kb = j;
				}
			}
			if (nccrow == 1) {
				s->grid[i][kr].value = cc+1;
				s->grid[i][kr].stat = stat_colortmp;
				remove_candidates(s,i,kr);
				nelim++;
			} else if (ncccol == 1) {
				s->grid[kc][i].value = cc+1;
				s->grid[kc][i].stat = stat_colortmp;
				remove_candidates(s,kc,i);
				nelim++;
			} else if (nccblk == 1) {
				col = (i / SIZE) * SIZE + (kb / SIZE);
				row = (i % SIZE) * SIZE + (kb % SIZE);
				s->grid[row][col].value = cc+1;
				s->grid[row][col].stat = stat_colortmp;
				remove_candidates(s,row,col);
				nelim++;
			}
			snelim += nelim;
		}
	} while (nelim > 0);

	/* return negative value if sudoku is not solvable
	 * else return number of hidden singles found */
	if ((ret = check_sudoku(*s)) >= 0) {
		ret = snelim;
	}

	return ret;
}

/* eliminate candidates with hidden tuple technique
 * 
 * tuple = 2  --> check for hidden doubles
 * tuple = 3  --> check for hidden triples
 * tuple = 4  --> check for hidden quadruples
 * 
 * doElim = 0 --> don't modify sudoku, but count the number of hidden tuples
 *                in sudoku
 * doElim = 1 --> eliminate candidates for 1st hidden tuple found, then return
 * 
 * return value: number of hidden tuples found
 ******************************************************************************/
int hidden_tuple(t_sudoku *sudoku, int tuple, int doElim, int verbosity) {
	
	int size = sudoku->size;
	int r, c, br, bc, *cc, ntup = 0;
	int rowVals[SIZE2], colVals[SIZE2], blkVals[SIZE2];

	cc = (int *) malloc(sizeof(int)*size);
	for (c = 0; c < size; c++) cc[c] = 0;
	
	for (c = 0; c < size; c++) {
		for (r = 0; r < size; r++) {
			rowVals[r] = colVals[r] = blkVals[r] = 0;
		}
		for (r = 0; r < size; r++) {
			if (sudoku->grid[r][c].value != 0) colVals[sudoku->grid[r][c].value-1] = 1;
			if (sudoku->grid[c][r].value != 0) rowVals[sudoku->grid[c][r].value-1] = 1;
			br = (c / SIZE) * SIZE + (r / SIZE);
			bc = (c % SIZE) * SIZE + (r % SIZE);
			if (sudoku->grid[br][bc].value != 0) blkVals[sudoku->grid[br][bc].value-1] = 1;
		}
		for (cc[0] = 0; cc[0] < size-tuple+1; cc[0]++) {
			find_hidden_tuple(sudoku,tuple,1,c,cc,colVals,rowVals,blkVals,doElim,verbosity,&ntup);
			if ((doElim == 1) && (ntup > 0)) goto done;
		}
	}

	done:
	free(cc);
	return(ntup);
}

/*
 ******************************************************************************/
void find_hidden_tuple(
	t_sudoku *sudoku,
	int tuple, int itup, int c, int *cc,
	int colVals[SIZE2], int rowVals[SIZE2], int blkVals[SIZE2],
	int doElim, int verbosity,
	int *ntup) {

	int size = sudoku->size;
	int iqc, checkTuple;
	
	for (cc[itup] = cc[itup-1]+1; cc[itup] < size-tuple+itup+1; cc[itup]++) {
		if (itup+1 < tuple) {
			find_hidden_tuple(sudoku,tuple,itup+1,c,cc,colVals,rowVals,blkVals,doElim,verbosity,ntup);
		} else {
			// columns
			checkTuple = 1;
			for (iqc = 0; iqc < tuple; iqc++) if (colVals[cc[iqc]] != 0)
				checkTuple = 0;
			if (checkTuple == 1) {
				hidden_tuple_crb(sudoku,tuple,cc,c,'c',doElim,verbosity,ntup);
				if ((doElim == 1) && (*ntup > 0)) return;
			}
			// rows
			checkTuple = 1;
			for (iqc = 0; iqc < tuple; iqc++) if (rowVals[cc[iqc]] != 0)
				checkTuple = 0;
			if (checkTuple == 1) {
				hidden_tuple_crb(sudoku,tuple,cc,c,'r',doElim,verbosity,ntup);
				if ((doElim == 1) && (*ntup > 0)) return;
			}
			// blocks
			checkTuple = 1;
			for (iqc = 0; iqc < tuple; iqc++) if (blkVals[cc[iqc]] != 0)
				checkTuple = 0;
			if (checkTuple == 1) {
				hidden_tuple_crb(sudoku,tuple,cc,c,'b',doElim,verbosity,ntup);
				if ((doElim == 1) && (*ntup > 0)) return;
			}
		}
	}
}

/*
 ******************************************************************************/
void hidden_tuple_crb(
	t_sudoku *sudoku,
	int tuple, int *cc, int c, char ent, int doElim, int verbosity,
	int *ntup) {

	int size = sudoku->size;
	int r, iqc, row, col, nelim;
	int numTupleCells, numTupleCandsWithinCell, numNonTupleCandsWithinTupleCells;
	int rows[SIZE2], cols[SIZE2];
	int tupCands[SIZE2];
	
	numTupleCells = numNonTupleCandsWithinTupleCells = 0;
	for (r = 0; r < size; r++) {
		switch (ent) {
		case 'c':
			col=c; row=r; break;
		case 'r':
			col=r; row=c; break;
		case 'b':
			row = (c / SIZE) * SIZE + (r / SIZE);
			col = (c % SIZE) * SIZE + (r % SIZE);
			break;
		}
		numTupleCandsWithinCell = 0;
		for (iqc = 0; iqc < tuple; iqc++) {
			if (sudoku->grid[row][col].cand[cc[iqc]] != 0) {
				if (numTupleCandsWithinCell == 0) {
					rows[numTupleCells] = row;
					cols[numTupleCells] = col;
					numTupleCells++;
				}
				numTupleCandsWithinCell++;
			}
		}
		if (numTupleCandsWithinCell > 0) {
			numNonTupleCandsWithinTupleCells +=
				sudoku->grid[row][col].ncand - numTupleCandsWithinCell;
		}
	}

	if ((numTupleCells == tuple)
		&& (numNonTupleCandsWithinTupleCells > 0)) {
	
		if (DEBUG) {
			fprintf(stderr,"[%c%d(%d",ent,c+1,cc[0]+1);
			for(iqc = 1; iqc < tuple; iqc++) fprintf(stderr,",%d",cc[iqc]+1);
			fprintf(stderr,")]");
		}

		(*ntup)++;
		
		if (doElim == 1) {
			
			// eliminate
			for (r = 0; r < size; r++) tupCands[r] = 0;
			for (iqc=0; iqc<tuple; iqc++) tupCands[cc[iqc]] = 1;

			nelim = 0;
			for (iqc=0; iqc<tuple; iqc++) {
				for (r = 0; r < size; r++) {
					if ((tupCands[r] == 0) && (sudoku->grid[rows[iqc]][cols[iqc]].cand[r] != 0)) {
						nelim++;
						sudoku->grid[rows[iqc]][cols[iqc]].cand[r]  = 0;
						sudoku->grid[rows[iqc]][cols[iqc]].ncand   -= 1;
					}
				}
			}

			// output
			if (verbosity > 0) {
				char entstr[8], tupstr[20], estr[512];

				switch (ent) {
				case 'c':	strcpy(entstr,"column"); break;
				case 'r':	strcpy(entstr,"row");    break;
				case 'b':	strcpy(entstr,"block");  break;
				}

				switch (tuple) {
				case 2: strcpy(tupstr,"hidden double");    break;
				case 3: strcpy(tupstr,"hidden triple");    break;
				case 4: strcpy(tupstr,"hidden quadruple"); break;
				}
			
				if (ent == 'b')
					sprintf(estr,"%c  %c%d%d, (%d",tupstr[7],entstr[0],c/3+1,c%3+1,cc[0]+1);
				else
					sprintf(estr,"%c  %c%d, (%d",tupstr[7],entstr[0],c+1,cc[0]+1);
				for (iqc=1; iqc<tuple; iqc++)
					sprintf(estr+strlen(estr)," %d",cc[iqc]+1);
				sprintf(estr+strlen(estr),") %s, -%d candidates",tupstr,nelim);
				fprint_explain(estr,*sudoku,verbosity);
			}

			// after elimination return
			return;
		}
	}
}

