/*
 * This file is part of sudognu.
 *
 * Copyright (C) 2007-2009 Jens Baaran, Germany.
 ******************************************************************************/

#include <limits.h>
#include <unistd.h>
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>

#include "sudoku.h"

// initialize external identifiers
char progname[100];
int action = 0;
int max_rating = INT_MAX;
int num_sudokus = 4;
int verbosity = 1;
int nthread = 0;
t_symmetry symmgrid = symm_none;
int dfcs[NUM_ET];

char fn_out[512] = "";
FILE *inFile = NULL;

pthread_mutex_t mut_input  = PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t mut_stdout = PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t mut_nsud   = PTHREAD_MUTEX_INITIALIZER;

char fn_wsudognu[MAX_STRLEN+1] = "./wsudognu.cgi";
FILE *explFile;
FILE *devurandom;

/*
 ******************************************************************************/
int getoptions(int argc, char *argv[]) {
	
	int i, iet, printerr;
	char *fn, sym;
	
	opterr = 0;
	
	/* initialize difficulties according to enum elim_type in sudoku.h */
	for (i = 0; i< NUM_ET; i++) {
		dfcs[i] = i;
	}
	
	while ((i = getopt (argc, argv, ":chNpstVwf:i:n:o:v:P:S:")) != -1) {
		switch (i) {
			case 'c':
				action = 1;
				break;
			case 'p':
				action = 2;
				break;
			case 's':
				action = 3;
				break;
			case 'w':
				action = 4;
				break;
			case 'N':
				action = 5;
				break;
			case 'S':
				if (sscanf(optarg,"%c",&sym) != 1) {
					return(1);
				}
				switch(sym) {
					case 'a':
						symmgrid = symm_a;
						break;
					case 'p':
						symmgrid = symm_p;
						break;
					default:
						fprintf(stderr,"ERROR: unkown symmetry identifier after -S option. Try 'a' or 'p'.\n\n");
						return(1);
				}
				break;
			case 't':
				nthread = sysconf(_SC_NPROCESSORS_ONLN);
				break;
			case 'f':
				if (strlen(optarg) > MAX_STRLEN) {
					fprintf(stderr,"ERROR: filename too long.\n\n");
					return(1);
				}
				if (sscanf(optarg,"%s",fn_wsudognu) != 1) {
					fprintf(stderr,"ERROR: cannot read file name from argument list\n\n");
					return(1);
				}
				break;
			case 'h':
				print_help(stdout);
				exit(0);
			case 'n':
				if ((sscanf(optarg,"%d",&num_sudokus) != 1) || (num_sudokus < 1)) {
					return(1);
				}
				break;
			case 'i':
				fn = optarg;
				if ((inFile = fopen(fn,"r")) == NULL) {
					fprintf(stderr,"ERROR: can't open input file %s\n\n",fn);
					return(1);
				}
				break;
			case 'o':
				strncpy(fn_out,optarg,511);
				fn_out[511] = '\0';
				break;
			case 'v':
				if ((sscanf(optarg,"%d",&verbosity) != 1) || (verbosity < 0)) {
					fprintf(stderr,"ERROR: verbosity out of range\n\n");
					return(1);
				}
				break;
			case 'V':
				print_version(stdout);
				exit(0);
				break;
			case 'P':
				if(strlen(optarg) <= (NUM_ET-2)) {
//					for (iet = 0; iet < NUM_ET; iet++) printf("%d ",dfcs[iet]);
					for (iet = 1; iet < NUM_ET-1; iet++) dfcs[iet] = -1;
					for (iet = 1; ( (iet < NUM_ET-1) && (optarg[iet-1] != '\0') ); iet++) {
						switch(optarg[iet-1]) {
							case '.':	dfcs[iet] = et_hs; break;
							case '-':	dfcs[iet] = et_s;  break;
							case 'd':	dfcs[iet] = et_hd; break;
							case 'D':	dfcs[iet] = et_d;  break;
							case 'b':	dfcs[iet] = et_b;  break;
							case 'B':	dfcs[iet] = et_l;  break;
							case 't':	dfcs[iet] = et_ht; break;
							case 'T':	dfcs[iet] = et_t;  break;
							case 'q':	dfcs[iet] = et_hq; break;
							case 'Q':	dfcs[iet] = et_q;  break;
							case 'W':	dfcs[iet] = et_xy; break;
							case 'Y':	dfcs[iet] = et_xyz; break;
							case 'X':	dfcs[iet] = et_xw; break;
							case 'c':	dfcs[iet] = et_2c; break;
							case 'S':	dfcs[iet] = et_sf; break;
							case 'J':	dfcs[iet] = et_jf; break;
							case 'F':	dfcs[iet] = et_fc; break;
							default: return(1);
						}
					}
//					for (iet = 0; iet < NUM_ET; iet++) printf("%d ",dfcs[iet]);
//					for (iet = 1; iet < NUM_ET-2; iet++)
//						if (dfcs[iet] == -1)
//							return(1);
					if ( (dfcs[1] != et_hs) && (dfcs[1] != et_s) ) {
						fprintf (stderr,"%s: ERROR: First solution technique must be `.' or `-'\n",progname);
						abort();
					}
				} else {
					fprintf (stderr,"%s: ERROR: length of argument string for -P option out of range.\n\n",progname);
					return(1);
				}
				break;
			case ':':
				if (isprint(optopt)) fprintf (stderr,"%s: ERROR: Missing parameter for option `-%c'.\n\n",progname,optopt);
				else                 fprintf (stderr,"%s: ERROR: Missing parameter for option character `\\x%x'.\n\n",progname,optopt);
				return(1);
			case '?':
				if (isprint(optopt)) fprintf (stderr,"%s: ERROR: Unknown option `-%c'.\n\n",progname,optopt);
				else                 fprintf (stderr,"%s: ERROR: Unknown option character `\\x%x'.\n\n",progname,optopt);
				return(1);
			default:
				abort ();
		}
	}
	for (i = optind; i < argc; i++)	fprintf (stderr,"Non-option argument %s\n", argv[i]);

	// no action = nothing to do
	if (action == 0) {
		return(1);
	}
	
	// reordering solution techniques is only allowed for solving or printing
	printerr = 0;
	if ((action != 2) && (action != 3)) {
		for (iet = 0; iet < NUM_ET; iet++) {
			if (dfcs[iet] != iet) printerr = iet;
			dfcs[iet] = iet;
		}
		if (printerr != 0) {
			fprintf (stderr,"%s: WARNING: -P option is ignored. Use -P only for solving and printing (%d).\n\n", argv[0],printerr);
		}
	}
	
	return(0);
}

/*
 ******************************************************************************/
void print_version(FILE *f) {
	fprintf(f,"%s version %s\n",progname,VERSION);
	fprintf(f,"\
Copyright (c) 2007-2009 Jens Baaran (http://baaran.de)\n\n\
This is free software. You may redistribute copies of this software\n\
under the terms of the GNU General Public License Version 2, June 1991\n\
(http://www.gnu.org/licenses/gpl-2.0.html). There is NO WARRANTY, to\n\
the extent permitted by applicable law. The authors of this software\n\
cannot be held responsible for any damage it may cause.\n\
");
}

/*
 ******************************************************************************/
void print_help(FILE *f) {

	print_usage(f);
	
	fprintf(f,"\
Sudoku format:\n\t\n\
The input and output format for sudokus uses one string per sudoku:\n\
concatenate the 81 numbers line by line from top left to bottom\n\
right without white space between the numbers. Empty fields are to be\n\
denoted by \"0\" (zero).\n\n\
");

	fprintf(f,"\
Sudoku solving:\n\t\n\
Various elimination techniques for solving sudokus are implemented, starting\n\
with easy techniques such as singles and hidden singles. Complex techniques\n\
for candidate elimination (tuple / hidden tuple, line-block interaction,\n\
forcing chain, X-wing, swordfish, jellyfish) are applied only if necessary,\n\
reverting to backtracking (guessing) at last resort. Solutions are checked for\n\
uniqueness.\n\n\
%s can display the solution process. When you set verbosity to 2, %s\n\
will display a string of characters, each one stands for one solution step:\n\
 .\t hidden single\n\
 -\t single\n\
 d\t hidden double\n\
 D\t double\n\
 b\t line-block interaction\n\
 B\t block-line interaction\n\
 t\t hidden triple\n\
 T\t triple\n\
 q\t hidden quadruple\n\
 Q\t quadruple\n\
 W\t xy-wing\n\
 Y\t xyz-wing\n\
 X\t X-wing\n\
 c\t two-color\n\
 S\t swordfish\n\
 J\t jellyfish\n\
 F\t forcing chain to a common value with length of forcing chain\n\
 f\t forcing chain to a contradiction for one of the branches\n\
 e\t forcing chain to the end of the sudoku\n\
 g\t guess during backtracking process\n\
 w\t last guess turned out to be wrong.\n\
\n\
The default order for applying the solution steps is to start with hidden\n\
singles and singles before trying to eliminate candidates with hidden tuples\n\
and tuples from doubles to quadruples. Line-block interactions are applied\n\
between doubles and triples. Then the more complex techniques X-wing, 2-color\n\
swordfish and jellyfish are used and if all these fail. Forcing chains are\n\
tried, before reverting to guessing and starting a backtracking process.\n\
\n\
You can influence the order, in which the techniques above are applied, by\n\
means of the -P option. As described above, the default order is\n\
\".-dDbBtTqQWYXcSJF\". If you want the steps to be applied in a different\n\
order, just shuffle these characters and pass them to %s with the -P\n\
flag. The sudoku solution performance can be improved greatly, if you leave\n\
out the 'q', 'Q' and 'J' techniques, which occur only very rarely, but need a\n\
lot of computation time.\n\
\n\
Forcing chains are not applied during backtracking (i.e. after the first\n\
value had to be guessed). While searching for forcing chains, only a subset\n\
of the available techniques is applied. Forcing chains are only tried for\n\
cells with two candidates. All of such cells are checked and %s will\n\
apply the shortest forcing chain it finds. Since only one solution path is\n\
evaluated for each possible starting cell of a forcing chain, %s does not\n\
always find the shortest forcing chain.\n\n\
",progname,progname,progname,progname,progname);

	fprintf(f,"\
Sudoku generation:\n\t\n\
All sudokus generated by %s are minimal sudokus, meaning that all givens\n\
are required to ensure a unique solution to the sudoku. Optionally symmetric\n\
sudokus can be generated by specifying '-S' on the command line. Generation\n\
of symmetric sudokus requires much more time than the generation of\n\
unsymmetric sudokus.\n\
\n\
It is not possible to specify a desired difficulty rating for generation of\n\
sudokus. To produce sudokus with a certain difficulty, just create a whole\n\
batch (say 100) and pick the ones with the desired difficulty.\n\n\
",progname);
	
	fprintf(f,"\
Sudoku rating:\n\t\n\
At first the sudoku is assigned a basic rating of 1, 10 or 100, depending on\n\
the most difficult technique required for its solution:\n\n\
 a basic rating of 1 is assigned for the techniques .-d\n\n\
 a basic rating of 10 is assigned for the techniques DbBtTqQ\n\n\
 a basic rating of 100 is assigned for the techniques WYXcSJFfeg.\n\n\
Next, each solution step is assigned a difficulty rating, starting with 1 for\n\
hidden singles up to %d for guesses during the backtracking process. In each\n\
step the two easiest applicable techniques are taken into account. The\n\
individual ratings for each solution step are added to the basic rating.\n\
Finally, 1 is subtracted from the sudoku rating for each cell to be solved. The\n\
smallest possible rating of 1 is assigned to a sudoku, if there are always at\n\
least two choices for the application of the hidden singles technique in the\n\
solution path chosen by %s.\n\n\
",NUM_ET-1,progname);
	
	print_contact(f);
}

/*
 ******************************************************************************/
void print_usage(FILE *f) {
	fprintf(f,"\
Usage: %s < command >  [ options ]\n\
%s can solve and create sudokus. It can print 4 sudokus to a pdf or an svg\n\
page and it comes with example cgi scripts for integration into www sites.\n\n\
",progname,progname);

	fprintf(f,"\
Examples:\n\
 %s -s -v 1 < sud.in > sol.out  \tsolve sudokus\n\
 %s -c -n 20 > sud.out     \tcreate 20 sudokus\n\
 %s -p -o sud < sud.in \tprint 4 sudokus to sud.pdf & sud.svg\n\n\
",progname,progname,progname);

	fprintf(f,"\
Commands (only one of these may be present):\n\
 -c\t create sudokus and print four of them to pdf/svg file\n\
 -p\t read sudokus from stdin and print four of them to pdf/svg file\n\
 -s\t read sudokus from stdin and solve them\n\
 -w\t read one sudoku and candidates from stdin and generate a html table\n\
 -N\t read sudokus from stdin and canonicalize them (output to stdout)\n\
 -h\t display usage information\n\
 -V\t display version and copyright information\n\n\
");
	
	fprintf(f,"\
Options (not all options work for every command):\n\
 -n num_sudokus\t number of sudokus to create\n\
 -i file\t read sudokus from file and not from stdin\n\
 -o file\t html or pdf file name for formatted output\n\
 -P sol_techn\t apply solution techniques according to sol_techn\n\
 \t The sol_techn string must consist of the characters \".-dDbBtTqQXcSJF\"\n\
 \t in arbitrary order. \".-dDbltTqQXcSJF\" is the default, when -P is not\n\
 \t present. For further explanation of the sol_techn string use the -h\n\
 \t command (or see \"Sudoku solving\" below, if you are looking at the\n\
 \t documentation).\n\
 -v verbosity\t for explaining solution techniques (one of 0, 1, 2, 10 or 11)\n\
 -f wsudognu\t location of wsudognu.cgi\n\
 \t wsudognu.cgi is a script necessary for processing the sudoku urls\n\
 \t created for the -s command with a verbosity of 11 (default value for\n\
 \t wsudognu is ./wsudognu.cgi)\n\
 -t\t use one thread for each available CPU (no pdf/svg file will be created)\n\
 -S symmetry\t create axi- or point-symmetrical sudokus\n\
 \t Slow; symmetry is 'a' or 'p'.\n\n\
");
}

/*
 ******************************************************************************/
void print_contact(FILE *f) {
	fprintf(f,"Report bugs to <%s_at_baaran_dot_de>\n\n",progname);
	fprintf(f,"Written by Jens Baaran\n\n");
}
