/*
 * This file is part of sudognu.
 *
 * Copyright (C) 2007 Jens Baaran, Germany.
 ******************************************************************************/

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include "sudoku.h"

/*
 ******************************************************************************/
int fish(t_sudoku *sudoku, int fish, int doElim, int verbosity) {
	
	int size = sudoku->size;
	int *r, *c, cc, nfish=0;
	
	// allocate
	r = (int *) malloc(sizeof(int)*fish);
	c = (int *) malloc(sizeof(int)*fish);
	
	// loop over candidates
	for (cc=0; cc<size; cc++) {

		// loop over row/column tupels
		for (r[0]=0; r[0]<size-fish+1; r[0]++) {
			for (c[0]=0; c[0]<size-fish+1; c[0]++) {
				find_fish(sudoku,fish,1,r,c,cc,doElim,verbosity,&nfish);
				if ((doElim == 1) && (nfish > 0)) goto done;
			}
		}
	}

	done:
	free(c);
	free(r);
	return(nfish);
}

/*
 ******************************************************************************/
void find_fish(t_sudoku *sudoku, int fish, int ifish, int *r, int *c, int cc, int doElim, int verbosity, int *nfish) {
	
	int size = sudoku->size;
	int rcc[4], ccc[4], nc, i, j;

	for (r[ifish]=r[ifish-1]+1; r[ifish]<size-fish+ifish+1; r[ifish]++) {
		for (c[ifish]=c[ifish-1]+1; c[ifish]<size-fish+ifish+1; c[ifish]++) {

			if (ifish+1 < fish) {

				// build tuple of row / columns for fish recursively
				find_fish(sudoku,fish,ifish+1,r,c,cc,doElim,verbosity,nfish);

			} else {

				// row column tuple is complete, start looking for some fish
				for (i=0; i<fish; i++) rcc[i] = ccc[i] = 0;
				for (i=0; i<fish; i++) {
					for (j=0; j<fish; j++) {
						if (sudoku->grid[r[i]][c[j]].cand[cc] > 0) rcc[i] = ccc[j] = 1;
					}
				}
				for (i=nc=0; i<fish; i++) nc += rcc[i] + ccc[i];

				// if each row and column of fish contains at least one candidate cc
				// look for candidates to eliminate
				if (nc == 2*fish) {
					*nfish += fish_rc(sudoku,fish,cc,r,c,"row",doElim,verbosity);
					if ((*nfish > 0) && (doElim == 1)) return;
					*nfish += fish_rc(sudoku,fish,cc,c,r,"column",doElim,verbosity);
					if ((*nfish > 0) && (doElim == 1)) return;
				}
				
			}
		}
	}
}
 
/*
 ******************************************************************************/
int fish_rc(t_sudoku *sudoku, int fish, int cc,
	int *r, int *c, const char ent[12], int doElim, int verbosity) {
	
	int size = sudoku->size;
	int fc[SIZE2], fr[SIZE2];
	int i, j, nelim=0, nfish=0;
	char estr[512], type[20];

	for (i=0; i < SIZE2; i++) fr[i] = fc[i] = 0;
	for (i=0; i < fish; i++)  fr[r[i]] = fc[c[i]] = 1;

	if (strcmp(ent,"row") != 0) transpose_sudoku(sudoku);

	// check row jellyfish: cand(cc) must not be present in r[0] - r[3]
	for (i=0; i<size; i++) {
		if (fc[i] == 0) {
			for (j=0; j<fish; j++) {
				if (sudoku->grid[r[j]][i].cand[cc] != 0) // check fish rows
					goto done;
			}
		}
	}
								
	// cand(cc) can then be eliminated from columns c[0] to c[3]
	// in rows other than r[0] to r[3]
	for (i=0; i<size; i++) {
		if (fr[i] == 0) {
			for (j=0; j<fish; j++) {
				if (sudoku->grid[i][c[j]].cand[cc] != 0) {
					if (doElim == 1) {
						nelim++;
						sudoku->grid[i][c[j]].cand[cc] = 0;
						sudoku->grid[i][c[j]].ncand -=1;
					} else {
						nfish = 1;
						goto done;
					}
				}
			}
		}
	}
	
	// output
	if (nelim > 0) {
		switch(fish) {
			case 2: strcpy(type,"x-wing");    break;
			case 3: strcpy(type,"swordfish"); break;
			case 4: strcpy(type,"jellyfish"); break;
			default: sprintf(type,"%d-fish",fish); break;
		}
		sprintf(estr,"%c  ",toupper(type[0]));
		if (strcmp(ent,"row") == 0) {
			for (i = 0; i < fish; i++) sprintf(estr+strlen(estr),"r%d",r[i]+1);
			for (i = 0; i < fish; i++) sprintf(estr+strlen(estr),"c%d",c[i]+1);
		} else {
			for (i = 0; i < fish; i++) sprintf(estr+strlen(estr),"r%d",c[i]+1);
			for (i = 0; i < fish; i++) sprintf(estr+strlen(estr),"c%d",r[i]+1);
		}
		sprintf(estr+strlen(estr),", (%d) %s %s, -%d candidates",cc+1,ent,type,nelim);
		fprint_explain(estr,*sudoku,verbosity);
		nfish = 1;
	}
	
	done:
	if (strcmp(ent,"row") != 0) transpose_sudoku(sudoku);
	if (DEBUG && (nelim > 0)) print_candidates_as_grid(*sudoku);
	return(nfish);
}
