/*
 * This file is part of sudognu.
 *
 * Copyright (C) 2007 Jens Baaran, Germany.
 ******************************************************************************/

#include <stdio.h>
#include <sys/times.h>
#include <time.h>
#include <stdlib.h>
#include <limits.h>
#include <pthread.h>
#include "sudoku.h"

extern pthread_mutex_t mut_input;
extern pthread_mutex_t mut_stdout;
extern pthread_mutex_t mut_nsud;

int nsud = 0;

/*
 ******************************************************************************/
int execute_thread_create(void *arg[]) {
	
	extern int num_sudokus;
	int tsud = 0;
	int v = 0;
	t_sudoku sudoku;
	int *verbosity = (int *) arg[0];
	
	if (*verbosity > 0) v = 1;

	reset_sudoku(&sudoku);
	while (nsud < num_sudokus) {
		pthread_mutex_lock(&mut_nsud);
		nsud++;
		pthread_mutex_unlock(&mut_nsud);
		tsud++;
		create_sudoku(&sudoku, v);
	}
	return tsud;
}

/*
 ******************************************************************************/
int create_sudoku(t_sudoku *sudoku, int verbosity) {

	int size;
	int r, c, ng, ngiv, t0, t1;
	int elim[NUM_ET], ret;
	t_sudoku csudoku, dsudoku;
	extern t_symmetry symmgrid;
	
	reset_sudoku(sudoku);
	size = sudoku->size;
	
	// repeat until grid with no extra givens is found (symmetric, if required)
	do {

		// add givens until solvable sudoku is found
		if (symmgrid == symm_none) {
			create_unsymm(&csudoku);
		} else if (symmgrid == symm_p) {
			create_psymm(&csudoku);
		} else if (symmgrid == symm_a) {
			create_asymm(&csudoku);
		}
		
		// remove all solved cells, only givens remain	
		reset_sudoku(sudoku);
		for (ng=r=0; r<size; r++) {
			for (c=0; c<size; c++) {
				if (csudoku.grid[r][c].stat == stat_given) {
					sudoku->grid[r][c].value = csudoku.grid[r][c].value;
					sudoku->grid[r][c].stat = stat_given;
					ng++;
				}
			}
		}

		// for symmetry try to remove quadruples & pairs of gratuituous givens
		if (symmgrid != symm_none) {
			remove_gratuituous_givens_symm(sudoku);
		}
		
		// finally try to remove givens one at a time
		remove_gratuituous_givens(sudoku);
		
	} while ((symmgrid != symm_none) && (check_sudoku_symmetry(*sudoku,&t0,&t1) == symm_none));

	// check symmetry
	if (check_sudoku_symmetry(*sudoku,&t0,&t1) == symm_v) {
		transpose_sudoku(sudoku);
	}
	if (symmgrid != symm_none) {
		beautify_symmetric_sudoku(sudoku);
	}
	
	// determine number of givens
	for (r=ngiv=0; r<size; r++) {
		for (c=0; c<size; c++) {
			if ((sudoku->grid[r][c].stat == stat_given) || (sudoku->grid[r][c].stat == stat_ggiven)) ngiv++;
		}
	}
	
	// archive sudoku
	copy_sudoku(*sudoku,&dsudoku);
	pthread_mutex_lock(&mut_stdout);                   // beginning of mutex protection
	if (verbosity >= 2) {
		ret = solve_sudoku(&dsudoku,elim,NUM_ET-1,2,1); // solve for rating and print solution steps, sudoku, canonized sudoku
		printf("\t%d\t%d\t%s\n",ngiv,(int)dsudoku.rating,VERSION); // print additional info
	} else if (verbosity == 1) {
		ret = solve_sudoku(&dsudoku,elim,NUM_ET-1,1,1); // solve for rating
		fprint_sudoku(stdout,*sudoku);                  // print sudoku
		printf("\t%d\t%d\t%s\n",ngiv,(int)dsudoku.rating,VERSION); // print additional info
	} else {
		ret = solve_sudoku(&dsudoku,elim,NUM_ET-1,0,1); // solve for rating, print nothing
	}
	pthread_mutex_unlock(&mut_stdout);                 // end of mutex protection
	
	sudoku->rating = dsudoku.rating;
	return(sudoku->rating);
}

/* select cells, sudoku is generally not gonna be symmetric
 ******************************************************************************/
int create_unsymm(t_sudoku *sudoku) {
	
	int size, r, c, cc, ng, index, elim[NUM_ET], ret;

	reset_sudoku(sudoku);  // no size before reset.
	size = sudoku->size;

	// loop until distribution of givens gives unique solution
	// uniqueness is ensured by not using forcing chains and guessing
	do {
		
		// start afresh
		reset_sudoku(sudoku);		
		ret = 1;
		ng = 0;

		// elimination loop
		do {

			// pick random free cell, pick candidate
			index = get_random_cell2(*sudoku,stat_free);
			r = index / size;
			c = index % size;
			if (ng < size-1) {
				cc = ng;
			} else {
				cc = get_random_cand(sudoku->grid[r][c]);		
			}
			if (DEBUG) fprintf(stderr,"%d. given: %d at r%dc%d\n",ng,cc+1,r+1,c+1);

			// insert given, get candidates and eliminate
			sudoku->grid[r][c].stat = stat_given;
			sudoku->grid[r][c].value = cc+1;
			ng++;
			get_candidates(sudoku);
			
			// elimination doesn't make sense for 1st size-1 givens, because
			// those givens are all different
			if (ng > size-1) {
				ret = eliminate(sudoku,elim,et_d,0,0);
			}

			// as long as ret > 0 elimination continues with additional given			
		} while(ret > 0);

		// result of elimination() is negative, if sudoku is unsolvable
	}  while(ret < 0);

	// elimination() returns 0, if solution is found. ret is always 0 here
	return(ret);
}

/* select cells for next given in a way that makes sudoku point-symmetric
 ******************************************************************************/
int create_psymm(t_sudoku *sudoku) {
	int size = sudoku->size;
	int r, c;
	int elim[NUM_ET], ret;
	int ng, index;

	reset_sudoku(sudoku);  // no size before reset.
	size = sudoku->size;

	// loop until distribution of givens gives unique solution
	// uniqueness is ensured by not using forcing chains and guessing
	do {

		// start afresh
		reset_sudoku(sudoku);		
		ret = 1;
		ng = 0;

		// elimination loop
		do {

//			ng = get_num_cell_with_stat1(*sudoku,stat_given);
//			if (DEBUG) fprintf(stderr,"%d givens ",ng);

			// find out symmetry status of sudoku
			check_sudoku_symmetry(*sudoku, &r, &c);

			// r and c are both < 0 only, if symmetry cannot be improved. in that case
			// choose random free cell for next given. else r & c will hold location
			// of the 1st symmetry breaking cell */
			if ((r < 0) && (c < 0)) {
				index = get_random_cell2(*sudoku,stat_free);
				r = index / size;
				c = index % size;
				if (DEBUG) fprintf(stderr," symmetry found, choose r%dc%d.\n",r+1,c+1);		
			} else {
				if (DEBUG) fprintf(stderr," 1st symmetry breaking empty cell is r%dc%d.\n",r+1,c+1);		
			}

			// select & insert candidate for 1st cell
			if (sudoku->grid[r][c].value < 1) {
				if (ng < 8) {
					sudoku->grid[r][c].value = ng + 1;
				} else {
					sudoku->grid[r][c].value = 1 + get_random_cand(sudoku->grid[r][c]);
				}
			}

			if (sudoku->grid[r][c].stat != stat_given) {
				ng++;
				sudoku->grid[r][c].stat = stat_given;
			}
			get_candidates(sudoku);
			// do NOT use forcing chains or guess for elimination, because
			// uniqueness is not checked here!!
			if (ng > size-1) {
				ret = eliminate(sudoku,elim,et_d,0,0);
				if (ret < 0) break;
			}

			// select & insert candidate for 2nd cell
			r = size - 1 - r;
			c = size - 1 - c;
			if (sudoku->grid[r][c].value < 1) {
				if (ng < 8) {
					sudoku->grid[r][c].value = ng + 1;
				} else {
					sudoku->grid[r][c].value = 1 + get_random_cand(sudoku->grid[r][c]);
				}
			}
			if (sudoku->grid[r][c].stat != stat_given) {
				ng++;
				sudoku->grid[r][c].stat = stat_given;
			}
			get_candidates(sudoku);
			// do NOT use forcing chains or guess for elimination, because
			// uniqueness is not checked here!!
			if (ng > size-1) {
				ret = eliminate(sudoku,elim,et_d,0,0);
			}

			// as long as ret > 0 elimination continues with additional given			
		} while(ret > 0);

		// result of elimination() is negative, if sudoku is unsolvable
	}  while(ret < 0);

	// elimination() returns 0, if solution is found. ret is always 0 here
	return(ret);
}

/* select cells for next given in a way that makes sudoku axi-symmetric
 ******************************************************************************/
int create_asymm(t_sudoku *sudoku) {
	int size = sudoku->size;
	int gr, gc, r, c, ir, i;
	int elim[NUM_ET], ret;
	int ng, index, max_ncand;

	reset_sudoku(sudoku);  // no size before reset.
	size = sudoku->size;

	// loop until distribution of givens gives unique solution
	// uniqueness is ensured by not using forcing chains and guessing
	do {

		// start afresh
		reset_sudoku(sudoku);		
		ret = 1;
		ng = 0;

		// elimination loop
		do {


			// get number of givens
//			ng = get_num_cell_with_stat1(*sudoku,stat_given);
			max_ncand = -1;

			// select positions for next 4 candidates
			if (ng < 20) {
				// pick random cell not on symmetry axis
				do {
					index = get_random_cell2(*sudoku,stat_free);
					gr = index / size;
					gc = index % size;
				} while ((gr == size / 2) || (gc == size / 2));
			} else if (ng < 28) {
				// in top 4 rows of middle column find cell with maximum number of candidates
				for (ir=0; ir<size/2; ir++) {
					if ((sudoku->grid[ir][size/2].stat != stat_given) && (sudoku->grid[ir][size/2].ncand > max_ncand)) {
						max_ncand = sudoku->grid[ir][size/2].ncand;
						gr = ir;
						gc = size / 2;
					}
				}
			} else if (ng == 28) {
				gr = gc = size / 2;
			}

			if (DEBUG) fprintf(stderr,"ng %d   gr/gc (r%dc%d)\n",ng,gr+1,gc+1);

			for (i=0; i<4; i++) {

				// mirror position for next candidate
				if (ng < 20) {
					// this is for cells not on the symmetry axes
					switch(i) {
					case 0:
						r = gr;
						c = gc;
						break;
					case 1:
						r = gr;
						c = size - 1 - gc;
						break;
					case 2:
						r = size - 1 - gr;
						c = gc;
						break;
					case 3:
						r = size - 1 - gr;
						c = size - 1 - gc;
						break;
					}
				} else if (ng < 28) {
					// this is for cells on the symmetry axes
					switch(i) {
					case 0:
						r = gr;
						c = gc;
						break;
					case 1:
						r = (size-1) - gr;
						c = gc;
						break;
					case 2:
						r = gc;
						c = gr;
						break;
					case 3:
						r = gc;
						c = (size-1) - gr;
						break;
					}
				} else if (ng == 28) {
					r = gr;
					c = gc;			
				}

				// select random candidate
				// -----------------------
				if (sudoku->grid[r][c].value < 1) {
					if (ng < 8) {
						sudoku->grid[r][c].value = ng + 1;
					} else {
						sudoku->grid[r][c].value = 1 + get_random_cand(sudoku->grid[r][c]);
					}
				}
				if (sudoku->grid[r][c].stat != stat_given) {
					ng++;
					sudoku->grid[r][c].stat = stat_given;
				}
				get_candidates(sudoku);
				// do NOT use forcing chains or guess for elimination, because
				// uniqueness is not checked here!!
				if (ng > size-1) {
					ret = eliminate(sudoku,elim,et_l,0,0);
					if ((ng == 29) && (ret != 0)) ret = -1;
					if (ret < 0) break;
				}

			}

			if (DEBUG) fprint_sudoku_as_grid(stderr,*sudoku);


			// as long as ret > 0 elimination continues with additional given			
		} while(ret > 0);
		
		// result of elimination() is negative, if sudoku is unsolvable
	}  while(ret < 0);

	// elimination() returns 0, if solution is found. ret is always 0 here
	return(ret);
}

/* remove gratuitious givens
 ******************************************************************************/
void remove_gratuituous_givens(t_sudoku *sudoku) {
	int r, c, size = sudoku->size;
	int elim[NUM_ET], ret;
	t_sudoku csudoku, dsudoku;

	copy_sudoku(*sudoku,&csudoku);
	
	for (r=0; r<size; r++) {
		for (c=0; c<size; c++) {
			if (sudoku->grid[r][c].stat == stat_given) {
				copy_sudoku(*sudoku,&csudoku);
				// turn status to stat_free
				csudoku.grid[r][c].value = 0;
				csudoku.grid[r][c].stat = stat_free;
				copy_sudoku(csudoku,&dsudoku);
				// solve
				ret = solve_sudoku(&dsudoku,elim,et_d,0,0);
				if (check_uniqueness1(csudoku,dsudoku) == 0) {
					sudoku->grid[r][c].value = 0;
					sudoku->grid[r][c].stat = stat_free;
				}
			}
		}
	}
}

/* remove gratuitious givens in a way that maintains sudoku symmetry
 * and uniqueness of solution
 ******************************************************************************/
void remove_gratuituous_givens_symm(t_sudoku *sudoku) {
	int ng, size = sudoku->size;
	int r, c;
	int t0, t1;
	int elim[NUM_ET], ret;
	extern t_symmetry symmgrid;
	t_symmetry sym;
	t_sudoku csudoku, dsudoku;

	ng = get_num_cell_with_stat1(*sudoku,stat_given);

	if (symmgrid == symm_a) {
		// between symmetry axes
		sym = check_sudoku_symmetry(*sudoku,&t0,&t1);
		if ((sym == symm_v) || (sym == symm_hv)) {
			for (r=0; r<size; r++) {
				for (c=0; c<size/2; c++) {
					if (sudoku->grid[r][c].stat == stat_given) {
						copy_sudoku(*sudoku,&csudoku);
						csudoku.grid[r][c].value = csudoku.grid[r][size-1-c].value = 0;
						csudoku.grid[r][c].stat = csudoku.grid[r][size-1-c].stat = stat_free;
						copy_sudoku(csudoku,&dsudoku);
						ret = solve_sudoku(&dsudoku,elim,et_d,0,0);
						if (check_uniqueness1(csudoku,dsudoku) == 0) {
							sudoku->grid[r][c].value = sudoku->grid[r][size-1-c].value = 0;
							sudoku->grid[r][c].stat = sudoku->grid[r][size-1-c].stat = stat_free;
							ng -= 2;
						}
					}
				}
			}
		}
		sym = check_sudoku_symmetry(*sudoku,&t0,&t1);
		if ((sym == symm_h) || (sym == symm_hv)) {
			for (r=0; r<size/2; r++) {
				for (c=0; c<size; c++) {
					if (sudoku->grid[r][c].stat == stat_given) {
						copy_sudoku(*sudoku,&csudoku);
						csudoku.grid[r][c].value = csudoku.grid[size-1-r][c].value = 0;
						csudoku.grid[r][c].stat = csudoku.grid[size-1-r][c].stat = stat_free;
						copy_sudoku(csudoku,&dsudoku);
						ret = solve_sudoku(&dsudoku,elim,et_d,0,0);
						if (check_uniqueness1(csudoku,dsudoku) == 0) {
							sudoku->grid[r][c].value = sudoku->grid[size-1-r][c].value = 0;
							sudoku->grid[r][c].stat = sudoku->grid[size-1-r][c].stat = stat_free;
							ng -= 2;
						}
					}
				}
			}
		}
		// now try symmetry axes
		if (check_sudoku_symmetry(*sudoku,&t0,&t1) == symm_hv) {
			r = c = size/2;
			if (sudoku->grid[r][c].stat == stat_given) {
				copy_sudoku(*sudoku,&csudoku);
				csudoku.grid[r][c].value = 0;
				csudoku.grid[r][c].stat = stat_free;
				copy_sudoku(csudoku,&dsudoku);
				ret = solve_sudoku(&dsudoku,elim,et_d,0,0);
				if (check_uniqueness1(csudoku,dsudoku) == 0) {
					sudoku->grid[r][c].value = 0;
					sudoku->grid[r][c].stat = stat_free;
				}
			}
		}
		sym = check_sudoku_symmetry(*sudoku,&t0,&t1);
		if ((sym == symm_v) || (sym == symm_hv)) {
			c = size/2;
			for (r=0; r<size; r++) {
				if (sudoku->grid[r][c].stat == stat_given) {
					copy_sudoku(*sudoku,&csudoku);
					csudoku.grid[r][c].value = 0;
					csudoku.grid[r][c].stat = stat_free;
					copy_sudoku(csudoku,&dsudoku);
					ret = solve_sudoku(&dsudoku,elim,et_d,0,0);
					if (check_uniqueness1(csudoku,dsudoku) == 0) {
						sudoku->grid[r][c].value = 0;
						sudoku->grid[r][c].stat = stat_free;
					}
				}
			}
		}
		sym = check_sudoku_symmetry(*sudoku,&t0,&t1);
		if ((sym == symm_h) || (sym == symm_hv)) {
			r = size/2;
			for (c=0; c<size; c++) {
				if (sudoku->grid[r][c].stat == stat_given) {
					copy_sudoku(*sudoku,&csudoku);
					csudoku.grid[r][c].value = 0;
					csudoku.grid[r][c].stat = stat_free;
					copy_sudoku(csudoku,&dsudoku);
					ret = solve_sudoku(&dsudoku,elim,et_d,0,0);
					if (check_uniqueness1(csudoku,dsudoku) == 0) {
						sudoku->grid[r][c].value = 0;
						sudoku->grid[r][c].stat = stat_free;
					}
				}
			}
		}
	} else if ((symmgrid == symm_p) && (check_sudoku_symmetry(*sudoku,&t0,&t1) == symm_p)) {
		for (r=0; r<size/2+1; r++) {
			for (c=0; c<size; c++) {
				if (sudoku->grid[r][c].stat == stat_given) {
					copy_sudoku(*sudoku,&csudoku);
					csudoku.grid[r][c].value = csudoku.grid[size-1-r][size-1-c].value = 0;
					csudoku.grid[r][c].stat = csudoku.grid[size-1-r][size-1-c].stat = stat_free;
					copy_sudoku(csudoku,&dsudoku);
					ret = solve_sudoku(&dsudoku,elim,et_d,0,0);
					if (check_uniqueness1(csudoku,dsudoku) == 0) {
						sudoku->grid[r][c].value = sudoku->grid[size-1-r][size-1-c].value = 0;
						sudoku->grid[r][c].stat = sudoku->grid[size-1-r][size-1-c].stat = stat_free;
					}
				}
			}
		}
	}
}
